﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Text.RegularExpressions;

namespace Nn.Adl.Parsing
{
    internal class InvalidCharacterException : ApplicationException
    {
        public InvalidCharacterException()
            : base("invalid character")
        {
        }
    }

    public interface ITokenReader<T>
        where T : class
    {
        Tuple<T, int> ReadToken(string allText, int position);
        T MakeHereDocument(string text);
    }

    public class DefaultTokenStream<T>
        where T : class
    {
        public class TokenInfo
        {
            public T Value { get; set; }
            public int Position { get; set; }
            public int Length { get; set; }
        }

        private static TokenInfo s_EndOfFile = new TokenInfo { Value = null, Position = -1, Length = 0 };
        private static Regex s_WhiteSpace
            = new Regex(@"^\s+", RegexOptions.Compiled);
        private static Regex s_LineComment
            = new Regex(@"^(//|#).*?(\r\n|\r|\n)", RegexOptions.Compiled);
        private static Regex s_BlockComment
            = new Regex(@"^/\*.*?\*/", RegexOptions.Compiled | RegexOptions.Singleline);
        private static Regex s_HereDocBegin
            = new Regex(@"^<<([a-zA-Z_][a-zA-Z0-9_]*)", RegexOptions.Compiled);

        private ITokenReader<T> m_TokenReader;
        private string m_Text;
        private int m_Position;
        private int m_PreviousPosition;

        private bool m_HasActiveHereDocument;
        private int m_HereDocumentBegin;
        private int m_HereDocumentEnd;

        public string Text { get { return m_Text; } }

        public DefaultTokenStream(string text, ITokenReader<T> tokenReader)
        {
            this.m_TokenReader = tokenReader;
            this.m_Text = text;
            this.m_Position = 0;
            this.m_PreviousPosition = 0;
            this.m_HasActiveHereDocument = false;
        }

        public int Position
        {
            get { return m_PreviousPosition; }
        }

        private TokenInfo EOFToken
        {
            get { return s_EndOfFile; }
        }

        public TokenInfo GetNextToken()
        {
            return ReadNextToken();
        }

        private void Skip(int numSkip)
        {
            for (int i = 0; i < numSkip; ++i)
            {
                GetNextToken();
            }
        }

        private TokenInfo ReadNextToken()
        {
            while (true)
            {
                m_PreviousPosition = m_Position;
                if (!(m_Position < m_Text.Length))
                {
                    return EOFToken;
                }
                int pos = m_Position;
                if (ProcessWhiteSpace(m_Position))
                {
                    continue;
                }
                else
                {
                    var ret = GetTokenValue(m_Position);
                    m_Position += ret.Length;
                    ret.Position = pos;
                    return ret;
                }
            }
        }

        private bool ProcessWhiteSpace(int position)
        {
            var length = m_Text.Length - position;
            Match m;
            // 空白類
            if ((m = s_WhiteSpace.Match(m_Text, position, length)).Success)
            {
                // ヒアドキュメントの開始行であり
                // ヒットした空白の中に改行があったら
                // 現在位置をヒアドキュメントの終わりに移動する
                if (m_HasActiveHereDocument && m.Value.IndexOfLineHead(0) >= 0)
                {
                    m_Position = m_HereDocumentEnd;
                    m_HasActiveHereDocument = false;
                    return true;
                }
                else
                {
                    m_Position += m.Length;
                    return true;
                }
            }
            else if ((m = s_LineComment.Match(m_Text, position, length)).Success)
            {
                // 行コメント
                if (m_HasActiveHereDocument)
                {
                    m_Position = m_HereDocumentEnd;
                    m_HasActiveHereDocument = false;
                    return true;
                }
                else
                {
                    m_Position += m.Length;
                    return true;
                }
            }
            else if ((m = s_BlockComment.Match(m_Text, position, length)).Success)
            {
                // ブロックコメント
                m_Position += m.Length;
                return true;
            }
            else
            {
                return false;
            }
        }

        private TokenInfo GetTokenValue(int position)
        {
            var length = m_Text.Length - position;
            // ヒアドキュメント
            Match m;
            if ((m = s_HereDocBegin.Match(m_Text, position, length)).Success)
            {
                var marker = m.Groups[1].Value;
                var s = ProcessHereDocument(marker);
                var value = m_TokenReader.MakeHereDocument(s);
                return new TokenInfo
                {
                    Value = value,
                    Length = m.Length,
                };
            }
            else
            {
                var p = m_TokenReader.ReadToken(m_Text, position);
                return new TokenInfo
                {
                    Value = p.Item1,
                    Length = p.Item2,
                };
            }
        }

        private string ProcessHereDocument(string marker)
        {
            if (!m_HasActiveHereDocument)
            {
                var lineEnd = m_Text.IndexOfLineHead(m_Position);
                if (lineEnd < 0)
                {
                    throw new InvalidCharacterException();
                }

                m_HereDocumentBegin = lineEnd;
                m_HereDocumentEnd = m_HereDocumentBegin;

                m_HasActiveHereDocument = true;
            }

            int nextEnd = m_HereDocumentEnd;
            var sb = new StringBuilder();
            var re = new Regex("^" + marker + "(\r\n|\r|\n)");

            while (true)
            {
                int lineEnd = m_Text.IndexOfLineHead(nextEnd);
                if (lineEnd < 0)
                {
                    throw new InvalidCharacterException();
                }

                string line = m_Text.Substring(nextEnd, lineEnd - nextEnd);

                nextEnd = lineEnd;

                if (re.Match(line).Success)
                {
                    break;
                }

                sb.Append(line);
            }

            m_HereDocumentEnd = nextEnd;
            return Regex.Replace(sb.ToString(), "(\r\n|\r|\n)$", string.Empty);
        }
    }
}
