﻿using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Net.Sockets;
using System.Text;
using System.Text.RegularExpressions;
using System.Threading;
using System.Threading.Tasks;

namespace HalfAwakeLogMonitor
{
    enum MonitoringResult : int
    {
        Success = 0,

        FailurePatternDetected = 1,
        FailureTimeout         = 2,

        TargetDisconnected     = -1,
    }

    class Program
    {
        static void Main(string[] args)
        {
            var option = ParseCommandLineOptionOrDie(args);

            var result = MonitorLog(option);

            Environment.Exit((int)result);
        }

        private static CommandLineOption ParseCommandLineOptionOrDie(string[] args)
        {
            var option = default(CommandLineOption);
            try
            {
                option = CommandLineOption.Parse(args);

                if (option == null)
                {
                    // ヘルプメッセージを表示したので終了
                    Environment.Exit(0);
                }
            }
            catch (ArgumentException)
            {
                // コマンドラインオプションのパースに失敗したので終了
                Environment.Exit(1);
            }
            return option;
        }

        private static MonitoringResult MonitorLog(CommandLineOption option)
        {
            var task = Task.Run(() =>
            {
                using (var helper = new UartLogReader(option.Target, 10023))
                {
                    while (true)
                    {
                        var line = helper.ReadLine();
                        if (line == null)
                        {
                            return MonitoringResult.TargetDisconnected;
                        }

                        Console.WriteLine(line);

                        if (!string.IsNullOrEmpty(option.PatternSuccessExit) && Regex.IsMatch(line, option.PatternSuccessExit))
                        {
                            return MonitoringResult.Success;
                        }
                        else if (!string.IsNullOrEmpty(option.PatternFailureExit) && Regex.IsMatch(line, option.PatternFailureExit))
                        {
                            return MonitoringResult.FailurePatternDetected;
                        }
                    }
                }
            });

            var successTimeout = option.SuccessTimeout > 0 ? option.SuccessTimeout * 1000 : int.MaxValue;
            var failureTimeout = option.FailureTimeout > 0 ? option.FailureTimeout * 1000 : int.MaxValue;
            var timeout = Math.Min(successTimeout, failureTimeout);
            if (timeout == int.MaxValue)
            {
                timeout = Timeout.Infinite;
            }

            var completed = task.Wait(timeout);

            if (completed)
            {
                return task.Result;
            }
            else
            {
                if (successTimeout < failureTimeout)
                {
                    Console.Error.WriteLine("HalfAwakeLogMonitor success timeout.");
                    return MonitoringResult.Success;
                }
                else
                {
                    Console.Error.WriteLine("HalfAwakeLogMonitor failure timeout.");
                    return MonitoringResult.FailureTimeout;
                }
            }
        }
    }
}
