﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.IO;

namespace Nintendo.ManuHostTools
{
    internal class ProgramOption
    {
        public UInt64 ProgramId { get; private set; }
        public string[] Arguments { get; private set; }
        public int UsbDeviceIndex { get; private set; }
        public int TargetDetectTimeOutSec { get; private set; }
        public bool IsVerbose = true;
        public string UsbDevicePath = String.Empty;
        public bool Shutdown = false;
        public bool Reboot = false;

        private static string GetOptionParameter(string option, string[] args, int index)
        {
            if (index >= args.Length)
            {
                Console.WriteLine("[ERROR] {0} option need argument.", option);
                ShowUsage();
                Environment.Exit(1);
            }
            return args[index];
        }

        public static ProgramOption Parse(string[] args)
        {
            var option = new ProgramOption();
            option.ProgramId = UInt64.MaxValue;
            option.Arguments = new string[] { };
            option.UsbDeviceIndex = 0;
            option.TargetDetectTimeOutSec = 10;

            for (int i = 0; i < args.Length; ++i)
            {
                switch (args[i])
                {
                    case "--args":
                        {
                            i++;
                            option.Arguments = args.Skip(i).Take(args.Length - i).ToArray();
                            i = args.Length;
                        }
                        break;
                    case "--id":
                        {
                            var str = GetOptionParameter(args[i], args, i + 1);
                            option.ProgramId = Convert.ToUInt64(str, 16);
                            ++i;
                        }
                        break;
                    case "--usb_device_index":
                        {
                            var str = GetOptionParameter(args[i], args, i + 1);
                            option.UsbDeviceIndex = Convert.ToInt32(str, 10);
                            ++i;
                        }
                        break;
                    case "--timeout":
                        {
                            var str = GetOptionParameter(args[i], args, i + 1);
                            option.TargetDetectTimeOutSec = Convert.ToInt32(str, 10);
                            ++i;
                        }
                        break;
                    case "--usb_device_path":
                        {
                            option.UsbDevicePath = GetOptionParameter(args[i], args, i + 1);
                            ++i;
                        }
                        break;
                    case "--no_verbose":
                        {
                            option.IsVerbose = false;
                            ++i;
                        }
                        break;
                    case "--shutdown":
                        {
                            option.Shutdown = true;
                            ++i;
                        }
                        break;
                    case "--reboot":
                        {
                            option.Reboot = true;
                            ++i;
                        }
                        break;
                    default:
                        Console.WriteLine("Unknown option : {0}", args[i]);
                        ShowUsage();
                        Environment.Exit(1);
                        break;
                }
            }

            // shutdown と reboot は ProgramID不要
            if (option.ProgramId == UInt64.MaxValue && !option.Shutdown && !option.Reboot)
            {
                Console.WriteLine("--id option is not specified.");
                ShowUsage();
                Environment.Exit(1);
            }

            return option;
        }

        public static void ShowUsage()
        {
            var ProgramAssembly = System.Reflection.Assembly.GetEntryAssembly();
            var ProgramName = ProgramAssembly.ManifestModule.Name;

            Console.WriteLine("[{0} Ver.{1}]", ProgramName, ProgramAssembly.GetName().Version);
            Console.WriteLine("Usage : {0} --id <Program ID> [Option] --args <Program arguments>", ProgramName);
            Console.WriteLine("Option");
            Console.WriteLine("    --usb_device_index : USB Device Index (default : 0)");
            Console.WriteLine("    --usb_device_path  : USB Device Path");
            Console.WriteLine("    --timeout          : USB Device detect timeout seconds (default : 10 seconds)");
        }
    }
}
