﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Text.RegularExpressions;

namespace MakeSvcVeneer
{
    internal class CommandLineOptionParser
    {
        public enum ParameterTypes
        {
            None,
            Single,
        }

        public class Option
        {
            public string LongName { get; private set; }
            public string ShortName { get; private set; }
            public ParameterTypes ParameterType { get; private set; }
            public Action<string[]> Action { get; private set; }

            public Option(string l, string s, ParameterTypes t, Action<string[]> f)
            {
                LongName = l;
                ShortName = s;
                ParameterType = t;
                Action = f;
            }

            public Option(string l, Action<string[]> f)
                : this(l, null, ParameterTypes.None, f) { }

            public Option(string l, string s, Action f)
                : this(l, s, ParameterTypes.None, x => f()) { }

            public Option(string l, string s, Action<string> f)
                : this(l, s, ParameterTypes.Single, x => f(x[0])) { }

            public Option(string l, Action f)
                : this(l, null, ParameterTypes.None, x => f()) { }

            public Option(string l, Action<string> f)
                : this(l, null, ParameterTypes.Single, x => f(x[0])) { }
        }

        private static string[] ExtractParameter(string[] args, int index, int num)
        {
            if (index + num > args.Length)
            {
                throw new ErrorException(
                    string.Format("Command line option parameter required.\noption = '{0}'", args[index - 1]));
            }

            return Util.SubArray(args, index, num);
        }

        public static void Parse(string[] args, List<Option> options)
        {
            var optionMap = new Dictionary<string, Option>();

            options.ForEach(x =>
                {
                    if (x.LongName != null)
                    {
                        optionMap.Add(x.LongName, x);
                    }
                    if (x.ShortName != null)
                    {
                        optionMap.Add(x.ShortName, x);
                    }
                });

            for (int i = 0; i < args.Length; ++i)
            {
                var arg = args[i];

                if (arg[0] != '-')
                {
                }
                else
                {
                    string name = null;
                    Option opt;

                    int start = (arg[1] == '-') ? 2 : 1;
                    name = arg.Substring(start).Replace('-', '_');

                    if (!optionMap.TryGetValue(name, out opt))
                    {
                        throw new ErrorException(
                            string.Format("Unknown command line option\noption = '{0}'", arg));
                    }

                    switch (opt.ParameterType)
                    {
                    case ParameterTypes.None:
                        {
                            opt.Action(null);
                        }
                        break;

                    case ParameterTypes.Single:
                        {
                            var param = ExtractParameter(args, i + 1, 1);
                            i += 1;

                            opt.Action(param);
                        }
                        break;
                    }
                }
            }
        }
    }
}
