﻿using System;
using System.IO;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Runtime.InteropServices;
using System.Security.Cryptography;

namespace MakeNrr
{
    internal class SortHash :IComparer<byte[]>
    {
        public int Compare(byte[] x, byte[] y)
        {
            if (x.Length != y.Length)
            {
                throw new ArgumentException(Properties.Resources.MakeNroError_InvalidHashSize);
            }

            for (int i = 0; i < x.Length; i++)
            {
                int sub = x[i] - y[i];
                if (sub != 0)
                {
                    return sub;
                }
            }

            return 0;
        }
    }

    internal struct Certification
    {
        internal const int Reserved0x10Size = 0x10;
        internal const int PublicKeySize    = 0x100;
        internal const int SignAreaSize     = 0x120;
        internal const int SignSize         = 0x100;

        internal UInt64 ApplicationIdMask;
        internal UInt64 ApplicationIdPattern;

        [MarshalAs(UnmanagedType.ByValArray, SizeConst = Reserved0x10Size)]
        internal byte[] Reserved0x10;

        [MarshalAs(UnmanagedType.ByValArray, SizeConst = PublicKeySize)]
        internal byte[] PublicKey;

        [MarshalAs(UnmanagedType.ByValArray, SizeConst = SignSize)]
        internal byte[] Sign;
    }

    /// <summary>
    /// Nrr ファイルのヘッダ情報
    /// </summary>
    internal struct NrrHeader
    {
        internal const int SignatureSize        = 0x4;
        internal const int Reserved0x4Size      = 0xC;
        internal const int SignSize             = 0x100;
        internal const int Reserved0x33CSize    = 0x4;
        internal const int Reserved0x348Size    = 0x8;

        [MarshalAs(UnmanagedType.ByValArray, SizeConst = SignatureSize)]
        internal byte[] Signature;

        [MarshalAs(UnmanagedType.ByValArray, SizeConst = Reserved0x4Size)]
        internal byte[] Reserved0x4;

        internal Certification Cert;

        [MarshalAs(UnmanagedType.ByValArray, SizeConst = SignSize)]
        internal byte[] Sign;

        internal UInt64 ApplicationId;

        internal UInt32 Size;

        [MarshalAs(UnmanagedType.ByValArray, SizeConst = Reserved0x33CSize)]
        internal byte[] Reserved0x33C;

        internal UInt32 HashOffset;

        internal UInt32 NumHash;

        [MarshalAs(UnmanagedType.ByValArray, SizeConst = Reserved0x348Size)]
        internal byte[] Reserved0x348;

    }

    internal class Nrr
    {
        internal const int RsaKeySize           = 2048;
        internal const int HashSize             = 0x100;
        internal const int SignAreaOffset       = 0x330; // 署名を適用する領域
        internal const int SignOffset           = 0x230; // 署名を保存する領域

        /// <summary>
        /// 生成された NRR ファイルのバイナリデータ
        /// </summary>
        internal byte[] NroFileBinary;

        internal Nrr()
        {
            this.Header = new NrrHeader();
            this.Header.Signature = new byte[NrrHeader.SignatureSize];
            this.Header.Signature[0] = (byte)'N';
            this.Header.Signature[1] = (byte)'R';
            this.Header.Signature[2] = (byte)'R';
            this.Header.Signature[3] = (byte)'0';

            this.Header.Reserved0x4 = new byte[NrrHeader.Reserved0x4Size];

            this.Header.Cert = new Certification();
            this.Header.Cert.Reserved0x10 = new byte[Certification.Reserved0x10Size];
            this.Header.Cert.PublicKey = new byte[Certification.PublicKeySize];
            this.Header.Cert.Sign = new byte[Certification.SignSize];

            this.Header.Sign = new byte[NrrHeader.SignSize];

            this.Header.ApplicationId = 0;

            this.Header.Reserved0x33C = new byte[NrrHeader.Reserved0x33CSize];

            this.Header.Reserved0x348 = new byte[NrrHeader.Reserved0x348Size];

            this.NroHashList = null;
            this.NroFileBinary = null;
        }

        internal void GenerateNrr(string[] nroFiles)
        {
            this.NroHashList = GenerateNroHashList(nroFiles);

            int nrrHeaderSize = Marshal.SizeOf(this.Header);
            int hashListSize = this.NroHashList.Length;
            uint outputBinarySize = AlignUp((uint)nrrHeaderSize + (uint)hashListSize, 0x1000); // 格納するサイズの時点でオフセットを考慮した値にする

            this.Header.HashOffset = (uint)nrrHeaderSize;
            this.Header.NumHash = (uint)nroFiles.Length;
            this.Header.Size = (uint)outputBinarySize;

            byte[] outputBinary = new byte[outputBinarySize];
            GCHandle gch = GCHandle.Alloc(outputBinary, GCHandleType.Pinned);
            Marshal.StructureToPtr(this.Header, gch.AddrOfPinnedObject(), false);
            gch.Free();

            Array.Copy(this.NroHashList, 0, outputBinary, this.Header.HashOffset, hashListSize);

            this.NroFileBinary = outputBinary;
        }

        private byte[] GenerateNroHashList(string[] nroFiles)
        {
            List<byte[]> list = new List<byte[]>();
            foreach (var nro in nroFiles)
            {
                byte[] hash = CalcNroHash(nro);
                list.Add(hash);
            }
            SortHash sortHash = new SortHash();
            list.Sort(sortHash);

            using (MemoryStream ms = new MemoryStream())
            {
                foreach (var hash in list)
                {
                    ms.Write(hash, 0, hash.Length);
                }

                return ms.ToArray();
            }
        }

        private byte[] CalcNroHash(string filename)
        {
            using (FileStream fs = new FileStream(filename, FileMode.Open, FileAccess.Read))
            {
                using (MemoryStream ms = new MemoryStream())
                {
                    fs.CopyTo(ms);
                    byte[] contents = ms.ToArray();
                    SHA256Managed sha256 = new SHA256Managed();
                    return sha256.ComputeHash(contents);
                }
            }
        }

        private uint AlignUp(uint n, uint align)
        {
            return (n + align - 1) & ~(align - 1);
        }

        private NrrHeader Header;
        private byte[] NroHashList;
    }
}
