﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Xml;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

namespace MakeMeta
{
    /// <summary>
    /// Srv アクセス制御データ
    /// </summary>
    public class SacEntry
    {
        private const int IsServerFlag = 7;
        private string _Name;
        public string Name
        {
            get
            {
                return _Name;
            }
            set
            {
                if (value.Length > 8)
                {
                    throw new ArgumentException(string.Format(Properties.Resources.Message_InvalidStringLength, "Entry/Name", 9));
                }

                _Name = value;
            }
        }

        public bool IsServer { get; set; }

        public int BinarySize
        {
            get
            {
                return Name.Length + 1;
            }
        }

        public byte[] ExportBinary()
        {
            byte info = (byte)(Name.Length - 1);
            info |= (byte)((IsServer ? 1 : 0) << IsServerFlag);

            byte[] outputBinary = new byte[BinarySize];
            outputBinary[0] = info;
            Array.Copy(System.Text.Encoding.ASCII.GetBytes(Name), 0, outputBinary, 1, Name.Length);

            return outputBinary;
        }
    }

    /// <summary>
    /// Srv アクセス制御データに関する共通部分を管理するクラス
    /// </summary>
    public class SrvAccessControl
    {
        internal List<SacEntry> Descriptors { get; set; }
        protected int DescriptorSize;

        public bool IsImported { get; protected set; }

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public SrvAccessControl()
        {
            Descriptors = new List<SacEntry>();
            IsImported = false;
        }

        /// <summary>
        /// Srv アクセス制御に関するバイナリデータを出力します。
        /// </summary>
        /// <returns>Srv アクセス制御に関するバイナリデータ</returns>
        public byte[] ExportBinary()
        {
            if (DescriptorSize == 0)
            {
                return null;
            }

            byte[] outputData = new byte[DescriptorSize];
            int dataSize = 0;
            foreach (var data in Descriptors)
            {
                Array.Copy(data.ExportBinary(), 0, outputData, dataSize, data.BinarySize);
                dataSize += data.BinarySize;
            }
            return outputData;
        }
    }


    /// <summary>
    /// Srv アクセス制御データを管理するクラス
    /// </summary>
    public class SacData : SrvAccessControl
    {
        public SacData()
        {
        }

        /// <summary>
        /// SrvAccessControlData の読み込み
        /// <param name="model">SrvAccessControlData</param>
        /// </summary>
        public void ImportSaData(DescModel.SaDataModel model)
        {
            IsImported = true;

            if (model == null)
            {
                return;
            }


            foreach (var entry in model.Entries)
            {
                if (entry.IsServer == null)
                {
                    throw new ArgumentException(string.Format(Properties.Resources.Message_NotFoundElement, "Entry/IsServer"));
                }

                if (entry.Name == null)
                {
                    throw new ArgumentException(string.Format(Properties.Resources.Message_NotFoundElement, "Entry/Name"));
                }

                SacEntry data = new SacEntry();
                data.IsServer = entry.IsServerValue;
                data.Name = entry.Name;

                DescriptorSize += data.BinarySize;

                Descriptors.Add(data);
            }
        }
    }

    /// <summary>
    /// Srv アクセス制御ディスクリプタを管理するクラス
    /// </summary>
    public class SacDescriptor : SrvAccessControl
    {
        public SacDescriptor(DescModel.OutputDescModel model)
        {
            ImportDescFile(model);
        }

        /// <summary>
        /// Desc ファイルの読み込み
        /// <param name="model">読み込む Desc ファイル</param>
        /// </summary>
        public void ImportDescFile(DescModel.OutputDescModel model)
        {
            IsImported = true;

            if (model.SrvAccessControlDescriptor == null)
            {
                return;
            }

            foreach (var entry in model.SrvAccessControlDescriptor.Entries)
            {
                if (entry.IsServer == null)
                {
                    throw new ArgumentException(string.Format(MakeMeta.Properties.Resources.Message_NotFoundElement, "Entry/IsServer"));
                }

                if (entry.Name == null)
                {
                    throw new ArgumentException(string.Format(MakeMeta.Properties.Resources.Message_NotFoundElement, "Entry/Name"));
                }

                SacEntry data = new SacEntry();
                data.IsServer = entry.IsServerValue;
                data.Name = entry.Name;

                DescriptorSize += data.BinarySize;

                Descriptors.Add(data);
            }
        }

        public bool HasWildCard(string name)
        {
            return name[name.Length - 1] == '*';
        }

        public void CheckCapabilities(SacData data)
        {
            if (data == null)
            {
                return;
            }

            foreach (var entry in data.Descriptors)
            {
                bool found = false;
                foreach (var desc in Descriptors)
                {
                    if (entry.IsServer != desc.IsServer)
                    {
                        continue;
                    }

                    if (HasWildCard(desc.Name))
                    {
                        string cmpName = desc.Name.Substring(0, desc.Name.Length - 1);
                        if (entry.Name.IndexOf(cmpName) == 0)
                        {
                            found = true;
                            break;
                        }
                    }
                    else
                    {
                        if (desc.Name == entry.Name)
                        {
                            found = true;
                            break;
                        }
                    }
                }
                if (!found)
                {
                    throw new ArgumentException(
                        string.Format(MakeMeta.Properties.Resources.Message_DefaultNotPermitted, entry.Name, "SrvAccessControlDescriptor"));
                }
            }
        }
    }

    public class SacDataGenerator
    {
        public static SacData Merge(MetaModel.MetaModel meta, DescModel.OutputDescModel desc)
        {
            SacData sac = new SacData();
            if (meta != null && meta.SrvAccessControlData != null)
            {
                sac.ImportSaData(meta.SrvAccessControlData);
                return sac;
            }
            else if (desc != null && desc.Default != null && desc.Default.SrvAccessControlData != null)
            {
                sac.ImportSaData(desc.Default.SrvAccessControlData);
                return sac;
            }
            else
            {
                return null;
            }
        }

    }
}
