﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using Nintendo.Foundation.IO;
using System.IO;
using CommandUtility;
using System.Security.Cryptography;
using YamlDotNet.Serialization;

namespace MakeFirmwareArchive
{
    public class CalculateHashArguments
    {
        [CommandLineOption('o', "output", Description = " ", IsRequired = true)]
        public string OutputPath { get; set; }

        [CommandLineOption('i', "input", Description = " ", IsRequired = true)]
        public string InputPath { get; set; }

        [CommandLineOption("filter", Description = " ", IsRequired = false, DefaultValue = "*")]
        public string Filter { get; set; }
    }

    public class NspHash
    {
        public NspHash(string file, string hash)
        {
            this.Path = file;
            this.Hash = hash;
        }
        public string Path { get; set; }
        public string Hash { get; set; }
    }

    public class CalculateHashCommand
    {
        static string getMd5(FileInfo file)
        {
            var builder = new StringBuilder();
            using (var fs = file.OpenRead())
            {
                using (var md5 = new MD5CryptoServiceProvider())
                {
                    byte[] data = md5.ComputeHash(fs);
                    for (int i = 0; i < data.Length; i++)
                    {
                        builder.Append(data[i].ToString("x2"));
                    }
                }
            }

            return builder.ToString();
        }

        public static void RecursivelySearchDirectory(string sourcePath, string destinationPath, List<NspHash> hashList, string filter)
        {
            foreach (var file in Directory.GetFiles(sourcePath, filter))
            {
                var fileName = Path.Combine(destinationPath, Path.GetFileName(file));
                hashList.Add(new NspHash(fileName, getMd5(new FileInfo(file))));
            }

            foreach (var dir in Directory.GetDirectories(sourcePath))
            {
                RecursivelySearchDirectory(dir, Path.Combine(destinationPath, Path.GetFileName(dir)), hashList, filter);
            }
        }

        public static void MakeYamlList(FileInfo outputPath, List<NspHash> hashList)
        {
            var yamlSerializer = new Serializer();

            using (var streamWriter = new StreamWriter(outputPath.FullName))
            {
                streamWriter.AutoFlush = true;
                var textWriter = TextWriter.Synchronized(streamWriter);

                yamlSerializer.Serialize(textWriter, hashList);
            }
        }

        public static void CalculateHash(FileInfo outputPath, FileInfo inputPath, string filter)
        {
            using (var tempHolder = new TemporaryFileHolder("MakeFirmwareArhive"))
            {
                var extractedNfa = tempHolder.CreateTemporaryDirectory("ExtractedNfa");
                var archiveDirectory = FirmwareArchiveDirectory.FromFirmwareArchive(extractedNfa, inputPath);

                var hashList = new List<NspHash>();
                RecursivelySearchDirectory(extractedNfa.FullName, inputPath.FullName, hashList, filter);

                MakeYamlList(outputPath, hashList);
            }
        }
    }
}
