﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Xml;

namespace MakeDesc
{
    public class KernelCapabilityFlag
    {
        public uint FieldValue { get; set; }
        public uint EntryNumber { get; set; }
        public uint Flag
        {
            get
            {
                uint result = 0;
                uint Sign = (EntryNumber < 32) ? (1U << (int)EntryNumber) - 1 : (uint)0xFFFFFFFF;
                result = Sign;
                result |= FieldValue << (int)(EntryNumber + 1);
                return result;
            }
        }
    }

    // No. 3
    public class ThreadInfoEntry
    {
        private const byte LowestPriorityFieldSize = 6;
        private const byte HighestPriorityFieldSize = 6;
        private const byte MinCoreNumberFieldSize = 8;
        private const byte MaxCoreNumberFieldSize = 8;

        private byte _LowestPriority;
        public byte LowestPriority
        {
            get
            {
                return _LowestPriority;
            }
            set
            {
                if (value >= (1 << LowestPriorityFieldSize))
                {
                    throw new ArgumentException(string.Format(Properties.Resources.Message_ValueShouldBeUnsigned, "LowestPriority", LowestPriorityFieldSize));
                }
                _LowestPriority = value;
            }
        }

        private byte _HighestPriority;
        public byte HighestPriority
        {
            get
            {
                return _HighestPriority;
            }
            set
            {
                if (value >= (1 << HighestPriorityFieldSize))
                {
                    throw new ArgumentException(string.Format(Properties.Resources.Message_ValueShouldBeUnsigned, "HighestPriority", HighestPriorityFieldSize));
                }
                _HighestPriority = value;
            }
        }

        public byte MinCoreNumber { get; set; }
        public byte MaxCoreNumber { get; set; }

        private KernelCapabilityFlag Capability;

        public ThreadInfoEntry()
        {
            Capability = new KernelCapabilityFlag();
            Capability.EntryNumber = 3;
        }

        public uint CalcFlag()
        {
            Capability.FieldValue |= LowestPriority;
            Capability.FieldValue |= (uint)(HighestPriority << LowestPriorityFieldSize);
            Capability.FieldValue |= (uint)(MinCoreNumber << (LowestPriorityFieldSize + HighestPriorityFieldSize));
            Capability.FieldValue |= (uint)(MaxCoreNumber << (LowestPriorityFieldSize + HighestPriorityFieldSize + MinCoreNumberFieldSize));
            return Capability.Flag;
        }
    }

    // No. 4
    public class SystemCallEntry
    {
        private const byte MaxSystemCallNum = 128;
        private List<byte> _SystemCallIds;
        private KernelCapabilityFlag Capability;

        public const int FieldSize = 24;

        public int Index { get; set; }

        public SystemCallEntry()
        {
            Capability = new KernelCapabilityFlag();
            Capability.EntryNumber = 4;
            _SystemCallIds = new List<byte>();
        }

        public byte[] GetSystemCallIds() { return _SystemCallIds.ToArray(); }

        public void AddSystemCallId(byte id)
        {
            if (id >= MaxSystemCallNum)
            {
                throw new ArgumentException(string.Format(Properties.Resources.Message_InvalidSystemCallNumber, id));
            }
            int index = id / FieldSize;
            if (index != this.Index)
            {
                throw new IndexOutOfRangeException(string.Format(Properties.Resources.Message_InvalidSystemCallIndex, this.Index, id, index));
            }
            _SystemCallIds.Add(id);
        }

        public int GetNumIds() { return _SystemCallIds.Count; }

        public uint CalcFlag()
        {
            uint field = 0;
            foreach (byte id in _SystemCallIds)
            {
                field |= (1u << (int)(id % FieldSize));
            }
            Capability.FieldValue = (uint)(((uint)this.Index << FieldSize) | field);
            return Capability.Flag;
        }

    }

    // No. 6
    public class MemoryMapEntry
    {
        public ulong BeginAddress { get; set; }
        public uint Size { get; set; }
        public bool IsIo { get; set; }
        public bool IsRw { get; set; }

        private KernelCapabilityFlag AddrCapability;
        private KernelCapabilityFlag SizeCapability;

        private const int AddrFieldSize = 24;
        private const int SizeFieldSize = 20;
        private const int PageShift = 12;

        public MemoryMapEntry()
        {
            AddrCapability = new KernelCapabilityFlag();
            SizeCapability = new KernelCapabilityFlag();
            AddrCapability.EntryNumber = 6;
            SizeCapability.EntryNumber = 6;
        }

        public uint[] CalcFlag()
        {
            uint[] result = new uint[2];

            AddrCapability.FieldValue = (uint)(BeginAddress >> PageShift);

            // 24bit以上の値をクリアする
            AddrCapability.FieldValue &= (1U << AddrFieldSize) - 1;

            uint highBit = (uint)(IsRw ? 0 : 1);
            AddrCapability.FieldValue |= highBit << AddrFieldSize;
            result[0] = AddrCapability.Flag;

            highBit = (uint)(IsIo ? 0 : 1);
            SizeCapability.FieldValue = (uint)(Size >> PageShift);
            // 20bit以上の値をクリアする
            SizeCapability.FieldValue &= (1U << SizeFieldSize) - 1;
            SizeCapability.FieldValue |= highBit << 24;
            result[1] = SizeCapability.Flag;

            return result;
        }
    }

    // No. 7
    public class IoMemoryMapEntry
    {
        public ulong BeginAddress { get; set; }

        private KernelCapabilityFlag Capability;

        private const int AddrFieldSize = 24;
        private const int PageShift = 12;

        public IoMemoryMapEntry()
        {
            Capability = new KernelCapabilityFlag();
            Capability.EntryNumber = 7;
        }

        public uint CalcFlag()
        {
            uint[] result = new uint[2];

            Capability.FieldValue = (uint)(BeginAddress >> PageShift);

            // 24bit以上の値をクリアする
            Capability.FieldValue &= (1U << AddrFieldSize) - 1;
            return Capability.Flag;
        }
    }

    // No. 11
    public class InterruptEntry
    {
        public ushort this[int index]
        {
            get
            {
                if (index > 2 || index < 0)
                {
                    throw new IndexOutOfRangeException();
                }
                return InterruptNumber[index];
            }
            set
            {
                if (value >= (1 << 10))
                {
                    throw new ArgumentException(string.Format(Properties.Resources.Message_ValueShouldBeUnsigned, "EnableInterrupts", 10));
                }
                InterruptNumber[index] = value;
            }
        }
        private ushort[] InterruptNumber;
        private KernelCapabilityFlag Capability;

        public InterruptEntry()
        {
            Capability = new KernelCapabilityFlag();
            Capability.EntryNumber = 11;
            InterruptNumber = new ushort[2] { 1023, 1023 };
        }

        public uint CalcFlag()
        {
            Capability.FieldValue = (uint)(InterruptNumber[0] | InterruptNumber[1] << 10);
            return Capability.Flag;
        }
    }

    // No. 13
    public class MiscParamsEntry
   {
        public byte ProgramType { get; set; }

        private KernelCapabilityFlag Capability;

        public MiscParamsEntry()
        {
            Capability = new KernelCapabilityFlag();
            Capability.EntryNumber = 13;
        }

        public uint CalcFlag()
        {
            Capability.FieldValue = ProgramType;
            return Capability.Flag;
        }
    }

    // No. 14
    public class KernelVersionEntry
    {
        private const int MajorVersionFieldSize = 13;
        private const int MinorVersionFieldSize = 4;

        private ushort _MajorVersion;
        public ushort MajorVersion
        {
            get
            {
                return _MajorVersion;
            }
            set
            {
                if (value >= (1 << MajorVersionFieldSize))
                {
                    throw new ArgumentException(string.Format(Properties.Resources.Message_ValueShouldBeUnsigned, "KernelVersion/MajorVersion", MajorVersionFieldSize));
                }
                _MajorVersion = value;
            }
        }

        private byte _MinorVersion;
        public byte MinorVersion
        {
            get
            {
                return _MinorVersion;
            }
            set
            {
                if (value >= (1 << MinorVersionFieldSize))
                {
                    throw new ArgumentException(string.Format(Properties.Resources.Message_ValueShouldBeUnsigned, "KernelVersion/MinorVersion", MinorVersionFieldSize));
                }
                _MinorVersion = value;
            }
        }

        private KernelCapabilityFlag Capability;

        public KernelVersionEntry()
        {
            Capability = new KernelCapabilityFlag();
            Capability.EntryNumber = 14;
        }

        public uint CalcFlag()
        {
            Capability.FieldValue = (uint)(MajorVersion << 4) | MinorVersion;
            return Capability.Flag;
        }
    }

    // No. 15
    public class HandleTableSizeEntry
    {
        private const int FieldSize = 10;
        private ushort _HandleTableSize;
        public ushort HandleTableSize
        {
            get
            {
                return _HandleTableSize;
            }
            set
            {
                if (value >= (1 << FieldSize))
                {
                    throw new ArgumentException(string.Format(Properties.Resources.Message_ValueShouldBeUnsigned, "HandleTableSize", FieldSize));
                }
                _HandleTableSize = value;
            }
        }

        private KernelCapabilityFlag Capability;

        public HandleTableSizeEntry()
        {
            Capability = new KernelCapabilityFlag();
            Capability.EntryNumber = 15;
        }

        public uint CalcFlag()
        {
            Capability.FieldValue = HandleTableSize;
            return Capability.Flag;
        }
    }

    // No. 16
    public class MiscFlagsEntry
    {
        public bool EnableDebug { get; set; }
        public bool ForceDebug { get; set; }

        private KernelCapabilityFlag Capability;

        public MiscFlagsEntry()
        {
            Capability = new KernelCapabilityFlag();
            Capability.EntryNumber = 16;
        }

        public uint CalcFlag()
        {
            Capability.FieldValue = (uint)(EnableDebug ? 1 : 0);
            Capability.FieldValue |= (uint)((ForceDebug ? 1 : 0) << 1);
            return Capability.Flag;
        }
    }

    /// <summary>
    /// カーネルケイパビリティに関する共通部分を管理する
    /// </summary>
    public class KernelCapability
    {
        internal ThreadInfoEntry Thread { get; set; }
        internal List<byte> SystemCallList { get; set; }
        internal List<MemoryMapEntry> MapList { get; set; }
        internal List<IoMemoryMapEntry> IoMapList { get; set; }
        internal List<ushort> IntList { get; set; }
        internal MiscParamsEntry MiscParams;
        internal HandleTableSizeEntry HandleTableSize { get; set; }
        internal KernelVersionEntry KernelVersion { get; set; }
        internal MiscFlagsEntry Misc { get; set; }

        protected SystemCallInfo SyscallInfo;

        public bool IsImported { get; private set; }

        /// <summary>
        /// コンストラクタ
        /// </summary>
        /// <param name="parentNodeName">所属する親要素</param>
        public KernelCapability(SystemCallInfo info, string version)
        {
            Thread = null;
            IntList = new List<ushort>();
            SystemCallList = new List<byte>();
            HandleTableSize = null;
            MapList = new List<MemoryMapEntry>();
            IoMapList = new List<IoMemoryMapEntry>();
            MiscParams = null;
            KernelVersion = null;
            Misc = null;
            SyscallInfo = info;
            if (version != null)
            {
                ImportKernelVersion(version);
            }
        }

        /// <summary>
        /// Desc ファイルの読み込み
        /// <param name="model">読み込む Desc ファイル</param>
        /// </summary>
        public virtual void ImportDescFile(DescModel.InputDescModel model)
        {
            throw new NotImplementedException();
        }

        /// <summary>
        /// Desc ファイルの書き込み
        /// <param name="model">書き込む Desc ファイル</param>
        /// </summary>
        public virtual void ExportDescFile(ref DescModel.OutputDescModel model)
        {
            throw new NotImplementedException();
        }

        /// <summary>
        /// カーネルケイパビリティに関するバイナリデータを出力します。
        /// </summary>
        /// <returns>カーネルケイパビリティに関するバイナリデータ</returns>
        public byte[] ExportBinary()
        {
            List<uint> descriptors = new List<uint>();

            MakeDescriptorData(descriptors);

            if (descriptors.Count == 0)
            {
                return null;
            }

            uint[] array = descriptors.ToArray();
            int typeSize = sizeof(uint);
            byte[] resultArray = new byte[descriptors.Count * typeSize];
            int size = 0;

            foreach (uint desc in array)
            {
                Array.Copy(BitConverter.GetBytes(desc), 0, resultArray, size, typeSize);
                size += typeSize;
            }

            return resultArray;
        }

        protected void MakeDescriptorData(List<uint> descriptors)
        {
            // No. 3
            AddThreadInfo(descriptors);

            // No. 4
            AddEnableSystemCalls(descriptors);

            // No. 6
            AddMemoryMap(descriptors);

            // No. 7
            AddIoMemoryMap(descriptors);

            // No. 11
            AddEnableInterrupts(descriptors);

            // No. 13
            AddMiscParams(descriptors);

            // No.14
            AddKernelVersion(descriptors);

            // No. 15
            AddHandleTableSize(descriptors);

            // No.16
            AddMiscFlags(descriptors);
        }

        // No. 3
        protected void ImportThreadInfo(DescModel.KcThreadInfoModel model)
        {
            if (model == null)
            {
                return;
            }

            if (!model.CheckSuccessToRead())
            {
                return;
            }

            Thread = new ThreadInfoEntry();
            Thread.LowestPriority = model.LowestPriorityValue;
            Thread.HighestPriority = model.HighestPriorityValue;
            Thread.MinCoreNumber = model.MinCoreNumberValue;
            Thread.MaxCoreNumber = model.MaxCoreNumberValue;
            if (Thread.LowestPriority < Thread.HighestPriority)
            {
                throw new ArgumentException(string.Format(Properties.Resources.Message_LessThan, "LowestPriority", "HighestPriority"));
            }
            if (Thread.MinCoreNumber > Thread.MaxCoreNumber)
            {
                throw new ArgumentException(string.Format(Properties.Resources.Message_LessThan, "MaxCoreNumber", "MinCoreNumber"));
            }

            IsImported = true;
        }

        protected void AddThreadInfo(List<uint> descriptors)
        {
            if (Thread == null)
            {
                return;
            }

            descriptors.Add(Thread.CalcFlag());
        }

        protected DescModel.KcThreadInfoModel ExportThreadInfo()
        {
            if (Thread == null)
            {
                return null;
            }

            DescModel.KcThreadInfoModel model = new DescModel.KcThreadInfoModel();
            model.LowestPriority = Thread.LowestPriority.ToString();
            model.HighestPriority = Thread.HighestPriority.ToString();
            model.MinCoreNumber = Thread.MinCoreNumber.ToString();
            model.MaxCoreNumber = Thread.MaxCoreNumber.ToString();
            return model;
        }

        // No. 4
        protected void ImportEnableSystemCalls(List<string> syscalls)
        {
            if (syscalls == null || syscalls.Count == 0)
            {
                return;
            }

            if (SyscallInfo == null)
            {
                throw new ArgumentException(Properties.Resources.Message_NotSvcHeaderLoaded, "EnableSystemCalls");
            }

            foreach (var sysCall in syscalls)
            {
                if (!SyscallInfo.HasSystemCall(sysCall))
                {
                    throw new ArgumentException(string.Format(Properties.Resources.Message_NotFoundSvc, sysCall));
                }
                uint svcId = SyscallInfo.GetSystemCallNumber(sysCall);
                if (SystemCallList.Contains((byte)svcId))
                {
                    throw new ArgumentException(string.Format(Properties.Resources.Message_DuplicateDefinition, "EnableSystemCalls", sysCall));
                }
                SystemCallList.Add((byte)svcId);
            }

            // システムコールを並び替え
            SystemCallList = SystemCallList.OrderBy(svcNum => svcNum).ToList();

            IsImported = true;
        }

        protected void AddEnableSystemCalls(List<uint> descriptors)
        {
            if (SystemCallList.Count == 0)
            {
                return;
            }

            SystemCallList.Sort();

            SystemCallEntry entry = new SystemCallEntry();
            entry.Index = 0;
            foreach (var svcId in SystemCallList)
            {
                if (entry.Index != svcId / SystemCallEntry.FieldSize)
                {
                    if (entry.GetNumIds() > 0)
                    {
                        descriptors.Add(entry.CalcFlag());
                    }
                    entry = new SystemCallEntry();
                    entry.Index = svcId / SystemCallEntry.FieldSize;
                }
                entry.AddSystemCallId(svcId);
            }
            if (entry.GetNumIds() > 0)
            {
                descriptors.Add(entry.CalcFlag());
            }
        }

        protected List<DescModel.KcEnableSystemCallsModel> ExportEnableSystemCalls()
        {
            if (SystemCallList.Count == 0)
            {
                return null;
            }

            List<DescModel.KcEnableSystemCallsModel> list = new List<DescModel.KcEnableSystemCallsModel>();
            foreach (var syscallEntry in SystemCallList)
            {
                DescModel.KcEnableSystemCallsModel svcModel = new DescModel.KcEnableSystemCallsModel();
                svcModel.SystemCallId = syscallEntry.ToString();
                svcModel.Name = SyscallInfo.SearchSystemCallName(syscallEntry);
                list.Add(svcModel);
            }

            return list;
        }

        // No.6, 7
        protected void ImportMemoryMap(List<DescModel.KcMemoryMapModel> models)
        {
            if (models == null)
            {
                return;
            }

            foreach (var map in models)
            {
                map.CheckSuccessToRead();

                if ((map.BeginAddressValue & 0xFFF) > 0)
                {
                    throw new ArgumentException(string.Format(Properties.Resources.Message_InvalidAlign, "MemoryMap/BeginAddress"));
                }
                if ((map.BeginAddressValue >= 0x1000000000))
                {
                    throw new ArgumentException(string.Format(Properties.Resources.Message_ValueShouldBeUnsigned, "MemoryMap/BeginAddress", 35));
                }

                // サイズは uint で定義されているので、コンバート時にエラーになるはずなのでチェックしない

                if ((map.SizeValue & 0xFFF) > 0)
                {
                    throw new ArgumentException(string.Format(Properties.Resources.Message_InvalidAlign, "MemoryMap/Size", 0x1000));
                }
                if (map.SizeValue == 0)
                {
                    throw new ArgumentException(string.Format(Properties.Resources.Message_Invalid0, "MemoryMap/Size"));
                }

                if (map.SizeValue == 0x1000 && map.Permission.ToUpper() == DescModel.KcMemoryMapModel.Permission_Rw)
                {
                    IoMemoryMapEntry entry = new IoMemoryMapEntry();
                    entry.BeginAddress = map.BeginAddressValue;
                    IoMapList.Add(entry);
                }
                else
                {
                    MemoryMapEntry entry = new MemoryMapEntry();
                    entry.BeginAddress = map.BeginAddressValue;
                    entry.Size = map.SizeValue;
                    entry.IsRw = map.Permission.ToUpper() == DescModel.KcMemoryMapModel.Permission_Rw;
                    entry.IsIo = map.Type == DescModel.KcMemoryMapModel.Type_Io;
                    MapList.Add(entry);
                }
            }

            // IoMemoryMap の BeginAddress で並び替え
            IoMapList = IoMapList.OrderBy(map => map.BeginAddress)
                                 .ToList();

            // MemoryMap の BeginAddress で並び替え
            MapList = MapList.OrderBy(map => map.BeginAddress)
                             .ToList();

            IsImported = true;
        }

        protected void AddMemoryMap(List<uint> descriptors)
        {
            if (MapList.Count == 0)
            {
                return;
            }

            foreach (var map in MapList)
            {
                descriptors.AddRange(map.CalcFlag());
            }
        }

        protected void AddIoMemoryMap(List<uint> descriptors)
        {
            if (IoMapList.Count == 0)
            {
                return;
            }

            foreach (var map in IoMapList)
            {
                descriptors.Add(map.CalcFlag());
            }
        }

        protected List<DescModel.KcMemoryMapModel> ExportMemoryMap()
        {
            if (MapList.Count == 0)
            {
                return null;
            }

            List<DescModel.KcMemoryMapModel> models = new List<DescModel.KcMemoryMapModel>();

            foreach (var map in MapList)
            {
                DescModel.KcMemoryMapModel mapInfo = new DescModel.KcMemoryMapModel();
                mapInfo.BeginAddress = DescModel.ConvertUtils.ConvertToHexString(map.BeginAddress);
                mapInfo.Size = DescModel.ConvertUtils.ConvertToHexString(map.Size);
                mapInfo.Permission = map.IsRw? DescModel.KcMemoryMapModel.Permission_Rw : DescModel.KcMemoryMapModel.Permission_Ro;
                mapInfo.Type = map.IsIo ? DescModel.KcMemoryMapModel.Type_Io : DescModel.KcMemoryMapModel.Type_Static;

                models.Add(mapInfo);
            }

            return models;
        }

        protected List<DescModel.KcMemoryMapModel> ExportIoMemoryMap()
        {
            if (IoMapList.Count == 0)
            {
                return null;
            }

            List<DescModel.KcMemoryMapModel> models = new List<DescModel.KcMemoryMapModel>();

            foreach (var map in IoMapList)
            {
                DescModel.KcMemoryMapModel mapInfo = new DescModel.KcMemoryMapModel();
                mapInfo.BeginAddress = DescModel.ConvertUtils.ConvertToHexString(map.BeginAddress);
                mapInfo.Size = "0x1000";
                mapInfo.Permission = DescModel.KcMemoryMapModel.Permission_Rw;
                mapInfo.Type = DescModel.KcMemoryMapModel.Type_Io;

                models.Add(mapInfo);
            }
            return models;
        }

        // No. 11
        protected void ImportEnableInterrupts(List<string> interrupts, bool enableEntireInterrupt)
        {
            if (interrupts == null)
            {
                return;
            }

            foreach (var intr in interrupts)
            {
                if (intr == null)
                {
                    throw new ArgumentException(string.Format(Properties.Resources.Message_NotFoundElement, "EnableInterrupts"));
                }
                ushort intNum = (ushort)DescModel.ConvertUtils.ConvertDecimalString(intr, "EnableInterrupts");
                if (intNum == 0)
                {
                    throw new ArgumentException(string.Format(Properties.Resources.Message_Invalid0, "EnableInterrupts"));
                }
                if (intNum >= 1024)
                {
                    throw new ArgumentException(string.Format(Properties.Resources.Message_OutOfRange, "EnableInterrupts", intNum));
                }
                if (intNum == 1023)
                {
                    if (!enableEntireInterrupt)
                    {
                        throw new ArgumentException(string.Format(Properties.Resources.Message_OutOfRange, "EnableInterrupts", intNum));
                    }
                }
                IntList.Add(intNum);
            }

            // 割り込み番号でソート
            IntList = IntList.OrderBy(intNum => intNum).ToList();

            IsImported = true;
        }

        protected void AddEnableInterrupts(List<uint> descriptors)
        {
            if (IntList.Count == 0)
            {
                return;
            }

            int i = 0;
            if (IntList.Count % 2 != 0)
            {
                InterruptEntry entry = new InterruptEntry();
                entry[0] = IntList[0];
                descriptors.Add(entry.CalcFlag());
                i++;
            }

            for (; i < IntList.Count; i += 2)
            {
                InterruptEntry entry = new InterruptEntry();
                entry[0] = IntList[i];
                if (entry[0] == 1023)
                {
                    entry[1] = 1023;
                    descriptors.Add(entry.CalcFlag());
                    entry[0] = 1023;
                }
                entry[1] = IntList[i + 1];
                descriptors.Add(entry.CalcFlag());
                if (entry[1] == 1023)
                {
                    InterruptEntry entire = new InterruptEntry();
                    entire[0] = 1023;
                    entire[1] = 1023;
                    descriptors.Add(entire.CalcFlag());
                }
            }
        }

        protected List<string> ExportEnableInterrupts()
        {
            if (IntList.Count == 0)
            {
                return null;
            }

            List<string> models = new List<string>();

            foreach (var intr in IntList)
            {
                models.Add(intr.ToString());
            }

            return models;
        }

        // No.13
        protected void ImportMiscParams(DescModel.KcMiscParamsModel model)
        {
            if (model == null)
            {
                return;
            }

            if (!model.CheckSuccessToRead())
            {
                return;
            }

            MiscParamsEntry entry = new MiscParamsEntry();

            entry.ProgramType = model.ProgramTypeValue;
            if (entry.ProgramType >= 8)
            {
                throw new ArgumentException(string.Format(Properties.Resources.Message_OutOfRange, "MiscParams/ProgramType", entry.ProgramType));
            }

            MiscParams = entry;
            IsImported = true;
        }

        protected void AddMiscParams(List<uint> descriptors)
        {
            if (MiscParams == null)
            {
                return;
            }
            descriptors.Add(MiscParams.CalcFlag());
        }

        protected DescModel.KcMiscParamsModel ExportMiscParams()
        {
            if (MiscParams == null)
            {
                return null;
            }

            DescModel.KcMiscParamsModel miscParams = new DescModel.KcMiscParamsModel();
            miscParams.ProgramType = MiscParams.ProgramType.ToString();

            return miscParams;
        }


        // No.14
        protected void ImportKernelVersion(string info)
        {
            if (info == null)
            {
                return;
            }

            string[] versions = info.Split('.');
            if (versions.Length != 2)
            {
                throw new ArgumentException(Properties.Resources.Message_InvalidKernelVersionFormat);
            }
            KernelVersionEntry version = new KernelVersionEntry();

            version.MajorVersion = ushort.Parse(versions[0]);
            version.MinorVersion = byte.Parse(versions[1]);

            KernelVersion = version;
            IsImported = true;
        }

        protected void AddKernelVersion(List<uint> descriptors)
        {
            if (KernelVersion == null)
            {
                return;
            }

            descriptors.Add(KernelVersion.CalcFlag());
        }

        protected DescModel.KcKernelVersionModel ExportKernelVersion()
        {
            if (KernelVersion == null)
            {
                return null;
            }

            DescModel.KcKernelVersionModel versionInfo = new DescModel.KcKernelVersionModel();

            versionInfo.MajorVersion = KernelVersion.MajorVersion.ToString();
            versionInfo.MinorVersion = KernelVersion.MinorVersion.ToString();

            return versionInfo;
        }

        // No.15
        protected void ImportHandleTableSize(string handleTableSize)
        {
            if (handleTableSize == null)
            {
                return;
            }

            HandleTableSizeEntry entry = new HandleTableSizeEntry();
            entry.HandleTableSize = (ushort)DescModel.ConvertUtils.ConvertDecimalString(handleTableSize, "KernelCapabilityDescriptor/HandleTableSize");
            if (entry.HandleTableSize >= 1024)
            {
                throw new ArgumentException(string.Format(Properties.Resources.Message_OutOfRange, "HandleTableSize", entry.HandleTableSize));
            }

            HandleTableSize = entry;
            IsImported = true;
        }

        protected void AddHandleTableSize(List<uint> descriptors)
        {
            if (HandleTableSize == null)
            {
                return;
            }
            descriptors.Add(HandleTableSize.CalcFlag());
        }

        protected string ExportHandleTableSize()
        {
            if (HandleTableSize == null)
            {
                return null;
            }

            return HandleTableSize.HandleTableSize.ToString();
        }

        // No.16
        protected void ImportMiscFlags(DescModel.KcMiscFlags model)
        {
            if (model == null)
            {
                return;
            }

            if (!model.CheckSuccessToRead())
            {
                return;
            }

            MiscFlagsEntry flag = new MiscFlagsEntry();

            flag.EnableDebug = model.EnableDebugValue;
            flag.ForceDebug = model.ForceDebugValue;

            Misc = flag;
            IsImported = true;
        }

        protected void AddMiscFlags(List<uint> descriptors)
        {
            if (Misc == null)
            {
                return;
            }

            descriptors.Add(Misc.CalcFlag());
        }

        protected DescModel.KcMiscFlags ExportMiscFlags()
        {
            if (Misc == null)
            {
                return null;
            }

            DescModel.KcMiscFlags flags = new DescModel.KcMiscFlags();
            flags.EnableDebug = Misc.EnableDebug.ToString();
            flags.ForceDebug = Misc.ForceDebug.ToString();
            return flags;
        }
    }

    /// <summary>
    /// カーネルケイパビリティディスクリプタを管理する
    /// </summary>
    public class KcDescriptor : KernelCapability
    {
        public KcDescriptor(SystemCallInfo info, string version)
            : base(info, version)
        {
        }

        /// <summary>
        /// Desc ファイルの読み込み
        /// <param name="model">読み込む Desc ファイル</param>
        /// </summary>
        public override void ImportDescFile(DescModel.InputDescModel model)
        {
            if (model.KernelCapabilityDescriptor == null)
            {
                return;
            }

            // No 3
            ImportThreadInfo(model.KernelCapabilityDescriptor.ThreadInfo);

            // No 4
            ImportEnableSystemCalls(model.KernelCapabilityDescriptor.EnableSystemCalls);

            // No 6, No 7
            ImportMemoryMap(model.KernelCapabilityDescriptor.MemoryMap);

            // No 11
            ImportEnableInterrupts(model.KernelCapabilityDescriptor.EnableInterrupts, true);

            // No 13
            ImportMiscParams(model.KernelCapabilityDescriptor.MiscParams);

            // No 15
            ImportHandleTableSize(model.KernelCapabilityDescriptor.HandleTableSize);

            // No 16
            ImportMiscFlags(model.KernelCapabilityDescriptor.MiscFlags);
        }

        /// <summary>
        /// Desc ファイルの書き込み
        /// <param name="model">書き込む Desc ファイル</param>
        /// </summary>
        public override void ExportDescFile(ref DescModel.OutputDescModel model)
        {
            if (!IsImported)
            {
                return;
            }

            model.KernelCapabilityDescriptor = new DescModel.KcDescriptorModelWithSvcInfo();

            // No 3
            model.KernelCapabilityDescriptor.ThreadInfo = ExportThreadInfo();

            // No 4
            model.KernelCapabilityDescriptor.EnableSystemCalls = ExportEnableSystemCalls();

            // No 6, No 7
            List<DescModel.KcMemoryMapModel> memoryMap = ExportMemoryMap();
            List<DescModel.KcMemoryMapModel> ioMap = ExportIoMemoryMap();
            if (memoryMap != null)
            {
                if (ioMap != null)
                {
                    memoryMap.AddRange(ioMap);
                }
            }
            else
            {
                memoryMap = ioMap;
            }
            model.KernelCapabilityDescriptor.MemoryMap = memoryMap;

            // No 11
            model.KernelCapabilityDescriptor.EnableInterrupts = ExportEnableInterrupts();

            // No 13
            model.KernelCapabilityDescriptor.MiscParams = ExportMiscParams();

            // No 14
            model.KernelCapabilityDescriptor.KernelVersion = ExportKernelVersion();

            // No 15
            model.KernelCapabilityDescriptor.HandleTableSize = ExportHandleTableSize();

            // No 16
            model.KernelCapabilityDescriptor.MiscFlags = ExportMiscFlags();
        }

        public void CheckThreadInfo(ThreadInfoEntry entry)
        {
            if (entry == null)
            {
                return;
            }

            if (Thread == null)
            {
                throw new ArgumentException(string.Format(Properties.Resources.Message_OutOfCapability, "ThreadInfo"));
            }

            if (!(entry.LowestPriority >= Thread.HighestPriority && entry.LowestPriority <= Thread.LowestPriority))
            {
                throw new ArgumentException(string.Format(Properties.Resources.Message_OutOfCapability, "ThreadInfo/LowestPriority"));
            }
            if (!(entry.HighestPriority >= Thread.HighestPriority && entry.HighestPriority <= Thread.LowestPriority))
            {
                throw new ArgumentException(string.Format(Properties.Resources.Message_OutOfCapability, "ThreadInfo/HighestPriority"));
            }
            if (!(entry.MinCoreNumber >= Thread.MinCoreNumber && entry.MinCoreNumber <= Thread.MaxCoreNumber))
            {
                throw new ArgumentException(string.Format(Properties.Resources.Message_OutOfCapability, "ThreadInfo/MinCoreNumber"));
            }
            if (!(entry.MaxCoreNumber >= Thread.MinCoreNumber && entry.MaxCoreNumber <= Thread.MaxCoreNumber))
            {
                throw new ArgumentException(string.Format(Properties.Resources.Message_OutOfCapability, "ThreadInfo/MaxCoreNumber"));
            }
        }

        public void CheckEnableSystemCalls(List<byte> syscalls)
        {
            if (syscalls == null)
            {
                return;
            }

            foreach (var id in syscalls)
            {
                if (SystemCallList.IndexOf(id) < 0)
                {
                    throw new ArgumentException(string.Format(Properties.Resources.Message_OutOfCapabilityWith, "EnableSystemCalls", SyscallInfo.SearchSystemCallName(id)));
                }
            }
        }

        public void CheckMemoryMap(List<MemoryMapEntry> memoryMaps)
        {
            if (memoryMaps == null)
            {
                return;
            }

            const string MemoryMapName = "MemoryMap";
            foreach (var entry in memoryMaps)
            {
                bool found = false;
                foreach (var desc in MapList)
                {
                    if (entry == null)
                    {
                        return;
                    }

                    if ((entry.BeginAddress >= desc.BeginAddress) &&
                        ((entry.BeginAddress + entry.Size - 1) <= (desc.BeginAddress + desc.Size - 1)) &&
                        (entry.IsRw == desc.IsRw) &&
                        (entry.IsIo == desc.IsIo)
                        )
                    {
                        found = true;
                        break;
                    }
                }
                if (!found)
                {
                    throw new ArgumentException(string.Format(
                        Properties.Resources.Message_OutOfCapabilityWithMemoryMap,
                        "KernelCapabilityData", MemoryMapName, entry.BeginAddress, entry.Size,
                        entry.IsRw ? DescModel.KcMemoryMapModel.Permission_Rw : DescModel.KcMemoryMapModel.Permission_Ro,
                        entry.IsIo ? DescModel.KcMemoryMapModel.Type_Io : DescModel.KcMemoryMapModel.Type_Static));
                }
            }
        }

        public void CheckIoMap(List<IoMemoryMapEntry> ioMaps)
        {
            if (ioMaps == null)
            {
                return;
            }

            const string MemoryMapName = "MemoryMap";
            foreach (var entry in ioMaps)
            {
                bool found = false;
                foreach (var desc in IoMapList)
                {
                    if (entry == null)
                    {
                        return;
                    }

                    if (entry.BeginAddress == desc.BeginAddress)
                    {
                        found = true;
                        break;
                    }
                }
                if (!found)
                {
                    throw new ArgumentException(string.Format(
                        Properties.Resources.Message_OutOfCapabilityWithMemoryMap,
                        "KernelCapabilityData", MemoryMapName, entry.BeginAddress, 0x1000, DescModel.KcMemoryMapModel.Permission_Rw, DescModel.KcMemoryMapModel.Type_Io));
                }
            }
        }

        public void CheckEnableInterrupts(List<ushort> interrupts)
        {
            if (interrupts == null)
            {
                return;
            }

            // 全許可
            if (IntList.IndexOf(1023) >= 0)
            {
                return;
            }

            for (int i = 0; i < IntList.Count; i++)
            {
                if (i + 1 < IntList.Count && IntList[i] == 1023 && IntList[i + 1] == 1023)
                {
                    return;
                }
            }

            foreach (var intr in interrupts)
            {
                if (intr == 1023)
                {
                    continue;
                }

                if (IntList.IndexOf(intr) < 0)
                {
                    throw new ArgumentException(string.Format(Properties.Resources.Message_OutOfCapabilityWith, "EnableInterrupts", intr));
                }
            }
        }

        public void CheckMiscParams(MiscParamsEntry entry)
        {
            if (entry == null)
            {
                return;
            }
            if (MiscParams == null)
            {
                throw new ArgumentException(string.Format(Properties.Resources.Message_OutOfCapability, "MiscFlags"));
            }

            if (entry.ProgramType != MiscParams.ProgramType)
            {
                throw new ArgumentException(string.Format(Properties.Resources.Message_OutOfCapability, "MiscFlags/ProgramType"));
            }
        }

        public void CheckKernelVersion(KernelVersionEntry entry)
        {
            if (entry == null)
            {
                return;
            }
            if (KernelVersion == null)
            {
                throw new ArgumentException(string.Format(Properties.Resources.Message_OutOfCapability, "KernelVersion"));
            }

            if (entry.MajorVersion != KernelVersion.MajorVersion ||
                entry.MinorVersion != KernelVersion.MinorVersion)
            {
                throw new ArgumentException(string.Format(Properties.Resources.Message_OutOfCapability, "KernelVersion"));
            }
        }

        public void CheckHandleTableSize(HandleTableSizeEntry entry)
        {
            if (entry == null)
            {
                return;
            }
            if (HandleTableSize == null)
            {
                throw new ArgumentException(string.Format(Properties.Resources.Message_OutOfCapability, "KernelCapabilityData/HandleTableSize"));
            }

            if (entry.HandleTableSize > HandleTableSize.HandleTableSize)
            {
                throw new ArgumentException(string.Format(Properties.Resources.Message_OutOfCapability, "KernelCapabilityData/HandleTableSize"));
            }
        }

        public void CheckMiscFlags(MiscFlagsEntry entry)
        {
            if (entry == null)
            {
                return;
            }
            if (Misc == null)
            {
                throw new ArgumentException(string.Format(Properties.Resources.Message_OutOfCapability, "MiscFlags"));
            }

            if (entry.EnableDebug && !Misc.EnableDebug)
            {
                throw new ArgumentException(string.Format(Properties.Resources.Message_OutOfCapability, "MiscFlags/EnableDebug"));
            }
            if (entry.ForceDebug && !Misc.ForceDebug)
            {
                throw new ArgumentException(string.Format(Properties.Resources.Message_OutOfCapability, "MiscFlags/ForceDebug"));
            }
        }

        public void CheckCapabilities(KcData data)
        {
            if (data == null)
            {
                return;
            }

            // No. 3
            CheckThreadInfo(data.Thread);

            // No.4
            CheckEnableSystemCalls(data.SystemCallList);

            // No.6
            CheckMemoryMap(data.MapList);

            // No.7
            CheckIoMap(data.IoMapList);

            // No.11
            CheckEnableInterrupts(data.IntList);

            // No.13
            CheckMiscParams(data.MiscParams);

            // No.14
            CheckKernelVersion(data.KernelVersion);

            // No.15
            CheckHandleTableSize(data.HandleTableSize);

            // No.16
            CheckMiscFlags(data.Misc);
        }
    }

    /// <summary>
    /// カーネルケイパビリティを管理する
    /// </summary>
    public class KcData : KernelCapability
    {

        public KcData(SystemCallInfo info, string version)
            : base(info, version)
        {
        }

        /// <summary>
        /// Desc ファイルの読み込み
        /// <param name="model">読み込む Desc ファイル</param>
        /// </summary>
        public override void ImportDescFile(DescModel.InputDescModel model)
        {
            if (model.Default == null || model.Default.KernelCapabilityData == null)
            {
                return;
            }

            // No 3
            ImportThreadInfo(model.Default.KernelCapabilityData.ThreadInfo);

            // No 4
            ImportEnableSystemCalls(model.Default.KernelCapabilityData.EnableSystemCalls);

            // No 6, No 7
            ImportMemoryMap(model.Default.KernelCapabilityData.MemoryMap);

            // No 11
            ImportEnableInterrupts(model.Default.KernelCapabilityData.EnableInterrupts, false);

            // No 13
            ImportMiscParams(model.Default.KernelCapabilityData.MiscParams);

            // No 15
            ImportHandleTableSize(model.Default.KernelCapabilityData.HandleTableSize);

            // No 16
            ImportMiscFlags(model.Default.KernelCapabilityData.MiscFlags);
        }

        /// <summary>
        /// Desc ファイルの書き込み
        /// <param name="model">書き込む Desc ファイル</param>
        /// </summary>
        public override void ExportDescFile(ref DescModel.OutputDescModel model)
        {
            if (!IsImported)
            {
                return;
            }

            model.Default.KernelCapabilityData = new DescModel.KcDataModelWithSvcInfo();

            // No 3
            model.Default.KernelCapabilityData.ThreadInfo = ExportThreadInfo();

            // No 4
            model.Default.KernelCapabilityData.EnableSystemCalls = ExportEnableSystemCalls();

            // No 6, No 7
            List<DescModel.KcMemoryMapModel> memoryMap = ExportMemoryMap();
            List<DescModel.KcMemoryMapModel> ioMap = ExportIoMemoryMap();
            if (memoryMap != null)
            {
                if (ioMap != null)
                {
                    memoryMap.AddRange(ioMap);
                }
            }
            else
            {
                memoryMap = ioMap;
            }
            model.Default.KernelCapabilityData.MemoryMap = memoryMap;

            // No 11
            model.Default.KernelCapabilityData.EnableInterrupts = ExportEnableInterrupts();

            // No 13
            model.Default.KernelCapabilityData.MiscParams = ExportMiscParams();

            // No 14
            model.Default.KernelCapabilityData.KernelVersion = ExportKernelVersion();

            // No 15
            model.Default.KernelCapabilityData.HandleTableSize = ExportHandleTableSize();

            // No 16
            model.Default.KernelCapabilityData.MiscFlags = ExportMiscFlags();
        }
    }

}
