﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.IO;
using System.Linq;
using System.Net;
using System.Net.Sockets;
using System.Text;
using System.Threading;
using System.Threading.Tasks;

namespace HtcDaemon.Usb
{
    internal class UsbTargetDetector : ITargetDetector
    {
        public event EventHandler<TargetDetectedEventArgs> Detected;

        private bool disposed;
        private Thread probeThread;
        private CancellationTokenSource cancellationTokenSource = new CancellationTokenSource();
        private UsbProber prober;

        public UsbTargetDetector()
        {
            probeThread = new Thread(ProbeThread) { Name = "DirectSocketTargetDetector" };
            prober = new UsbProber(HtcConstants.UsbDeviceGuid);
        }

        ~UsbTargetDetector()
        {
            this.Dispose(false);
        }

        public void Dispose()
        {
            this.Dispose(true);
            GC.SuppressFinalize(this);
        }

        protected virtual void Dispose(bool disposing)
        {
            if (!this.disposed)
            {
                if (disposing)
                {
                    // ここでマネージドリソースをDisposeする
                    cancellationTokenSource.Cancel();
                    probeThread.Join();
                }

                // ここでアンマネージドリソースを解放する
                this.disposed = true;
            }

            //// 基底クラスのDisposeを呼び出す
            // base.Dispose(disposing);
        }

        public void Start()
        {
            probeThread.Start();
        }

        public void Stop()
        {
            Dispose();
        }

        private void ProbeThread()
        {
            var token = cancellationTokenSource.Token;
            while (!token.IsCancellationRequested)
            {
                Probe();
            }
        }

        private void Probe()
        {
            var token = cancellationTokenSource.Token;

            string devicePath = string.Empty;
            while (!token.IsCancellationRequested)
            {
                if (prober.Probe(out devicePath))
                {
                    break;
                }
                Thread.Sleep(1000);
            }
            if (token.IsCancellationRequested)
            {
                return;
            }

            try
            {
                var usbDevice = new UsbDevice(devicePath);
                var connection = new UsbHostTargetConnectionBuilder().Build(usbDevice);
                var disconnectedEvent = new AutoResetEvent(false);
                connection.Disconnected +=
                    (sender, e) => disconnectedEvent.Set();

                RaiseDetected(connection);

                // 切断か Stop を待つ
                var waitHandles = new WaitHandle[]
                {
                    disconnectedEvent,
                    token.WaitHandle
                };
                WaitHandle.WaitAny(waitHandles);
            }
            catch (OperationCanceledException)
            {
                Console.WriteLine("USB connection canceled");
                // USBの接続が切断された後、デバイスとしては存在しているがあらゆる操作が失敗する状態が少し続くようである。
                // ビジーループでポーリングしてしまわないよう、少し待つ。
                Thread.Sleep(1000);
            }
        }

        private void RaiseDetected(HostTargetConnection connection)
        {
            if (Detected != null)
            {
                Detected(this, new TargetDetectedEventArgs(connection));
            }
        }
    }
}
