﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

namespace Nintendo.FsFileCacheSimulator.Simulation.Algorithm
{
    internal abstract class LeastRecentlyUsedCache<TKey> : ICacheSystem
    {
        private class LeastRecentlyUsedCacheEntry
        {
            public bool ReadAhead { get; }
            public bool CacheUsed { get; set; }
            public LeastRecentlyUsedCacheEntry(bool readAhead)
            {
                ReadAhead = readAhead;
                CacheUsed = false;
            }
        }

        private ulong m_CacheSize;
        private uint m_PageSize;
        private LruList<TKey, LeastRecentlyUsedCacheEntry> m_LruList;

        private bool m_ReadAheadEnabled;
        private ulong m_ReadAheadSize;

        public LeastRecentlyUsedCache(ulong cacheSize, uint pageSize, bool enableReadAhead)
        {
            m_CacheSize = cacheSize;
            m_PageSize = pageSize;
            try
            {
                m_LruList = new LruList<TKey, LeastRecentlyUsedCacheEntry>(Convert.ToInt32(cacheSize / pageSize));
            }
            catch (OverflowException ex)
            {
                throw new ArgumentException("ページ数が int 型で保持できる最大値を超えました。キャッシュサイズが大きすぎるか、またはページサイズが小さすぎます", ex);
            }
            m_ReadAheadEnabled = enableReadAhead;
        }

        public void ClearCache()
        {
            m_LruList.Clear();
        }

        public void ReadFile(string path, ulong offset, ulong size, SimulatedFileSystem fileSystem, CacheStatistics statistics)
        {
            var readAheadSize = 0UL;
            var readAheadAdjusted = false;
            var maxReadAheadSize = 64 * 1024UL;

            var absoluteOffset = CalculateAbsoluteOffset(path, offset);

            var toCacheRegion = FileRegion.Zero;
            var toCopyRegion = FileRegion.Zero;
            var originalRequest = new FileRegion(absoluteOffset, size);

            for (var position = BitUtility.AlignDown(absoluteOffset, m_PageSize); position < (absoluteOffset + size + readAheadSize); position += m_PageSize)
            {
                var key = CreateKey(path, position);
                var node = m_LruList.Find(key);
                statistics.CacheSearchCount++;

                if (node != null)
                {
                    if (toCopyRegion == FileRegion.Zero)
                    {
                        toCopyRegion = new FileRegion(position, m_PageSize);
                    }
                    else
                    {
                        if (toCopyRegion.EndOffset != position)
                        {
                            throw new ImplementationErrorException();
                        }
                        toCopyRegion = new FileRegion(toCopyRegion.Offset, toCopyRegion.Size + m_PageSize);
                    }
                }
                else
                {
                    if (m_ReadAheadEnabled && !readAheadAdjusted)
                    {
                        // シーケンシャルリードが発生している時、またはファイルの先頭から読み取りを開始する時に先読みをする
                        var sequentialReadingSeemsOccuring = EqualsLastReadParameters(path, absoluteOffset);
                        var readingStartsFromFileHead = offset == 0;

                        if (sequentialReadingSeemsOccuring || readingStartsFromFileHead)
                        {
                            if (m_ReadAheadSize == 0 || readingStartsFromFileHead)
                            {
                                // 最初は適当にページサイズ分
                                m_ReadAheadSize = m_PageSize;
                            }
                            else
                            {
                                // 徐々に増やす
                                m_ReadAheadSize *= 2;
                                if (m_ReadAheadSize > maxReadAheadSize)
                                {
                                    m_ReadAheadSize = BitUtility.AlignUp(maxReadAheadSize, m_PageSize);
                                }
                            }

                            // 先読みサイズより要求サイズの方が大きい場合は、そちらを先読みサイズとして採用
                            readAheadSize = BitUtility.AlignUp(Math.Max(m_ReadAheadSize, Math.Min(size, maxReadAheadSize)), m_PageSize);

                            // ファイルの終端を超えて先読みを行わない
                            if (TryGetAbsoluteEndOffset(path, out ulong endOffset))
                            {
                                if (absoluteOffset + size + readAheadSize > endOffset)
                                {
                                    readAheadSize = BitUtility.AlignUp(endOffset - absoluteOffset - size, m_PageSize);
                                }
                            }
                        }
                        else
                        {
                            m_ReadAheadSize = 0;
                            readAheadSize = 0;
                        }

                        readAheadAdjusted = true;
                        SaveLastReadParameters(path, absoluteOffset + size);
                    }

                    if (toCacheRegion == FileRegion.Zero)
                    {
                        if (toCopyRegion != FileRegion.Zero)
                        {
                            DoCopy(path, toCopyRegion, originalRequest, statistics);
                            toCopyRegion = FileRegion.Zero;
                        }
                        toCacheRegion = new FileRegion(position, m_PageSize);
                    }
                    else
                    {
                        if (toCopyRegion == FileRegion.Zero)
                        {
                            if (toCacheRegion.EndOffset != position)
                            {
                                throw new ImplementationErrorException();
                            }
                            toCacheRegion = new FileRegion(toCacheRegion.Offset, toCacheRegion.Size + m_PageSize);
                        }
                        else
                        {
                            var toCacheRegion2 = new FileRegion(position, m_PageSize);
                            if (!(toCacheRegion.EndOffset == toCopyRegion.Offset && toCopyRegion.EndOffset == toCacheRegion2.Offset))
                            {
                                throw new ImplementationErrorException();
                            }

                            if (ShouldPerformCoalescedRead(toCacheRegion, toCacheRegion2))
                            {
                                toCacheRegion = FileRegion.GetInclusion(toCacheRegion, toCacheRegion2);
                                toCopyRegion = FileRegion.Zero;
                            }
                            else
                            {
                                DoCopy(path, toCopyRegion, originalRequest, statistics);
                                toCopyRegion = FileRegion.Zero;

                                DoCache(path, toCacheRegion, originalRequest, fileSystem, statistics);
                                toCacheRegion = toCacheRegion2;
                            }
                        }
                    }
                }
            }

            if (toCopyRegion != FileRegion.Zero)
            {
                DoCopy(path, toCopyRegion, originalRequest, statistics);
            }
            if (toCacheRegion != FileRegion.Zero)
            {
                DoCache(path, toCacheRegion, originalRequest, fileSystem, statistics);
            }
        }

        private void DoCopy(string path, FileRegion copyRegion, FileRegion originalRequest, CacheStatistics statistics)
        {
            for (var position = copyRegion.Offset; position < copyRegion.EndOffset; position += m_PageSize)
            {
                var key = CreateKey(path, position);
                var node = m_LruList.Find(key);
                AddReadPageSearchTime(statistics);

                AddReadCacheCopyTime(m_PageSize, statistics);

                m_LruList.Promote(node);
                AddReadPagePromotionTime(statistics);

                statistics.CacheHitCount++;
                if (node.Value.Item2.ReadAhead && !node.Value.Item2.CacheUsed)
                {
                    statistics.ReadAheadHitCount++;
                }
                node.Value.Item2.CacheUsed = true;
            }
        }

        private void DoCache(string path, FileRegion cacheRegion, FileRegion originalRequest, SimulatedFileSystem fileSystem, CacheStatistics statistics)
        {
            // 要求は 2 つ
            //   * cacheRegion で示された領域をキャッシュする
            //   * cacheRegion と originalRequest の重なっている領域をユーザバッファにコピーする

            var requestedRegion = FileRegion.GetIntersection(cacheRegion, originalRequest);
            if (requestedRegion == FileRegion.Zero)
            {
                // 重なっていないなら、キャッシュだけすれば良い
                // CacheSize を超えないようにできるだけキャッシュする
                var actuallyCachedRegion = cacheRegion.GetEndRegionWithSizeLimit(m_CacheSize);
                fileSystem.ReadFile(path, CalculateRelativeOffset(actuallyCachedRegion.Offset), actuallyCachedRegion.Size);
                CommitPages(path, actuallyCachedRegion, originalRequest, statistics);
                return;
            }

            // 重なっている場合は、IPC が 2 回にならないように最善を尽す
            if (requestedRegion.Includes(cacheRegion))
            {
                // ユーザバッファにファイルを読み込んでから、キャッシュにコピーする流れ
                // ただし CacheSize を超えてはいけない
                fileSystem.ReadFile(path, CalculateRelativeOffset(requestedRegion.Offset), requestedRegion.Size);
                var actuallyCachedRegion = cacheRegion.GetEndRegionWithSizeLimit(m_CacheSize);
                CommitPages(path, actuallyCachedRegion, originalRequest, statistics);
                AddReadCacheCopyTime(actuallyCachedRegion.Size, statistics);
            }
            else if (cacheRegion.Includes(requestedRegion))
            {
                // 基本的にはキャッシュにファイルを読み込んでからユーザバッファにコピーすれば良いが
                // すべてをキャッシュし切れない場合は、きちんとユーザバッファを含むようにしながらキャッシュしないといけない

                var actuallyCachedRegion = cacheRegion;
                if (actuallyCachedRegion.Size > m_CacheSize)
                {
                    // とりあえず末尾を再選択して、userRequestedRegion を含むか確認する
                    actuallyCachedRegion = cacheRegion.GetEndRegionWithSizeLimit(m_CacheSize);
                    if (!actuallyCachedRegion.Includes(requestedRegion))
                    {
                        // userRequestedRegion を含む CacheSize 分の領域を再選択
                        actuallyCachedRegion = new FileRegion(requestedRegion.Offset, m_CacheSize);
                    }
                }

                if (actuallyCachedRegion.Includes(requestedRegion))
                {
                    fileSystem.ReadFile(path, CalculateRelativeOffset(actuallyCachedRegion.Offset), actuallyCachedRegion.Size);
                    CommitPages(path, actuallyCachedRegion, originalRequest, statistics);
                    AddReadCacheCopyTime(requestedRegion.Size, statistics);
                }
                else
                {
                    // 調整してもダメなら、多分 requestedRegion が大きすぎてうまくキャッシュし切れない
                    if (requestedRegion.ExpandAndAlign(m_PageSize).Size <= m_CacheSize)
                    {
                        throw new ImplementationErrorException();
                    }
                    // ユーザバッファに読み込んでからキャッシュにコピーする方針に切り替える
                    fileSystem.ReadFile(path, CalculateRelativeOffset(requestedRegion.Offset), requestedRegion.Size);
                    actuallyCachedRegion = requestedRegion.ShrinkAndAlign(m_PageSize).GetEndRegionWithSizeLimit(m_CacheSize);
                    CommitPages(path, actuallyCachedRegion, originalRequest, statistics);
                    AddReadCacheCopyTime(actuallyCachedRegion.Size, statistics);
                }
            }
            else
            {
                // 中途半端に重なっていて最も起きてほしくないパターン
                // 仕方ないのでユーザバッファにファイルを読み込んだ後、cacheRegion と重なっているところだけキャッシュする
                fileSystem.ReadFile(path, CalculateRelativeOffset(requestedRegion.Offset), requestedRegion.Size);
                var actuallyCachedRegion = requestedRegion.ShrinkAndAlign(m_PageSize).GetEndRegionWithSizeLimit(m_CacheSize);
                CommitPages(path, actuallyCachedRegion, originalRequest, statistics);
                AddReadCacheCopyTime(actuallyCachedRegion.Size, statistics);
            }
        }

        private void CommitPages(string path, FileRegion cacheRegion, FileRegion originalRequest, CacheStatistics statistics)
        {
            for (var position = cacheRegion.Offset; position < cacheRegion.EndOffset; position += m_PageSize)
            {
                var key = CreateKey(path, position);

                if (m_LruList.ContainsKey(key))
                {
                    m_LruList.Remove(key);
                    AddReadCacheCopyTime(m_PageSize, statistics);
                }
                AddReadPageSearchTime(statistics);

                if (m_LruList.CountFree() == 0)
                {
                    m_LruList.Remove();
                    AddReadCacheCopyTime(m_PageSize, statistics);
                }

                var readingAhead = !originalRequest.Intersects(new FileRegion(position, m_PageSize));
                if (readingAhead)
                {
                    statistics.ReadAheadCount++;
                }
                m_LruList.Add(key, new LeastRecentlyUsedCacheEntry(readingAhead));
            }
        }

        public void WriteFile(string path, ulong offset, ulong size, SimulatedFileSystem fileSystem, CacheStatistics statistics)
        {
            // ライトスルーキャッシュ
            fileSystem.WriteFile(path, offset, size);

            // 書き込み開始位置と終了位置がページ境界と一致しない場合、端の領域はキャッシュしない
            // Read してキャッシュしても良いが全体のファイルアクセス時間が却って増大してしまう場合が多い
            var absoluteOffset = CalculateAbsoluteOffset(path, offset);
            var cacheRegion = (new FileRegion(absoluteOffset, size)).ShrinkAndAlign(m_PageSize).GetEndRegionWithSizeLimit(m_CacheSize);
            AddWriteCacheCopyTime(cacheRegion.Size, statistics);

            for (var position = cacheRegion.Offset; position < cacheRegion.EndOffset; position += m_PageSize)
            {
                var key = CreateKey(path, position);
                var node = m_LruList.Find(key);
                AddWritePageSearchTime(statistics);

                if (node != null)
                {
                    m_LruList.Promote(node);
                    AddWritePagePromotionTime(statistics);
                }
                else
                {
                    if (m_LruList.CountFree() == 0)
                    {
                        m_LruList.Remove();
                    }
                    m_LruList.Add(key, new LeastRecentlyUsedCacheEntry(false));
                }
            }
        }

        private bool ShouldPerformCoalescedRead(FileRegion region1, FileRegion region2)
        {
            ulong cacheMissRegionSizeSum = region1.Size + region2.Size;
            ulong coalescedRegionSize = FileRegion.GetInclusion(region1, region2).Size;
            if (coalescedRegionSize < cacheMissRegionSizeSum)
            {
                throw new ImplementationErrorException();
            }
            ulong cacheHitRegionSize = coalescedRegionSize - cacheMissRegionSizeSum;

            if (cacheMissRegionSizeSum < 16 * 1024)
            {
                return cacheHitRegionSize < (cacheMissRegionSizeSum * 2 / 3);
            }
            else if (cacheMissRegionSizeSum < 64 * 1024)
            {
                return cacheHitRegionSize < (cacheMissRegionSizeSum * 2 / 4);
            }
            else if (cacheMissRegionSizeSum < 128 * 1024)
            {
                return cacheHitRegionSize < (cacheMissRegionSizeSum * 2 / 5);
            }
            else
            {
                return cacheHitRegionSize < (cacheMissRegionSizeSum / 10);
            }
        }

        private void AddReadCacheCopyTime(ulong size, CacheStatistics statistics)
        {
            statistics.ReadCacheOverhead += Utility.GetMemoryCopyTime(size);
        }

        private void AddWriteCacheCopyTime(ulong size, CacheStatistics statistics)
        {
            statistics.WriteCacheOverhead += Utility.GetMemoryCopyTime(size);
        }

        private void AddReadPageSearchTime(CacheStatistics statistics)
        {
            statistics.ReadCacheOverhead += TimeSpan.FromTicks(1);
        }

        private void AddWritePageSearchTime(CacheStatistics statistics)
        {
            statistics.WriteCacheOverhead += TimeSpan.FromTicks(1);
        }

        private void AddReadPagePromotionTime(CacheStatistics statistics)
        {
            statistics.ReadCacheOverhead += TimeSpan.FromTicks(1);
        }

        private void AddWritePagePromotionTime(CacheStatistics statistics)
        {
            statistics.WriteCacheOverhead += TimeSpan.FromTicks(1);
        }

        protected abstract ulong CalculateAbsoluteOffset(string path, ulong offset);
        protected abstract ulong CalculateRelativeOffset(ulong absoluteOffset);
        protected abstract bool TryGetAbsoluteEndOffset(string path, out ulong endOffset);
        protected abstract TKey CreateKey(string path, ulong absoluteOffset);
        protected abstract void SaveLastReadParameters(string path, ulong absoluteEndOffset);
        protected abstract bool EqualsLastReadParameters(string path, ulong absoluteEndOffset);
    }

    internal class FileOffsetBasedLeastRecentlyUsedCacheKey : IEquatable<FileOffsetBasedLeastRecentlyUsedCacheKey>
    {
        public string Path { get; }
        public ulong Offset { get; }
        public FileOffsetBasedLeastRecentlyUsedCacheKey(string path, ulong offset)
        {
            Path = path;
            Offset = offset;
        }
        public bool Equals(FileOffsetBasedLeastRecentlyUsedCacheKey other)
        {
            return Path == other.Path && Offset == other.Offset;
        }
        public override bool Equals(object obj)
        {
            if (obj is FileOffsetBasedLeastRecentlyUsedCacheKey)
            {
                return Equals((FileOffsetBasedLeastRecentlyUsedCacheKey)obj);
            }
            return false;
        }
        public override int GetHashCode()
        {
            return Path.GetHashCode() ^ Offset.GetHashCode();
        }
        public override string ToString()
        {
            return $"path: {Path}, offset: {Offset}";
        }
        public static bool operator==(FileOffsetBasedLeastRecentlyUsedCacheKey lhs, FileOffsetBasedLeastRecentlyUsedCacheKey rhs)
        {
            return lhs.Equals(rhs);
        }
        public static bool operator!=(FileOffsetBasedLeastRecentlyUsedCacheKey lhs, FileOffsetBasedLeastRecentlyUsedCacheKey rhs)
        {
            return !lhs.Equals(rhs);
        }
    }

    internal class FileOffsetBasedLeastRecentlyUsedCache : LeastRecentlyUsedCache<FileOffsetBasedLeastRecentlyUsedCacheKey>
    {
        private string m_LastReadPath;
        private ulong m_LastReadEndOffset;

        public FileOffsetBasedLeastRecentlyUsedCache(ulong cacheSize, uint pageSize, bool enableReadAhead)
            : base(cacheSize, pageSize, enableReadAhead)
        {
        }

        protected override ulong CalculateAbsoluteOffset(string path, ulong offset)
        {
            return offset;
        }

        protected override ulong CalculateRelativeOffset(ulong absoluteOffset)
        {
            return absoluteOffset;
        }

        protected override bool TryGetAbsoluteEndOffset(string path, out ulong endOffset)
        {
            endOffset = 0;
            return false;
        }

        protected override FileOffsetBasedLeastRecentlyUsedCacheKey CreateKey(string path, ulong absoluteOffset)
        {
            return new FileOffsetBasedLeastRecentlyUsedCacheKey(path, absoluteOffset);
        }

        protected override void SaveLastReadParameters(string path, ulong absoluteEndOffset)
        {
            m_LastReadPath = path;
            m_LastReadEndOffset = absoluteEndOffset;
        }

        protected override bool EqualsLastReadParameters(string path, ulong absoluteEndOffset)
        {
            return path == m_LastReadPath && absoluteEndOffset == m_LastReadEndOffset;
        }
    }

    internal class StorageOffsetBasedLeastRecentlyUsedCache : LeastRecentlyUsedCache<ulong>
    {
        private IReadOnlyDictionary<string, FileRegion> m_OffsetTable;
        private SortedList<ulong, string> m_ReverseOffsetTable;

        private ulong m_LastReadEndOffset;

        public StorageOffsetBasedLeastRecentlyUsedCache(ulong cacheSize, uint pageSize, IReadOnlyDictionary<string, FileRegion> offsetTable, bool enableReadAhead)
            : base(cacheSize, pageSize, enableReadAhead)
        {
            m_OffsetTable = offsetTable;

            m_ReverseOffsetTable = new SortedList<ulong, string>();
            foreach (var kv in offsetTable)
            {
                // サイズ 0 のファイルはオフセットが同一になる
                //   ---> 異なるファイルが同じオフセットに配置されることがあり得る (SortedList.Add メソッドは使用不可)
                //
                // 代入にすると、シミュレーションされた FsAccessLog において ReadFile 関数の結果が正しくない結果になる場合があるが、
                // サイズ 0 なのでアクセス頻度分析には影響しないはずなので不問にする
                m_ReverseOffsetTable[kv.Value.Offset] = kv.Key;
            }
        }

        protected override ulong CalculateAbsoluteOffset(string path, ulong offset)
        {
            return m_OffsetTable[FileSystem.Utility.GetFileSystemName(path)].Offset + offset;
        }

        protected override ulong CalculateRelativeOffset(ulong absoluteOffset)
        {
            // 二分探索で path を見つける
            var startIndex = 0;
            var endIndex = m_ReverseOffsetTable.Keys.Count;
            while (endIndex - startIndex > 1)
            {
                var index = startIndex + (endIndex - startIndex) / 2;
                if (m_ReverseOffsetTable.Keys[index] < absoluteOffset)
                {
                    startIndex = index;
                }
                else
                {
                    endIndex = index;
                }
            }
            var path = m_ReverseOffsetTable[m_ReverseOffsetTable.Keys[startIndex]];
            var baseOffset = m_OffsetTable[FileSystem.Utility.GetFileSystemName(path)].Offset;

            if (absoluteOffset < baseOffset)
            {
                // ROM FS の先頭にあるファイルがページ境界に沿っていない場合はここに来る
                // どうしようもないので 0 を返す...
                return 0;
            }
            return absoluteOffset - baseOffset;
        }

        protected override bool TryGetAbsoluteEndOffset(string path, out ulong endOffset)
        {
            endOffset = m_OffsetTable[FileSystem.Utility.GetFileSystemName(path)].EndOffset;
            return true;
        }

        protected override ulong CreateKey(string path, ulong absoluteOffset)
        {
            return absoluteOffset;
        }

        protected override void SaveLastReadParameters(string path, ulong absoluteEndOffset)
        {
            m_LastReadEndOffset = absoluteEndOffset;
        }

        protected override bool EqualsLastReadParameters(string path, ulong absoluteEndOffset)
        {
            return absoluteEndOffset == m_LastReadEndOffset;
        }
    }
}
