﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using Nintendo.FsFileCacheSimulator.FileSystem;

namespace Nintendo.FsFileCacheSimulator.Simulation
{
    internal class AccessSimulator
    {
        public static SimulationResult DoAccessSimulation(
            AccessLog accessLog,
            Func<string, FileSystemType, StorageType> storageTypeGetter,
            Func<string, FileSystemType, ICacheSystem> cacheSystemGetter,
            ulong splitResultPerMilliseconds)
        {
            var fileSystem = new AccessSimulator(accessLog, storageTypeGetter, cacheSystemGetter, splitResultPerMilliseconds);
            var results = fileSystem.DoAccessSimulation();
            return new SimulationResult(results, fileSystem.m_SimulatedAccessLogEntries);
        }

        private AccessLog m_AccessLog;
        private Func<string, FileSystemType, StorageType> m_StorageTypeGetter;
        private Func<string, FileSystemType, ICacheSystem> m_CacheSystemGetter;
        private ulong m_SplitResultPerMilliseconds;

        private Dictionary<string, SimulatedFileSystem> m_MountedFileSystem;
        private List<SimulatedFileSystem> m_UnmountedFileSystem;

        // Key は参照同値性で問題ないはず
        private Dictionary<SimulatedFileSystem, ICacheSystem> m_CacheSystems;
        private Dictionary<SimulatedFileSystem, CacheStatistics> m_CacheStatistics;
        private Dictionary<SimulatedFileSystem, TimeSpan> m_LogBasedReadTimes;
        private Dictionary<SimulatedFileSystem, TimeSpan> m_LogBasedWriteTimes;

        private AccessLog.Entry m_LastAccessLogEntry;
        private List<AccessLog.Entry> m_SimulatedAccessLogEntries;

        private AccessSimulator(
            AccessLog accessLog,
            Func<string, FileSystemType, StorageType> storageTypeGetter,
            Func<string, FileSystemType, ICacheSystem> cacheSystemGetter,
            ulong splitResultPerMilliseconds)
        {
            m_AccessLog = accessLog;
            m_StorageTypeGetter = storageTypeGetter;
            m_CacheSystemGetter = cacheSystemGetter;
            m_SplitResultPerMilliseconds = splitResultPerMilliseconds;

            m_MountedFileSystem = new Dictionary<string, SimulatedFileSystem>();
            m_UnmountedFileSystem = new List<SimulatedFileSystem>();
            m_CacheSystems = new Dictionary<SimulatedFileSystem, ICacheSystem>();
            m_CacheStatistics = new Dictionary<SimulatedFileSystem, CacheStatistics>();
            m_LogBasedReadTimes = new Dictionary<SimulatedFileSystem, TimeSpan>();
            m_LogBasedWriteTimes = new Dictionary<SimulatedFileSystem, TimeSpan>();
            m_SimulatedAccessLogEntries = new List<AccessLog.Entry>();
        }

        private bool ShouldSplitResultByTime => m_SplitResultPerMilliseconds > 0;

        private void StartFileSystemSimulation(string name, FileSystemType fileSystemType)
        {
            if (m_MountedFileSystem.ContainsKey(name))
            {
                throw new ArgumentException($"同一のマウント名 '{name}' で複数のファイルシステムが同時にマウントされました");
            }

            SimulatedFileSystem fileSystem;
            switch (fileSystemType)
            {
                case FileSystemType.Rom:
                    fileSystem = SimulatedFileSystem.MountRom(name, m_StorageTypeGetter(name, fileSystemType));
                    break;
                case FileSystemType.SaveData:
                    fileSystem = SimulatedFileSystem.MountSaveData(name, m_StorageTypeGetter(name, fileSystemType));
                    break;
                case FileSystemType.CacheStorage:
                    fileSystem = SimulatedFileSystem.MountCacheStorage(name, m_StorageTypeGetter(name, fileSystemType));
                    break;
                case FileSystemType.TemporaryStorage:
                    fileSystem = SimulatedFileSystem.MountTemporaryStorage(name, m_StorageTypeGetter(name, fileSystemType));
                    break;
                default:
                    throw new NotImplementedException();
            }
            fileSystem.ReadFileEvent += UpdateStatisticsOnReadFileEvent;
            fileSystem.WriteFileEvent += UpdateStatisticsOnWriteFileEvent;

            m_MountedFileSystem.Add(name, fileSystem);
            m_CacheSystems.Add(fileSystem, m_CacheSystemGetter(name, fileSystemType));
            m_CacheStatistics.Add(fileSystem, new CacheStatistics());
            m_LogBasedReadTimes.Add(fileSystem, TimeSpan.Zero);
            m_LogBasedWriteTimes.Add(fileSystem, TimeSpan.Zero);
        }

        private FileSystemSimulationResult EndFileSystemSimulation(string name, ulong startTime, ulong endTime)
        {
            if (!m_MountedFileSystem.ContainsKey(name))
            {
                throw new ArgumentException($"マウント名 '{name}' でファイルシステムはマウントされていません");
            }

            var fileSystem = m_MountedFileSystem[name];

            var ret = new FileSystemSimulationResult(
                fileSystem.MountName,
                fileSystem.FileSystemType,
                fileSystem.Statistics,
                startTime,
                endTime,
                m_LogBasedReadTimes[fileSystem],
                m_LogBasedWriteTimes[fileSystem],
                m_CacheSystems[fileSystem],
                m_CacheStatistics[fileSystem]);

            m_MountedFileSystem.Remove(name);
            m_CacheSystems.Remove(fileSystem);
            m_CacheStatistics.Remove(fileSystem);
            m_LogBasedReadTimes.Remove(fileSystem);
            m_LogBasedWriteTimes.Remove(fileSystem);

            return ret;
        }

        private FileSystemSimulationResult SplitSimulationResult(string name, ulong startTime, ulong endTime)
        {
            if (!m_MountedFileSystem.ContainsKey(name))
            {
                throw new ArgumentException($"マウント名 '{name}' でファイルシステムはマウントされていません");
            }

            var fileSystem = m_MountedFileSystem[name];

            var ret = new FileSystemSimulationResult(
                fileSystem.MountName,
                fileSystem.FileSystemType,
                fileSystem.Statistics,
                startTime,
                endTime,
                m_LogBasedReadTimes[fileSystem],
                m_LogBasedWriteTimes[fileSystem],
                m_CacheSystems[fileSystem],
                m_CacheStatistics[fileSystem]);

            fileSystem.Statistics = new FileSystemStatistics();
            m_CacheStatistics[fileSystem] = new CacheStatistics();
            m_LogBasedReadTimes[fileSystem] = TimeSpan.Zero;
            m_LogBasedWriteTimes[fileSystem] = TimeSpan.Zero;

            return ret;
        }

        private IEnumerable<FileSystemSimulationResult> DoAccessSimulation()
        {
            var results = new List<FileSystemSimulationResult>();
            var openFiles = new Dictionary<ulong, string>();

            var splitStartTime = m_AccessLog.StartTimeMillisecond;
            var splitEndTime = ShouldSplitResultByTime ? splitStartTime + m_SplitResultPerMilliseconds : m_AccessLog.EndTimeMillisecond;

            foreach (var accessLogEntry in m_AccessLog.Entries)
            {
                if (ShouldSplitResultByTime && accessLogEntry.Start > splitEndTime)
                {
                    while (splitEndTime < accessLogEntry.Start)
                    {
                        foreach (var mountName in m_MountedFileSystem.Keys)
                        {
                            results.Add(SplitSimulationResult(mountName, splitStartTime, splitEndTime));
                        }
                        splitStartTime += m_SplitResultPerMilliseconds;
                        splitEndTime += m_SplitResultPerMilliseconds;
                    }
                }

                if (!accessLogEntry.IsResultSuccess)
                {
                    m_SimulatedAccessLogEntries.Add(accessLogEntry);
                    continue;
                }

                switch (accessLogEntry)
                {
                    case AccessLog.MountRomEntry mountRomEntry:
                        StartFileSystemSimulation(mountRomEntry.Name, FileSystemType.Rom);
                        m_SimulatedAccessLogEntries.Add(accessLogEntry);
                        break;
                    case AccessLog.MountSaveDataEntry mountSaveDataEntry:
                        StartFileSystemSimulation(mountSaveDataEntry.Name, FileSystemType.SaveData);
                        m_SimulatedAccessLogEntries.Add(accessLogEntry);
                        break;
                    case AccessLog.MountCacheStorageEntry mountCacheStorageEntry:
                        StartFileSystemSimulation(mountCacheStorageEntry.Name, FileSystemType.CacheStorage);
                        m_SimulatedAccessLogEntries.Add(accessLogEntry);
                        break;
                    case AccessLog.MountTemporaryStorageEntry mountTemporaryStorageEntry:
                        StartFileSystemSimulation(mountTemporaryStorageEntry.Name, FileSystemType.TemporaryStorage);
                        m_SimulatedAccessLogEntries.Add(accessLogEntry);
                        break;
                    case AccessLog.UnmountEntry unmountEntry:
                        if (m_MountedFileSystem.ContainsKey(unmountEntry.Name))
                        {
                            results.Add(EndFileSystemSimulation(unmountEntry.Name, splitStartTime, splitEndTime));
                        }
                        else
                        {
                            Log.Default.WarningLine($"不明なマウント名 '{unmountEntry.Name}' をアンマウントしようとしました。無視します");
                        }
                        m_SimulatedAccessLogEntries.Add(accessLogEntry);
                        break;
                    case AccessLog.OpenFileEntry openFileEntry:
                        {
                            var mountName = FileSystem.Utility.GetMountName(openFileEntry.Path);
                            if (m_MountedFileSystem.ContainsKey(mountName))
                            {
                                openFiles.Add(openFileEntry.Handle, openFileEntry.Path);
                            }
                            else
                            {
                                Log.Default.WarningLine($"マウント名 '{mountName}' が登録されていません。ファイル '{openFileEntry.Path}' のオープンを無視します");
                            }
                            m_SimulatedAccessLogEntries.Add(accessLogEntry);
                        }
                        break;
                    case AccessLog.CloseFileEntry closeFileEntry:
                        if (openFiles.ContainsKey(closeFileEntry.Handle))
                        {
                            openFiles.Remove(closeFileEntry.Handle);
                        }
                        else
                        {
                            Log.Default.WarningLine($"ファイルハンドル '0x{closeFileEntry.Handle:x16}' はオープンされていません。このハンドルのクローズを無視します");
                        }
                        m_SimulatedAccessLogEntries.Add(accessLogEntry);
                        break;
                    case AccessLog.ReadFileEntry readFileEntry:
                        {
                            if (openFiles.TryGetValue(readFileEntry.Handle, out string path))
                            {
                                var mountName = FileSystem.Utility.GetMountName(path);

                                if (m_MountedFileSystem.TryGetValue(mountName, out SimulatedFileSystem fileSystem))
                                {
                                    m_LastAccessLogEntry = readFileEntry;  // UpdateStatisticsOnReadFileEvent で Start と End の時刻を記録するため
                                    m_CacheSystems[fileSystem].ReadFile(path, readFileEntry.Offset, readFileEntry.Size, fileSystem, m_CacheStatistics[fileSystem]);
                                }
                                else
                                {
                                    throw new ImplementationErrorException();
                                }
                            }
                            else
                            {
                                Log.Default.WarningLine($"ファイルハンドル '0x{readFileEntry.Handle:x16}' はオープンされていません。このファイルの読み取りを無視します");
                                m_SimulatedAccessLogEntries.Add(accessLogEntry);
                            }
                        }
                        break;
                    case AccessLog.WriteFileEntry writeFileEntry:
                        {
                            if (openFiles.TryGetValue(writeFileEntry.Handle, out string path))
                            {
                                var mountName = FileSystem.Utility.GetMountName(path);

                                if (m_MountedFileSystem.TryGetValue(mountName, out SimulatedFileSystem fileSystem))
                                {
                                    m_LastAccessLogEntry = writeFileEntry;  // UpdateStatisticsOnWriteFileEvent で Start と End の時刻を記録するため
                                    m_CacheSystems[fileSystem].WriteFile(path, writeFileEntry.Offset, writeFileEntry.Size, fileSystem, m_CacheStatistics[fileSystem]);
                                }
                                else
                                {
                                    throw new ImplementationErrorException();
                                }
                            }
                            else
                            {
                                Log.Default.WarningLine($"ファイルハンドル '0x{writeFileEntry.Handle:x16}' はオープンされていません。このファイルへの書き込みを無視します");
                                m_SimulatedAccessLogEntries.Add(accessLogEntry);
                            }
                        }
                        break;
                    default:
                        m_SimulatedAccessLogEntries.Add(accessLogEntry);
                        break;
                }
            }

            if (ShouldSplitResultByTime)
            {
                while (splitStartTime < m_AccessLog.EndTimeMillisecond)
                {
                    foreach (var mountName in m_MountedFileSystem.Keys)
                    {
                        results.Add(SplitSimulationResult(mountName, splitStartTime, Math.Min(splitEndTime, m_AccessLog.EndTimeMillisecond)));
                    }
                    splitStartTime += m_SplitResultPerMilliseconds;
                    splitEndTime += m_SplitResultPerMilliseconds;
                }
            }
            else
            {
                foreach (var mountName in m_MountedFileSystem.Keys.ToArray())
                {
                    results.Add(EndFileSystemSimulation(mountName, splitStartTime, splitEndTime));
                }
            }

            return results;
        }

        public void UpdateStatisticsOnReadFileEvent(object sender, SimulatedFileSystem.AccessEventArgs args)
        {
            var fileSystem = (SimulatedFileSystem)sender;

            m_LogBasedReadTimes[fileSystem] += TimeSpan.FromMilliseconds(m_LastAccessLogEntry.End - m_LastAccessLogEntry.Start);

            const uint ResultSuccess = 0;
            m_SimulatedAccessLogEntries.Add(new AccessLog.ReadFileEntry(
                m_LastAccessLogEntry.Start,
                m_LastAccessLogEntry.End,
                ResultSuccess,
                m_LastAccessLogEntry.Handle,
                args.Offset,
                args.Size));
        }

        public void UpdateStatisticsOnWriteFileEvent(object sender, SimulatedFileSystem.AccessEventArgs args)
        {
            var fileSystem = (SimulatedFileSystem)sender;

            m_LogBasedWriteTimes[fileSystem] += TimeSpan.FromMilliseconds(m_LastAccessLogEntry.End - m_LastAccessLogEntry.Start);

            const uint ResultSuccess = 0;
            m_SimulatedAccessLogEntries.Add(new AccessLog.WriteFileEntry(
                m_LastAccessLogEntry.Start,
                m_LastAccessLogEntry.End,
                ResultSuccess,
                m_LastAccessLogEntry.Handle,
                args.Offset,
                args.Size));
        }
    }
}
