﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Runtime.Serialization;
using ContentsUploader.Assistants;

namespace ContentsUploader.Models
{
    public static class Osiris
    {
        public interface IMergableRomList<T> where T : IEquatable<T>
        {
            int total { get; set; }
            int length { get; set; }
            int offset { get; set; }
            List<T> roms { get; set; }
        }

        public static bool MergeEntity<T>(IMergableRomList<T> dst, IMergableRomList<T> src) where T : IEquatable<T>
        {
            // どちらかが null なら失敗、同じものなら何もせずに成功
            if (object.ReferenceEquals(dst, null) || object.ReferenceEquals(src, null))
            {
                return false;
            }
            else if (object.ReferenceEquals(dst, src))
            {
                return true;
            }

            // 下記条件なら失敗
            //  ・マージ先の開始位置が 0 以外
            //  ・マージ先とマージ元に連続性がない
            //  ・マージ先よりマージ元の総数が少ない
            if (dst.offset != 0 || dst.length < src.offset || dst.total > src.total)
            {
                return false;
            }

            // 重複部分の一致確認
            for (var index = 0; index < (dst.length - src.offset); index++)
            {
                if (!dst.roms[src.offset + index].Equals(src.roms[index]))
                {
                    return false;
                }
            }

            // 非重複部分を追加
            for (var index = dst.length - src.offset; index < src.length; index++)
            {
                dst.roms.Add(src.roms[index]);
            }
            dst.length = dst.roms.Count;
            dst.total = src.total;
            return true;
        }

        [DataContract]
        public class RomSetEntity : IMergableRomList<RomDetailsEntity>
        {
            public RomSetEntity()
            {
                roms = new List<RomDetailsEntity>();
            }

            public RomSetEntity(int inputTotal, int inputLength, int inputOffset, List<RomDetailsEntity> inputRoms)
            {
                total = inputTotal;
                length = inputLength;
                offset = inputOffset;
                roms = inputRoms;
            }

            public bool IsPartial { get { return offset != 0 || total != length; } }
            public bool IsComplete { get { return !(IsPartial); } }

            [DataMember(Name = "total")]
            public int total { get; set; }

            [DataMember(Name = "length")]
            public int length { get; set; }

            [DataMember(Name = "offset")]
            public int offset { get; set; }

            [DataMember(Name = "roms")]
            public List<RomDetailsEntity> roms { get; set; }
        }

        [DataContract]
        public class RomDetailsEntity : IEquatable<RomDetailsEntity>
        {
            [DataMember(Name = "rom_id")]
            public string romId { get; set; }

            [DataMember(Name = "rom_type")]
            public string romType { get; set; }

            [DataMember(Name = "application_id")]
            public string applicationId { get; set; }

            [DataMember(Name = "original_file_name")]
            public string originalFileName { get; set; }

            [DataMember(Name = "status")]
            public string status { get; set; }

            [DataMember(Name = "updated_date")]
            public string updatedData { get; set; }

            public override bool Equals(object obj)
            {
                return Equals(obj as RomDetailsEntity);
            }

            public bool Equals(RomDetailsEntity other)
            {
                if (object.ReferenceEquals(other, null))
                {
                    return false;
                }
                if (object.ReferenceEquals(other, this))
                {
                    return true;
                }
                return ((other.romId == this.romId) &&
                        (other.romType == this.romType) &&
                        (other.applicationId == this.applicationId) &&
                        (other.originalFileName == this.originalFileName) &&
                        (other.status == this.status) &&
                        (other.updatedData == this.updatedData));
            }

            public override int GetHashCode()
            {
                return romId.GetHashCode();
            }

            public override string ToString()
            {
                return $"{applicationId},{romType},{status},{romId}";
            }

            public static bool operator ==(RomDetailsEntity lhs, RomDetailsEntity rhs)
            {
                if (object.ReferenceEquals(lhs, null))
                {
                    return object.ReferenceEquals(rhs, null);
                }
                return lhs.Equals(rhs);
            }

            public static bool operator !=(RomDetailsEntity lhs, RomDetailsEntity rhs)
            {
                return !(lhs == rhs);
            }
        }

        [DataContract]
        public class RomTitlePairSetEntity
        {
            [DataMember(Name = "rom_titles")]
            public List<RomTitlePairEntity> romTitles { get; set; }
        }

        [DataContract]
        public class RomTitlePairEntity
        {
            public static readonly RomTitlePairEntity Invalid = new RomTitlePairEntity("0000000000000000", 0, "");

            public RomTitlePairEntity(string titleId, int titleVersion, string updatedData)
            {
                this.titleId = titleId;
                this.titleVersion = titleVersion;
                this.updatedData = updatedData;
            }

            // レスポンスに含まれていなかったので削除
            //[DataMember(Name = "rom_id")]
            //public string romId { get; set; }

            [DataMember(Name = "title_id")]
            public string titleId { get; set; }

            [DataMember(Name = "title_version")]
            public int titleVersion { get; set; }

            [DataMember(Name = "updated_date")]
            public string updatedData { get; set; }

            public override string ToString()
            {
                return $"{titleId},{titleVersion},{updatedData}";
            }

            public static int CompareByUpdateDateDescending(RomTitlePairEntity lhs, RomTitlePairEntity rhs)
            {
                return -1 * ToolUtility.Compare(lhs, rhs,
                    (l, r) =>
                    {
                        return ToolUtility.Compare(l.updatedData, r.updatedData);
                    });
            }
        }

        [DataContract]
        public class TitleDetailsEntity
        {
            [DataMember(Name = "aoc_index")]
            public string aocIndex { get; set; }

            [DataMember(Name = "application_id")]
            public string applicationId { get; set; }

            [DataMember(Name = "status")]
            public string status { get; set; }

            [DataMember(Name = "status_date")]
            public string statusDate { get; set; }

            [DataMember(Name = "title_id")]
            public string titleId { get; set; }

            [DataMember(Name = "title_type")]
            public string titleType { get; set; }

            // TODO: このパラメータを参照する要求が生まれたら実装
            //[DataMember(Name = "title_versions")]
            //public TitleDetailsVersion titleVersions { get; set; }

            [DataMember(Name = "updated_date")]
            public string updatedData { get; set; }
        }

        [DataContract]
        public class RomTitleSetEntity : IMergableRomList<RomTitleEntity>
        {
            public RomTitleSetEntity()
            {
                roms = new List<RomTitleEntity>();
            }

            public bool IsPartial { get { return offset != 0 || total != length; } }
            public bool IsComplete { get { return !(IsPartial); } }

            [DataMember(Name = "total")]
            public int total { get; set; }

            [DataMember(Name = "length")]
            public int length { get; set; }

            [DataMember(Name = "offset")]
            public int offset { get; set; }

            [DataMember(Name = "roms")]
            public List<RomTitleEntity> roms { get; set; }
        }

        [DataContract]
        public class RomTitleEntity : IEquatable<RomTitleEntity>
        {
            [DataMember(Name = "rom_id")]
            public string romId { get; set; }

            [DataMember(Name = "rom_type")]
            public string romType { get; set; }

            [DataMember(Name = "title_id")]
            public string titleId { get; set; }

            [DataMember(Name = "title_version")]
            public int titleVersion { get; set; }

            [DataMember(Name = "application_id")]
            public string applicationId { get; set; }

            [DataMember(Name = "status")]
            public string status { get; set; }

            [DataMember(Name = "updated_date")]
            public string updatedData { get; set; }

            public override bool Equals(object obj)
            {
                return Equals(obj as RomTitleEntity);
            }

            public bool Equals(RomTitleEntity other)
            {
                if (object.ReferenceEquals(other, null))
                {
                    return false;
                }
                if (object.ReferenceEquals(other, this))
                {
                    return true;
                }
                return ((other.romId == this.romId) &&
                        (other.romType == this.romType) &&
                        (other.titleId == this.titleId) &&
                        (other.titleVersion == this.titleVersion) &&
                        (other.applicationId == this.applicationId) &&
                        (other.status == this.status) &&
                        (other.updatedData == this.updatedData));
            }

            public override int GetHashCode()
            {
                return romId.GetHashCode();
            }

            public override string ToString()
            {
                return $"{applicationId},{titleId},{titleVersion},{romType},{status},{romId}";
            }

            public static bool operator ==(RomTitleEntity lhs, RomTitleEntity rhs)
            {
                if (object.ReferenceEquals(lhs, null))
                {
                    return object.ReferenceEquals(rhs, null);
                }
                return lhs.Equals(rhs);
            }

            public static bool operator !=(RomTitleEntity lhs, RomTitleEntity rhs)
            {
                return !(lhs == rhs);
            }
        }

        public class RomTitleInfo
        {
            public enum RomType
            {
                Application = 0,
                Patch = 1,
                Combined = 2,
                Unknown = 100,
            }

            public RomType romType { get; }
            public Id64 applicationId { get; }
            public Id64 contentMetaId { get; }
            public int contentMetaVersion { get; }

            public string romId { get; }
            public string updateDate { get; }
            public string status { get; }

            public RomTitleInfo(RomDetailsEntity rom, RomTitlePairEntity title)
            {
                romType = ToolUtility.ToEnumValue(rom.romType, RomType.Unknown);
                applicationId = new Id64(rom.applicationId);
                contentMetaId = new Id64(title.titleId);
                contentMetaVersion = title.titleVersion;

                romId = rom.romId;
                updateDate = rom.updatedData;
                status = rom.status;
            }

            public RomTitleInfo(RomTitleEntity rom)
            {
                romType = ToolUtility.ToEnumValue(rom.romType, RomType.Unknown);
                applicationId = new Id64(rom.applicationId);
                contentMetaId = new Id64(rom.titleId);
                contentMetaVersion = rom.titleVersion;

                romId = rom.romId;
                updateDate = rom.updatedData;
                status = rom.status;
            }

            public override string ToString()
            {
                return $"{applicationId},{contentMetaId},{contentMetaVersion},{romType},{status},{romId}";
            }

            public static int CompareForRevokeRom(RomTitleInfo lhs, RomTitleInfo rhs)
            {
                return -1 * ToolUtility.Compare(lhs, rhs,
                    (l, r) =>
                    {
                        int result = 0;
                        if (false) { } // 単なる位置合わせ
                        else if (ToolUtility.Compare(out result, l.romType, r.romType)) { }
                        else if (ToolUtility.Compare(out result, l.applicationId, r.applicationId)) { }
                        else if (ToolUtility.Compare(out result, l.contentMetaId, r.contentMetaId)) { }
                        else if (ToolUtility.Compare(out result, l.contentMetaVersion, r.contentMetaVersion)) { }
                        else if (ToolUtility.Compare(out result, l.updateDate, r.updateDate)) { }
                        return result;
                    });
            }
        }
    }
}
