﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;

namespace ContentsUploader
{
    [Flags]
    public enum Languages : ulong
    {
        en_US = 1 << 0,
        fr_US = 1 << 1,
        pt_US = 1 << 2,
        es_US = 1 << 3,
        en_AU = 1 << 4,
        en = 1 << 5,
        nl = 1 << 6,
        fr = 1 << 7,
        de = 1 << 8,
        it = 1 << 9,
        ja = 1 << 10,
        ko = 1 << 11,
        pt = 1 << 12,
        ru = 1 << 13,
        zh_CN = 1 << 14,
        es = 1 << 15,
        zh_TW = 1 << 16,
    }

    public class Country
    {
        public string Code { get; }
        public string Name { get; }

        public Country(string code, string name)
        {
            Code = code;
            Name = name;
        }

        public override string ToString()
        {
            return $"{Code},{Name}";
        }
    }

    public static class Constants
    {
        public const int ApproveRomTimeoutMinutes = 10;         // タイムアウト(minute)
        public const int ApproveRomRetryCountMax = 3;           // 試行回数
        public const int ApproveRomIntervalMilliseconds = 5000; // 間隔（msec）

        public const int RevokeRomTimeoutMinutes = 10;          // タイムアウト(minute)
        public const int RevokeRomRetryCountMax = 3;            // 試行回数
        public const int RevokeRomIntervalMilliseconds = 5000;  // 間隔（msec）

        public const int ApprovePriceRetryCountMax = 2;         // 試行回数

        public static readonly Languages LanguagesNone;
        public static readonly Languages LanguagesAll;

        public static readonly Languages DemoLanguagesRequired;
        public static readonly Languages DemoLanguagesDefault;
        public static readonly Languages DemoLanguagesAll;
        public static readonly List<string> DemoCountries;

        public static readonly Languages TitleLanguagesRequired;
        public static readonly Languages TitleLanguagesDefault;
        public static readonly Languages TitleLanguagesAll;
        public static readonly List<string> TitleCountries;

        public static readonly Dictionary<string, string> LanguageNames;
        public static readonly Dictionary<string, Country> Countries;
        private static Country[] CountriesAll = {
                new Country("JP", "Japan"),
                new Country("AI", "Anguilla"),
                new Country("AG", "Antigua and Barbuda"),
                new Country("AR", "Argentina"),
                new Country("AW", "Aruba"),
                new Country("BS", "Commonwealth of The Bahamas"),
                new Country("BB", "Barbados"),
                new Country("BZ", "Belize"),
                new Country("BO", "Republic of Bolivia"),
                new Country("BR", "Brazil"),
                new Country("VG", "British Virgin Islands"),
                new Country("CA", "Canada"),
                new Country("KY", "Cayman Islands"),
                new Country("CL", "Chile"),
                new Country("CO", "Colombia"),
                new Country("CR", "Costa Rica"),
                new Country("DM", "Commonwealth of Dominica"),
                new Country("DO", "Dominican Republic"),
                new Country("EC", "Ecuador"),
                new Country("SV", "Republic of El Salvador"),
                new Country("GF", "Guiana"),
                new Country("GD", "Grenada"),
                new Country("GP", "Guadeloupe"),
                new Country("GT", "Guatemala"),
                new Country("GY", "Co-operative Republic of Guyana"),
                new Country("HT", "Republic of Haiti"),
                new Country("HN", "Honduras"),
                new Country("JM", "Jamaica"),
                new Country("MQ", "Martinique"),
                new Country("MX", "Mexico"),
                new Country("MS", "Montserrat"),
                new Country("AN", "Nederlandse Antillen"),
                new Country("NI", "Nicaragua"),
                new Country("PA", "Panama"),
                new Country("PY", "Paraguay"),
                new Country("PE", "Peru"),
                new Country("KN", "Saint Christopher and Nevis"),
                new Country("LC", "Saint Lucia"),
                new Country("VC", "Saint Vincent and the Grenadines"),
                new Country("SR", "Republic of Suriname"),
                new Country("TT", "Republic of Trinidad and Tobago"),
                new Country("TC", "Turks and Caicos Islands"),
                new Country("US", "United States"),
                new Country("UY", "Uruguay"),
                new Country("VI", "Virgin Islands of the United States"),
                new Country("VE", "Venezuela"),
                new Country("AL", "Republic of Albania"),
                new Country("AU", "Australia"),
                new Country("AT", "Austria"),
                new Country("BE", "Belgium"),
                new Country("BA", "Bosnia and Herzegovina"),
                new Country("BW", "Republic of Botswana"),
                new Country("BG", "Republic of Bulgaria"),
                new Country("HR", "Republic of Croatia"),
                new Country("CY", "Republic of Cyprus"),
                new Country("CZ", "Czech Republic"),
                new Country("DK", "Denmark"),
                new Country("EE", "Republic of Estonia"),
                new Country("FI", "Finland"),
                new Country("FR", "France"),
                new Country("DE", "Germany"),
                new Country("GR", "Greece"),
                new Country("HU", "Republic of Hungary"),
                new Country("IS", "Republic of Iceland"),
                new Country("IE", "Ireland"),
                new Country("IT", "Italy"),
                new Country("LV", "Republic of Latvia"),
                new Country("LS", "Kingdom of Lesotho"),
                new Country("LI", "Principality of Liechtenstein"),
                new Country("LT", "Republic of Lithuania"),
                new Country("LU", "Luxembourg"),
                new Country("MK", "Former Yugoslav Republic of Macedonia"),
                new Country("MT", "Republic of Malta"),
                new Country("ME", "Republic of Montenegro"),
                new Country("MZ", "Republic of Mozambique"),
                new Country("NA", "Republic of Namibia"),
                new Country("NL", "Netherlands"),
                new Country("NZ", "New Zealand"),
                new Country("NO", "Norway"),
                new Country("PL", "Poland"),
                new Country("PT", "Portugal"),
                new Country("RO", "Romania"),
                new Country("RU", "Russia"),
                new Country("RS", "Republic of Serbia & Republic of Kosovo"),
                new Country("SK", "Slovak Republic"),
                new Country("SI", "Republic of Slovenia"),
                new Country("ZA", "South Africa"),
                new Country("ES", "Spain"),
                new Country("SZ", "Kingdom of Swaziland"),
                new Country("SE", "Sweden"),
                new Country("CH", "Switzerland"),
                new Country("TR", "Turkey"),
                new Country("GB", "United Kingdom"),
                new Country("ZM", "Republic of Zambia"),
                new Country("ZW", "Republic of Zimbabwe"),
                new Country("AZ", "Azerbaijan"),
                new Country("MR", "Mauritania"),
                new Country("ML", "Mali"),
                new Country("NE", "Niger"),
                new Country("TD", "Chad"),
                new Country("SD", "Sudan"),
                new Country("ER", "Eritrea"),
                new Country("DJ", "Djibouti"),
                new Country("SO", "Somalia"),
                new Country("AD", "Andorra"),
                new Country("GI", "Gibraltar"),
                new Country("GG", "Guernsey"),
                new Country("IM", "Isle of Man"),
                new Country("JE", "Jersey"),
                new Country("MC", "Monaco"),
                new Country("TW", "Taiwan"),
                new Country("KR", "Korea"),
                new Country("HK", "Hong Kong"),
                new Country("SG", "Republic of Singapore"),
                new Country("MY", "Malaysia"),
                new Country("AE", "United Arab Emirates"),
                new Country("IN", "India"),
                new Country("SA", "Kingdom of Saudi Arabia"),
                new Country("SM", "San Marino"),
                new Country("VA", "Vatican City"),
                new Country("BM", "Bermuda")
            };

        static Constants()
        {
            {
                // 言語指定なし
                LanguagesNone = (Languages)0;

                // 全言語
                var all = LanguagesNone;
                foreach (var value in Enum.GetValues(typeof(Languages)))
                {
                    all |= (Languages)value;
                }
                LanguagesAll = all;

                // 言語名
                LanguageNames = new Dictionary<string, string>();
                LanguageNames.Add(Languages.en_US.ToString(), "American English");
                LanguageNames.Add(Languages.fr_US.ToString(), "American French");
                LanguageNames.Add(Languages.pt_US.ToString(), "American Portuguese");
                LanguageNames.Add(Languages.es_US.ToString(), "American Spanish");
                LanguageNames.Add(Languages.en_AU.ToString(), "Australian English");
                LanguageNames.Add(Languages.en.ToString(), "British English");
                LanguageNames.Add(Languages.nl.ToString(), "Dutch");
                LanguageNames.Add(Languages.fr.ToString(), "French");
                LanguageNames.Add(Languages.de.ToString(), "German");
                LanguageNames.Add(Languages.it.ToString(), "Italian");
                LanguageNames.Add(Languages.ja.ToString(), "Japanese");
                LanguageNames.Add(Languages.ko.ToString(), "Korean");
                LanguageNames.Add(Languages.pt.ToString(), "Portuguese");
                LanguageNames.Add(Languages.ru.ToString(), "Russian");
                LanguageNames.Add(Languages.zh_CN.ToString(), "Simplified Chinese");
                LanguageNames.Add(Languages.es.ToString(), "Spanish");
                LanguageNames.Add(Languages.zh_TW.ToString(), "Traditional Chinese");

                // 国
                Countries = new Dictionary<string, Country>();
                foreach (var country in CountriesAll)
                {
                    Countries.Add(country.Code, country);
                }
            }

            // Demo
            {
                // 必須言語
                DemoLanguagesRequired = Languages.en_US | Languages.es_US | Languages.en | Languages.ja;

                // 既定言語
                DemoLanguagesDefault = DemoLanguagesRequired;

                // 全言語
                // ※ ko, zh_CN, zh_TW は UI 上からは登録できるが、Web API から反映できないため除外しておく
                DemoLanguagesAll = LanguagesAll & ~(Languages.ko | Languages.zh_CN | Languages.zh_TW);

                // 対象国
                DemoCountries = new List<string> { "JP", "GB", "US" };
            }

            // Title
            {
                // 必須言語
                TitleLanguagesRequired = Languages.en_US | Languages.fr_US | Languages.es_US | Languages.en | Languages.de | Languages.ja;

                // 既定言語
                TitleLanguagesDefault = TitleLanguagesRequired;

                // 全言語
                // ※ ko, zh_CN, zh_TW は UI 上からも登録できないので除外しておく
                TitleLanguagesAll = LanguagesAll & ~(Languages.ko | Languages.zh_CN | Languages.zh_TW);

                // 対象国
                TitleCountries = new List<string> { "CA", "DE", "JP", "GB", "US" };
            }
        }

        public static string ToLanguageName(Languages language)
        {
            return ToLanguageName(language.ToString());
        }

        public static string ToLanguageName(string language)
        {
            var name = "";
            if (LanguageNames.ContainsKey(language))
            {
                name = LanguageNames[language];
            }
            return name;
        }
    }
}
