﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.IO;

namespace CommandUtility
{
    public class FileUtility
    {
        public static string LoadText(string path)
        {
            return LoadText(new FileInfo(path));
        }

        public static string LoadText(FileInfo path)
        {
            using (var textReader = path.OpenText())
            {
                return textReader.ReadToEnd();
            }
        }

        public static string GetRelativePath(string targetPath, DirectoryInfo originPath)
        {
            var target = new Uri(targetPath);
            var origin = new Uri(originPath.FullName + Path.DirectorySeparatorChar);
            return Uri.UnescapeDataString(origin.MakeRelativeUri(target).ToString().Replace('/', Path.DirectorySeparatorChar));
        }

        public static string GetRelativePath(string targetPath, string basePath)
        {
            Uri pathUri = new Uri(targetPath);
            if (!basePath.EndsWith(Path.DirectorySeparatorChar.ToString()))
            {
                basePath += Path.DirectorySeparatorChar;
            }
            Uri baseUri = new Uri(basePath);
            return Uri.UnescapeDataString(baseUri.MakeRelativeUri(pathUri).ToString().Replace('/', Path.DirectorySeparatorChar));
        }

        public static void WriteToStream(Stream writer, FileInfo file)
        {
            using (var reader = file.OpenRead())
            {
                reader.CopyTo(writer);
            }
        }

        public static void WriteBytes(Stream writer, byte[] bytes)
        {
            writer.Write(bytes, 0, bytes.Length);
        }

        public static void WriteZeroToStream(Stream writer, long size)
        {
            if (size == 0)
            {
                return;
            }

            var pad = new byte[(int)size];
            writer.Write(pad, 0, (int)size);
        }

        public static void WriteToStreamWithPad(FileStream writer, FileInfo file, long alignedSize)
        {
            EnsureMaxFileSize(file, alignedSize);

            WriteToStream(writer, file);
            WriteZeroToStream(writer, alignedSize - file.Length);
        }

        public static byte[] ReadRange(Stream reader, long offset, long size)
        {
            reader.Seek(offset, SeekOrigin.Begin);
            return BinaryUtility.ReadBytes(reader, (int)size);
        }

        public static void EnsureFileSize(FileInfo file, long expectedSize)
        {
            if (file.Length != expectedSize)
            {
                throw new Exception(string.Format(
                    "Invalid file size: file={0}, actualSize={1}, expectedSize={2}",
                    file.FullName, file.Length, expectedSize));
            }
        }

        public static void WritePaddingToAddress(Stream writer, long address)
        {
            var paddingSize = address - writer.Position;
            ErrorHandling.Verify(0 <= paddingSize, $"exceed data. expected: position <= {address}, actualPosition = {writer.Position}");
            WriteZeroToStream(writer, paddingSize);
        }

        public static void WritePaddingToAddressIfNeed(Stream writer, long address)
        {
            var paddingSize = address - writer.Position;
            if(0 < paddingSize)
            {
                WriteZeroToStream(writer, paddingSize);
            }
        }

        public static void EnsureMaxFileSize(FileInfo file, long expectedMaxSize)
        {
            if (expectedMaxSize < file.Length)
            {
                throw new Exception(string.Format(
                    "Invalid file size: file={0}, actualSize={1}, maxSize={2}",
                    file.FullName, file.Length, expectedMaxSize));
            }
        }

        public static FileInfo MakeExistedFileInfo(string argumentName, string fileName)
        {
            var fileInfo = new FileInfo(fileName);
            if (!fileInfo.Exists)
            {
                throw new Exception(string.Format("{0}({1}) is not found.", argumentName, fileInfo.FullName));
            }

            return fileInfo;
        }

        public static void OpenDirectory(DirectoryInfo directory)
        {
            if (!Directory.Exists(directory.FullName))
            {
                throw new DirectoryNotFoundException($"{directory.FullName} is not found.");
            }

            System.Diagnostics.Process.Start(directory.FullName);
        }

        public static void OpenFile(FileInfo fileInfo)
        {
            if (!File.Exists(fileInfo.FullName))
            {
                throw new FileNotFoundException($"{fileInfo.FullName} is not found.");
            }

            System.Diagnostics.Process.Start(fileInfo.FullName);
        }
    }

    public static class FileSystemInfoExtensions
    {
        public static string GetRelativePath(this FileSystemInfo targetPath, FileInfo basePath)
        {
            return FileUtility.GetRelativePath(
                targetPath.FullName, basePath.GetDirectoryPath().FullName);
        }

        public static string GetRelativePath(this FileSystemInfo targetPath, DirectoryInfo basePath)
        {
            return FileUtility.GetRelativePath(
                targetPath.FullName, basePath.FullName);
        }
    }

    public static class FileInfoExtensions
    {
        public static void Create(this FileInfo fileInfo, string text)
        {
            fileInfo.GetDirectoryPath().Create();
            using (var writer = fileInfo.CreateText())
            {
                writer.Write(text);
            }
        }

        public static DirectoryInfo GetDirectoryPath(this FileInfo fileInfo)
        {
            return new DirectoryInfo(Path.GetDirectoryName(fileInfo.FullName));
        }

        public static FileInfo CopyTo(this FileInfo fileInfo, FileInfo target)
        {
            fileInfo.CopyTo(target.FullName);

            return target;
        }

        public static FileInfo CopyTo(this FileInfo fileInfo, DirectoryInfo targetDirectory)
        {
            var targetFile = targetDirectory.CombineAsFile(fileInfo.Name);
            fileInfo.CopyTo(targetFile);

            return targetFile;
        }

        public static void CopyTo(this FileInfo fileInfo, Stream writer)
        {
            using (var reader = fileInfo.OpenRead())
            {
                reader.CopyTo(writer);
            }
        }
    }

    public static class DirectoryInfoExtensions
    {
        public static FileInfo CombineAsFile(this DirectoryInfo directoryInfo, string filename)
        {
            return new FileInfo(Path.Combine(directoryInfo.FullName, filename));
        }

        public static DirectoryInfo CombineAsDirectory(this DirectoryInfo directoryInfo, string directoryName)
        {
            return new DirectoryInfo(Path.Combine(directoryInfo.FullName, directoryName));
        }

        public static void DeleteWithRetry(this DirectoryInfo directory, bool recursive)
        {
            RetryUtility.Do(() =>
            {
                directory.Delete(recursive);
            },
            e =>
            {
                Console.Error.WriteLine($"[ERROR] Failed to delete directory. path={directory.FullName}. exception={e.GetType().Name}:{e.Message}");
            },
            5,
            TimeSpan.FromSeconds(1));
        }
    }
}
