﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
namespace ElfLibrary
{
    using System;

    public enum ElfSegmentType
    {
        PT_NULL = 0,
        PT_LOAD = 1,
        PT_DYNAMIC = 2,
        PT_INTERP = 3,
        PT_NOTE = 4,
        PT_SHLIB = 5,
        PT_PHDR = 6,
        PT_LOPROC = 0x70000000,
        PT_HIPROC = 0x7fffffff,
    }

    /// <summary>
    /// Elf ProgramHeader のベースクラス
    /// </summary>
    public abstract class ElfProgramHeader : IElfFile
    {
        // 順番が 32-bit と異なる
        public uint P_Type { get; set; }
        public uint P_Flags { get; set; }
        public ulong P_Offset { get; set; }
        public ulong P_Vaddr { get; set; }
        public ulong P_Paddr { get; set; }
        public ulong P_FileSz { get; set; }
        public ulong P_MemSz { get; set; }
        public ulong P_Align { get; set; }

        /// <summary>
        /// ELFプログラムエントリの位置を計算。
        /// </summary>
        /// <returns>エントリの終了地点のオフセット</returns>
        public ulong SetPosition(ulong offset)
        {
            if (!(1 == P_Align || 0 == P_Align % 2))
            {
                throw new InvalidOperationException();
            }
            P_Offset = NumericConvert.RoundUp(offset, P_Align);
            return (P_Offset + P_FileSz);
        }

        /// <summary>
        /// ProgramHeader を読み込みます。
        /// </summary>
        /// <param name="reader">ElfBinaryReader</param>
        public abstract void ReadElfFile(ElfBinaryReader reader);

        /// <summary>
        /// ProgramHeader を書きだします。
        /// </summary>
        /// <param name="reader">ElfBinaryWriter</param>
        public abstract void WriteElfData(ElfBinaryWriter writer);
    }
}
