﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
namespace Bin2Obj
{
    using System;
    using System.IO;
    using System.Text;
    using System.Collections.Generic;
    using System.Collections;
    using ElfLibrary;

    /// <summary>
    /// Elf形式オブジェクト
    /// </summary>
    public class ElfObject
    {
        /// <summary>
        /// データセクション
        /// </summary>
        private class DataSection
        {
            public int Index { get; set; }
            public byte[] Section { get; set; }
        }

        // Elf Header
        private ElfHeader elfHeader;

        // Section Header
        private List<ElfSectionHeader> sectionList;

        // Section String Section (.shstrtab)
        private byte[] sectionName;

        // Symbol Section (.symtab)
        private int symbolIndex;
        private List<ElfSymbolTableEntry> symbolList;

        // Symbol String Section (.strtab)
        private int symbolNameIndex;
        private byte[] symbolName;

        // Data Section (.data) or (.rodata)
        private List<DataSection> dataList;

        public ElfObject(ElfMachineType machine, ElfByteOrderType byteOrder, ElfArchitectureType architecture)
        {
            sectionName = new byte[0];
            symbolName = new byte[0];
            symbolIndex = 0;
            symbolNameIndex = 0;

            sectionList = new List<ElfSectionHeader>();
            symbolList = new List<ElfSymbolTableEntry>();
            dataList = new List<DataSection>();

            switch (architecture)
            {
                case ElfArchitectureType.Elf32:
                    elfHeader = new ElfLibrary.Elf32.Elf32Header();
                    elfHeader.Initialize((ushort)machine, (byte)byteOrder);
                    break;
                case ElfArchitectureType.Elf64:
                    elfHeader = new ElfLibrary.Elf64.Elf64Header();
                    elfHeader.Initialize((ushort)machine, (byte)byteOrder);
                    break;
                default:
                    throw new ArgumentException();
            }

            checked
            {
                // デフォルトセクションの追加
                // index=0 のセクションはダミー = NULL
                AddSection(null, (uint)ElfSectionAttributeFlags.NULL, 0, 0, 0);
                // index=1 セクション名テーブル
                elfHeader.StringTableIndex = (ushort)AddSection(".shstrtab", ElfSectionType.STRTAB, ElfSectionAttributeFlags.NULL, 0, 0);
                // index=2 シンボル情報構造体
                symbolIndex = AddSection(".symtab", ElfSectionType.SYMTAB, ElfSectionAttributeFlags.NULL, 0, 4);
                // index=3  シンボル名テーブル
                symbolNameIndex = AddSection(".strtab", ElfSectionType.STRTAB, ElfSectionAttributeFlags.NULL, 0, 1);

                if (symbolIndex < 0)
                {
                    throw new InvalidCastException();
                }

                // シンボル名テーブルのインデックスをリンクする
                sectionList[symbolIndex].Link = (uint)symbolNameIndex;

                // グローバルバインドされた最初のシンボルの index を入れる
                // 最初のNULLシンボル以外はグローバルなシンボルなので、1 を入れる
                sectionList[symbolIndex].Info = 1;

                switch (architecture)
                {
                    case ElfArchitectureType.Elf32:
                        sectionList[symbolIndex].EntrySize = ElfLibrary.Elf32.Elf32SymbolTableEntry.Elf32HeaderSize;
                        break;
                    case ElfArchitectureType.Elf64:
                        sectionList[symbolIndex].EntrySize = ElfLibrary.Elf64.Elf64SymbolTableEntry.Elf64HeaderSize;
                        break;
                    default:
                        throw new ArgumentException();
                }

                // シンボルテーブルの初期化
                // index=0 NULLシンボルを登録
                AddSymbol(string.Empty, 0, 0, 0);
            }
        }

        /// <summary>
        /// オブジェクトファイルをファイルに出力する
        /// <param name="fs">出力するファイルストリーム</param>
        /// </summary>
        public void OutputObject(FileStream fs)
        {
            ulong offset = 0;
            int n;

            using (BinaryWriter binaryWriter = new BinaryWriter(fs, System.Text.Encoding.UTF8, true))
            {
                ElfBinaryWriter writer = new ElfBinaryWriter(binaryWriter, elfHeader.ByteOrder);
                // Elf Header
                offset = 0;

                elfHeader.WriteElfData(writer);

                // Section Header
                offset = elfHeader.SectionOffset;
                OutputNull(writer, fs.Length, offset);
                foreach (ElfSectionHeader sectionHeader in sectionList)
                {
                    sectionHeader.WriteElfData(writer);
                }

                // Section: ShStrTab
                n = elfHeader.StringTableIndex;
                offset = sectionList[n].Offset;
                OutputNull(writer, fs.Length, offset);
                writer.WriteBytes(sectionName);

                // Section: Symbol
                n = symbolIndex;
                offset = sectionList[n].Offset;
                OutputNull(writer, fs.Length, offset);
                foreach (ElfSymbolTableEntry symbolEntry in symbolList)
                {
                    symbolEntry.WriteElfData(writer);
                }

                // Section: StrTab
                n = symbolNameIndex;
                offset = sectionList[n].Offset;
                OutputNull(writer, fs.Length, offset);
                writer.WriteBytes(symbolName);

                // Section: Object
                foreach (DataSection dataSection in dataList)
                {
                    n = dataSection.Index;
                    offset = sectionList[n].Offset;
                    OutputNull(writer, fs.Length, offset);
                    writer.WriteBytes(dataSection.Section);
                }
            }
        }

        /// <summary>
        /// データセクションを追加する
        /// </summary>
        /// <param name="param">データセクションのパラメータ</param>
        public void AddDataSection(ElfDataSectionParameter param)
        {
            if (null == param.Data)
            {
                throw new ArgumentNullException();
            }

            DataSection dataSection = new DataSection();
            dataSection.Section = param.Data;

            // データセクションの登録
            dataSection.Index = AddSection(
                (param.IsWritable) ? param.SectionRwData : param.SectionRoData,
                ElfSectionType.PROGBITS,
                (param.IsWritable) ? ElfSectionAttributeFlags.WRITE | ElfSectionAttributeFlags.ALLOC : ElfSectionAttributeFlags.ALLOC,
                (ulong)dataSection.Section.Length,
                param.Align);

            checked
            {
                // シンボルの登録
                string symbolBegin = CreateSymbolString(param.BinaryFileName, param.SymbolBegin);
                string symbolEnd = CreateSymbolString(param.BinaryFileName, param.SymbolEnd);
                AddSymbol(symbolBegin, 0, (uint)dataSection.Section.Length, (byte)dataSection.Index);
                AddSymbol(symbolEnd, (uint)dataSection.Section.Length, 0, (byte)dataSection.Index);
                dataList.Add(dataSection);
            }
        }

        /// <summary>
        /// セクションを追加します
        /// </summary>
        /// <param name="name">シンボル名（もし空文字列なら NULL SECTION にする）</param>
        /// <param name="type">SHT_*</param>
        /// <param name="flags">SHF_*</param>
        /// <param name="size">セクションのサイズ</param>
        /// <param name="align">セクションのアラインメント</param>
        /// <returns>登録したindex</returns>
        private int AddSection(string name, ElfSectionType type, ElfSectionAttributeFlags flags, ulong size, ulong align)
        {
            checked
            {
                ushort n = elfHeader.SectionNumber;
                elfHeader.SectionNumber = (ushort)(n + 1);
                ElfSectionHeader sectionHeader;
                switch (elfHeader.Architecture)
                {
                    case ElfArchitectureType.Elf32:
                        sectionHeader = new ElfLibrary.Elf32.Elf32SectionHeader();
                        break;
                    case ElfArchitectureType.Elf64:
                        sectionHeader = new ElfLibrary.Elf64.Elf64SectionHeader();
                        break;
                    default:
                        throw new InvalidOperationException();
                }
                sectionList.Add(sectionHeader);

                if (string.IsNullOrEmpty(name))
                {
                    AddSectionName(string.Empty);
                    return n;
                }
                else
                {
                    int intName = AddSectionName(name);
                    sectionHeader.Name = (uint)intName;
                    sectionHeader.Type = (uint)type;
                    sectionHeader.Flags = (ulong)flags;
                    sectionHeader.Size = size;
                    sectionHeader.AddressAlign = align;
                    return n;
                }
            }
        }

        /// <summary>
        /// セクション名を追加します。
        /// </summary>
        /// <param name="name">追加するセクション名</param>
        /// <returns>追加したセクション名のセクション名テーブルでの先頭位置</returns>
        private int AddSectionName(string name)
        {
            int pos = CatString(ref sectionName, name);
            if (0 == elfHeader.StringTableIndex)
            {
                return pos;
            }

            sectionList[elfHeader.StringTableIndex].Size = checked((ushort)sectionName.Length);
            return pos;
        }

        /// <summary>
        /// シンボルを追加します
        /// </summary>
        /// <param name="name">シンボル名 (もし空文字列なら NULL SYMBOL にする)</param>
        /// <param name="value">シンボルの値</param>
        /// <param name="size">シンボルのサイズ</param>
        /// <param name="relatedSection">関連セクション</param>
        /// <returns>登録したindex</returns>
        private void AddSymbol(string name, uint value, uint size, byte relatedSection)
        {
            ElfSymbolTableEntry symbolEntry;
            switch (elfHeader.Architecture)
            {
                case ElfArchitectureType.Elf32:
                    symbolEntry = new ElfLibrary.Elf32.Elf32SymbolTableEntry();
                    break;
                case ElfArchitectureType.Elf64:
                    symbolEntry = new ElfLibrary.Elf64.Elf64SymbolTableEntry();
                    break;
                default:
                    throw new InvalidOperationException();
            }

            if (symbolIndex > 0 && symbolIndex < 0xff)
            {
                checked
                {
                    // シンボルテーブル更新
                    sectionList[symbolIndex].Size = (uint)(symbolList.Count + 1) * sectionList[symbolIndex].EntrySize; // すべてのSymbolTableEntryを足し合わせたサイズ
                    symbolList.Add(symbolEntry);

                    if (!string.IsNullOrEmpty(name))
                    {
                        symbolEntry.Name = (uint)AddSymbolName(name);
                        symbolEntry.Value = value;
                        symbolEntry.Size = size;
                        symbolEntry.Info = ElfSymbolTableEntryOperation.MakeInfo(ElfSymbolBinding.GLOBAL, ElfSymbolType.OBJECT);
                        symbolEntry.Other = 0;
                        symbolEntry.SectionIndex = relatedSection;
                    }
                    else
                    {
                        AddSymbolName(string.Empty);
                    }
                }
            }
            else
            {
                Console.Error.WriteLine("Warning: no symbol section. {0} is skipped.\n", name);
            }
        }

        /// <summary>
        /// シンボル名を追加します。
        /// </summary>
        /// <param name="name">シンボル名</param>
        /// <returns>追加したシンボル名のセクション名テーブルでの先頭位置</returns>
        private int AddSymbolName(string name)
        {
            int pos = CatString(ref symbolName, name);
            if (0 == symbolNameIndex)
            {
                return pos;
            }
            sectionList[symbolNameIndex].Size = checked((ushort)symbolName.Length);
            return pos;
        }

        /// <summary>
        /// セクションの位置情報を書き込む
        /// </summary>
        public void MapSection()
        {
            ulong offset = elfHeader.CalcSectionPosition();

            for (int i = 1; i < elfHeader.SectionNumber; i++)
            {
                offset = sectionList[i].SetPosition(offset);
            }
        }

        /// <summary>
        /// 文字列を後ろに追加し、追加した文字列の先頭位置を返す。
        /// </summary>
        /// <param name="baseline">ベースとなる文字列</param>
        /// <param name="addline">追加される文字列</param>
        /// <returns>追加した文字列の位置
        /// Array.Length の型が int なので、それ以上大きな値を保持することは今のところ想定していない。
        /// </returns>
        private int CatString(ref byte[] baseline, string addline)
        {
            if (null == baseline)
            {
                Console.Error.WriteLine("Warning: CatString is skipped, because baseline is null");
                return 0;
            }
            int size = baseline.Length;
            int index = size;
            size += addline.Length + 1; // '\0' のため +1 をする
            byte[] newArray = new byte[size];
            baseline.CopyTo(newArray, 0);
            baseline = new byte[size];
            newArray.CopyTo(baseline, 0);
            if (0 == addline.Length)
            {
                return index;
            }
            char[] charArray = addline.ToCharArray();
            for (int i = 0; i < charArray.Length; i++)
            {
                baseline[index + i] = checked((byte)charArray[i]);
            }
            return index;
        }

        /// <summary>
        /// offsetまでのデータを0で埋めます。
        /// </summary>
        /// <param name="fs">ファイルストリーム</param>
        /// <param name="writer">書き込みインスタンス</param>
        /// <param name="fileLength">ファイルの現在の大きさ</param>
        /// <param name="offset">オフセット</param>
        private void OutputNull(ElfBinaryWriter writer, long fileLength, ulong offset)
        {
            checked
            {
                if (offset < (ulong)fileLength)
                {
                    throw new ArgumentException();
                }

                byte[] zeroArray = new byte[offset - (ulong)fileLength];
                writer.WriteBytes(zeroArray);
            }
        }

        /// <summary>
        /// ファイル名とシンボル名のフォーマット文字列からシンボル名を作成する
        /// </summary>
        /// <param name="filename">ファイル名</param>
        /// <param name="symbolFormat">シンボル名のフォーマット文字列</param>
        /// <returns>シンボル名</returns>
        private string CreateSymbolString(string filename, string symbolFormat)
        {
            string baseName = Path.GetFileNameWithoutExtension(filename);
            string extension = Path.GetExtension(filename);
            string symbolName = symbolFormat;

            // format 指定の中で %f, %b, %e という文字列を対応するファイル名情報に置き換える
            symbolName = symbolName.Replace("%f", baseName + extension);
            symbolName = symbolName.Replace("%t", baseName + extension);
            symbolName = symbolName.Replace("%b", baseName);
            symbolName = symbolName.Replace("%e", extension.Replace(".", string.Empty));

            StringBuilder newSymbolName = new StringBuilder();

            // シンボル名として使用できない文字が含まれているなら、その文字を _ にする
            foreach (char c in symbolName)
            {
                newSymbolName.Append((char.IsLetterOrDigit(c)) ? c : '_');
            }

            // 最初の１文字目が数字の場合は _ にする
            if (char.IsDigit(newSymbolName[0]))
            {
                newSymbolName[0] = '_';
            }

            return newSymbolName.ToString();
        }
    }
}
