﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/gc/gc.h>
#include <nn/gc/detail/gc_Types.h>
#include <nn/util/util_BitPack.h>
#include <msclr/marshal.h>

#include "../Util/DeclareAlive.h"

namespace Nintendo { namespace Authoring { namespace FileSystemMetaLibrary {

using namespace System;
using namespace System::Collections;
using namespace System::Collections::Generic;
using namespace System::IO;
using namespace System::Security::Cryptography;
using namespace System::Runtime::InteropServices;
using namespace msclr::interop;

using namespace nn;

    public ref class XciInfo
    {
    public:
        // サイズ定義
        static const UInt32 PageSize = gc::GcPageSize;
        static const UInt32 CardKeyAreaPageCount = 0x8;
        static const UInt32 CardHeaderPageCount = gc::GcCardHeaderPageCount;
        static const UInt32 ReservedAreaPageCount = gc::GcReservedAreaPageCount;
        static const UInt32 CertAreaStartPageAddress = CardKeyAreaPageCount + CardHeaderPageCount + ReservedAreaPageCount;
        static const UInt32 CertAreaPageCount = gc::GcCertAreaPageCount;
        static const UInt32 NormalAreaStartPageAddress = CardKeyAreaPageCount + CardHeaderPageCount + ReservedAreaPageCount + CertAreaPageCount;

        static const UInt32 RomAreaStartPageCountAlignment = 0x40; // 32 KB

        // 以下、外部入力が必要なゲームカードヘッダのパラメータ
        UInt32 romAreaStartPageAddress; // Secure 領域の開始ページアドレス（KeyArea 分のページを除く）

        Byte kekIndex;
        static const Byte KekIndexVersion0 = 0x0;
        static const Byte KekIndexVersionForDev = 0x1;

        Byte romSize;
        static const Byte RomSize1GB = gc::detail::MemoryCapacity::MemoryCapacity_1GB;
        static const Byte RomSize2GB = gc::detail::MemoryCapacity::MemoryCapacity_2GB;
        static const Byte RomSize4GB = gc::detail::MemoryCapacity::MemoryCapacity_4GB;
        static const Byte RomSize8GB = gc::detail::MemoryCapacity::MemoryCapacity_8GB;
        static const Byte RomSize16GB = gc::detail::MemoryCapacity::MemoryCapacity_16GB;
        static const Byte RomSize32GB = gc::detail::MemoryCapacity::MemoryCapacity_32GB;

        Byte flags;
        static const Byte AutoBootFlag = 0x01;
        static const Byte HistoryEraseFlag = 0x02;
        static const Byte RepairToolFlag = 0x04;

        UInt64 packageId;

        UInt32 validDataEndAddress; // 最終ページアドレス

        array<Byte>^ iv; // カードヘッダ暗号化に使った IV
        static const int IvSize = 16;

        UInt64 partitionFsHeaderAddress;
        UInt64 partitionFsHeaderSize;
        array<Byte>^ partitionFsHeaderHash;
        static const int PartitionFsHeaderHashSize = 32;

        UInt32 selSec;
        static const UInt32 SelSecForT1 = gc::detail::SelectSecurityMode::SelectSecurityMode_T1;
        static const UInt32 SelSecForT2 = gc::detail::SelectSecurityMode::SelectSecurityMode_T2;

        UInt64 fwVersion; // Lotus3 の Fw Version
        static const UInt64 FwVersionForDev = 0x0;
        static const UInt64 FwVersion = 0x1;
        static const UInt64 FwVersionSince4_0_0NUP = 0x2;

        UInt32 accCtrl1;
        static const UInt32 AccCtrl1_25MHz = 0x00A10011;
        static const UInt32 AccCtrl1_50MHz = 0x00A10010; // RomSize が 8 GB から利用可能

        UInt32 fwMode;

        UInt32 cupVersion;
        UInt64 cupId;

        array<Byte>^ uppHash;
        static const int UppHashSize = 8;

        static array<Int32>^ RomSizeTable = { 1, 2, 4, 8, 16, 32 };
        static array<Int32>^ ClockRateTable = { 25, 50 };

        static const Int64 HeaderReservedSize = (Int64)NormalAreaStartPageAddress * PageSize; // 64 KB
        static const Int64 UpdatePartitionMaximumLimitSize = 512 * 1024 * 1024 - HeaderReservedSize; // HeaderReservedSize と併せて 512 MB まで
        static const Int64 ApplicationUsableAreaSizeOnUpdatePartition =
        // 128 * 1024 * 1024; // for NX Addon 0.12.0
           144 * 1024 * 1024; // for NX Addon 3.4.0
        static const Int64 UpdatePartitionLimitSize = UpdatePartitionMaximumLimitSize - ApplicationUsableAreaSizeOnUpdatePartition;
        static const Int64 EmptyUpdatePartitionSize = static_cast<Int64>(PageSize); // 一応 1 ページ分

        static Int32 InvalidRomSize = -1;
        static Int32 InvalidClockRate = -1;

        static Byte ConvertRomSizeToRomSizeByte(Int32 romSize)
        {
            switch (romSize)
            {
            case  1: return RomSize1GB;
            case  2: return RomSize2GB;
            case  4: return RomSize4GB;
            case  8: return RomSize8GB;
            case 16: return RomSize16GB;
            case 32: return RomSize32GB;
            default:
                throw gcnew ArgumentException();
            }
        }

        static UInt32 ConvertClockRateToAccCtrl1(Int32 clockRate)
        {
            switch (clockRate)
            {
            case 25: return AccCtrl1_25MHz;
            case 50: return AccCtrl1_50MHz;
            default:
                throw gcnew ArgumentException();
            }
        }

        static Int32 GetClockRateFromRomSize(Int32 romSize)
        {
            // 4GB 以下は XtraROM の実力に依存し、25 MHz 動作しかできない。
            return romSize >= 8 ? 50 : 25;
        }

        static Int64 GetUnusedAreaSize(Int32 romSize)
        {
            Int64 megaBytes = 1024 * 1024;
            switch (romSize)
            {
            case  1: return   72 * megaBytes;
            case  2: return  144 * megaBytes;
            case  4: return  288 * megaBytes;
            case  8: return  576 * megaBytes;
            case 16: return 1152 * megaBytes;
            case 32: return 2304 * megaBytes;
            default:
                throw gcnew ArgumentException();
            }
        }

        static UInt64 GetFwVersion(UInt32 cupVersion)
        {
            // CupVersion = (NN_FIRMWARE_VERSION_MAJOR << 26 | NN_FIRMWARE_VERSION_MINOR << 20 | NN_FIRMWARE_VERSION_MICRO << 16 | 0);
            const UInt32 CupVersionFor4_0_0NUP = 0x10000000;
            if (cupVersion >= CupVersionFor4_0_0NUP)
            {
                return FwVersionSince4_0_0NUP;
            }
            else
            {
                return FwVersion;
            }
        }
    };

    public ref class XciMeta
    {
    public:
        XciMeta(array<Byte>^ initialData) : m_initialData(initialData)
        {
            GC::KeepAlive(this);
        }

        static Int32 GetIvOffset()
        {
            return offsetof(struct gc::detail::CardHeader, iv[0]);
        }

        static Int32 GetEncryptionTargetOffset()
        {
            return offsetof(struct gc::detail::CardHeader, fwVersion[0]);
        }

        static Int32 GetEncryptionTargetSize()
        {
            return sizeof(gc::detail::CardHeader) - GetEncryptionTargetOffset();
        }

        array<Byte>^ CreateHeader(XciInfo^ xciInfo)
        {
            if (m_initialData == nullptr)
            {
                throw gcnew InvalidOperationException("Invalid initial data.");
            }

            gc::detail::CardHeader xciHeader;
            memset(&xciHeader, 0x0, sizeof(gc::detail::CardHeader));

            // デフォルト値はここで設定
            CopyByteArray(&(xciHeader.magicCode), MagicCode, MagicCode->Length);
            xciHeader.backupAreaStartPageAddress = BackupAreaStartPageAddress;
            xciHeader.version = CardHeaderVersion;
            xciHeader.selT1Key = SelT1Key;
            xciHeader.selKey = SelKey;
            xciHeader.wait1TimeRead = Wait1TimeRead;
            xciHeader.wait2TimeRead = Wait2TimeRead;
            xciHeader.wait1TimeWrite = Wait1TimeWrite;
            xciHeader.wait2TimeWrite = Wait2TimeWrite;

            // パラメータ設定
            xciHeader.romAreaStartPageAddress = xciInfo->romAreaStartPageAddress;

            xciHeader.keyIndex.Set<gc::detail::CardHeaderMap_KeyIndex::kekIndex>(xciInfo->kekIndex);

            xciHeader.romSize = xciInfo->romSize;

            xciHeader.flags = xciInfo->flags;

            auto packageIdByteArray = BitConverter::GetBytes(xciInfo->packageId);
            CopyByteArray(&(xciHeader.packageId), packageIdByteArray, packageIdByteArray->Length);

            xciHeader.validDataEndAddress = xciInfo->validDataEndAddress;

            for (int i = 0; i < xciInfo->iv->Length; i++) // IV はビッグエンディアン
            {
                xciHeader.iv[i] = xciInfo->iv[xciInfo->iv->Length - 1 - i];
            }

            xciHeader.partitionFsHeaderAddress = xciInfo->partitionFsHeaderAddress;
            xciHeader.partitionFsHeaderSize = xciInfo->partitionFsHeaderSize;
            CopyByteArray(&(xciHeader.partitionFsHeaderHash), xciInfo->partitionFsHeaderHash, xciInfo->partitionFsHeaderHash->Length);

            auto initialDataHash = GetInitialDataHash();
            CopyByteArray(&(xciHeader.initialDataHash), initialDataHash, initialDataHash->Length);

            xciHeader.selSec = xciInfo->selSec;

            xciHeader.limArea = xciInfo->romAreaStartPageAddress; // limArea は romAreaStartPageAddress と同値

            auto fwVersionArray = BitConverter::GetBytes(xciInfo->fwVersion);
            CopyByteArray(&(xciHeader.fwVersion), fwVersionArray, fwVersionArray->Length);

            xciHeader.accCtrl1 = xciInfo->accCtrl1;

            xciHeader.fwMode = xciInfo->fwMode;

            xciHeader.cupVersionReserve = xciInfo->cupVersion;
            xciHeader.cupId = xciInfo->cupId;

            CopyByteArray(&(xciHeader.uppHashForLotCheck), xciInfo->uppHash, xciInfo->uppHash->Length);

            array<Byte>^ buf = gcnew array<Byte>(sizeof(gc::detail::CardHeader));
            {
                pin_ptr<unsigned char> ptr = &buf[0];
                memcpy(ptr, &xciHeader, sizeof(gc::detail::CardHeader));
                ptr = nullptr;
            }

            return buf;
        }

        static array<Byte>^ GetRootPartitionHash(array<Byte>^ buffer)
        {
            System::Diagnostics::Debug::Assert(buffer->Length == sizeof(gc::detail::CardHeader));

            gc::detail::CardHeader xciHeader;
            {
                pin_ptr<unsigned char> ptr = &buffer[0];
                memcpy(&xciHeader, ptr, sizeof(gc::detail::CardHeader));
                ptr = nullptr;
            }

            auto bytes = gcnew array<Byte>(XciInfo::PartitionFsHeaderHashSize);
            {
                pin_ptr<unsigned char> ptr = &bytes[0];
                memcpy(ptr, xciHeader.partitionFsHeaderHash, XciInfo::PartitionFsHeaderHashSize);
                ptr = nullptr;
            }

            return bytes;
        }

    internal:
        static void CopyByteArray(void* dst, array<Byte>^ src, int size)
        {
            pin_ptr<unsigned char> ptr = &src[0];
            memcpy(dst, ptr, size);
            ptr = nullptr;
        }

    public:
        static const int TitleKey1Size = 0x8;
        static const int TitleKey2Size = 0x8;

        static const int InitialDataSize = 0x200;
        static const int InitialDataASize = 0x10;
        static const int InitialDataMacSize = 0x10;
        static const int InitialDataNonceSize = 0xC;

        static const int KeyAreaEncryptionTargetOffset = 0x200;
        static const int KeyAreaEncryptionTargetSize = 0xD00;

    private:
        array<Byte>^ GetInitialDataHash()
        {
            if (m_initialData == nullptr)
            {
                throw gcnew InvalidOperationException();
            }
            auto hashCalculator = gcnew SHA256CryptoServiceProvider();
            return hashCalculator->ComputeHash(m_initialData, 0, m_initialData->Length);
        }

        array<Byte>^ m_initialData;

        static array<Byte>^ MagicCode = { 'H', 'E', 'A', 'D' };
        static const uint32_t BackupAreaStartPageAddress = 0xFFFFFFFF;
        static const uint8_t CardHeaderVersion = 0x0;
        static const uint8_t SelT1Key = 0x2;
        static const uint8_t SelKey = 0x0;
        static const uint32_t Wait1TimeRead = 5000;//(200us/40ns)
        static const uint32_t Wait2TimeRead = 0x0;
        static const uint32_t Wait1TimeWrite = 0x0;
        static const uint32_t Wait2TimeWrite = 0x0;
    };

    public ref class XciHeaderReader
    {
    public:
        XciHeaderReader(array<Byte>^ buffer)
        {
            System::Diagnostics::Debug::Assert(buffer->Length == sizeof(gc::detail::CardHeader));
            m_XciHeader = new gc::detail::CardHeader();
            pin_ptr<unsigned char> ptr = &buffer[0];
            memcpy(m_XciHeader, ptr, sizeof(gc::detail::CardHeader));
            ptr = nullptr;
        }

        ~XciHeaderReader()
        {
            this->!XciHeaderReader();
        }

        !XciHeaderReader()
        {
            delete m_XciHeader;
        }

        UInt32 GetRomAreaStartPageAddress()
        {
            return m_XciHeader->romAreaStartPageAddress;
        }

        UInt32 GetBackupAreaStartPageAddress()
        {
            return m_XciHeader->backupAreaStartPageAddress;
        }

        Byte GetKekIndex()
        {
            return m_XciHeader->keyIndex.Get<gc::detail::CardHeaderMap_KeyIndex::kekIndex>();
        }

        Byte GetTitleKeyDecIndex()
        {
            return m_XciHeader->keyIndex.Get<gc::detail::CardHeaderMap_KeyIndex::titleKeyDecIndex>();
        }

        Byte GetRomSize()
        {
            return m_XciHeader->romSize;
        }

        Byte GetCardHeaderVersion()
        {
            return m_XciHeader->version;
        }

        Byte GetFlags()
        {
            return m_XciHeader->flags;
        }

        UInt64 GetPackageId()
        {
            auto value = m_XciHeader->packageId;
            return BitConverter::ToUInt64(CopyToByte(value, 8), 0);
        }

        UInt32 GetValidDataEndAddress()
        {
            return m_XciHeader->validDataEndAddress;
        }

        array<Byte>^ GetIv()
        {
            auto value = m_XciHeader->iv;
            return CopyToByte(value, 0x10);
        }

        UInt64 GetPartitionFsHeaderAddress()
        {
            return m_XciHeader->partitionFsHeaderAddress;
        }

        UInt64 GetPartitionFsHeaderSize()
        {
            return m_XciHeader->partitionFsHeaderSize;
        }

        array<Byte>^ GetPartitionFsHeaderHash()
        {
            auto value = m_XciHeader->partitionFsHeaderHash;
            return CopyToByte(value, 0x20);
        }

        array<Byte>^ GetInitialDataHash()
        {
            auto value = m_XciHeader->initialDataHash;
            return CopyToByte(value, 0x20);
        }

        UInt32 GetSelSec()
        {
            return m_XciHeader->selSec;
        }

        UInt32 GetSelT1Key()
        {
            return m_XciHeader->selT1Key;
        }

        UInt32 GetSelKey()
        {
            return m_XciHeader->selKey;
        }

        UInt32 GetLimArea()
        {
            return m_XciHeader->limArea;
        }

        UInt64 GetFwVersion()
        {
            auto value = m_XciHeader->fwVersion;
            return BitConverter::ToUInt64(CopyToByte(value, 8), 0);
        }

        UInt32 GetAccCtrl1()
        {
            return m_XciHeader->accCtrl1;
        }

        UInt32 GetWait1TimeRead()
        {
            return m_XciHeader->wait1TimeRead;
        }

        UInt32 GetWait2TimeRead()
        {
            return m_XciHeader->wait2TimeRead;
        }

        UInt32 GetWait1TimeWrite()
        {
            return m_XciHeader->wait1TimeWrite;
        }

        UInt32 GetWait2TimeWrite()
        {
            return m_XciHeader->wait2TimeWrite;
        }

        UInt32 GetFwMode()
        {
            return m_XciHeader->fwMode;
        }

        UInt32 GetUppVersion()
        {
            return m_XciHeader->cupVersionReserve;
        }

        array<Byte>^ GetUppHash()
        {
            auto value = m_XciHeader->uppHashForLotCheck;
            return CopyToByte(value, 8);
        }

        UInt64 GetUppId()
        {
            return m_XciHeader->cupId;
        }

    private:
        array<Byte>^ CopyToByte(void* src, size_t size)
        {
            auto dst = gcnew array<Byte>(size);
            {
                pin_ptr<unsigned char> ptr = &dst[0];
                memcpy(ptr, src, size);
                ptr = nullptr;
            }
            return dst;
        }

        gc::detail::CardHeader* m_XciHeader;
    };

}}}
