﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <string>
#include <nn/nn_Result.h>
#include <nn/result/result_HandlingUtility.h>

#include <nn/fs/fs_File.h>
#include <nn/fs/fs_RomFsFileSystem.h>

#include "../Util/DeclareAlive.h"
#include "FileSystemArchiveReaderImplBase.h"
#include "RomFsFileSystemArchiveReader.h"
#include "ManagedStreamStorage.h"
#include "BufferPool.h"

namespace Nintendo { namespace Authoring { namespace FileSystemMetaLibrary {

using namespace System;
using namespace System::IO;
using namespace System::Collections;
using namespace System::Collections::Generic;
using namespace System::Runtime::InteropServices;

using namespace nn;


    private class RomFsFileSystemArchiveReaderImpl : public FileSystemArchiveReaderImplBase
    {
    public:
        RomFsFileSystemArchiveReaderImpl(std::shared_ptr<fs::IStorage> storage)
            : FileSystemArchiveReaderImplBase(std::move(storage))
            , m_FileSystem(std::make_shared<nn::fs::RomFsFileSystem>())
        {
            EnsureBufferPool();

            auto result = m_FileSystem->Initialize(m_Storage.get(), nullptr, 0, false); // TORIAEZU
            if(result.IsFailure())
            {
                throw gcnew ArgumentException(String::Format("Failed to Initialize RomFsFileSystem 0x{0:X8}.", result.GetInnerValueForDebug()));
            }
            FileSystemArchiveReaderImplBase::Initialize(m_FileSystem.get());
        }

        List<Tuple<Int64, Int64>^>^ GetFileFragmentList(String^ fileName)
        {
            auto name = static_cast<char*>(GetUtf8CharsFromString(fileName).ToPointer());
            NN_UTIL_SCOPE_EXIT{ Marshal::FreeHGlobal(static_cast<IntPtr>(name)); };

            // パッチ対応で断片化するようになったら要修正
            int64_t offset;
            auto result = m_FileSystem->GetFileBaseOffset(&offset, (std::string("/") + name).c_str());
            if (result.IsFailure())
            {
                throw gcnew ArgumentException(String::Format("Failed to OpenFileStorage 0x{0:X8}.", result.GetInnerValueForDebug()));
            }

            auto fragmentList = gcnew List<Tuple<Int64, Int64>^>();
            fragmentList->Add(gcnew Tuple<Int64, Int64>(offset, this->GetFileSize(fileName)));
            return fragmentList;
        }

        Int64 GetFileOffset(String^ fileName)
        {
            auto name = static_cast<char*>(GetUtf8CharsFromString(fileName).ToPointer());
            NN_UTIL_SCOPE_EXIT{ Marshal::FreeHGlobal(static_cast<IntPtr>(name)); };

            int64_t offset = 0;
            auto result = m_FileSystem->GetFileBaseOffset(&offset, (std::string("/") + name).c_str());
            if (result.IsFailure())
            {
                throw gcnew ArgumentException(String::Format("Failed to GetFileBaseOffset 0x{0:X8}.", result.GetInnerValueForDebug()));
            }

            return offset;
        }
    private:
        std::shared_ptr<nn::fs::RomFsFileSystem> m_FileSystem;
    };

    RomFsFileSystemArchiveReader::RomFsFileSystemArchiveReader(Stream^ stream)
    {
        auto storage = std::shared_ptr<fs::IStorage>(new ManagedStreamStorage(stream));
        m_Impl = new RomFsFileSystemArchiveReaderImpl(storage);
        GC::KeepAlive(this);
    }

    RomFsFileSystemArchiveReader::~RomFsFileSystemArchiveReader()
    {
        this->!RomFsFileSystemArchiveReader();
    }

    RomFsFileSystemArchiveReader::!RomFsFileSystemArchiveReader()
    {
        delete m_Impl;
    }

    List<Tuple<String^, Int64>^>^ RomFsFileSystemArchiveReader::ListFileInfo()
    {
        return Util::ReturnAndDeclareAlive(this, m_Impl->ListFileInfo());
    }

    List<Tuple<String^, Int64>^>^ RomFsFileSystemArchiveReader::ListFileInfo(String^ rootPath)
    {
        return Util::ReturnAndDeclareAlive(this, m_Impl->ListFileInfo(rootPath));
    }

    List<Tuple<Int64, Int64>^>^ RomFsFileSystemArchiveReader::GetFileFragmentList(String^ fileName)
    {
        return Util::ReturnAndDeclareAlive(this, m_Impl->GetFileFragmentList(fileName));
    }

    int64_t RomFsFileSystemArchiveReader::GetFileSize(String^ fileName)
    {
        return Util::ReturnAndDeclareAlive(this, m_Impl->GetFileSize(fileName));
    }

    int64_t RomFsFileSystemArchiveReader::GetFileOffset(String^ fileName)
    {
        return Util::ReturnAndDeclareAlive(this, m_Impl->GetFileOffset(fileName));
    }

    array<byte>^ RomFsFileSystemArchiveReader::ReadFile(String^ fileName, int64_t offset, int64_t size)
    {
        return Util::ReturnAndDeclareAlive(this, m_Impl->ReadFile(fileName, offset, size));
    }

    int64_t RomFsFileSystemArchiveReader::GetBaseSize()
    {
        return Util::ReturnAndDeclareAlive(this, m_Impl->GetBaseSize());
    }

    array<byte>^ RomFsFileSystemArchiveReader::ReadBase(int64_t offset, int64_t size)
    {
        return Util::ReturnAndDeclareAlive(this, m_Impl->ReadBase(offset, size));
    }

    RomFsFileSystemArchiveReader::RomFsFileSystemArchiveReader(std::shared_ptr<fs::IStorage> storage)
    {
        m_Impl = new RomFsFileSystemArchiveReaderImpl(storage);
        GC::KeepAlive(this);
    }

}}}
