﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <string>
#include <msclr/marshal.h>
#include <nn/nn_Result.h>
#include <nn/nn_Abort.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/fs/fs_IStorage.h>
#include <nn/fs/fs_FileStorage.h>
#include <nn/fssystem/fs_PartitionFileSystem.h>
#include <nn/fssystem/fs_AsynchronousAccess.h>

#include "../Util/DeclareAlive.h"
#include "FileSystemArchiveReaderImplBase.h"
#include "PartitionFileSystemArchiveReader.h"
#include "ManagedStreamStorage.h"
#include "BufferPool.h"

namespace Nintendo { namespace Authoring { namespace FileSystemMetaLibrary {

using namespace System;
using namespace System::IO;
using namespace System::Collections;
using namespace System::Collections::Generic;
using namespace System::Runtime::InteropServices;

using namespace nn;
using namespace nn::fs;

    private class PartitionFileSystemArchiveReaderImpl : public FileSystemArchiveReaderImplBase
    {
    public:
        PartitionFileSystemArchiveReaderImpl(std::shared_ptr<fs::IStorage> storage)
            : FileSystemArchiveReaderImplBase(std::move(storage))
            , m_FileSystem(std::make_shared<PartitionFileSystem>())
        {
            EnsureBufferPool();

            auto result = m_FileSystem->Initialize(m_Storage);
            if(result.IsFailure())
            {
                throw gcnew ArgumentException(String::Format("Failed to Initialize PartitionFileSystem 0x{0:X8}.", result.GetInnerValueForDebug()));
            }
            FileSystemArchiveReaderImplBase::Initialize(m_FileSystem.get());
        }

        List<Tuple<Int64, Int64>^>^ GetFileFragmentList(String^ fileName)
        {
            auto name = static_cast<char*>(GetUtf8CharsFromString(fileName).ToPointer());
            NN_UTIL_SCOPE_EXIT{ Marshal::FreeHGlobal(static_cast<IntPtr>(name)); };

            // partitionFs は断片化しない
            int64_t offset;
            auto result = m_FileSystem->GetFileBaseOffset(&offset, (std::string("/") + name).c_str());
            if (result.IsFailure())
            {
                throw gcnew ArgumentException(String::Format("Failed to OpenFileStorage 0x{0:X8}.", result.GetInnerValueForDebug()));
            }

            auto fragmentList = gcnew List<Tuple<Int64, Int64>^>();
            fragmentList->Add(gcnew Tuple<Int64, Int64>(offset, this->GetFileSize(fileName)));
            return fragmentList;
        }

        void OpenFileStorage(std::shared_ptr<fs::IStorage>* outValue, String^ fileName)
        {
            auto name = static_cast<char*>(GetUtf8CharsFromString(fileName).ToPointer());
            NN_UTIL_SCOPE_EXIT{ Marshal::FreeHGlobal(static_cast<IntPtr>(name)); };

            std::shared_ptr<FileStorageBasedFileSystem> fileStorage = std::make_shared<FileStorageBasedFileSystem>();
            auto copyFileSystem = m_FileSystem;
            auto result = fileStorage->Initialize(std::move(copyFileSystem), (std::string("/") + name).c_str(), nn::fs::OpenMode_Read);
            if (result.IsFailure())
            {
                throw gcnew ArgumentException(String::Format("Failed to OpenFileStorage 0x{0:X8}.", result.GetInnerValueForDebug()));
            }

            *outValue = std::move(fileStorage);
        }

    private:
        std::shared_ptr<PartitionFileSystem> m_FileSystem;
    };

    PartitionFileSystemArchiveReader::PartitionFileSystemArchiveReader(Stream^ stream)
    {
        auto storage = std::shared_ptr<fs::IStorage>(new ManagedStreamStorage(stream));
        m_Impl = new PartitionFileSystemArchiveReaderImpl(storage);
        GC::KeepAlive(this);
    }

    PartitionFileSystemArchiveReader::~PartitionFileSystemArchiveReader()
    {
        this->!PartitionFileSystemArchiveReader();
    }

    PartitionFileSystemArchiveReader::!PartitionFileSystemArchiveReader()
    {
        delete m_Impl;
    }

    List<Tuple<String^, Int64>^>^ PartitionFileSystemArchiveReader::ListFileInfo()
    {
        return Util::ReturnAndDeclareAlive(this, m_Impl->ListFileInfo());
    }

    List<Tuple<String^, Int64>^>^ PartitionFileSystemArchiveReader::ListFileInfo(String^ rootPath)
    {
        return Util::ReturnAndDeclareAlive(this, m_Impl->ListFileInfo(rootPath));
    }

    List<Tuple<Int64, Int64>^>^ PartitionFileSystemArchiveReader::GetFileFragmentList(String^ fileName)
    {
        return Util::ReturnAndDeclareAlive(this, m_Impl->GetFileFragmentList(fileName));
    }

    int64_t PartitionFileSystemArchiveReader::GetFileSize(String^ fileName)
    {
        return Util::ReturnAndDeclareAlive(this, m_Impl->GetFileSize(fileName));
    }

    array<byte>^ PartitionFileSystemArchiveReader::ReadFile(String^ fileName, int64_t offset, int64_t size)
    {
        return Util::ReturnAndDeclareAlive(this, m_Impl->ReadFile(fileName, offset, size));
    }

    int64_t PartitionFileSystemArchiveReader::GetBaseSize()
    {
        return Util::ReturnAndDeclareAlive(this, m_Impl->GetBaseSize());
    }

    array<byte>^ PartitionFileSystemArchiveReader::ReadBase(int64_t offset, int64_t size)
    {
        return Util::ReturnAndDeclareAlive(this, m_Impl->ReadBase(offset, size));
    }

    PartitionFileSystemArchiveReader::PartitionFileSystemArchiveReader(std::shared_ptr<fs::IStorage> storage)
    {
        m_Impl = new PartitionFileSystemArchiveReaderImpl(storage);
        GC::KeepAlive(this);
    }

    void PartitionFileSystemArchiveReader::OpenFileStorage(std::shared_ptr<fs::IStorage>* outValue, String^ fileName)
    {
        m_Impl->OpenFileStorage(outValue, fileName);
        GC::KeepAlive(this);
    }

}}}
