﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/escore/escore.h>
#include <nn/escore/estypes.h>
#include <nn/publish/publish.h>
#include <nn/istorage/storage.h>

namespace Nintendo { namespace Authoring { namespace ETicketLibrary {

    using namespace System;

    public ref class TicketPublication {
    public:
        // 元のライブラリのデータ型
        // ticketBuf: u8[1024](Max), ticketLen: u32, titleKey: IOSCAesKey=u8[16], deviceId: ESDeviceId=u64, ticketId: ESTitleId=u64, commonKeyId: u8, rightId: u8[16]
        static UInt32 PublishTicket(array<Byte>^ ticketBuffer, UInt32 ticketLength, array<Byte>^ titleKey, UInt64 deviceId, UInt64 ticketId, array<Byte>^ rightsId, Byte commonKeyId, array<Byte>^ issuerName) {

            pin_ptr<unsigned char> pinTicketBuffer = &ticketBuffer[0];
            pin_ptr<unsigned char> pinTitleKey = &titleKey[0];
            pin_ptr<unsigned char> pinRightsId = &rightsId[0];
            pin_ptr<unsigned char> pinIssuerName = &issuerName[0];

            bcc::publish::ESError rv = ES_ERR_OK;
            bcc::publish::PublishTicket ticket;

            if (ticket.SetTitleKey(pinTitleKey) != ES_ERR_OK){
                throw gcnew ApplicationException;
            }

            if (ticket.SetDeviceId(deviceId) != ES_ERR_OK) {
                throw gcnew ApplicationException;
            }

            if (ticket.SetTicketId(ticketId) != ES_ERR_OK) {
                throw gcnew ApplicationException;
            }

            if (ticket.SetRightsId(pinRightsId) != ES_ERR_OK) {
                throw gcnew ApplicationException;
            }

            if (ticket.SetCommonKeyId(commonKeyId) != ES_ERR_OK) {
                throw gcnew ApplicationException;
            }

            if (ticket.SetIssuer(pinIssuerName) != ES_ERR_OK) {
                throw gcnew ApplicationException;
            }

            if (ticket.OutputTicket(pinTicketBuffer, ticketLength) != ES_ERR_OK) {
                throw gcnew ApplicationException;
            }

            UInt32 length;
            if (ticket.GetTicketLen(&length) != ES_ERR_OK) {
                throw gcnew ApplicationException;
            }

            return length;
        }

        static void SignTicket(array<Byte>^ ticketBuffer, UInt32 ticketLength, array<Byte>^ signatureBuffer, Int32 signatureLength)
        {
            pin_ptr<unsigned char> pinTicketBuffer = &ticketBuffer[0];
            pin_ptr<unsigned char> pinSignatureBuffer = &signatureBuffer[0];

            bcc::publish::ESError rv = ES_ERR_OK;
            bcc::publish::PublishTicket ticket(pinTicketBuffer, ticketLength);

            if (ticket.SetSignData(pinSignatureBuffer) != ES_ERR_OK)
            {
                throw gcnew ApplicationException;
            }

            if (ticket.OutputTicket(pinTicketBuffer, ticketLength) != ES_ERR_OK) {
                throw gcnew ApplicationException;
            }
        }

        static int GetTicketDataForSign(array<Byte>^ outTicketDataBuffer, UInt32 ticketDataLength, array<Byte>^ ticketBuffer, UInt32 ticketLength)
        {
            pin_ptr<unsigned char> pinOutTicketDataBuffer = &outTicketDataBuffer[0];
            pin_ptr<unsigned char> pinTicketBuffer = &ticketBuffer[0];

            bcc::publish::ESError rv = ES_ERR_OK;
            bcc::publish::PublishTicket ticket(pinTicketBuffer, ticketLength);

            if (ticket.GetTicketDataForSign(pinOutTicketDataBuffer, ticketDataLength) != ES_ERR_OK)
            {
                throw gcnew ApplicationException;
            }

            UInt32 length;
            if (ticket.GetTicketDataForSignLen(&length) != ES_ERR_OK) {
                throw gcnew ApplicationException;
            }

            return length;
        }
    };

    public ref class TicketDataReader {
    public:
        TicketDataReader(array<Byte>^ ticketDataBuffer, UInt32 ticketLength) : m_TicketDataBuffer(ticketDataBuffer), m_TicketLength(ticketLength) { }

        array<Byte>^ GetTitleKey()
        {
            pin_ptr<unsigned char> pinTicketDataBuffer = &m_TicketDataBuffer[0];

            nn::escore::MemoryInputStream ticketReader(pinTicketDataBuffer, static_cast<unsigned int>(m_TicketLength));

            bcc::publish::ESError rv = ES_ERR_OK;
            nn::escore::ETicket ticket;

            if (ticket.Set(ticketReader, nullptr, 0, false) != ES_ERR_OK)
            {
                throw gcnew ApplicationException;
            }

            nn::escore::ESTitleKeyType esTitleKeyType;
            if (ticket.GetTitleKeyType(&esTitleKeyType) != ES_ERR_OK)
            {
                throw gcnew ApplicationException;
            }

            nn::escore::ESTitleKey esTitleKey;
            if (ticket.GetTitleKey(&esTitleKey) != ES_ERR_OK)
            {
                throw gcnew ApplicationException;
            }

            if (esTitleKeyType == ES_TITLE_KEY_TYPE_AES)
            {
                auto titleKey = gcnew array<Byte>(sizeof(nn::escore::ESTitleKey::aesKey));
                pin_ptr<unsigned char> pinTitleKey = &titleKey[0];

                memcpy(pinTitleKey, &esTitleKey, sizeof(nn::escore::ESTitleKey::aesKey));

                return titleKey;
            }
            else if (esTitleKeyType == ES_TITLE_KEY_TYPE_RSA2048)
            {
                auto titleKey = gcnew array<Byte>(sizeof(nn::escore::ESTitleKey::rsaKey));
                pin_ptr<unsigned char> pinTitleKey = &titleKey[0];

                memcpy(pinTitleKey, &esTitleKey, sizeof(nn::escore::ESTitleKey::rsaKey));

                return titleKey;
            }
            else
            {
                throw gcnew ApplicationException;
            }
        }

        UInt64 GetDeviceId()
        {
            pin_ptr<unsigned char> pinTicketDataBuffer = &m_TicketDataBuffer[0];

            nn::escore::MemoryInputStream ticketReader(pinTicketDataBuffer, static_cast<unsigned int>(m_TicketLength));

            bcc::publish::ESError rv = ES_ERR_OK;
            nn::escore::ETicket ticket;

            if (ticket.Set(ticketReader, nullptr, 0, false) != ES_ERR_OK)
            {
                throw gcnew ApplicationException;
            }

            nn::escore::ESDeviceId deviceId;
            if (ticket.GetDeviceId(&deviceId) != ES_ERR_OK)
            {
                throw gcnew ApplicationException;
            }

            return static_cast<UInt64>(deviceId);
        }

        UInt64 GetTicketId()
        {
            pin_ptr<unsigned char> pinTicketDataBuffer = &m_TicketDataBuffer[0];

            nn::escore::MemoryInputStream ticketReader(pinTicketDataBuffer, static_cast<unsigned int>(m_TicketLength));

            bcc::publish::ESError rv = ES_ERR_OK;
            nn::escore::ETicket ticket;

            if (ticket.Set(ticketReader, nullptr, 0, false) != ES_ERR_OK)
            {
                throw gcnew ApplicationException;
            }

            nn::escore::ESTicketId ticketId;
            if (ticket.GetTicketId(&ticketId) != ES_ERR_OK)
            {
                throw gcnew ApplicationException;
            }

            return static_cast<UInt64>(ticketId);
        }

        array<Byte>^ GetRightsId()
        {
            pin_ptr<unsigned char> pinTicketDataBuffer = &m_TicketDataBuffer[0];

            auto rightsId = gcnew array<Byte>(sizeof(nn::escore::ESRightsId));
            pin_ptr<unsigned char> pinRightsId = &rightsId[0];

            nn::escore::MemoryInputStream ticketReader(pinTicketDataBuffer, static_cast<unsigned int>(m_TicketLength));

            bcc::publish::ESError rv = ES_ERR_OK;
            nn::escore::ETicket ticket;

            if (ticket.Set(ticketReader, nullptr, 0, false) != ES_ERR_OK)
            {
                throw gcnew ApplicationException;
            }

            nn::escore::ESRightsId esRightsId;
            if (ticket.GetRightsId(&esRightsId) != ES_ERR_OK)
            {
                throw gcnew ApplicationException;
            }

            memcpy(pinRightsId, &esRightsId, sizeof(nn::escore::ESRightsId));

            return rightsId;
        }

        Byte GetCommonKeyId()
        {
            pin_ptr<unsigned char> pinTicketDataBuffer = &m_TicketDataBuffer[0];

            nn::escore::MemoryInputStream ticketReader(pinTicketDataBuffer, static_cast<unsigned int>(m_TicketLength));

            bcc::publish::ESError rv = ES_ERR_OK;
            nn::escore::ETicket ticket;

            if (ticket.Set(ticketReader, nullptr, 0, false) != ES_ERR_OK)
            {
                throw gcnew ApplicationException;
            }

            u8 commonKeyId;
            if (ticket.GetCommonKeyId(&commonKeyId) != ES_ERR_OK)
            {
                throw gcnew ApplicationException;
            }

            return static_cast<Byte>(commonKeyId);
        }

    private:
        array<Byte>^ m_TicketDataBuffer;
        UInt32 m_TicketLength;
    };
}}}
