﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Globalization;
using System.IO;
using System.Linq;
using System.Text;
using System.Text.RegularExpressions;
using System.Threading;
using System.Xml.Serialization;
using Nintendo.ApplicationControlProperty;
using Nintendo.Authoring.FileSystemMetaLibrary;

namespace Nintendo.Authoring.AuthoringLibrary
{
    /// <summary>
    /// UnpublishableErrorを取得するクラスです
    /// </summary>
    public class UnpublishableError
    {
        static public UnpublishableErrorModel GetList(UnpublishableErrorCheckData checkData)
        {
            List<UnpublishableErrorDetailModel> errorDetailModelList =
                UnpublishableErrorCheck.GetUnpublishableErrorCheck(checkData);

            if (errorDetailModelList.Count() == 0)
            {
                // エラーが発生していない場合は、
                // xml出力時に<Error />を表示する為に、エラー要因無しのモデルを1つ追加する。
                errorDetailModelList.Add(UnpublishableErrorDetailModel.Create(null, null));
            }

            // 収集したエラー情報を元に UnpublishableError のXMLモデルを作成する
            var id = string.Empty;
            var type = string.Empty;
            var aocIndex = string.Empty;

            if (checkData.Nsp.ContentMeta != null)
            {
                // チェック対象がnsp
                id = checkData.Nsp.ContentMeta.Id;
                type = checkData.Nsp.ContentMeta.Type;
                if (type == NintendoContentMetaConstant.ContentMetaTypeAddOnContent)
                {
                    aocIndex = (checkData.Nsp.ContentMeta as AddOnContentContentMetaModel).Index.ToString();
                }
            }
            else if (checkData.Nmeta.AddOnContent != null)
            {
                // チェック対象がnmeta(Aoc)
                id = checkData.Nmeta.AddOnContent.Id;
                type = NintendoContentMetaConstant.ContentMetaTypeAddOnContent;
                aocIndex = checkData.Nmeta.AddOnContent.Index.ToString();
                if (!string.IsNullOrEmpty(id) && !string.IsNullOrEmpty(checkData.Nmeta.AddOnContent.ApplicationId))
                {
                    aocIndex = (Convert.ToUInt64(id, 16) - IdConverter.ConvertToAocBaseId(Convert.ToUInt64(checkData.Nmeta.AddOnContent.ApplicationId, 16))).ToString();
                }
            }
            else if (checkData.Nsp.MultiApplicationCardInfo != null)
            {
                // チェック対象がマルチアプリケーションカードのnsp
                id = checkData.Nsp.MultiApplicationCardInfo.Id;
                type = NintendoContentMetaConstant.ContentMetaTypeApplication;
            }

            var programIndex = checkData.ApplicationControlProperty?.ProgramIndex;

            List<UnpublishableErrorContentModel> contents =
                new List<UnpublishableErrorContentModel>();
            contents.Add(UnpublishableErrorContentModel.Create(
                id, type, programIndex, aocIndex, errorDetailModelList));
            UnpublishableErrorModel unpublishError = UnpublishableErrorModel.Create();
            unpublishError.Contents.AddRange(contents);

            return unpublishError;
        }

        static public void GetUnpublishableErrorCount(out int outErrorCount, out int outWarningCount, UnpublishableErrorModel model)
        {
            outErrorCount = 0;
            outWarningCount = 0;
            if (model == null)
            {
                return;
            }
            foreach (var content in model.Contents)
            {
                outErrorCount += content.ErrorList.FindAll(e => e.Type == UnpublishableErrorCheck.ErrorType.Error.ToString()).ToList().Count();
                outWarningCount += content.ErrorList.FindAll(e => e.Type == UnpublishableErrorCheck.ErrorType.Warning.ToString()).ToList().Count();
            }

            return;
        }

        public static void VerifyNmetaUnpublishableError(UnpublishableErrorCheckData checkData, UnpublishableErrorCheck.CheckItemId[] checkList)
        {
            VerifyNmetaUnpublishableError(new List<UnpublishableErrorCheckData>() { checkData }, checkList);
        }

        public static void VerifyNmetaUnpublishableError(List<UnpublishableErrorCheckData> checkDataList, UnpublishableErrorCheck.CheckItemId[] checkList)
        {
            // --error-unpublishable の有無に関係なく実施するチェック項目のリストを取得する
            var mandatoryList = UnpublishableErrorCheck.GetMandatoryCheckList(checkList);
            // 無視するチェック項目をファイルから取得する
            var ignoreList = UnpublishableErrorCheck.ReadCheckList(GlobalSettings.IgnoreErrorUnpublishable);

            UnpublishableErrorModel unpublishableErrorModel = UnpublishableErrorModel.Create();
            foreach (var checkData in checkDataList)
            {
                checkData.CheckTarget = UnpublishableErrorCheckData.CheckTargetType.Nmeta;
                // 指定した項目をアプリケーションタイプに関係なくチェックする
                checkData.CheckContentType = UnpublishableErrorCheckData.CheckContentTypeForce;

                checkData.CheckInfo.CheckList = checkList;
                checkData.CheckInfo.IgnoreList = ignoreList;
                // 指定されたチェック対象に対して実施可能なUnpublishableErrorチェックを実施して結果を取得
                unpublishableErrorModel.Contents.AddRange(UnpublishableError.GetList(checkData).Contents);
            }

            if (!GlobalSettings.ErrorUnpublishable)
            {
                // --error-unpublishableが指定されていない場合はWarning項目を削除する
                UnpublishableErrorCheck.DeleteUnpublishableErrorType(unpublishableErrorModel,
                                                                     UnpublishableErrorCheck.ErrorType.Warning);
                // --error-unpublishableが指定されていない場合はMandatoryCheckList以外のチェック項目はErrorをWarning扱いにする
                UnpublishableErrorCheck.ConvertUnpublishableErrorType(unpublishableErrorModel,
                                                                      UnpublishableErrorCheck.ErrorType.Error,
                                                                      UnpublishableErrorCheck.ErrorType.Warning,
                                                                      mandatoryList);
            }

            // UnpublishableErrorエラーの発生数を取得する
            int errorCount = 0;
            int warningCount = 0;
            UnpublishableError.GetUnpublishableErrorCount(out errorCount, out warningCount, unpublishableErrorModel);

            // UnpublishableErrorエラー情報を表示する
            if (errorCount + warningCount > 0)
            {
                UnpublishableError.ShowUnpublishableError(unpublishableErrorModel, GlobalSettings.ShowUnpublishableErroFormatXml);
            }

            if (GlobalSettings.IgnoreErrorUnpublishable == null && errorCount > 0)
            {
                throw new ArgumentException("Found one or more unpublishable error in nmeta.");
            }
        }

        public static void ShowUnpublishableError(UnpublishableErrorModel unpublishableError, bool showXml)
        {
            if (unpublishableError.Contents.Where(entry => !string.IsNullOrEmpty(entry.ProgramIndex) && Convert.ToByte(entry.ProgramIndex) != 0).Count() == 0)
            {
                // ProgramIndex=0 以外が存在しない場合はProgramIndexを表示させないようにするためにnullを設定
                unpublishableError.Contents.ForEach(entry => entry.ProgramIndex = null);
            }
            if (showXml == true)
            {
                // エラー情報をXML形式で表示
                ShowUnpublishableErrorWithXml(unpublishableError);
            }
            else
            {
                // エラー情報をText形式で表示
                ShowUnpublishableErrorWithText(unpublishableError);
            }
        }

        private static void ShowUnpublishableErrorWithXml(UnpublishableErrorModel unpublishableError)
        {
            var nameSpace = new XmlSerializerNamespaces();
            nameSpace.Add(string.Empty, string.Empty);

            // xmlをUTF-8で出力する為にConsoleのエンコードをUTF8にする
            var originalEncoding = Console.OutputEncoding;
            Console.OutputEncoding = Encoding.UTF8;

            var sw = Console.Out;
            var serializer = new XmlSerializer(typeof(UnpublishableErrorModel));
            serializer.Serialize(sw, unpublishableError, nameSpace);
            Console.WriteLine();

            Console.OutputEncoding = originalEncoding;
        }
        private static void ShowUnpublishableErrorWithText(UnpublishableErrorModel unpublishableError)
        {
            // 現在のスレッドのカルチャを取得
            CultureInfo currentCultureUi = Thread.CurrentThread.CurrentUICulture;

            const int IndentSize = 2;
            const int FirstIndentCount = 2;
            int indentCount = 0;
            Console.WriteLine("{0}UnpublishableError:", new string(' ', indentCount * IndentSize));

            foreach (var content in unpublishableError.Contents)
            {
                // Content情報の表示
                indentCount = FirstIndentCount;
                if (content.ProgramIndex != null)
                {
                    // ProgramIndexを表示
                    Console.WriteLine("{0}ProgramIndex:{1}",
                        new string(' ', indentCount * IndentSize), content.ProgramIndex);
                }
                if (content.Type == NintendoContentMetaConstant.ContentMetaTypeAddOnContent)
                {
                    // Aocのインデックスを表示
                    Console.WriteLine("{0}Index:{1}",
                        new string(' ', indentCount * IndentSize), content.AocIndex);
                }
                int errorCount = 0;
                foreach (var error in content.ErrorList)
                {
                    if (error.Type == null)
                    {
                        continue;
                    }
                    // Content情報の表示
                    string title = string.Empty;
                    string description = string.Empty;
                    if (currentCultureUi.TwoLetterISOLanguageName == "ja")
                    {
                        title = error.Message.Title.Japanese;
                        description = error.Message.Description.Japanese;
                    }
                    else
                    {
                        title = error.Message.Title.English;
                        description = error.Message.Description.English;
                    }

                    indentCount++;
                    Console.WriteLine("{0}[{1}] ({2}) {3}",
                        new string(' ', indentCount * IndentSize),
                        error.Type, error.Message.Id, title);
                    {
                        indentCount++;
                        // 改行がある場合に、改行後にインデントを入れる
                        Regex replaceRegex = new Regex(@"([\r\n]+)");
                        description = replaceRegex.Replace(description, @"$1{0}");
                        description = string.Format(description,
                            new string(' ', indentCount * IndentSize));

                        Console.WriteLine("{0}{1}",
                            new string(' ', indentCount * IndentSize),
                            description);
                        indentCount--;
                    }
                    indentCount--;
                    errorCount++;
                }
                if (errorCount == 0)
                {
                    // エラーが発生しなかった場合は、その旨を表示
                    indentCount++;
                    Console.WriteLine("{0}{1}",
                        new string(' ', indentCount * IndentSize),
                        AuthoringTool.Resources.ResourceManager.GetString("Result_ValidContent"));
                    indentCount--;
                    continue;
                }
            }
        }
    }
}
