﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.IO;

namespace Nintendo.Authoring.AuthoringLibrary
{
    /// <summary>
    /// Stream を指定したオフセットと長さでを切り出して扱うクラスです。
    /// </summary>
    public class SubStream : Stream
    {
        public override bool CanRead { get { return true; } }
        public override bool CanSeek { get { return true; } }
        public override bool CanWrite { get { return true; } }
        public override long Length { get { return m_baseEndPosition - m_baseOffset; } }
        public override long Position
        {
            get
            {
                return m_baseStream.Position - m_baseOffset;
            }
            set
            {
                this.Seek(value, SeekOrigin.Begin);
            }
        }

        private Stream m_baseStream;
        private long m_baseOffset;
        private long m_baseEndPosition;

        private bool m_disposed;

        /// <summary>
        /// stream を指定したオフセットと長さでを切り出した stream を提供します。
        /// 長さに 0 を指定した場合は、元の stream の長さの拡張が可能になります。
        /// </summary>
        public SubStream(Stream stream, long offset, long length)
        {
            m_baseStream = stream;
            m_baseOffset = offset;
            if (length >= 0)
            {
                m_baseEndPosition = m_baseOffset + length;
            }
            else
            {
                throw new ArgumentOutOfRangeException();
            }
            m_baseStream.Seek(m_baseOffset, SeekOrigin.Begin);
        }
        public override int Read(byte[] buffer, int offset, int count)
        {
            // baseStream の範囲外を読みだそうとしたら例外
            if (m_baseStream.Length != m_baseEndPosition && (m_baseStream.Length - m_baseStream.Position) < count)
            {
                throw new ArgumentOutOfRangeException();
            }
            long restLength = m_baseEndPosition - m_baseStream.Position;
            if (restLength < (long)count)
            {
                return m_baseStream.Read(buffer, offset, (int)restLength);
            }
            else
            {
                return m_baseStream.Read(buffer, offset, count);
            }
        }
        public override void Write(byte[] buffer, int offset, int count)
        {
            long restLength = m_baseEndPosition - m_baseStream.Position;
            if (restLength < (long)count)
            {
                throw new ArgumentOutOfRangeException();
            }
            else
            {
                m_baseStream.Write(buffer, offset, count);
                if ((restLength < (long)count))
                {
                    m_baseStream.SetLength(m_baseStream.Position);
                }
            }
        }
        public override long Seek(long offset, SeekOrigin origin)
        {
            long nextPosition = offset;
            if (origin == SeekOrigin.Begin)
            {
                nextPosition += m_baseOffset;
            }
            else if (origin == SeekOrigin.Current)
            {
                nextPosition += this.Position + m_baseOffset;
            }
            else if (origin == SeekOrigin.End)
            {
                nextPosition += m_baseEndPosition;
            }

            if (nextPosition < m_baseOffset || m_baseEndPosition < nextPosition)
            {
                throw new ArgumentOutOfRangeException();
            }

            long ret = m_baseStream.Seek(nextPosition, SeekOrigin.Begin);
            if (m_baseStream.Length < nextPosition)
            {
                m_baseStream.SetLength(m_baseStream.Position);
            }
            return ret;
        }
        public override void SetLength(long value)
        {
            throw new NotSupportedException();
        }
        public override void Flush()
        {
            m_baseStream.Flush();
        }
        protected override void Dispose(bool disposing)
        {
            if (!m_disposed)
            {
                if (disposing)
                {
                    m_baseStream.Dispose();
                }
                m_disposed = true;
            }
            base.Dispose(disposing);
        }
    }
}
