﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text;
using System.Xml.Serialization;
using YamlDotNet.RepresentationModel;
using Nintendo.Authoring.FileSystemMetaLibrary;
using Nintendo.Authoring.CryptoLibrary;

namespace Nintendo.Authoring.AuthoringLibrary
{
    public static class PatchConstructionUtility
    {
        internal static Tuple<string[], IndirectStorageStream.BinaryMatchHint[]> GetHintsFromNcaFiles(NintendoContentArchiveReader originalNcaReader, NintendoContentArchiveReader currentNcaReader, int fsIndex)
        {
            // オフセット、サイズのアラインメント計算が不要であることを確認するアサーション
            System.Diagnostics.Debug.Assert(RomFsAdfWriter.AlignmentSize == IndirectStorageSource.BlockSize);

            if (originalNcaReader == null || currentNcaReader == null)
            {
                return Tuple.Create(new string[0], new IndirectStorageStream.BinaryMatchHint[0]);
            }

            if (!originalNcaReader.GetExistentFsIndices().Contains(fsIndex) || !currentNcaReader.GetExistentFsIndices().Contains(fsIndex))
            {
                return Tuple.Create(new string[0], new IndirectStorageStream.BinaryMatchHint[0]);
            }

            long originalHashTargetOffset;
            using (var header = originalNcaReader.GetFsHeaderInfo(fsIndex))
            {
                originalHashTargetOffset = (long)header.GetHashTargetOffset();
            }

            long currentHashTargetOffset;
            using (var header = currentNcaReader.GetFsHeaderInfo(fsIndex))
            {
                currentHashTargetOffset = (long)header.GetHashTargetOffset();
            }

            var returnValue = Tuple.Create(new List<string>(), new List<IndirectStorageStream.BinaryMatchHint>());
            using (var originalFs = originalNcaReader.OpenFileSystemArchiveReader(fsIndex))
            {
                using (var currentFs = currentNcaReader.OpenFileSystemArchiveReader(fsIndex))
                {
                    // 共通のパスを含むファイルを列挙
                    var pathList = new List<string>();
                    {
                        var originalList = originalFs.ListFileInfo();
                        originalList.Sort();
                        var currentList = currentFs.ListFileInfo();
                        currentList.Sort();

                        int originalIndex = 0;
                        int currentIndex = 0;
                        while (originalIndex < originalList.Count && currentIndex < currentList.Count)
                        {
                            var result = originalList[originalIndex].Item1.CompareTo(currentList[currentIndex].Item1);
                            if (result < 0)
                            {
                                ++originalIndex;
                            }
                            else if (result > 0)
                            {
                                ++currentIndex;
                            }
                            else
                            {
                                pathList.Add(originalList[originalIndex].Item1);
                                ++originalIndex;
                                ++currentIndex;
                            }
                        }
                    }

                    // パスを元にヒント情報を列挙
                    var hintList = new List<Tuple<string, IndirectStorageStream.BinaryMatchHint>>();
                    foreach (var path in pathList)
                    {
                        var originalFragments = originalFs.GetFileFragmentList(path);
                        var currentFragments = currentFs.GetFileFragmentList(path);

                        int originalIndex = 0;
                        int currentIndex = 0;
                        while (originalIndex < originalFragments.Count && currentIndex < currentFragments.Count)
                        {
                            // NOTE: 論物変換が必要なら、ここでそれぞれ変換する必要がある
                            //       今は必要ない nca しか渡らないので、何もしない。

                            long originalSizeAligned = BitUtility.AlignUp(originalFragments[originalIndex].Item2, RomFsAdfWriter.AlignmentSize);
                            long currentSizeAligned = BitUtility.AlignUp(currentFragments[currentIndex].Item2, RomFsAdfWriter.AlignmentSize);
                            var originalOffset = originalFragments[originalIndex].Item1;
                            var currentOffset = currentFragments[currentIndex].Item1;

                            if (originalSizeAligned != 0 &&
                                currentSizeAligned != 0 &&
                                BitUtility.IsAligned(originalOffset, RomFsAdfWriter.AlignmentSize) &&
                                BitUtility.IsAligned(currentOffset, RomFsAdfWriter.AlignmentSize))
                            {
                                var hint = new IndirectStorageStream.BinaryMatchHint
                                {
                                    oldOffset = originalHashTargetOffset + originalOffset,
                                    oldSize = originalSizeAligned,
                                    newOffset = currentHashTargetOffset + currentOffset,
                                    newSize = currentSizeAligned,
                                };
                                hintList.Add(Tuple.Create(path, hint));
                            }

                            var originalEndOffset = originalOffset + originalSizeAligned;
                            var currentEndOffset = currentOffset + currentSizeAligned;

                            if (originalEndOffset != currentEndOffset)
                            {
                                // もっと救えるケースもあるが、この関数に渡される nca に関しては、
                                // フラグメントが複数あるケース自体が実在しないので、これで十分。
                                break;
                            }
                            else
                            {
                                ++originalIndex;
                                ++currentIndex;
                            }
                        }
                    }

                    // ヒント情報を newOffset の昇順にソート
                    hintList.Sort((left, right) =>
                    {
                        if (left.Item2.newOffset < right.Item2.newOffset)
                        {
                            return -1;
                        }
                        if (left.Item2.newOffset > right.Item2.newOffset)
                        {
                            return 1;
                        }
                        return 0;
                    });

                    if (0 < hintList.Count)
                    {
                        returnValue.Item1.Add(hintList[0].Item1);
                        returnValue.Item2.Add(hintList[0].Item2);

                        // 連続したデータをマージ
                        int mergeIndex = 0;
                        for (int i = 1; i < hintList.Count; ++i)
                        {
                            var previous = returnValue.Item2.Last();
                            var current = hintList[i].Item2;

                            long previousOriginalEndOffset = previous.oldOffset + previous.oldSize;
                            long previousCurrentEndOffset = previous.newOffset + previous.newSize;

                            if (current.oldOffset == previousOriginalEndOffset && current.newOffset == previousCurrentEndOffset)
                            {
                                var hint = new IndirectStorageStream.BinaryMatchHint
                                {
                                    oldOffset = previous.oldOffset,
                                    oldSize = previous.oldSize + current.oldSize,
                                    newOffset = previous.newOffset,
                                    newSize = previous.newSize + current.newSize,
                                };

                                returnValue.Item2[mergeIndex] = hint;
                            }
                            else
                            {
                                returnValue.Item1.Add(hintList[i].Item1);
                                returnValue.Item2.Add(hintList[i].Item2);

                                ++mergeIndex;
                            }
                        }
                    }
                }
            }

            return Tuple.Create(returnValue.Item1.ToArray(), returnValue.Item2.ToArray());
        }
    }

    public class PatchableNintendoContentArchive : IConnector
    {
        public List<Connection> ConnectionList { get; private set; }

        public PatchableNintendoContentArchive(IReadableSink outSink, string ncmContentType, NintendoContentArchiveReader originalReader, NintendoContentArchiveReader currentReader, string descFilePath, KeyConfiguration keyConfig)
        {
            var patchedNcaInfo = ArchiveReconstructionUtils.GetPatchedNcaInfo(ncmContentType, originalReader, currentReader, descFilePath, NintendoContentMetaConstant.ContentMetaTypePatch, 0);
            var patchedNca = new NintendoContentArchiveSource(patchedNcaInfo, keyConfig, false);
            outSink.SetSize(patchedNca.Size);
            ConnectionList = new List<Connection>();
            ConnectionList.Add(new Connection(patchedNca, outSink));
        }

        public ISource GetSource()
        {
            throw new NotImplementedException();
        }
    }

    internal class PatchableNintendoSubmissionPackageArchive : IConnector
    {
        public List<Connection> ConnectionList { get; private set; }
        private KeyConfiguration m_KeyConfig;

        public PatchableNintendoSubmissionPackageArchive(IReadableSink outSink, NintendoSubmissionPackageReader originalReader, NintendoSubmissionPackageReader currentReader, NintendoSubmissionPackageReader previousReader, string metaFilePath, string descFilePath, KeyConfiguration keyConfig)
        {
            m_KeyConfig = keyConfig;

            var patchedNspInfo = ArchiveReconstructionUtils.GetPatchedNspInfo(originalReader, currentReader, previousReader, metaFilePath, descFilePath, m_KeyConfig);

            // 与えられた desc に対する ACID のチェック
            foreach (var entry in patchedNspInfo.Entries)
            {
                foreach (var programContent in entry.Contents.Where(x => x.ContentType == NintendoContentMetaConstant.ContentTypeProgram))
                {
                    var npdmData = NintendoSubmissionPackageArchiveUtils.GetNpdmBinary(programContent, m_KeyConfig);
                    ArchiveReconstructionUtils.CheckAcid(npdmData, descFilePath);
                }
            }

            var patchedNsp = new NintendoSubmissionPackageArchive(outSink, patchedNspInfo, m_KeyConfig);
            ConnectionList = new List<Connection>(patchedNsp.ConnectionList);
        }

        public ISource GetSource()
        {
            throw new NotImplementedException();
        }
    }

    internal class IscPatchableNintendoSubmissionPackageArchive : IConnector
    {
        public List<Connection> ConnectionList { get; private set; }
        private KeyConfiguration m_KeyConfig;

        public IscPatchableNintendoSubmissionPackageArchive(IReadableSink outSink, NintendoSubmissionPackageReader previousReader, NintendoSubmissionPackageReader currentReader, string descFilePath, KeyConfiguration keyConfig)//, string outputPath)
        {
            m_KeyConfig = keyConfig;

            var patchedNspInfo = ArchiveReconstructionUtils.GetIscPatchedNspInfo(previousReader, currentReader, descFilePath, m_KeyConfig);

            // 与えられた desc に対する ACID のチェック
            foreach (var entry in patchedNspInfo.Entries)
            {
                foreach (var programContent in entry.Contents.Where(x => x.ContentType == NintendoContentMetaConstant.ContentTypeProgram))
                {
                    var npdmData = NintendoSubmissionPackageArchiveUtils.GetNpdmBinary(programContent, m_KeyConfig);
                    ArchiveReconstructionUtils.CheckAcid(npdmData, descFilePath);
                }
            }

            var patchedNsp = new NintendoSubmissionPackageArchive(outSink, patchedNspInfo, m_KeyConfig);
            ConnectionList = new List<Connection>(patchedNsp.ConnectionList);
        }

        public ISource GetSource()
        {
            throw new NotImplementedException();
        }
    }

    internal class SparseOriginalNintendoSubmissionPackageArchive : IConnector
    {
        public List<Connection> ConnectionList { get; private set; }

        public SparseOriginalNintendoSubmissionPackageArchive(IReadableSink outSink, NintendoSubmissionPackageReader originalReader, NintendoSubmissionPackageReader patchReader, NintendoSubmissionPackageReader previousReader, KeyConfiguration keyConfig, Dictionary<string, NintendoContentArchiveReader> sparseInfos)
        {
            if (!ArchiveReconstructionUtils.ReadContentMetaInNsp(originalReader).All(p => p.Type == NintendoContentMetaConstant.ContentMetaTypeApplication))
            {
                throw new ArgumentException("original nsp is not Application");
            }
            if (!ArchiveReconstructionUtils.ReadContentMetaInNsp(patchReader).All(p => p.Type == NintendoContentMetaConstant.ContentMetaTypePatch))
            {
                throw new ArgumentException("patch nsp is not Patch");
            }
            if (previousReader != null && !ArchiveReconstructionUtils.ReadContentMetaInNsp(previousReader).All(p => p.Type == NintendoContentMetaConstant.ContentMetaTypePatch))
            {
                throw new ArgumentException("previous nsp is not Patch");
            }

            var nspComparer = new NintendoSubmissionPackageComparer(originalReader, patchReader, keyConfig);
            var sparseStorages = ArchiveReconstructionUtils.RetrieveSparseStorageStream(previousReader, keyConfig);
            var sparsifiedNspInfo = ArchiveReconstructionUtils.GetNspInfo(originalReader, keyConfig, true);

            foreach (var entryInfo in sparsifiedNspInfo.Entries)
            {
                entryInfo.HasTicket = ModifiableArchiveUtil.HasTicket(originalReader, entryInfo.ContentMetaInfo.Model.GetUInt64Id());

                if (entryInfo.MetaType == NintendoContentMetaConstant.ContentMetaTypeApplication)
                {
                    foreach (var content in entryInfo.Contents)
                    {
                        var originalNcaName = content.ContentId + ".nca";
                        NintendoContentArchiveReader originalNcaReader;
                        NintendoContentArchiveReader patchNcaReader;
                        if (nspComparer.FindPatch(originalNcaName, out originalNcaReader, out patchNcaReader))
                        {
                            var fsInfo = (NintendoContentFileSystemInfo)content.FsInfo;
                            for (int i = 0; i < fsInfo.existentFsIndices.Count; ++i)
                            {
                                var fsIndex = fsInfo.existentFsIndices[i];
                                var fsEntry = fsInfo.fsEntries[i];

                                fsEntry.startOffset = (ulong)originalNcaReader.GetFsStartOffset(fsIndex);
                                fsEntry.endOffset = (ulong)originalNcaReader.GetFsEndOffset(fsIndex);

                                // SparseFs 向けに書き換え
                                if (ArchiveReconstructionUtils.IsSparseTargetPartition(content.ContentType, (NintendoContentArchivePartitionType)fsIndex))
                                {
                                    if (patchNcaReader == null || !patchNcaReader.HasFsInfo(fsIndex))
                                    {
                                        throw new ArgumentNullException();
                                    }

                                    var info = new FileSystemArchiveSparseSource.SparseInfo()
                                    {
                                        originalReader = originalNcaReader,
                                        patchReader = patchNcaReader,
                                    };

                                    // 直前のパッチからスパース情報を抜き取る
                                    if (!sparseInfos.ContainsKey(content.ContentType))
                                    {
                                        ArchiveReconstructionUtils.AttachSparseStorageStream(content.ContentType, sparseStorages, originalNcaReader);

                                        sparseInfos.Add(content.ContentType, originalNcaReader);
                                    }

                                    fsEntry.formatType = "SparseFs";
                                    fsEntry.sourceInterface = new FileSystemArchiveSparseSource(info);

                                    fsInfo.fsEntries[i] = fsEntry;
                                }
                            }
                        }
                    }
                }
            }

            var sparsifiedNsp = new NintendoSubmissionPackageArchive(outSink, sparsifiedNspInfo, keyConfig);
            ConnectionList = new List<Connection>(sparsifiedNsp.ConnectionList);
        }

        public ISource GetSource()
        {
            throw new NotImplementedException();
        }
    }

    internal class SparsePatchNintendoSubmissionPackageArchive : IConnector
    {
        public List<Connection> ConnectionList { get; private set; }

        public SparsePatchNintendoSubmissionPackageArchive(IReadableSink outSink, NintendoSubmissionPackageReader sparseReader, NintendoSubmissionPackageReader patchReader, KeyConfiguration keyConfig, Dictionary<string, NintendoContentArchiveReader> sparseInfos)
        {
            var originalNspInfo = ArchiveReconstructionUtils.GetNspInfo(sparseReader, keyConfig);
            var sparsifiedNspInfo = ArchiveReconstructionUtils.GetNspInfo(patchReader, keyConfig);

            foreach (var entryInfo in sparsifiedNspInfo.Entries)
            {
                // パッチはデフォルトでチケット有り
                entryInfo.HasTicket = true;

                if (entryInfo.MetaType == NintendoContentMetaConstant.ContentMetaTypePatch)
                {
                    var patchMetaModel = entryInfo.ContentMetaInfo.Model as PatchContentMetaModel;
                    var sparseMetaModel = ArchiveReconstructionUtils.ReadContentMetaInNsp(sparseReader).Find(value => value.Type == NintendoContentMetaConstant.ContentMetaTypeApplication) as ApplicationContentMetaModel;

                    // スパース済みオリジナル nsp の履歴を追加
                    patchMetaModel.HistoryList.Add(new PatchHistoryModel()
                    {
                        Type = NintendoContentMetaConstant.ContentMetaTypeApplication,
                        Id = sparseMetaModel.Id,
                        Version = 0,
                        Digest = sparseMetaModel.Digest,
                        ContentList = sparseMetaModel.ContentList,
                    });

                    patchMetaModel.Sort();

                    var streams = new List<SparseStorageArchiveStream>();
                    foreach (var info in sparseInfos)
                    {
                        if (info.Value.HasSparseStorageArchive())
                        {
                            streams.Add(new SparseStorageArchiveStream(info.Key, info.Value));
                        }
                    }
                    patchMetaModel.SparseStorages = streams;

                    // オリジナル側のソースを設定
                    var appEntryInfo = originalNspInfo.Entries.Where(value => value.MetaType == NintendoContentMetaConstant.ContentMetaTypeApplication).Single();
                    foreach (var content in entryInfo.Contents)
                    {
                        content.OriginalSource = appEntryInfo.Contents.Find(value => content.ContentType == value.ContentType && content.IdOffset == value.IdOffset)?.Source;
                    }
                }
            }

            var sparsifiedNsp = new NintendoSubmissionPackageArchive(outSink, sparsifiedNspInfo, keyConfig);
            ConnectionList = new List<Connection>(sparsifiedNsp.ConnectionList);
        }

        public ISource GetSource()
        {
            throw new NotImplementedException();
        }
    }
}
