﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.Text;
using YamlDotNet.RepresentationModel;
using Nintendo.Authoring.FileSystemMetaLibrary;

namespace Nintendo.Authoring.AuthoringLibrary
{
    public class NintendoContentPatchFileSystemInfo : FileSystemMetaLibrary.FileSystemInfo
    {
        public ISource OldSource { get; private set; }
        public ISource NewSource { get; private set; }
        public List<IndirectStorageStream.ExcludeRange> OldExcludeRanges { get; private set; }
        public List<IndirectStorageStream.ExcludeRange> NewExcludeRanges { get; private set; }
        public byte[] IndirectHeader { get; private set; }
        public byte[] IndirectTable { get; private set; }
        public ISource DifferenceSource { get; private set; } // IndirectHeader, IndirectTable とともに生成した差分データ
        public byte[] AesCtrExTable { get; private set; } // パッチファイルシステムとは分離すべき

        public byte[] MasterHash { get; set; }
        public Int64 HashTargetOffset { get; set; } // NewSource 上のオフセット

        public string DebugLogOutputFilePath { get; set; }

        public bool NeedsBuild()
        {
            return (IndirectHeader == null || IndirectTable == null);
        }

        public void UpdateNewSource(ISource updatedNewSource)
        {
            Trace.Assert(NewSource.Size == updatedNewSource.Size);
            NewSource = updatedNewSource;
        }

        public NintendoContentPatchFileSystemInfo(ISource oldSource, ISource newSource, List<IndirectStorageStream.ExcludeRange> oldExcludeRanges, List<IndirectStorageStream.ExcludeRange> newExcludeRanges)
        {
            OldSource = oldSource;
            NewSource = newSource;
            OldExcludeRanges = oldExcludeRanges;
            NewExcludeRanges = newExcludeRanges;
        }

        public NintendoContentPatchFileSystemInfo(byte[] indirectHeader, byte[] indirectTable, byte[] aesCtrExTable, ISource oldSource, ISource differenceSource)
        {
            IndirectHeader = indirectHeader;
            IndirectTable = indirectTable;
            AesCtrExTable = aesCtrExTable;
            OldSource = oldSource;
            DifferenceSource = differenceSource;

            NewSource = new StorageArchiveSource(new IndirectStorageArchiveReader(IndirectHeader, IndirectTable, new SourceBasedStream(OldSource), new SourceBasedStream(DifferenceSource)));
        }
    }

    public class NintendoContentPatchFsArchiveSource : ISource
    {
        public long Size { get; private set; }
        private ISource m_Source;
        private byte[] m_IndirectHeader;

        public NintendoContentPatchFsArchiveSource(NintendoContentPatchFileSystemInfo fileSystemInfo)
        {
            var builder = new IndirectStorageArchiveBuilder(fileSystemInfo.OldSource, fileSystemInfo.NewSource);
            if (fileSystemInfo.NeedsBuild())
            {
                Debug.Assert(fileSystemInfo.OldExcludeRanges != null && fileSystemInfo.NewExcludeRanges != null);
                builder.Build(fileSystemInfo.OldExcludeRanges, fileSystemInfo.NewExcludeRanges);
                // TODO: デバッグログ出力
            }
            else
            {
                builder.Import(fileSystemInfo.IndirectHeader, fileSystemInfo.IndirectTable, fileSystemInfo.DifferenceSource);
            }

            m_IndirectHeader = builder.GetHeader();

            var differenceSource = builder.GetDifferenceSource();
            ISource indirectTableSource;
            {
                var table = builder.GetTable();
                indirectTableSource = new MemorySource(table, 0, table.Length);
            }

            ISource aesCtrExTableSource;
            {
                aesCtrExTableSource = new MemorySource(fileSystemInfo.AesCtrExTable, 0, fileSystemInfo.AesCtrExTable.Length);
            }

            var elements = new List<ConcatenatedSource.Element>();
            elements.Add(new ConcatenatedSource.Element(differenceSource, "data", 0));
            // W/A: builder.GetDifferenceSource() で再構成した差分データと元の差分データでパディングサイズが異なることがある
            Trace.Assert(differenceSource.Size <= fileSystemInfo.DifferenceSource.Size);
            var indirectOffset = fileSystemInfo.DifferenceSource.Size;
            elements.Add(new ConcatenatedSource.Element(indirectTableSource, "indirectTable", indirectOffset));
            elements.Add(new ConcatenatedSource.Element(aesCtrExTableSource, "aesCtrExTable", indirectOffset + indirectTableSource.Size));

            m_Source = new ConcatenatedSource(elements);
            Size = m_Source.Size;
        }

        public ByteData PullData(long offset, int size)
        {
            return m_Source.PullData(offset, size);
        }

        public SourceStatus QueryStatus()
        {
            return m_Source.QueryStatus();
        }
    }

// TODO: 実装
#if false
    public class NintendoContentOptimizedPatchFileSystemInfo : FileSystemMetaLibrary.FileSystemInfo
    {
        public NintendoContentPatchFsArchiveSource PreviousSource { get; private set; }
        public NintendoContentPatchFsArchiveSource CurrentSource { get; private set; }

        public NintendoContentOptimizedPatchFileSystemInfo(NintendoContentPatchFsArchiveSource previousSource, NintendoContentPatchFsArchiveSource currentSource)
        {
            PreviousSource = previousSource;
            CurrentSource = currentSource;
        }
    }

    public class NintendoContentOptimizedPatchFsArchiveSource : ISource
    {
        public long Size { get; private set; }
        private ISource m_Source;

        public NintendoContentOptimizedPatchFsArchiveSource(NintendoContentOptimizedPatchFileSystemInfo fileSystemInfo)
        {
        }

        public ByteData PullData(long offset, int size)
        {
            return m_Source.PullData(offset, size);
        }

        public SourceStatus QueryStatus()
        {
            return m_Source.QueryStatus();
        }
    }
#endif

}
