﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.IO;
using System.Text;
using YamlDotNet.RepresentationModel;
using Nintendo.Authoring.FileSystemMetaLibrary;

namespace Nintendo.Authoring.AuthoringLibrary
{
    public class PartitionFsAdfWriter
    {
        private string m_adfPath;

        public PartitionFsAdfWriter(string adfPath)
        {
            m_adfPath = adfPath;
        }

        public void Write(string dirPath)
        {
            using (var adf = new StreamWriter(m_adfPath, false, Encoding.UTF8))
            {
                adf.WriteLine("formatType : {0}", NintendoContentFileSystemMetaConstant.FormatTypePartitionFs);
                adf.WriteLine("version : 0");
                adf.WriteLine("entries :");
                long offset = 0;
                foreach (var file in Directory.EnumerateFiles(dirPath, "*", SearchOption.AllDirectories))
                {
                    adf.WriteLine("  - type : file");
                    adf.WriteLine("    name : \"{0}\"", file.Replace("\\", "/").Replace(dirPath + "/", string.Empty));
                    adf.WriteLine("    offset : {0}", offset);
                    FileInfo fi = new FileInfo(file);
                    offset += fi.Length;
                    adf.WriteLine("    path : {0}", Path.GetFullPath(file.Replace("\\", "/")));
                }
            }
        }
    }

    public class PartitionFsAdfReader
    {
        private string m_adfPath;

        public PartitionFsAdfReader(string adfPath)
        {
            m_adfPath = adfPath;
        }

        public PartitionFileSystemInfo GetFileSystemInfo()
        {
            PartitionFileSystemInfo fileSystemInfo = new PartitionFileSystemInfo();

            using (var adf = new StreamReader(m_adfPath, Encoding.UTF8))
            {
                var yamlStream = new YamlStream();
                yamlStream.Load(adf);

                YamlMappingNode rootNode;
                try
                {
                    rootNode = (YamlMappingNode)yamlStream.Documents[0].RootNode;
                    YamlScalarNode formatType = (YamlScalarNode)rootNode.Children[new YamlScalarNode("formatType")];
                    if (formatType.Value != NintendoContentFileSystemMetaConstant.FormatTypePartitionFs)
                    {
                        throw new ArgumentException();
                    }
                }
                catch
                {
                    throw new ArgumentException("invalid format .adf file.");
                }

                // エントリ情報の読み込み
                YamlSequenceNode entries;
                try
                {
                    entries = (YamlSequenceNode)rootNode.Children[new YamlScalarNode("entries")];
                }
                catch
                {
                    throw new ArgumentException("invalid format .adf file. no \"entries\"");
                }

                foreach (YamlMappingNode entry in entries)
                {
                    PartitionFileSystemInfo.EntryInfo entryInfo = new PartitionFileSystemInfo.EntryInfo();
                    foreach (var child in entry)
                    {
                        switch (((YamlScalarNode)child.Key).Value)
                        {
                            case "type":
                                entryInfo.type = ((YamlScalarNode)child.Value).Value;
                                break;
                            case "name":
                                entryInfo.name = ((YamlScalarNode)child.Value).Value;
                                break;
                            case "offset":
                                entryInfo.offset = Convert.ToUInt64(((YamlScalarNode)child.Value).Value);
                                break;
                            case "path":
                                entryInfo.path = ((YamlScalarNode)child.Value).Value;
                                break;
                            default:
                                throw new ArgumentException("invalid format .adf file. invalid key is specified\n" + entry.ToString());
                        }
                    }
                    if (entryInfo.type == null || entryInfo.name == null || entryInfo.path == null)
                    {
                        throw new ArgumentException("invalid format .adf file. \"type\" or \"name\" or \"path\" is not specified.\n" + entry.ToString());
                    }
                    if (entryInfo.type != "file")
                    {
                        throw new ArgumentException("invalid format .adf file. \"type\" should be \"file\".\n" + entry.ToString());
                    }
                    FileInfo fi = new FileInfo(entryInfo.path);
                    entryInfo.size = (ulong)fi.Length;
                    fileSystemInfo.entries.Add(entryInfo);
                }
            }

            return fileSystemInfo;
        }
    }

    public class PartitionFsArchiveSource : ISource
    {
        public long Size { get; private set; }

        private ISource m_source;

        public PartitionFsArchiveSource(PartitionFileSystemInfo fileSystemInfo)
        {
            PartitionFileSystemMeta metaMgr = new PartitionFileSystemMeta();
            List<ConcatenatedSource.Element> elements = new List<ConcatenatedSource.Element>();
            long m_metaOffset;

            {
                byte[] buffer = metaMgr.Create(fileSystemInfo);
                ConcatenatedSource.Element element = new ConcatenatedSource.Element(
                    new MemorySource(buffer, 0, buffer.Length), "meta", 0);
                elements.Add(element);
                m_metaOffset = element.Source.Size;
            }

            foreach (var entryInfo in fileSystemInfo.entries)
            {
                var entrySource = entryInfo.sourceInterface != null ? (ISource)entryInfo.sourceInterface : new FileSource(entryInfo.path, 0, (long)entryInfo.size);

                ConcatenatedSource.Element element = new ConcatenatedSource.Element(
                    entrySource, entryInfo.name, (long)entryInfo.offset + m_metaOffset);
                elements.Add(element);
            }

            m_source = new ConcatenatedSource(elements);
            Size = m_source.Size;
        }
        public ByteData PullData(long offset, int size)
        {
            return m_source.PullData(offset, size);
        }
        public SourceStatus QueryStatus()
        {
            return m_source.QueryStatus();
        }
    }
}
