﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text;
using System.Text.RegularExpressions;
using System.Xml.Serialization;
using Nintendo.Authoring.FileSystemMetaLibrary;

namespace Nintendo.Authoring.AuthoringLibrary
{
    internal class OnCardAddOnContentNintendoSubmissionPackageArchive : IConnector
    {
        public List<Connection> ConnectionList { get; private set; }

        public OnCardAddOnContentNintendoSubmissionPackageArchive(IReadableSink outSink, List<NintendoSubmissionPackageReader> nspReaders, NintendoSubmissionPackageReader baseNspReader, int? cardSize, int? cardClockRate, byte? cardLaunchFlags, KeyConfiguration keyConfig)
        {
            var mergedNspInfo = new NintendoSubmissionPackageFileSystemInfo();
            {
                foreach (var nspInfo in nspReaders.Select(x => ArchiveReconstructionUtils.GetNspInfo(x, keyConfig, true)))
                {
                    if (nspInfo.Entries.Exists(x => x.MetaType != NintendoContentMetaConstant.ContentMetaTypeAddOnContent))
                    {
                        throw new ArgumentException("Input nsp includes contents that is not AddOnContent.");
                    }
                    // NCA を内部鍵で再暗号化
                    for (int i = 0; i < nspInfo.Entries.Count; i++)
                    {
                        var entry = nspInfo.Entries[i];
                        for (int j = 0; j < entry.Contents.Count; j++)
                        {
                            var content = entry.Contents[j];
                            var fsInfo = (content.FsInfo as NintendoContentFileSystemInfo);
                            fsInfo.rightsId = null;
                            content.SetFsInfo(content.ContentType, null, content.FsInfo);
                            entry.Contents[j] = content;
                        }
                        nspInfo.Entries[i] = entry;
                    }
                    mergedNspInfo.Entries.AddRange(nspInfo.Entries);
                }

                // 重複した ID があるとエラー
                {
                    var distinctCount = mergedNspInfo.Entries.Select(x => x.ContentMetaInfo.Model.Id).Distinct().Count();
                    if (distinctCount != mergedNspInfo.Entries.Count)
                    {
                        throw new ArgumentException("AddOnContent IDs in on-card-aoc cannot be duplicated.");
                    }
                }

                // OnCardAoc 用情報の収集
                mergedNspInfo.OnCardAddOnContentInfo = new OnCardAddOnContentInfoModel();
                {
                    var baseContentMetas = ArchiveReconstructionUtils.ReadContentMetaInNsp(baseNspReader).Where(x => x.Type == NintendoContentMetaConstant.ContentMetaTypePatch);
                    if (!baseContentMetas.Any())
                    {
                        // パッチがない場合はアプリケーションを基準にする
                        baseContentMetas = ArchiveReconstructionUtils.ReadContentMetaInNsp(baseNspReader).Where(x => x.Type == NintendoContentMetaConstant.ContentMetaTypeApplication);
                        if (!baseContentMetas.Any())
                        {
                            // パッチもアプリも base に指定されていなければエラー
                            throw new ArgumentException("Base nsp should be Application or Patch.");
                        }
                        mergedNspInfo.OnCardAddOnContentInfo.SetApplication(baseContentMetas.ToList());
                    }
                    else
                    {
                        mergedNspInfo.OnCardAddOnContentInfo.SetPatchAndApplication(baseContentMetas.Select(x => x as PatchContentMetaModel).ToList());
                    }
                    mergedNspInfo.OnCardAddOnContentInfo.SetOriginalAddOnContentMeta(mergedNspInfo.Entries.Select(x => x.ContentMetaInfo.Model).ToList());
                    mergedNspInfo.OnCardAddOnContentInfo.CheckRequiredVersion();
                }

                // カード設定の外部入力
                {
                    mergedNspInfo.CardSize = cardSize ?? 0;
                    mergedNspInfo.CardClockRate = cardClockRate ?? 0;
                    mergedNspInfo.CardLaunchFlags = cardLaunchFlags;
                }
            }

            var mergedNsp = new NintendoSubmissionPackageArchive(outSink, mergedNspInfo, keyConfig);
            ConnectionList = new List<Connection>(mergedNsp.ConnectionList);
        }

        public ISource GetSource()
        {
            throw new NotImplementedException();
        }
    }

    [XmlRoot("OnCardAocInfo", IsNullable = false)]
    public class OnCardAddOnContentInfoModel
    {
        [XmlElement("OriginalAddOnContentMeta")]
        public OnCardAocInfoContentMetaListModel OriginalAddOnContentMeta { get; set; }

        [XmlElement("Application")]
        public OnCardAocInfoContentMetaListModel Application { get; set; }

        [XmlElement("Patch")]
        public OnCardAocInfoContentMetaListModel Patch { get; set; }
        [XmlIgnore]
        public bool PatchSpecified { get { return Patch != null; } }

        public void SetOriginalAddOnContentMeta(List<ContentMetaModel> contentMetaList)
        {
            OriginalAddOnContentMeta = GetContentMetaList(contentMetaList);
        }

        public void SetApplication(List<ContentMetaModel> contentMetaList)
        {
            Application = GetContentMetaList(contentMetaList);
        }

        public void SetPatchAndApplication(List<PatchContentMetaModel> patchContentMetaList)
        {
            Patch = GetContentMetaList(patchContentMetaList.Select(x => x as ContentMetaModel).ToList());
            var apps = patchContentMetaList.SelectMany(x => x.HistoryList.Where(y => y.Type == NintendoContentMetaConstant.ContentMetaTypeApplication));
            Application = new OnCardAocInfoContentMetaListModel();
            Application.ContentMetaList = new List<OnCardAocInfoContentMetaModel>();
            Application.ContentMetaList.AddRange(apps.Select(x =>
            {
                var contentMeta = new OnCardAocInfoContentMetaModel();
                contentMeta.Type = x.Type;
                contentMeta.Id = x.Id;
                contentMeta.Version = x.Version;
                contentMeta.Digest = x.Digest;
                contentMeta.ApplicationId = x.Id;
                contentMeta.ContentList = x.ContentList;
                return contentMeta;
            }));
        }

        public void CheckRequiredVersion()
        {
            var versionInfo = new Dictionary<string, uint>();
            foreach (var contentMeta in PatchSpecified ? Patch.ContentMetaList : Application.ContentMetaList)
            {
                if (!versionInfo.ContainsKey(contentMeta.ApplicationId))
                {
                    versionInfo.Add(contentMeta.ApplicationId, contentMeta.Version);
                }
                else if (versionInfo[contentMeta.ApplicationId] < contentMeta.Version) // 重複しないはずだが念のため
                {
                    versionInfo[contentMeta.ApplicationId] = contentMeta.Version;
                }
            }

            foreach (var contentMeta in OriginalAddOnContentMeta.ContentMetaList)
            {
                if (!versionInfo.ContainsKey(contentMeta.ApplicationId))
                {
                    throw new ArgumentException(string.Format("There is no Application or Patch that corresponds to AddOnContent {0}.", contentMeta.Id));
                }
                else if (versionInfo[contentMeta.ApplicationId] < contentMeta.RequiredApplicationVersion)
                {
                    throw new ArgumentException(string.Format("The verision of base Application or Patch is lower than RequiredApplicationVersion of AddOnContent {0}.", contentMeta.Id));
                }
            }
        }

        private OnCardAocInfoContentMetaListModel GetContentMetaList(List<ContentMetaModel> refContentMetaList)
        {
            var contentMetaList = new OnCardAocInfoContentMetaListModel();
            contentMetaList.ContentMetaList = new List<OnCardAocInfoContentMetaModel>();
            foreach (var refContentMeta in refContentMetaList)
            {
                var contentMeta = new OnCardAocInfoContentMetaModel();
                contentMeta.Type = refContentMeta.Type;
                contentMeta.Id = refContentMeta.Id;
                contentMeta.Version = refContentMeta.Version.Value;
                contentMeta.Digest = refContentMeta.Digest;
                contentMeta.ContentList = refContentMeta.ContentList;
                switch (refContentMeta.Type)
                {
                    case NintendoContentMetaConstant.ContentMetaTypeApplication:
                        contentMeta.ApplicationId = refContentMeta.Id;
                        break;
                    case NintendoContentMetaConstant.ContentMetaTypePatch:
                        contentMeta.ApplicationId = (refContentMeta as PatchContentMetaModel).ApplicationId;
                        break;
                    case NintendoContentMetaConstant.ContentMetaTypeAddOnContent:
                        var aocMeta = (refContentMeta as AddOnContentContentMetaModel);
                        contentMeta.Tag = aocMeta.Tag;
                        contentMeta.ApplicationId = aocMeta.ApplicationId;
                        contentMeta.RequiredApplicationVersion = aocMeta.RequiredApplicationVersion;
                        break;
                    default:
                        throw new ArgumentException(string.Format("Not supported type {0}", contentMeta.Type));
                }
                contentMetaList.ContentMetaList.Add(contentMeta);
            }
            return contentMetaList;
        }

        public byte[] GetBytes()
        {
            var nameSpace = new XmlSerializerNamespaces();
            nameSpace.Add(String.Empty, String.Empty);

            using (var memoryStream = new MemoryStream())
            {
                var sw = new StreamWriter(memoryStream, Encoding.UTF8);
                var serializer = new XmlSerializer(typeof(OnCardAddOnContentInfoModel));
                serializer.Serialize(sw, this, nameSpace);
                return memoryStream.ToArray();
            }
        }
    }

    public class OnCardAocInfoContentMetaListModel
    {
        [XmlElement("ContentMeta")]
        public List<OnCardAocInfoContentMetaModel> ContentMetaList { get; set; }
    }

    public class OnCardAocInfoContentMetaModel : ContentMetaLiteModel
    {
        override public bool ContentListSpecified { get { return false; } }

        [XmlElement("Tag")]
        public string Tag { get; set; }
        [XmlIgnore]
        public bool TagSpecified { get { return Tag != null; } }

        [XmlIgnore]
        public string ApplicationId { get; set; }

        [XmlIgnore]
        public UInt32 RequiredApplicationVersion { get; set; }
    }

    public class OnCardAddOnContentInfoXmlSource : ISource
    {
        public long Size { get; set; }
        private ISource m_source;

        public OnCardAddOnContentInfoXmlSource(OnCardAddOnContentInfoModel model)
        {
            var bytes = model.GetBytes();
            m_source = new MemorySource(bytes, 0, bytes.Length);
            Size = m_source.Size;
        }

        public ByteData PullData(long offset, int size)
        {
            return m_source.PullData(offset, size);
        }

        public SourceStatus QueryStatus()
        {
            return m_source.QueryStatus();
        }
    }

}
