﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.Xml;
using System.Xml.Linq;
using System.Xml.Serialization;
using System.Text;
using YamlDotNet.RepresentationModel;
using Nintendo.Authoring.FileSystemMetaLibrary;

namespace Nintendo.Authoring.AuthoringLibrary
{
    [Flags]
    public enum ContentMetaAttribute
    {
        None = 0,
        IncludesExFatDriver = 1 << 0,
        Rebootless = 1 << 1,
    }

    static class ContentMetaAttributeExtension
    {
        public static ContentMetaAttribute FromStringList(IEnumerable<string> list)
        {
            var attributes = ContentMetaAttribute.None;
            foreach (var attribute in list)
            {
                attributes |= (ContentMetaAttribute)Enum.Parse(typeof(ContentMetaAttribute), attribute);
            }

            return attributes;
        }

        public static List<string> ToStringList(this ContentMetaAttribute attributes)
        {
            var list = new List<string>();
            if (attributes.HasFlag(ContentMetaAttribute.IncludesExFatDriver))
            {
                var attribute = ContentMetaAttribute.IncludesExFatDriver;
                list.Add(attribute.ToString());
            }
            if (attributes.HasFlag(ContentMetaAttribute.Rebootless))
            {
                var attribute = ContentMetaAttribute.Rebootless;
                list.Add(attribute.ToString());
            }

            return list;
        }
    }

    // ContentId, Hash 等の要素を含まないコンテンツメタの静的な構成情報
    internal class ContentMetaDescriptor
    {
        public ContentMetaDescriptor(string type, UInt64 id, UInt32 version, Byte attributes, UInt64 applicationId, UInt32 requiredDownloadSystemVersion, List<NintendoContentDescriptor> contentDescriptorList, List<NintendoContentMetaInfo> contentMetaInfoList, NintendoMetaExtendedHeader extendedHeader, NintendoMetaExtendedData extendedData, byte[] digest, UInt32? originalRequiredSystemVersion, bool? keepGeneration)
        {
            Type = type;
            Id = id;
            Version = version;
            Attributes = attributes;
            ContentDescriptorList = contentDescriptorList;
            ContentMetaInfoList = contentMetaInfoList;
            ExtendedHeader = extendedHeader;
            ExtendedData = extendedData;
            RequiredDownloadSystemVersion = requiredDownloadSystemVersion;
            ApplicationId = applicationId;
            Digest = digest;
            OriginalRequiredSystemVersion = originalRequiredSystemVersion;
            KeepGeneration = keepGeneration;
        }

        public string Type { get; set; }
        public UInt64 Id { get; set; }
        public UInt32 Version { get; set; }
        public Byte Attributes { get; set; }
        public UInt64 ApplicationId { get; set; }
        public UInt32 RequiredDownloadSystemVersion { get; set; }
        public List<NintendoContentDescriptor> ContentDescriptorList { get; set; }
        public List<NintendoContentMetaInfo> ContentMetaInfoList { get; set; }
        public NintendoMetaExtendedHeader ExtendedHeader { get; set; }
        public NintendoMetaExtendedData ExtendedData { get; set; }
        public byte[] Digest { get; set; }
        public UInt32? OriginalRequiredSystemVersion { get; set; }
        public bool? KeepGeneration { get; set; }
    }

    public class NintendoContentMetaBase
    {
        private NintendoContentMeta m_contentMeta;
        private ContentMetaDescriptor m_descriptor;
        private List<Tuple<ISource, NintendoContentDescriptor>> m_contentSourceList;
        private NintendoContentMetaBaseSource m_source;

        public NintendoContentMetaBase(string type, ContentMetaModel model)
            : this(ConvertContentMetaDescriptor(type, model), false)
        {
        }

        // TODO: 利用が適切でないので利用箇所と併せて修正
        public NintendoContentMetaBase(string type, string metaPath)
            : this(new List<Tuple<ISource, NintendoContentDescriptor>>(), type, metaPath, null)
        {
        }

        // TODO: buffer はもう不要なはずなので修正
        public NintendoContentMetaBase(List<Tuple<ISource, NintendoContentDescriptor>> contentSourceList, byte[] buffer, ContentMetaModel model, bool isProdEncryption)
            : this(contentSourceList, ExtractContentMetaDescriptor(buffer, model, contentSourceList.Select(tuple => tuple.Item2).ToList()), isProdEncryption)
        {
        }

        public NintendoContentMetaBase(ContentMetaModel model, bool isProdEncryption)
            : this(ExtractContentMetaDescriptor(model), isProdEncryption)
        {
        }

        public NintendoContentMetaBase(List<Tuple<ISource, NintendoContentDescriptor>> contentSourceList, string type, string metaPath, List<IContentMetaExtendedData> extendedData, int contentIndex = 0)
            : this(contentSourceList, ExtractContentMetaDescriptor(type, metaPath, contentSourceList.Select(tuple => tuple.Item2).ToList(), extendedData, contentIndex), false)
        {
        }

        internal NintendoContentMetaBase(ContentMetaDescriptor descriptor, bool isProdEncryption)
            : this(new List<Tuple<ISource, NintendoContentDescriptor>>(), descriptor, isProdEncryption)
        {
        }

        internal NintendoContentMetaBase(List<Tuple<ISource, NintendoContentDescriptor>> contentSourceList, ContentMetaDescriptor descriptor, bool isProdEncryption)
        {
            m_contentSourceList = contentSourceList;
            m_descriptor = descriptor;

            var type = m_descriptor.Type;
            var id = m_descriptor.Id;
            var version = m_descriptor.Version;
            var attributes = m_descriptor.Attributes;
            var applicationId = m_descriptor.ApplicationId;
            var requiredDownloadSystemVersion = m_descriptor.RequiredDownloadSystemVersion;
            var contentInfoList = m_descriptor.ContentDescriptorList;
            var contentMetaInfoList = m_descriptor.ContentMetaInfoList;
            var digest = m_descriptor.Digest;

            // コンテンツメタデータ作成
            switch(type)
            {
                case NintendoContentMetaConstant.ContentMetaTypeApplication:
                    {
                        var applicationExtendedHeader = (NintendoApplicationMetaExtendedHeader)descriptor.ExtendedHeader;
                        m_contentMeta = NintendoContentMeta.CreateApplicationMeta(id, version, attributes, requiredDownloadSystemVersion, applicationExtendedHeader, contentInfoList, contentMetaInfoList, digest);
                    }
                    break;
                case NintendoContentMetaConstant.ContentMetaTypeAddOnContent:
                    {
                        var addOnContentExtendedHeader = (NintendoAddOnContentMetaExtendedHeader)descriptor.ExtendedHeader;
                        m_contentMeta = NintendoContentMeta.CreateAddOnContentMeta(id, version, attributes, requiredDownloadSystemVersion, addOnContentExtendedHeader, contentInfoList, contentMetaInfoList, digest);
                    }
                    break;
                case NintendoContentMetaConstant.ContentMetaTypePatch:
                    {
                        var patchExtendedHeader = (NintendoPatchMetaExtendedHeader)descriptor.ExtendedHeader;
                        var patchExtendedData = descriptor.ExtendedData as NintendoPatchMetaExtendedData;
                        m_contentMeta = NintendoContentMeta.CreatePatchMeta(id, version, attributes, requiredDownloadSystemVersion, patchExtendedHeader, contentInfoList, contentMetaInfoList, patchExtendedData, digest);
                    }
                    break;
                case NintendoContentMetaConstant.ContentMetaTypeDelta:
                    {
                        var deltaExtendedHeader = (NintendoDeltaMetaExtendedHeader)descriptor.ExtendedHeader;
                        var deltaExtendedData = descriptor.ExtendedData as NintendoDeltaMetaExtendedData; // null の場合もある
                        m_contentMeta = NintendoContentMeta.CreateDeltaMeta(id, version, attributes, requiredDownloadSystemVersion, deltaExtendedHeader, contentInfoList, contentMetaInfoList,deltaExtendedData, digest);
                    }
                    break;
                case NintendoContentMetaConstant.ContentMetaTypeSystemUpdate:
                    {
                        m_contentMeta = NintendoContentMeta.CreateSystemUpdateMeta(id, version, type, attributes, requiredDownloadSystemVersion, contentInfoList, contentMetaInfoList, digest);
                    }
                    break;
                default:
                    {
                        m_contentMeta = NintendoContentMeta.CreateDefault(id, version, type, attributes, requiredDownloadSystemVersion, contentInfoList, contentMetaInfoList, digest);
                    }
                    break;
            }

            m_source = new NintendoContentMetaBaseSource(this, !isProdEncryption);
        }

        public byte[] GetBytes()
        {
            return m_contentMeta.GetBytes();
        }

        public ISource GetSource()
        {
            return m_source;
        }

        public Int64 GetContentInfoHashOffset(Int32 index)
        {
            return m_contentMeta.GetContentInfoHashOffset(index);
        }

        public Int64 GetContentInfoHashSize()
        {
            return NintendoContentMeta.GetContentInfoHashSize();
        }

        public Int64 GetContentInfoIdDataOffset(Int32 index)
        {
            return m_contentMeta.GetContentInfoIdDataOffset(index);
        }

        public Int64 GetContentInfoIdDataSize()
        {
            return NintendoContentMeta.GetContentInfoIdDataSize();
        }

        public Int64 GetDigestOffset()
        {
            return m_contentMeta.GetDigestOffset();
        }

        public Int64 GetDigestSize()
        {
            return NintendoContentMeta.GetDigestSize();
        }

        public string GetEntryName()
        {
            return m_descriptor.Type + "_" + string.Format("{0:x16}", m_descriptor.Id) + ".cnmt";
        }

        public byte GetRepresentKeyGeneration()
        {
            var reader = new NintendoContentMetaReader(GetBytes());
            if (!reader.CanUpdateContentMetaKeyGeneration())
            {
                // 古いシステムが参照する可能性があるコンテンツメタは鍵世代を 0 にする
                return (byte)0;
            }
            byte keyGeneration = (byte)0;
            foreach (var contentDescriptor in m_descriptor.ContentDescriptorList)
            {
                if (keyGeneration < contentDescriptor.ContentInfo.KeyGeneration)
                {
                    keyGeneration = contentDescriptor.ContentInfo.KeyGeneration;
                }
            }
            return keyGeneration;
        }

        internal ContentMetaDescriptor GetContentMetaDescryptor()
        {
            return m_descriptor;
        }

        internal List<Tuple<ISource, NintendoContentDescriptor>> GetContentSourceList()
        {
            return m_contentSourceList;
        }
        static private UInt64 ParseId(string s)
        {
            return UInt64.Parse(s.Replace("0x", ""), System.Globalization.NumberStyles.HexNumber);
        }
        static private byte[] ParseBytes(string s)
        {
            var strings = new List<string>();
            for (int i = 0; i < s.Count();  i += 2)
            {
                strings.Add(s.Substring(i, 2));
            }
            return strings.Select(p => byte.Parse(p, System.Globalization.NumberStyles.HexNumber)).ToArray();
        }

        static private void SetContentMetaExtended(ref NintendoMetaExtendedHeader exHeader, ref NintendoMetaExtendedData exData, ContentMetaModel model)
        {
            switch (model.Type)
            {
                case NintendoContentMetaConstant.ContentMetaTypeApplication:
                    {
                        var applicationModel = model as ApplicationContentMetaModel;
                        exHeader = new NintendoApplicationMetaExtendedHeader(applicationModel.GetUInt64PatchId(), applicationModel.RequiredSystemVersion);
                        exData = null;
                    }
                    break;
                case NintendoContentMetaConstant.ContentMetaTypeAddOnContent:
                    {
                        var aocModel = model as AddOnContentContentMetaModel;
                        exHeader = new NintendoAddOnContentMetaExtendedHeader(aocModel.GetUInt64ApplicationId(), aocModel.RequiredApplicationVersion, aocModel.Tag);
                        exData = null;
                    }
                    break;
                case NintendoContentMetaConstant.ContentMetaTypePatch:
                    {
                        // パッチは、マージ対応のために model から再構成する
                        var patchModel = model as PatchContentMetaModel;

                        var patchExHeader = new NintendoPatchMetaExtendedHeader(patchModel.GetUInt64ApplicationId(), patchModel.RequiredSystemVersion);
                        var patchExData = new NintendoPatchMetaExtendedData();
                        if (patchModel.HistoryList == null)
                        {
                            patchExData.Histories = new List<NintendoPatchMetaExtendedData.NintendoPatchMetaHisotry>();
                        }
                        else
                        {
                            patchExData.Histories = patchModel.HistoryList.Select(p =>
                            {
                                var history = new NintendoPatchMetaExtendedData.NintendoPatchMetaHisotry();
                                history.Type = p.Type;
                                history.ProgramId = ParseId(p.Id);
                                history.Version = p.Version;
                                history.Digest = ParseBytes(p.Digest);
                                history.ContentInfos = p.ContentList.Select(p2 =>
                                {
                                    return new NintendoContentInfo(p2.Type, ParseBytes(p2.Id), p2.Size, ParseBytes(p2.Hash), (byte)p2.KeyGeneration, p2.IdOffset);
                                }).ToList();
                                return history;
                            }).ToList();
                        }

                        if (patchModel.DeltaHistoryList == null)
                        {
                            patchExData.DeltaHistories = new List<NintendoPatchMetaExtendedData.NintendoPatchMetaDeltaHisotry>();
                        }
                        else
                        {
                            patchExData.DeltaHistories = patchModel.DeltaHistoryList.Select(p =>
                            {
                                var history = new NintendoPatchMetaExtendedData.NintendoPatchMetaDeltaHisotry();
                                history.Delta = new NintendoDeltaMetaExtendedData();
                                history.Delta.Source = new NintendoDeltaMetaExtendedData.NintendoDeltaMetaPatchIndicator()
                                {
                                    PatchId = ParseId(p.Source.PatchId),
                                    Version = p.Source.Version
                                };
                                history.Delta.Destination = new NintendoDeltaMetaExtendedData.NintendoDeltaMetaPatchIndicator()
                                {
                                    PatchId = ParseId(p.Destination.PatchId),
                                    Version = p.Destination.Version
                                };
                                history.DownloadSize = p.DownloadSize;
                                return history;
                            }).ToList();
                        }

                        if (patchModel.DeltaList == null)
                        {
                            patchExData.Deltas = new List<NintendoPatchMetaExtendedData.NintendoPatchMetaDelta>();
                        }
                        else
                        {
                            patchExData.Deltas = patchModel.DeltaList.Select(p =>
                            {
                                var delta = new NintendoPatchMetaExtendedData.NintendoPatchMetaDelta();
                                delta.ContentInfos = p.ContentList.Select(p2 =>
                                {
                                    return new NintendoContentInfo(p2.Type, ParseBytes(p2.Id), p2.Size, ParseBytes(p2.Hash), (byte)p2.KeyGeneration, p2.IdOffset);
                                }).ToList();

                                delta.Delta = new NintendoDeltaMetaExtendedData();
                                delta.Delta.Source = new NintendoDeltaMetaExtendedData.NintendoDeltaMetaPatchIndicator()
                                {
                                    PatchId = ParseId(p.Source.PatchId),
                                    Version = p.Source.Version
                                };
                                delta.Delta.Destination = new NintendoDeltaMetaExtendedData.NintendoDeltaMetaPatchIndicator()
                                {
                                    PatchId = ParseId(p.Destination.PatchId),
                                    Version = p.Destination.Version
                                };
                                delta.Delta.FragmentSets = p.FragmentSetList.Select(p2 =>
                                {
                                    var fragmentSet = new NintendoDeltaMetaExtendedData.NintendoDeltaMetaFragmentSet();
                                    fragmentSet.FragmentTargetContentType = p2.FragmentTargetContentType;
                                    fragmentSet.UpdateType = p2.UpdateType;
                                    fragmentSet.SourceContentId = ParseBytes( p2.Source.ContentId);
                                    fragmentSet.SourceSize = p2.Source.Size;
                                    fragmentSet.DestinationContentId = ParseBytes(p2.Destination.ContentId);
                                    fragmentSet.DestinationSize = p2.Destination.Size;

                                    fragmentSet.Fragments = p2.FragmentList.Select(p3 =>
                                    {
                                        return new NintendoDeltaMetaExtendedData.NintendoDeltaMetaFragmentIndicator()
                                        {
                                            FragmentIndex = p3.FragmentIndex,
                                            ContentInfoIndex = p3.ContentInfoIndex
                                        };
                                    }).ToList();
                                    return fragmentSet;
                                }).ToList();

                                return delta;
                            }).ToList();
                        }

                        patchExHeader.ExtendedDataSize = (UInt32) patchExData.GetSize();
                        exHeader = patchExHeader;
                        exData = patchExData;
                    }
                    break;
            }
        }

        // TODO: 削除
        static private ContentMetaDescriptor ExtractContentMetaDescriptor(byte[] buffer, ContentMetaModel model, List<NintendoContentDescriptor> contentDescriptorList)
        {
            var reader = new NintendoContentMetaReader(buffer);
            NintendoMetaExtendedHeader exHeader = reader.GetExtendedHeader();
            NintendoMetaExtendedData exData = reader.GetExtendedData();
            SetContentMetaExtended(ref exHeader, ref exData, model);
            return new ContentMetaDescriptor(reader.GetType(), reader.GetId(), reader.GetVersion(), reader.GetAttributes(), reader.GetApplicationId(), reader.GetRequiredDownloadSystemVersion(), contentDescriptorList, reader.GetContentMetaInfoList(), exHeader, exData, reader.GetDigest(), (model as PatchContentMetaModel)?.OriginalRequiredSystemVersion, model.KeepGeneration);
        }

        static private ContentMetaDescriptor ExtractContentMetaDescriptor(ContentMetaModel model)
        {
            NintendoMetaExtendedHeader exHeader = null;
            NintendoMetaExtendedData exData = null;
            SetContentMetaExtended(ref exHeader, ref exData, model);
            return new ContentMetaDescriptor(model.Type, model.GetUInt64Id(), model.Version.Value, (Byte)ContentMetaAttributeExtension.FromStringList(model.AttributeList), 0, model.RequiredDownloadSystemVersion, model.GetContentDescriptorList(), model.GetContentMetaInfoList(), exHeader, exData, model.GetDigestBytes(), (model as PatchContentMetaModel)?.OriginalRequiredSystemVersion, model.KeepGeneration);
        }

        static private UInt32 GetDefaultDownloadSystemVersion()
        {
            return 0; // TODO: バージョン系のヘッダを参照して返す
        }

        static public UInt32 MakeVersion(UInt32 version, UInt16 releaseVersion, UInt16 privateVersion)
        {
            UInt32 tmpVersion = (UInt32)((releaseVersion << 16) | privateVersion);
            if (version > 0 && tmpVersion > 0)
            {
                throw new ArgumentException("<Version> cannot be specified with <ReleaseVersion>");
            }
            return (version > 0) ? version : (UInt32)((releaseVersion << 16) | privateVersion);
        }
        static private XDocument ReadContentMetaInNsp(string nsp)
        {
            using (var readerStream = new FileStream(nsp, FileMode.Open, FileAccess.Read))
            {
                var reader = new NintendoSubmissionPackageReader(readerStream);
                foreach (var info in reader.ListFileInfo())
                {
                    var fileName = info.Item1;
                    var fileSize = info.Item2;

                    if (fileName.EndsWith(".cnmt.xml"))
                    {
                        var buffer = reader.ReadFile(fileName, 0, fileSize);
                        using (var memoryStream = new MemoryStream(buffer))
                        {
                            return XDocument.Load(memoryStream);
                        }
                    }
                }
            }
            return null;
        }
        static private byte[] HexStringToBytes(string hex)
        {
            var hexStr = new List<string>();
            for(var i = 0; i < hex.Length; i += 2)
            {
                hexStr.Add(hex.Substring(i, 2));
            }
            return hexStr.Select(h => byte.Parse(h, System.Globalization.NumberStyles.HexNumber)).ToArray();
        }

        static private NintendoDeltaMetaExtendedData.NintendoDeltaMetaPatchIndicator GetIndicator(XDocument xml)
        {
            // パッチのコンテントメタ xml を読み込んで、indicator を構築する
            var indicator = new NintendoDeltaMetaExtendedData.NintendoDeltaMetaPatchIndicator();
            indicator.PatchId = UInt64.Parse(xml.Element("ContentMeta").Element("Id").Value.Replace("0x", ""), System.Globalization.NumberStyles.AllowHexSpecifier);
            indicator.Version = UInt32.Parse(xml.Element("ContentMeta").Element("Version").Value);

            return indicator;
        }
        static private byte[] GetContentIdBytes(string contentId)
        {
            var list = new List<string>();
            for (int i = 0; i < contentId.Length; i += 2)
            {
                list.Add(contentId.Substring(i, 2));
            }
            return list.Select(p => byte.Parse(p, System.Globalization.NumberStyles.HexNumber)).ToArray();
        }
        static private List<NintendoDeltaMetaExtendedData.NintendoDeltaMetaFragmentSet> GetFragmentSets(List<NintendoContentDescriptor> contentDescriptorList)
        {
            var fragmentSets = new List<NintendoDeltaMetaExtendedData.NintendoDeltaMetaFragmentSet>();

            // ContentId でグルーピング
            var uniqueContentIds = contentDescriptorList.Select(p => p.ContentFragmentInfo.DestinationContentId).Distinct();
            foreach(var contentId in uniqueContentIds)
            {
                var filterd = contentDescriptorList.Where(p => p.ContentFragmentInfo.DestinationContentId == contentId);
                var first = filterd.First();

                // filter した中で、Source / Destination の情報が同じことを確認できる
                if(filterd.All(p => p.ContentFragmentInfo.SourceContentId == first.ContentFragmentInfo.SourceContentId) == false
                   || filterd.All(p => p.ContentFragmentInfo.DestinationContentId == first.ContentFragmentInfo.DestinationContentId) == false
                   || filterd.All(p => p.ContentFragmentInfo.SourceSize == first.ContentFragmentInfo.SourceSize) == false
                   || filterd.All(p => p.ContentFragmentInfo.DestinationSize == first.ContentFragmentInfo.DestinationSize) == false
                   || filterd.All(p => p.ContentFragmentInfo.FragmentTargetContentType == first.ContentFragmentInfo.FragmentTargetContentType) == false
                   || filterd.All(p => p.ContentFragmentInfo.UpdateType == first.ContentFragmentInfo.UpdateType) == false)
                {
                    throw new ArgumentException("Invalid content for delta");
                }

                var fragmentSet = new NintendoDeltaMetaExtendedData.NintendoDeltaMetaFragmentSet();

                fragmentSet.SourceContentId = GetContentIdBytes(first.ContentFragmentInfo.SourceContentId);
                fragmentSet.DestinationContentId = GetContentIdBytes(first.ContentFragmentInfo.DestinationContentId);
                fragmentSet.SourceSize = first.ContentFragmentInfo.SourceSize;
                fragmentSet.DestinationSize = first.ContentFragmentInfo.DestinationSize;
                fragmentSet.FragmentTargetContentType = first.ContentFragmentInfo.FragmentTargetContentType;
                fragmentSet.UpdateType = first.ContentFragmentInfo.UpdateType; // TODO

                fragmentSet.Fragments = new List<NintendoDeltaMetaExtendedData.NintendoDeltaMetaFragmentIndicator>();

                for (var i = 0; i < contentDescriptorList.Count(); ++i)
                {
                    if(contentDescriptorList[i].ContentFragmentInfo.DestinationContentId == contentId)
                    {
                        var fragmentIndicator = new NintendoDeltaMetaExtendedData.NintendoDeltaMetaFragmentIndicator(){
                            ContentInfoIndex = (UInt16)i,
                            FragmentIndex = contentDescriptorList[i].ContentFragmentInfo.FragmentIndex
                        };
                        fragmentSet.Fragments.Add(fragmentIndicator);
                    }
                }

                System.Diagnostics.Debug.Assert(fragmentSet.Fragments.Count > 0);

                fragmentSets.Add(fragmentSet);
            }

            return fragmentSets;
        }

        static private UInt64 GetContentMetaId(ContentMetaBase metaBase, string type)
        {
            if(type == NintendoContentMetaConstant.ContentMetaTypePatch)
            {
                var patchMeta = (PatchMeta)metaBase;
                return patchMeta.Id;
            }
            return metaBase.Id;
        }

        static private ContentMetaDescriptor ExtractContentMetaDescriptor(string type, string metaPath, List<NintendoContentDescriptor> contentDescriptorList, List<IContentMetaExtendedData> metaExtendedData, int contentIndex)
        {
            UInt64 id = 0;
            UInt32 version = 0;
            UInt64 applicationId = 0;
            Byte attributes = 0;
            NintendoMetaExtendedHeader extendedHeader = null;
            NintendoMetaExtendedData extendedData = null;
            var contentMetaInfoList = new List<NintendoContentMetaInfo>();
            var requiredDownloadSystemVersion = GetDefaultDownloadSystemVersion();
            var reader = new MetaFileReader(metaPath, type);

            if (type.Equals(NintendoContentMetaConstant.ContentMetaTypeSystemUpdate))
            {
                var systemUpdateMeta = GetDeserializedModel<SystemUpdateModel>(metaPath, false).ContentMeta;
                if(systemUpdateMeta.Type != NintendoContentMetaConstant.ContentMetaTypeSystemUpdate)
                {
                    throw new ArgumentException("The content meta type is not SystemUpdate.");
                }

                id = systemUpdateMeta.GetUInt64Id();
                if (systemUpdateMeta.Version.HasValue && (systemUpdateMeta.ReleaseVersion.HasValue || systemUpdateMeta.PrivateVersion.HasValue))
                {
                    throw new ArgumentException("<Version> cannot be specified with <ReleaseVersion>");
                }
                version = MakeVersion(
                    systemUpdateMeta.Version.GetValueOrDefault(),
                    systemUpdateMeta.ReleaseVersion.GetValueOrDefault(),
                    systemUpdateMeta.PrivateVersion.GetValueOrDefault());
                foreach(var m in systemUpdateMeta.ContentMetaList)
                {
                    contentMetaInfoList.Add(new NintendoContentMetaInfo(m.Type, m.GetUInt64Id(), m.Version.GetValueOrDefault(), (Byte)ContentMetaAttributeExtension.FromStringList(m.AttributeList)));
                }
            }
            else
            {
                var meta = reader.GetContentMetaList().ElementAt(contentIndex);
                id = GetContentMetaId(meta, type);

                if (type == NintendoContentMetaConstant.ContentMetaTypeSystemData ||
                    type == NintendoContentMetaConstant.ContentMetaTypeSystemProgram)
                {
                    var defaultVersion = NintendoContentMeta.GetDefaultVersion();
                    version = reader.GetVersion().GetValueOrDefault(defaultVersion);
                }
                else
                {
                    version = MakeVersion(meta.Version, meta.ReleaseVersion, meta.PrivateVersion);
                }

                attributes = meta.ContentMetaAttributes;
                requiredDownloadSystemVersion = reader.GetRequiredDownloadSystemVersion().GetValueOrDefault(requiredDownloadSystemVersion);
                switch (type)
                {
                    case NintendoContentMetaConstant.ContentMetaTypeApplication:
                        {
                            // TODO: 設定ファイルから書けるように
                            var patchId = id + 0x0800;

                            var programMeta = (ApplicationMeta)meta;
                            var debugRequiredSystemVersion = programMeta.RequiredSystemVersion;
                            var requiredSystemVersion = debugRequiredSystemVersion.HasValue ?
                                debugRequiredSystemVersion.Value :
                                NintendoContentMeta.GetRequiredSystemVersion();

                            extendedHeader = new NintendoApplicationMetaExtendedHeader(
                                patchId,
                                requiredSystemVersion
                                );
                        }
                        break;
                    case NintendoContentMetaConstant.ContentMetaTypeAddOnContent:
                        {
                            var aocMeta = (AddOnContentMeta)meta;
                            var requiredApplicationReleaseVersion = aocMeta.RequiredApplicationReleaseVersion;
                            var tag = aocMeta.Tag;

                            extendedHeader = new NintendoAddOnContentMetaExtendedHeader(
                                aocMeta.ApplicationId,
                                MakeVersion(0, requiredApplicationReleaseVersion, 0),
                                tag);
                        }
                        break;
                    case NintendoContentMetaConstant.ContentMetaTypePatch:
                        {
                            var patchMeta = (PatchMeta)meta;
                            applicationId = patchMeta.ApplicationId;
                            var debugRequiredSystemVersion = patchMeta.RequiredSystemVersion;
                            var requiredSystemVersion = debugRequiredSystemVersion.HasValue ?
                                debugRequiredSystemVersion.Value :
                                NintendoContentMeta.GetRequiredSystemVersion();

                            // 最初に作った段階では、history/delta は考慮しない
                            // TODO: 将来的には、パッチ元に指定したものをすべて取り込む
                            extendedHeader = new NintendoPatchMetaExtendedHeader(
                                applicationId,
                                requiredSystemVersion);
                        }
                        break;
                    case NintendoContentMetaConstant.ContentMetaTypeDelta:
                        {
                            var nspDeltaMetaExtendedData = metaExtendedData != null ? metaExtendedData.Find(p => p as DeltaMetaExtendedData != null) as DeltaMetaExtendedData : null;
                            if (nspDeltaMetaExtendedData != null)
                            {
                                var sourceMeta = ReadContentMetaInNsp(nspDeltaMetaExtendedData.Source);
                                var destinationMeta = ReadContentMetaInNsp(nspDeltaMetaExtendedData.Destination);
                                applicationId = UInt64.Parse( sourceMeta.Descendants("ApplicationId").First().Value.Replace("0x", ""), System.Globalization.NumberStyles.HexNumber);

                                // とりあえず全部のコンテントを突っ込んでおく
                                var count = sourceMeta.Descendants("Content").Count();

                                var deltaExtendedData = new NintendoDeltaMetaExtendedData();
                                deltaExtendedData.Source = GetIndicator(sourceMeta);
                                deltaExtendedData.Destination = GetIndicator(destinationMeta);
                                deltaExtendedData.FragmentSets = GetFragmentSets(contentDescriptorList);

                                extendedData = deltaExtendedData;
                                extendedHeader = new NintendoDeltaMetaExtendedHeader(applicationId, (UInt32)(deltaExtendedData.GetSize()));
                            }
                            else
                            {
                                extendedHeader = new NintendoDeltaMetaExtendedHeader(0, 0);
                            }
                        }
                        break;
                    default:
                        break;
                }
            }

            return new ContentMetaDescriptor(type, id, version, attributes, applicationId, requiredDownloadSystemVersion, contentDescriptorList, contentMetaInfoList, extendedHeader, extendedData, new byte[NintendoContentMeta.GetDigestSize()], null, reader.GetKeepGeneration());
        }

        static private ContentMetaDescriptor ConvertContentMetaDescriptor(string type, ContentMetaModel model)
        {
            var descriptor = ExtractContentMetaDescriptor(model);

            // パッチはアプリにコンバート
            if (type == NintendoContentMetaConstant.ContentMetaTypePatch)
            {
                var baseExHeader = (NintendoApplicationMetaExtendedHeader)descriptor.ExtendedHeader;
                var exHeader = new NintendoPatchMetaExtendedHeader(descriptor.Id, baseExHeader.RequiredSystemVersion);

                // ContentDescriptor は無効な値なので初期化しておく
                return new ContentMetaDescriptor(type, baseExHeader.PatchId, descriptor.Version, descriptor.Attributes, descriptor.ApplicationId, descriptor.RequiredDownloadSystemVersion, new List<NintendoContentDescriptor>(), descriptor.ContentMetaInfoList, exHeader, null, descriptor.Digest, descriptor.OriginalRequiredSystemVersion, descriptor.KeepGeneration);
            }
            // パッチ以外はそのまま
            else
            {
                return descriptor;
            }
        }

        static private ModelT GetDeserializedModel<ModelT>(string metaPath, bool nullable)
        {
            var document = new XmlDocument();
            document.Load(metaPath);
            var xmlRoot = (XmlRootAttribute)typeof(ModelT).GetCustomAttributes(typeof(XmlRootAttribute), false)[0];
            var node = document.SelectSingleNode("//" + xmlRoot.ElementName);
            if (node == null)
            {
                if (nullable)
                {
                    return default(ModelT);
                }
                else
                {
                    throw new ArgumentException(string.Format("Could not find <{0}> element in {1}.", xmlRoot.ElementName, metaPath));
                }
            }

            var reader = XmlReader.Create(new StringReader(node.OuterXml));
            var serializer = new XmlSerializer(typeof(ModelT));
            return (ModelT)serializer.Deserialize(reader);
        }
    }

    public class NintendoContentMetaBaseSource : ISource
    {
        public long Size { get; private set; }

        private NintendoContentMetaBase m_contentMetaBase;
        private ISource m_source;
        private bool m_isUpdateDigest;

        internal NintendoContentMetaBaseSource(NintendoContentMetaBase contentMetaBase, bool isUpdateDigest)
        {
            m_isUpdateDigest = isUpdateDigest;
            m_contentMetaBase = contentMetaBase;

            ISource metaBaseSource;
            {
                {
                    var buffer = contentMetaBase.GetBytes();
                    metaBaseSource = new MemorySource(buffer, 0, buffer.Length);
                }

                // TORIAEZU: contentHashSourceList の要素が無い場合、Content ID, Hash 等は書き換わらずに使われる
                var contentHashSourceList = contentMetaBase.GetContentSourceList().Select(tuple => tuple.Item1).ToList();

                for (int i = 0; i < contentHashSourceList.Count; i++)
                {
                    var idSize = contentMetaBase.GetContentInfoIdDataSize();
                    var idOffset = contentMetaBase.GetContentInfoIdDataOffset(i);
                    metaBaseSource = new AdaptedSource(metaBaseSource, new SubSource(contentHashSourceList[i], 0, idSize), idOffset, idSize);

                    var hashSize = contentMetaBase.GetContentInfoHashSize();
                    var hashOffset = contentMetaBase.GetContentInfoHashOffset(i);
                    metaBaseSource = new AdaptedSource(metaBaseSource, contentHashSourceList[i], hashOffset, hashSize);
                }

                if (isUpdateDigest)
                {
                    var digestSize = contentMetaBase.GetDigestSize();
                    var digestOffset = contentMetaBase.GetDigestOffset();
                    metaBaseSource = new AdaptedSource(metaBaseSource, new Sha256StreamHashSource(new SubSource(metaBaseSource, 0, digestOffset)), digestOffset, digestSize);
                }
            }

            m_source = metaBaseSource;
            Size = m_source.Size;
        }

        public ByteData PullData(long offset, int size)
        {
            return m_source.PullData(offset, size);
        }

        public SourceStatus QueryStatus()
        {
            return m_source.QueryStatus();
        }
    }

    // TOOD: コンテンツメタの階層化を直接的に扱う
    //       コンテンツメタの中にコンテンツとコンテンツメタが入るようにして、それを引数に取る
    public class NintendoContentMetaArchiveSource : ISource
    {
        public long Size { get; private set; }

        private ISource m_source;

        internal NintendoContentMetaArchiveSource(NintendoContentMetaBase contentMeta, int keyEncryptionKeyIndex, KeyConfiguration config, bool isProdEncryption, bool isGameCard, bool noEncryption)
        {
            Trace.Assert(!(isProdEncryption && noEncryption));

            var reader = new NintendoContentMetaReader(contentMeta.GetBytes());
            var contentMetaBaseSource = contentMeta.GetSource();

            // コンテンツメタデータをファイルとして含む PartitionFsArchiveSource を作成
            ISource partFsSource;
            {
                PartitionFileSystemInfo partFsInfo = new PartitionFileSystemInfo();
                partFsInfo.version = 0;
                PartitionFileSystemInfo.EntryInfo entry = new PartitionFileSystemInfo.EntryInfo();
                entry.type = "source";
                entry.name = contentMeta.GetEntryName();
                entry.offset = 0;
                entry.size = (ulong)contentMetaBaseSource.Size;
                partFsInfo.entries.Add(entry);

                PartitionFileSystemMeta partFsMetaMgr = new PartitionFileSystemMeta();
                List<ConcatenatedSource.Element> partFsElements = new List<ConcatenatedSource.Element>();
                {
                    byte[] buffer = partFsMetaMgr.Create(partFsInfo);
                    ConcatenatedSource.Element headerElement = new ConcatenatedSource.Element(
                        new MemorySource(buffer, 0, buffer.Length), "meta", 0);
                    ConcatenatedSource.Element bodyElement = new ConcatenatedSource.Element(
                        contentMetaBaseSource, "body", headerElement.Source.Size);
                    partFsElements.Add(headerElement);
                    partFsElements.Add(bodyElement);
                }
                partFsSource = new ConcatenatedSource(partFsElements);
            }

            // コンテンツメタデータをファイルとして含む NintendoContentArchive を作成
            ISource ncaSource;
            {
                NintendoContentFileSystemInfo ncaInfo = new NintendoContentFileSystemInfo();
                ncaInfo.distributionType = isGameCard ? NintendoContentFileSystemMetaConstant.DistributionTypeGameCard
                                                      : NintendoContentFileSystemMetaConstant.DistributionTypeDownload;
                ncaInfo.contentType = (byte)NintendoContentArchiveContentType.Meta;
                ncaInfo.keyGeneration = contentMeta.GetRepresentKeyGeneration();
                ncaInfo.programId = reader.GetId();
                ncaInfo.contentIndex = 0;
                ncaInfo.keyAreaEncryptionKeyIndex = (byte)keyEncryptionKeyIndex;
                ncaInfo.isProdEncryption = isProdEncryption;
                ncaInfo.partitionAlignmentType = (int)NintendoContentArchiveUtil.GetAlignmentType(reader.GetType(), NintendoContentMetaConstant.ContentTypeMeta);
                if (noEncryption)
                {
                    ncaInfo.headerEncryptionType = (byte)NintendoContentArchiveEncryptionType.None;
                }

                NintendoContentFileSystemInfo.EntryInfo fsEntry = new NintendoContentFileSystemInfo.EntryInfo();
                fsEntry.type = "source";
                fsEntry.formatType = NintendoContentFileSystemMetaConstant.FormatTypePartitionFs;
                fsEntry.sourceInterface = new CliCompatibleSource(partFsSource);
                fsEntry.version = 2;
                fsEntry.hashType = (byte)NintendoContentArchiveHashType.Auto;
                fsEntry.encryptionType = noEncryption ? (byte)NintendoContentArchiveEncryptionType.None : (byte)NintendoContentArchiveEncryptionType.Auto;
                fsEntry.partitionIndex = 0;

                ncaInfo.fsEntries.Add(fsEntry);
                ncaInfo.GenerateExistentFsIndicesFromFsEntries();
                ncaSource = new NintendoContentArchiveSource(ncaInfo, config);
            }

            m_source = ncaSource;
            Size = m_source.Size;
        }

        public ByteData PullData(long offset, int size)
        {
            return m_source.PullData(offset, size);
        }

        public SourceStatus QueryStatus()
        {
            return m_source.QueryStatus();
        }
    }

    internal class NintendoContentMetaXmlSource : ISource
    {
        private NintendoContentMetaBase m_contentMetaBase;

        // Attribute 付きで初期化した XmlSerializer はインスタンス毎にアセンブリロードされメモリを圧迫するので静的確保し再利用する
        private static Object m_ContentMetaSerializerLock = new Object();
        private static readonly XmlSerializer m_ContentMetaSerializer = new XmlSerializer(typeof(ContentMetaModel), ContentMetaModel.GetSerializeAttributes());
        private static readonly XmlSerializer m_ApplicationContentMetaSerializer = new XmlSerializer(typeof(ApplicationContentMetaModel), ContentMetaModel.GetSerializeAttributes());
        private static readonly XmlSerializer m_PatchContentMetaSerializer = new XmlSerializer(typeof(PatchContentMetaModel), ContentMetaModel.GetSerializeAttributes());
        private static readonly XmlSerializer m_AddOnContentContentMetaSerializer = new XmlSerializer(typeof(AddOnContentContentMetaModel), ContentMetaModel.GetSerializeAttributes());
        private static readonly XmlSerializer m_DeltaContentMetaSerializer = new XmlSerializer(typeof(DeltaContentMetaModel), ContentMetaModel.GetSerializeAttributes());

        public NintendoContentMetaXmlSource(NintendoContentMetaBase contentMetaBase, ISource metaHashSource, long metaSize)
        {
            m_contentMetaBase = contentMetaBase;
            m_MetaHashSource = metaHashSource;

            // 事前サイズ計算

            var reader = new NintendoContentMetaReader(contentMetaBase.GetBytes());
            var contentMetaModel = PrepareContentMetaModel(contentMetaBase.GetContentMetaDescryptor()); // バイナリに含まれない情報もあるため descriptor を引数にとる
            contentMetaModel.Type = reader.GetType();
            contentMetaModel.SetUInt64Id(reader.GetId());
            contentMetaModel.Version = reader.GetVersion();
            contentMetaModel.AttributeList = ((ContentMetaAttribute)reader.GetAttributes()).ToStringList();
            contentMetaModel.RequiredDownloadSystemVersion = reader.GetRequiredDownloadSystemVersion();

            {
                var emptyDigest = new byte[NintendoContentMeta.GetDigestSize()];
                contentMetaModel.SetDigestBytes(emptyDigest);
            }

            var keyGenerationList = new List<byte>();

            var contentDescriptorList = contentMetaBase.GetContentMetaDescryptor().ContentDescriptorList;
            if (contentDescriptorList.Count > 0)
            {
                contentMetaModel.ContentList = new List<ContentModel>();
                foreach (var descriptor in contentDescriptorList)
                {
                    var model = new ContentModel();
                    model.Type = descriptor.ContentInfo.Type;
                    model.Size = descriptor.ContentInfo.Size;
                    model.KeyGeneration = descriptor.ContentInfo.KeyGeneration;
                    model.IdOffset = descriptor.ContentInfo.IdOffset;
                    keyGenerationList.Add((byte)model.KeyGeneration);
                    // プレースホルダ的に空の ID・Hash を設定
                    var emptyHash = new byte[NintendoContentMeta.GetContentInfoHashSize()];
                    model.SetIdBytes(emptyHash);
                    model.SetHashBytes(emptyHash);
                    contentMetaModel.ContentList.Add(model);
                }
            }

            // INFO: ContentMeta 自身の情報の追加
            {
                contentMetaModel.ContentList = contentMetaModel.ContentList ?? new List<ContentModel>();
                m_MetaModel = new ContentModel();
                contentMetaModel.ContentList.Add(m_MetaModel);
                m_MetaModel.Type = NintendoContentMetaConstant.ContentTypeMeta;
                m_MetaModel.Size = metaSize;
                m_MetaModel.KeyGeneration = m_contentMetaBase.GetRepresentKeyGeneration();
                m_MetaModel.IdOffset = 0;
                keyGenerationList.Add((byte)m_MetaModel.KeyGeneration);
                var emptyHash = new byte[NintendoContentMeta.GetContentInfoHashSize()];
                m_MetaModel.SetIdBytes(emptyHash);
                m_MetaModel.SetHashBytes(emptyHash);
            }

            var contentMetaInfoList = reader.GetContentMetaInfoList();
            if (contentMetaInfoList.Count > 0)
            {
                contentMetaModel.ContentMetaList = new List<ContentMetaModel>();
                foreach (var info in contentMetaInfoList)
                {
                    var model = new ContentMetaModel();
                    model.Type = info.Type;
                    model.SetUInt64Id(info.Id);
                    model.Version = info.Version;
                    model.AttributeList = ((ContentMetaAttribute)info.Attributes).ToStringList();
                    contentMetaModel.ContentMetaList.Add(model);
                }
            }

            contentMetaModel.KeyGenerationMin = keyGenerationList.Min();

            m_model = contentMetaModel;
            Size = GetBytes().Length;
        }

        public long Size { get; private set; }

        public SourceStatus QueryStatus()
        {
            var sourceList = new List<ISource>() { m_contentMetaBase.GetSource(), m_MetaHashSource };
            foreach(var source in sourceList)
            {
                var rangeList = source.QueryStatus().AvailableRangeList;
                if(rangeList.Count == 0 || rangeList[0].Size != source.Size)
                {
                    return new SourceStatus();
                }
            }

            var status = new SourceStatus();
            status.AvailableRangeList.MergingAdd(new Range(0, Size));
            return status;
        }

        public ByteData PullData(long offset, int size)
        {
            {
                var metaBaseSource = m_contentMetaBase.GetSource();
                var metaBin = metaBaseSource.PullData(0, (int)metaBaseSource.Size);
                Debug.Assert(metaBin.Buffer.Count == (int)metaBaseSource.Size);
                var reader = new NintendoContentMetaReader(metaBin.Buffer.Array);
                var contentDescriptorList = reader.GetContentDescriptorList();
                for (int i = 0; i < contentDescriptorList.Count; i++)
                {
                    var info = contentDescriptorList[i].ContentInfo;
                    m_model.ContentList[i].SetIdBytes(info.Id);
                    m_model.ContentList[i].SetHashBytes(info.Hash);
                }

                m_model.SetDigestBytes(reader.GetDigest());
            }

            {
                var metaHash = m_MetaHashSource.PullData(0, NintendoContentMeta.GetContentInfoHashSize());
                Debug.Assert(metaHash.Buffer.Count == NintendoContentMeta.GetContentInfoHashSize());
                m_MetaModel.SetIdBytes(metaHash.Buffer.Array);
                m_MetaModel.SetHashBytes(metaHash.Buffer.Array);
            }

            var bytes = GetBytes();
            return new MemorySource(bytes, 0, bytes.Length).PullData(offset, size);
        }

        static private XmlSerializer GetContentMetaXmlSerializer(Type type)
        {
            if (type.Equals(typeof(ContentMetaModel)))
            {
                return m_ContentMetaSerializer;
            }
            else if (type.Equals(typeof(ApplicationContentMetaModel)))
            {
                return m_ApplicationContentMetaSerializer;
            }
            else if (type.Equals(typeof(PatchContentMetaModel)))
            {
                return m_PatchContentMetaSerializer;
            }
            else if (type.Equals(typeof(AddOnContentContentMetaModel)))
            {
                return m_AddOnContentContentMetaSerializer;
            }
            else if (type.Equals(typeof(DeltaContentMetaModel)))
            {
                return m_DeltaContentMetaSerializer;
            }

            throw new NotImplementedException();
        }

        private byte[] GetBytes()
        {
            var nameSpace = new XmlSerializerNamespaces();
            nameSpace.Add(String.Empty, String.Empty);

            using (var memoryStream = new MemoryStream())
            {
                var sw = new StreamWriter(memoryStream, Encoding.UTF8);
                lock (m_ContentMetaSerializerLock)
                {
                    var serializer = GetContentMetaXmlSerializer(m_ModelType);
                    serializer.Serialize(sw, m_model, nameSpace);
                }
                return memoryStream.ToArray();
            }
        }
        private List<ContentModel> GetContentModelList(List<NintendoContentInfo> contentInfo)
        {
            return contentInfo.Select(p =>
            {
                var m = new ContentModel();
                m.Type = p.Type;
                m.SetIdBytes(p.Id);
                m.Size = p.Size;
                if (p.Hash != null)
                {
                    m.SetHashBytes(p.Hash);
                }
                m.KeyGeneration = p.KeyGeneration;
                m.IdOffset = p.IdOffset;
                return m;
            })
            .ToList();
        }

        private List<DeltaFragmentSetModel> GetFragmentSetModelList(List<NintendoDeltaMetaExtendedData.NintendoDeltaMetaFragmentSet> fragmentSets)
        {
            return fragmentSets.Select(p =>
            {
                var m = new DeltaFragmentSetModel();
                m.FragmentTargetContentType = p.FragmentTargetContentType;
                m.UpdateType = p.UpdateType;
                m.Source = new DeltaContentIndicatorModel();
                m.Destination = new DeltaContentIndicatorModel();
                m.Source.SetContentId(p.SourceContentId);
                m.Source.Size = p.SourceSize;
                m.Destination.SetContentId(p.DestinationContentId);
                m.Destination.Size = p.DestinationSize;
                m.FragmentList = p.Fragments.Select(q =>
                {
                    var m2 = new DeltaFragmentIndicatorModel();
                    m2.ContentInfoIndex = q.ContentInfoIndex;
                    m2.FragmentIndex = q.FragmentIndex;

                    return m2;
                }
                ).ToList();

                return m;
            }
            ).ToList();
        }

        private ContentMetaModel PrepareContentMetaModel(ContentMetaDescriptor descriptor)
        {
            ContentMetaModel contentMetaModel = null;
            switch(descriptor.Type)
            {
                case NintendoContentMetaConstant.ContentMetaTypeApplication:
                    {
                        var exHeader = (NintendoApplicationMetaExtendedHeader)descriptor.ExtendedHeader;

                        var model = new ApplicationContentMetaModel();
                        model.RequiredSystemVersion = exHeader.RequiredSystemVersion;
                        model.SetUInt64PatchId(exHeader.PatchId);

                        contentMetaModel = model;
                        m_ModelType = typeof(ApplicationContentMetaModel);

                    }
                    break;
                case NintendoContentMetaConstant.ContentMetaTypePatch:
                    {
                        var exHeader = (NintendoPatchMetaExtendedHeader)descriptor.ExtendedHeader;

                        var model = new PatchContentMetaModel();
                        model.RequiredSystemVersion = exHeader.RequiredSystemVersion;
                        model.OriginalRequiredSystemVersion = descriptor.OriginalRequiredSystemVersion;
                        model.SetUInt64ApplicationId(exHeader.ApplicationId);

                        var exData = descriptor.ExtendedData as NintendoPatchMetaExtendedData;
                        if(exData != null)
                        {
                            model.HistoryList = exData.Histories.Select(p =>
                            {
                                var m = new PatchHistoryModel();
                                m.Type = p.Type;
                                m.SetId(p.ProgramId);
                                m.Version = p.Version;
                                m.SetDigestBytes(p.Digest);
                                m.ContentList = GetContentModelList(p.ContentInfos);
                                return m;
                            })
                            .ToList();

                            model.DeltaHistoryList = exData.DeltaHistories.Select(p =>
                            {
                                var m = new PatchDeltaHistoryModel();
                                m.Source = new DeltaPatchIndicatorModel();
                                m.Source.SetPatchId(p.Delta.Source.PatchId);
                                m.Source.Version = p.Delta.Source.Version;
                                m.Destination = new DeltaPatchIndicatorModel();
                                m.Destination.SetPatchId(p.Delta.Destination.PatchId);
                                m.Destination.Version = p.Delta.Destination.Version;
                                m.DownloadSize = p.DownloadSize;
                                return m;
                            })
                            .ToList();
                            model.DeltaList = exData.Deltas.Select(p =>
                            {
                                var m = new PatchDeltaModel();
                                m.ContentList = GetContentModelList(p.ContentInfos);

                                m.Source = new DeltaPatchIndicatorModel();
                                m.Source.SetPatchId(p.Delta.Source.PatchId);
                                m.Source.Version = p.Delta.Source.Version;
                                m.Destination = new DeltaPatchIndicatorModel();
                                m.Destination.SetPatchId(p.Delta.Destination.PatchId);
                                m.Destination.Version = p.Delta.Destination.Version;

                                m.FragmentSetList = GetFragmentSetModelList(p.Delta.FragmentSets);

                                return m;
                            })
                            .ToList();

                        }
                        contentMetaModel = model;
                        m_ModelType = typeof(PatchContentMetaModel);

                    }
                    break;
                case NintendoContentMetaConstant.ContentMetaTypeAddOnContent:
                    {
                        var exHeader = (NintendoAddOnContentMetaExtendedHeader)descriptor.ExtendedHeader;

                        var model = new AddOnContentContentMetaModel();
                        model.RequiredApplicationVersion = exHeader.RequiredApplicationVersion;
                        model.SetUInt64ApplicationId(exHeader.ApplicationId);
                        model.Tag = exHeader.Tag;
                        model.Index = GetAddOnContentIndex(descriptor, exHeader);

                        contentMetaModel = model;
                        m_ModelType = typeof(AddOnContentContentMetaModel);
                    }
                    break;
                case NintendoContentMetaConstant.ContentMetaTypeDelta:
                    {
                        var exHeader = descriptor.ExtendedHeader as NintendoDeltaMetaExtendedHeader;
                        var model = new DeltaContentMetaModel();
                        model.SetUInt64ApplicationId(descriptor.ApplicationId);
                        var exData = descriptor.ExtendedData as NintendoDeltaMetaExtendedData;
                        if (exData != null)
                        {
                            model.Source = new DeltaPatchIndicatorModel();
                            model.Source.SetPatchId(exData.Source.PatchId);
                            model.Source.Version = exData.Source.Version;

                            model.Destination = new DeltaPatchIndicatorModel();
                            model.Destination.SetPatchId(exData.Destination.PatchId);
                            model.Destination.Version = exData.Destination.Version;

                            model.FragmentSetList = GetFragmentSetModelList(exData.FragmentSets);
                        }
                        contentMetaModel = model;
                        m_ModelType = typeof(DeltaContentMetaModel);

                    }
                    break;
                default:
                    {
                        contentMetaModel = new ContentMetaModel();
                    }
                    break;
            }
            contentMetaModel.KeepGeneration = descriptor.KeepGeneration;
            return contentMetaModel;
        }

        private UInt32 GetAddOnContentIndex(ContentMetaDescriptor descriptor, NintendoAddOnContentMetaExtendedHeader exHeader)
        {
            // INFO: descriptor が Aoc のものなので、ここでの Id は AddOnContentId を示す。
            return (uint)(descriptor.Id - IdConverter.ConvertToAocBaseId(exHeader.ApplicationId));
        }

        public ContentMetaModel Model { get { return m_model; } }

        private ContentMetaModel m_model;
        private Type m_ModelType = typeof(ContentMetaModel);
        private ContentModel m_MetaModel;
        private ISource m_MetaHashSource;
    }

    public class ContentMetaModelUtils
    {
        public static byte[] GetStringBytes(string str, int size)
        {
            Trace.Assert(str.Length == size * 2);
            var bytes = new byte[size];
            int j = 0;
            for (int i = 0; i < bytes.Length; i++)
            {
                bytes[i] = Convert.ToByte(str.Substring(j, 2), 16);
                j += 2;
            }
            return bytes;
        }
    }

    [XmlRoot("Content", IsNullable = false)]
    public class ContentModel
    {
        [XmlElement("Type")]
        public string Type { get; set; }

        [XmlElement("Id")]
        public string Id { get; set; }

        [XmlElement("Size")]
        public Int64 Size { get; set; }

        [XmlElement("Hash")]
        public string Hash { get; set; }

        [XmlElement("KeyGeneration")]
        public Int32 KeyGeneration { get; set; }

        [XmlElement("IdOffset")]
        public byte IdOffset { get; set; }

        public void SetIdBytes(byte[] bytes)
        {
            Debug.Assert(bytes.Length >= NintendoContentMeta.GetContentInfoIdDataSize());

            Id = BitConverter.ToString(bytes, 0, NintendoContentMeta.GetContentInfoIdDataSize()).Replace("-", string.Empty).ToLower();
        }

        public byte[] GetIdBytes()
        {
            return ContentMetaModelUtils.GetStringBytes(Id, NintendoContentMeta.GetContentInfoIdDataSize());
        }

        public void SetHashBytes(byte[] bytes)
        {
            Debug.Assert(bytes.Length >= NintendoContentMeta.GetContentInfoHashSize());

            Hash = BitConverter.ToString(bytes, 0, NintendoContentMeta.GetContentInfoHashSize()).Replace("-", string.Empty).ToLower();
        }

        public byte[] GetHashBytes()
        {
            return ContentMetaModelUtils.GetStringBytes(Hash, NintendoContentMeta.GetContentInfoHashSize());
        }
    }

    [XmlRoot("ContentMeta", IsNullable = false)]
    [XmlInclude(typeof (ApplicationContentMetaModel))]
    [XmlInclude(typeof (PatchContentMetaModel))]
    [XmlInclude(typeof (AddOnContentContentMetaModel))]
    [XmlInclude(typeof (DeltaContentMetaModel))]
    public class ContentMetaModel
    {
        [XmlElement("Type")]
        public string Type { get; set; }

        [XmlElement("Id")]
        public string Id { get; set; }

        [XmlElement("Version")]
        public UInt32? Version { get; set; }

        [XmlElement("ReleaseVersion")]
        public UInt16? ReleaseVersion { get; set; }

        [XmlElement("PrivateVersion")]
        public UInt16? PrivateVersion { get; set; }

        [XmlElement("ContentMetaAttribute")]
        public List<string> AttributeList { get; set; }

        [XmlElement("RequiredDownloadSystemVersion")]
        public UInt32 RequiredDownloadSystemVersion { get; set; }

        [XmlElement("Content")]
        public List<ContentModel> ContentList { get; set; }

        [XmlElement("ContentMeta")]
        public List<ContentMetaModel> ContentMetaList { get; set; }

        [XmlElement("Digest")]
        public string Digest { get; set; }

        [XmlElement("KeyGenerationMin")]
        public UInt32 KeyGenerationMin { get; set; }

        [XmlElement("KeepGeneration")]
        public bool? KeepGeneration { get; set; }
        [XmlIgnore]
        public bool KeepGenerationSpecified { get { return KeepGeneration != null; } }

        public UInt64 GetUInt64Id()
        {
            return Convert.ToUInt64(Id, 16);
        }

        public void SetUInt64Id(UInt64 id)
        {
            Id = "0x" + id.ToString("x16");
        }

        public void SetDigestBytes(byte[] bytes)
        {
            Debug.Assert(bytes.Length >= NintendoContentMeta.GetDigestSize());

            Digest = BitConverter.ToString(bytes, 0, NintendoContentMeta.GetDigestSize()).Replace("-", string.Empty).ToLower();
        }

        public byte[] GetDigestBytes()
        {
            return ContentMetaModelUtils.GetStringBytes(Digest, NintendoContentMeta.GetDigestSize());
        }

        public void AreValidValues(string parentTagName)
        {
            if (Version != null && (ReleaseVersion != null || PrivateVersion != null))
            {
                throw new ArgumentException("<Version> cannot be specified with <ReleaseVersion> and <PrivateVersion>.");
            }

            IdConverter.ConvertStringToId(Id, "Id", parentTagName);
        }

        public List<NintendoContentDescriptor> GetContentDescriptorList()
        {
            var contentDescriptors = new List<NintendoContentDescriptor>();
            foreach (var content in ContentList)
            {
                var contentDescriptor = new NintendoContentDescriptor();
                contentDescriptor.ContentInfo = new NintendoContentInfo(content.Type, content.GetIdBytes(), content.Size, content.GetHashBytes(), (byte)content.KeyGeneration, content.IdOffset);
                contentDescriptors.Add(contentDescriptor);
            }
            return contentDescriptors;
        }

        public List<NintendoContentMetaInfo> GetContentMetaInfoList()
        {
            var contentMetaInfos = new List<NintendoContentMetaInfo>();
            foreach (var contentMeta in ContentMetaList)
            {
                contentMetaInfos.Add(new NintendoContentMetaInfo(contentMeta.Type, contentMeta.GetUInt64Id(), contentMeta.Version.Value, (Byte)ContentMetaAttributeExtension.FromStringList(AttributeList)));
            }
            return contentMetaInfos;
        }

        static public XmlAttributeOverrides GetSerializeAttributes()
        {
            var overrideAttrs = new XmlAttributeOverrides();
            var ignoreAttr = new XmlAttributes();
            ignoreAttr.XmlIgnore = true;
            overrideAttrs.Add(typeof(ContentMetaModel), "ReleaseVersion", ignoreAttr);
            overrideAttrs.Add(typeof(ContentMetaModel), "PrivateVersion", ignoreAttr);

            return overrideAttrs;
        }
    }

    public class ContentMetaLiteModel
    {
        [XmlElement("Type")]
        public string Type { get; set; }
        [XmlIgnore]
        virtual public bool TypeSpecified { get { return true; } }

        [XmlElement("Id")]
        public string Id { get; set; }

        [XmlElement("Version")]
        public UInt32 Version { get; set; }

        [XmlElement("Digest")]
        public string Digest { get; set; }

        [XmlElement("Content")]
        public List<ContentModel> ContentList { get; set; }
        [XmlIgnore]
        virtual public bool ContentListSpecified { get { return true; } }

        public byte[] GetDigestBytes()
        {
            return ContentMetaModelUtils.GetStringBytes(Digest, NintendoContentMeta.GetDigestSize());
        }
    }

    [XmlRoot("ContentMeta", IsNullable = false)]
    public class ApplicationContentMetaModel : ContentMetaModel
    {
        [XmlElement("RequiredSystemVersion")]
        public UInt32 RequiredSystemVersion { get; set; }

        [XmlElement("PatchId")]
        public string PatchId { get; set; }

        public UInt64 GetUInt64PatchId()
        {
            return Convert.ToUInt64(PatchId, 16);
        }

        public void SetUInt64PatchId(UInt64 id)
        {
            PatchId = "0x" + id.ToString("x16");
        }
    }

    [XmlRoot("ContentMeta", IsNullable = false)]
    public class AddOnContentContentMetaModel : ContentMetaModel
    {
        [XmlElement("RequiredApplicationVersion")]
        public UInt32 RequiredApplicationVersion { get; set; }

        [XmlElement("ApplicationId")]
        public string ApplicationId { get; set; }

        [XmlElement("Tag")]
        public string Tag { get; set; }

        [XmlElement("Index")]
        public UInt32 Index { get; set; }

        public UInt64 GetUInt64ApplicationId()
        {
            return Convert.ToUInt64(ApplicationId, 16);
        }

        public void SetUInt64ApplicationId(UInt64 id)
        {
            ApplicationId = "0x" + id.ToString("x16");
        }
    }
    public class PatchHistoryModel : ContentMetaLiteModel
    {
        public void SetId(UInt64 id)
        {
            Id = "0x" + id.ToString("x16");
        }
        public void SetDigestBytes(byte[] bytes)
        {
            Debug.Assert(bytes.Length >= NintendoContentMeta.GetDigestSize());

            Digest = BitConverter.ToString(bytes, 0, NintendoContentMeta.GetDigestSize()).Replace("-", string.Empty).ToLower();
        }
    }
    public class PatchDeltaHistoryModel
    {
        [XmlElement("Source")]
        public DeltaPatchIndicatorModel Source { get; set; }

        [XmlElement("Destination")]
        public DeltaPatchIndicatorModel Destination { get; set; }

        [XmlElement("DownloadSize")]
        public UInt64 DownloadSize { get; set; }
    }
    public class PatchDeltaModel
    {
        [XmlElement("Content")]
        public List<ContentModel> ContentList { get; set; }

        [XmlElement("Source")]
        public DeltaPatchIndicatorModel Source { get; set; }

        [XmlElement("Destination")]
        public DeltaPatchIndicatorModel Destination { get; set; }

        [XmlElement("FragmentSet")]
        public List<DeltaFragmentSetModel> FragmentSetList { get; set; }

        public PatchDeltaHistoryModel GetHistory()
        {
            var deltaHistory = new PatchDeltaHistoryModel();
            deltaHistory.Source = Source;
            deltaHistory.Destination = Destination;
            UInt64 downloadSize = 0;
            ContentList.ForEach(p => downloadSize += (UInt64)p.Size);
            deltaHistory.DownloadSize = downloadSize;

            return deltaHistory;
        }
    }

    [XmlRoot("ContentMeta", IsNullable = false)]
    public class PatchContentMetaModel : ContentMetaModel
    {
        [XmlElement("RequiredSystemVersion")]
        public UInt32 RequiredSystemVersion { get; set; }

        [XmlElement("OriginalRequiredSystemVersion")]
        public UInt32? OriginalRequiredSystemVersion { get; set; }
        [XmlIgnore]
        public bool OriginalRequiredSystemVersionSpecified { get { return OriginalRequiredSystemVersion != null; } }

        [XmlElement("ApplicationId")]
        public string ApplicationId { get; set; }

        public UInt64 GetUInt64ApplicationId()
        {
            return Convert.ToUInt64(ApplicationId, 16);
        }

        public void SetUInt64ApplicationId(UInt64 id)
        {
            ApplicationId = "0x" + id.ToString("x16");
        }

        [XmlElement("History")]
        public List<PatchHistoryModel> HistoryList { get; set; }

        [XmlElement("DeltaHistory")]
        public List<PatchDeltaHistoryModel> DeltaHistoryList { get; set; }

        [XmlElement("Delta")]
        public List<PatchDeltaModel> DeltaList { get; set; }

        [XmlIgnore]
        public List<SparseStorageArchiveStream> SparseStorages;

        public void Sort()
        {
            if (HistoryList != null)
            {
                // バージョン順にする
                HistoryList.Sort((x, y) => x.Version.CompareTo(y.Version));
            }
            if (DeltaHistoryList != null)
            {
                // source バージョン順にする。同じ場合は delta のバージョン順
                DeltaHistoryList.Sort((x, y) => {
                    if (x.Source.Version == y.Source.Version)
                    {
                        return x.Destination.Version.CompareTo(y.Destination.Version);
                    }
                    else
                    {
                        return x.Source.Version.CompareTo(y.Source.Version);
                    }
                });
            }
        }
    }

    [XmlRoot("SystemUpdate", IsNullable = false)]
    public class SystemUpdateModel
    {
        [XmlElement("ContentMeta")]
        public ContentMetaModel ContentMeta { get; set; }

        public void AreValidValues()
        {
            ContentMeta.AreValidValues("SystemUpdate");
        }
    }

    public class DeltaPatchIndicatorModel
    {
        [XmlElement("PatchId")]
        public string PatchId { get; set; }

        [XmlElement("Version")]
        public UInt32 Version { get; set; }

        public void SetPatchId(UInt64 id)
        {
            PatchId = "0x" + id.ToString("x16");
        }
    }
    public class DeltaFragmentIndicatorModel
    {
        [XmlElement("ContentInfoIndex")]
        public UInt16 ContentInfoIndex { get; set; }
        [XmlElement("FragmentIndex")]
        public UInt16 FragmentIndex { get; set; }

    }
    public class DeltaContentIndicatorModel
    {
        [XmlElement("ContentId")]
        public string ContentId { get; set; }
        [XmlElement("Size")]
        public Int64 Size { get; set; }
        public void SetContentId(byte[] contentId)
        {
            ContentId = BitConverter.ToString(contentId, 0, NintendoContentMeta.GetContentInfoIdDataSize()).Replace("-", string.Empty).ToLower();
        }
    }
    public class DeltaFragmentSetModel
    {
        [XmlElement("TargetContentType")]
        public string FragmentTargetContentType { get; set; }
        [XmlElement("Source")]
        public DeltaContentIndicatorModel Source { get; set; }
        [XmlElement("Destination")]
        public DeltaContentIndicatorModel Destination { get; set; }
        [XmlElement("UpdateType")]
        public string UpdateType { get; set; }
        [XmlElement("Fragment")]
        public List<DeltaFragmentIndicatorModel> FragmentList { get; set; }

    }

    [XmlRoot("ContentMeta", IsNullable = false)]
    public class DeltaContentMetaModel : ContentMetaModel
    {
        [XmlElement("ApplicationId")]
        public string ApplicationId { get; set; }

        public UInt64 GetUInt64ApplicationId()
        {
            return Convert.ToUInt64(ApplicationId, 16);
        }

        public void SetUInt64ApplicationId(UInt64 id)
        {
            ApplicationId = "0x" + id.ToString("x16");
        }

        [XmlElement("Source")]
        public DeltaPatchIndicatorModel Source { get; set; }

        [XmlElement("Destination")]
        public DeltaPatchIndicatorModel Destination { get; set; }

        [XmlElement("FragmentSet")]
        public List<DeltaFragmentSetModel> FragmentSetList { get; set; }
    }

}
