﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.Xml.Linq;
using System.Xml;
using System.Text;
using System.Text.RegularExpressions;
using System.Security.Cryptography;
using System.Resources;
using YamlDotNet.RepresentationModel;
using Nintendo.Authoring.FileSystemMetaLibrary;
using Nintendo.Authoring.CryptoLibrary;

namespace Nintendo.Authoring.AuthoringLibrary
{
    using EntryFilterRule = Pair<FilterType, string>;
    using DirectoryConnector = Pair<string, string>;

    public enum NintendoContentArchiveContentType
    {
        Program = NintendoContentFileSystemMetaConstant.ContentTypeProgram,
        Meta = NintendoContentFileSystemMetaConstant.ContentTypeMeta,
        Control = NintendoContentFileSystemMetaConstant.ContentTypeControl,
        Manual = NintendoContentFileSystemMetaConstant.ContentTypeManual,
        Data = NintendoContentFileSystemMetaConstant.ContentTypeData,
        PublicData = NintendoContentFileSystemMetaConstant.ContentTypePublicData,
        Invalid = 255,
    }

    public enum NintendoContentArchiveHashType
    {
        Auto = NintendoContentFileSystemMetaConstant.HashTypeAuto,
        HierarchicalSha256 = NintendoContentFileSystemMetaConstant.HashTypeHierarchicalSha256,
        HierarchicalIntegrity = NintendoContentFileSystemMetaConstant.HashTypeHierarchicalIntegrity,
    }

    public enum NintendoContentArchiveEncryptionType
    {
        Auto = NintendoContentFileSystemMetaConstant.EncryptionTypeAuto,
        None = NintendoContentFileSystemMetaConstant.EncryptionTypeNone,
        AesCtr = NintendoContentFileSystemMetaConstant.EncryptionTypeAesCtr,
        AesCtrEx = NintendoContentFileSystemMetaConstant.EncryptionTypeAesCtrEx,
    }

    public enum NintendoContentArchiveHeaderEncryptionType
    {
        Auto = NintendoContentFileSystemMetaConstant.HeaderEncryptionTypeAuto,
        None = NintendoContentFileSystemMetaConstant.HeaderEncryptionTypeNone,
    }

    public enum NintendoContentArchiveEncryptionKeyIndex
    {
        AesCtr   = NintendoContentFileSystemMetaConstant.DecryptionKeyIndexAesCtr,
        AesCtrHw = NintendoContentFileSystemMetaConstant.DecryptionKeyIndexAesCtrHw,
    }

    public enum NintendoContentArchivePartitionType
    {
        Code,
        Data,
        Logo
    }

    public enum NintendoContentArchivePartitionAlignmentType
    {
        AlignHashBlockSize = NintendoContentFileSystemMetaConstant.PartitionAlignmentTypeHashBlockSize,
        AlignSectorOrEncryptionBlockSize = NintendoContentFileSystemMetaConstant.PartitionAlignmentTypeSectorOrEncryptionBlockSize,
    }

    public class NintendoContentArchiveUtil
    {
        static public bool NeedsAesCtrGeneration(NintendoContentFileSystemInfo.EntryInfo entryInfo)
        {
            return entryInfo.encryptionType == (byte)NintendoContentArchiveEncryptionType.AesCtr ||
                   entryInfo.encryptionType == (byte)NintendoContentArchiveEncryptionType.AesCtrEx;
        }

        static public bool NeedsAesCtrSecureValue(NintendoContentFileSystemInfo.EntryInfo entryInfo)
        {
            return entryInfo.encryptionType == (byte)NintendoContentArchiveEncryptionType.AesCtr ||
                   entryInfo.encryptionType == (byte)NintendoContentArchiveEncryptionType.AesCtrEx;
        }

        static public uint GetAesCtrSecureValue(string contentType, int partitionIndex, byte programIndex)
        {
            return GetAesCtrSecureValue(contentType, partitionIndex) + ((uint)programIndex << 24);
        }

        static private uint GetAesCtrSecureValue(string contentType, int partitionIndex)
        {
            switch (contentType)
            {
                case NintendoContentMetaConstant.ContentTypeProgram:
                    switch ((NintendoContentArchivePartitionType)partitionIndex)
                    {
                        case NintendoContentArchivePartitionType.Code:
                            return 1; // TODO: 調整の必要あり
                        case NintendoContentArchivePartitionType.Data:
                            return 2; // TODO: 調整の必要あり
                    }
                    break;

                case NintendoContentMetaConstant.ContentTypeHtmlDocument:
                    return (uint)NintendoContentMeta.ConvertContentType(contentType); // TODO: 調整の必要あり
                case NintendoContentMetaConstant.ContentTypeLegalInformation:
                    return (uint)NintendoContentMeta.ConvertContentType(contentType); // TODO: 調整の必要あり
            }
            return 0;
        }

        static public NintendoContentArchivePartitionAlignmentType GetAlignmentType(string metaType, string ncmContentType)
        {
            if (metaType == null)
            {
                throw new ArgumentException("meta type must not be null or empty.");
            }

            // アプリ、パッチ、アドオンコンテンツ
            if (metaType == NintendoContentMetaConstant.ContentMetaTypeApplication || metaType == NintendoContentMetaConstant.ContentMetaTypePatch)
            {
                if (ncmContentType == NintendoContentMetaConstant.ContentTypeProgram || ncmContentType == NintendoContentMetaConstant.ContentTypeHtmlDocument)
                {
                    return NintendoContentArchivePartitionAlignmentType.AlignHashBlockSize;
                }
            }
            else if (metaType == NintendoContentMetaConstant.ContentMetaTypeAddOnContent)
            {
                if (ncmContentType == NintendoContentMetaConstant.ContentTypeData || ncmContentType == NintendoContentMetaConstant.ContentTypePublicData)
                {
                    return NintendoContentArchivePartitionAlignmentType.AlignHashBlockSize;
                }
            }

            // デルタフラグメント
            if (metaType == NintendoContentMetaConstant.ContentMetaTypeDelta || metaType == NintendoContentMetaConstant.ContentMetaTypePatch)
            {
                if (ncmContentType == NintendoContentMetaConstant.ContentTypeDeltaFragment)
                {
                    return NintendoContentArchivePartitionAlignmentType.AlignHashBlockSize;
                }
            }

            // その他
            return NintendoContentArchivePartitionAlignmentType.AlignSectorOrEncryptionBlockSize;
        }

        // TODO: NintendoContentArchiveSource の static 関数をこちらに移動する
    }

    public class NintendoContentArchiveBuildLog
    {
        public NintendoContentArchiveBuildLog()
        {
            ContentIndex = -1;
            ContentType = string.Empty;
            OriginalPath = string.Empty;
            TargetPath = string.Empty;
            OutputPath = string.Empty;
            CacheDirectory = null;
            NeedsOutputLog = false;
        }

        public NintendoContentArchiveBuildLog(string targetPath, bool needsOutput, string cacheDir)
        {
            Debug.Assert(!string.IsNullOrEmpty(targetPath));

            ContentIndex = -1;
            ContentType = string.Empty;
            OriginalPath = string.Empty;
            TargetPath = targetPath;
            OutputPath = targetPath;
            CacheDirectory = cacheDir;
            NeedsOutputLog = needsOutput;

            CreateCacheDirectory();
        }

        public NintendoContentArchiveBuildLog(string originalPath, string targetPath, bool needsOutput, string cacheDir)
        {
            Debug.Assert(!string.IsNullOrEmpty(targetPath));

            ContentIndex = -1;
            ContentType = string.Empty;
            OriginalPath = originalPath;
            TargetPath = targetPath;
            OutputPath = targetPath;
            CacheDirectory = cacheDir;
            NeedsOutputLog = needsOutput;

            CreateCacheDirectory();
        }

        public int ContentIndex { get; private set; }
        public string ContentType { get; private set; }
        public string OriginalPath { get; private set; }
        public string TargetPath { get; private set; }
        private string OutputPath { get; set; }
        public string CacheDirectory { get; private set; }
        public bool NeedsOutputLog { get; private set; }
        public bool NeedsOutputCache { get { return !string.IsNullOrEmpty(CacheDirectory); } }

        public void SetContent(int contentIndex, string contentType)
        {
            Debug.Assert((0 <= contentIndex) == !string.IsNullOrEmpty(contentType));

            ContentIndex = contentIndex;
            ContentType = contentType;

            if (0 <= ContentIndex)
            {
                OutputPath = string.Format("{0}.c{1}.{2}", TargetPath, ContentIndex, ContentType);
            }
        }

        public string GetOutputPath(string extension)
        {
            return OutputPath + extension;
        }

        private void CreateCacheDirectory()
        {
            if (!string.IsNullOrEmpty(CacheDirectory) && !Directory.Exists(CacheDirectory) )
            {
                Directory.CreateDirectory(CacheDirectory);
            }
        }
    }

    public class NintendoContentAdfWriter
    {
        private string m_adfPath;
        private string m_type;
        private ulong m_ProgramId;
        private byte m_ProgramIndex;
        private byte[] m_RightsId;
        private string m_DescPath;
        private int m_KeyAreaEncryptionKeyIndex;
        private byte m_KeyGeneration;
        private bool m_IsHardwareEncryptionKeyEmbedded;
        private bool m_NoEncryption; // TODO: Configuration としてまとめる？

        public NintendoContentAdfWriter(string adfPath, string type, string metaPath, string descPath, int keyAreaEncryptionKeyIndex, byte keyGeneration, ulong programId, byte programIndex, bool isHardwareEncryptionKeyEmbedded, bool noEncryption)
        {
            m_ProgramId = programId;
            m_RightsId = null;
            m_adfPath = adfPath;
            m_type = type;
            m_DescPath = descPath;
            m_KeyAreaEncryptionKeyIndex = noEncryption ? 0 : keyAreaEncryptionKeyIndex;
            m_KeyGeneration = noEncryption ? (byte)0 : keyGeneration;
            m_IsHardwareEncryptionKeyEmbedded = noEncryption ? false : isHardwareEncryptionKeyEmbedded;
            m_NoEncryption = noEncryption;
        }

        public NintendoContentAdfWriter(string adfPath, string type, string metaPath, string descPath, int keyAreaEncryptionKeyIndex, byte keyGeneration, ulong programId, byte programIndex, byte[] rightsId, bool isHardwareEncryptionKeyEmbedded, bool noEncryption)
        {
            m_ProgramId = programId;
            m_ProgramIndex = programIndex;
            m_RightsId = noEncryption ? null : rightsId;
            m_adfPath = adfPath;
            m_type = type;
            m_DescPath = descPath;
            m_KeyAreaEncryptionKeyIndex = noEncryption ? 0 : keyAreaEncryptionKeyIndex;
            m_KeyGeneration = noEncryption ? (byte)0 : keyGeneration;
            m_IsHardwareEncryptionKeyEmbedded = noEncryption ? false : isHardwareEncryptionKeyEmbedded;
            m_NoEncryption = noEncryption;
        }

        public void Write(List<DirectoryConnector> dirPaths, List<EntryFilterRule> filterRules, List<Tuple<string, Int64>> originalRomFsListFileInfo = null)
        {
            using (var adf = new StreamWriter(m_adfPath, false, Encoding.UTF8))
            {
                adf.WriteLine("formatType : NintendoContent");
                adf.WriteLine("version : 0");
                adf.WriteLine("contentType : {0}", m_type);
                adf.WriteLine("keyGeneration : {0}", m_KeyGeneration);
                adf.WriteLine("programId : {0:X16}", m_ProgramId);
                adf.WriteLine("programIndex : {0}", m_ProgramIndex);
                if (m_RightsId != null)
                {
                    adf.WriteLine("rightsId : {0:X32}", BitConverter.ToString(m_RightsId).Replace("-", string.Empty));
                }
                adf.WriteLine("contentIndex : 0");
                if (m_DescPath != null)
                {
                    adf.WriteLine("descFilePath : " + Path.GetFullPath(m_DescPath));
                }
                if (m_KeyAreaEncryptionKeyIndex != 0)
                {
                    adf.WriteLine("keyAreaEncryptionKeyIndex : " + m_KeyAreaEncryptionKeyIndex);
                }
                if (m_IsHardwareEncryptionKeyEmbedded)
                {
                    adf.WriteLine("hardwareEncryptionKey : true");
                }
                if (m_NoEncryption)
                {
                    adf.WriteLine("headerEncryptionType : {0}", (int)NintendoContentArchiveHeaderEncryptionType.None);
                }

                adf.WriteLine("entries :");

                switch (m_type)
                {
                    case NintendoContentMetaConstant.ContentTypeProgram:
                        {
                            var basePath = Path.GetFullPath(Path.GetDirectoryName(m_adfPath)) + "\\" +
                                Path.GetFileNameWithoutExtension(m_adfPath);

                            adf.WriteLine("  - type : format");
                            adf.WriteLine("    partitionType : code");
                            adf.WriteLine("    formatType : {0}", NintendoContentFileSystemMetaConstant.FormatTypePartitionFs);
                            adf.WriteLine("    version : 2");
                            adf.WriteLine("    hashType : {0}", (int)NintendoContentArchiveHashType.HierarchicalSha256);
                            adf.WriteLine("    encryptionType : {0}", m_NoEncryption ? (int)NintendoContentArchiveEncryptionType.None : (int)NintendoContentArchiveEncryptionType.AesCtr);

                            string codeAdfPath = basePath + ".code.adf";
                            adf.WriteLine("    path : {0}", codeAdfPath);
                            PartitionFsAdfWriter codeAdfWriter = new PartitionFsAdfWriter(codeAdfPath);
                            codeAdfWriter.Write(dirPaths[0].first);

                            if (dirPaths.Count >= 2)
                            {
                                adf.WriteLine("  - type : format");
                                adf.WriteLine("    partitionType : data");
                                adf.WriteLine("    formatType : {0}", NintendoContentFileSystemMetaConstant.FormatTypeRomFs);
                                adf.WriteLine("    version : 2");
                                adf.WriteLine("    hashType : {0}", (int)NintendoContentArchiveHashType.HierarchicalIntegrity);
                                adf.WriteLine("    encryptionType : {0}", m_NoEncryption ? (int)NintendoContentArchiveEncryptionType.None : (int)NintendoContentArchiveEncryptionType.AesCtr);

                                string romAdfPath = basePath + ".rom.adf";
                                adf.WriteLine("    path : {0}", romAdfPath);
                                RomFsAdfWriter romAdfWriter = new RomFsAdfWriter(romAdfPath);
                                romAdfWriter.Write(dirPaths[1], filterRules, originalRomFsListFileInfo);
                            }

                            if (dirPaths.Count >= 3)
                            {
                                adf.WriteLine("  - type : format");
                                adf.WriteLine("    partitionType : logo");
                                adf.WriteLine("    formatType : {0}", NintendoContentFileSystemMetaConstant.FormatTypePartitionFs);
                                adf.WriteLine("    version : 2");
                                adf.WriteLine("    hashType : {0}", (int)NintendoContentArchiveHashType.HierarchicalSha256);
                                adf.WriteLine("    encryptionType : {0}", (int)NintendoContentArchiveEncryptionType.None);

                                string logoAdfPath = basePath + ".logo.adf";
                                adf.WriteLine("    path : {0}", logoAdfPath);
                                var logoAdfWriter = new PartitionFsAdfWriter(logoAdfPath);
                                logoAdfWriter.Write(dirPaths[2].first);
                            }
                        }
                        break;
                    case NintendoContentMetaConstant.ContentTypeControl:
                    case NintendoContentMetaConstant.ContentTypeData:
                    case NintendoContentMetaConstant.ContentTypePublicData:
                    case NintendoContentMetaConstant.ContentTypeHtmlDocument:
                    case NintendoContentMetaConstant.ContentTypeLegalInformation:
                    case NintendoContentMetaConstant.ContentTypeDeltaFragment:
                        {
                            adf.WriteLine("  - type : format");
                            adf.WriteLine("    formatType : {0}", NintendoContentFileSystemMetaConstant.FormatTypeRomFs);
                            adf.WriteLine("    version : 2");
                            adf.WriteLine("    hashType : {0}", (int)NintendoContentArchiveHashType.HierarchicalIntegrity);
                            adf.WriteLine("    encryptionType : {0}", m_NoEncryption ? (int)NintendoContentArchiveEncryptionType.None : (int)NintendoContentArchiveEncryptionType.AesCtr);
                            string romAdfPath = Path.GetFullPath(Path.GetDirectoryName(m_adfPath)) + "\\" +
                                Path.GetFileNameWithoutExtension(m_adfPath) + ".rom.adf";
                            adf.WriteLine("    path : {0}", romAdfPath);
                            RomFsAdfWriter romAdfWriter = new RomFsAdfWriter(romAdfPath);
                            if (m_type == NintendoContentMetaConstant.ContentTypeData || m_type ==  NintendoContentMetaConstant.ContentTypePublicData)
                            {
                                romAdfWriter.Write(dirPaths, filterRules);
                            }
                            else
                            {
                                romAdfWriter.Write(dirPaths, null);
                            }
                        }
                        break;
                    default:
                        throw new NotImplementedException();
                }
            }
        }
    }

    public class NintendoContentAdfReader
    {
        private string m_adfPath;

        public NintendoContentAdfReader(string adfPath)
        {
            m_adfPath = adfPath;
        }

        internal static Byte ConvertToContentTypeByte(string contentType)
        {
            switch(contentType)
            {
                case NintendoContentMetaConstant.ContentTypeProgram: return (Byte)NintendoContentFileSystemMetaConstant.ContentTypeProgram;
                case NintendoContentMetaConstant.ContentTypeControl: return (Byte)NintendoContentFileSystemMetaConstant.ContentTypeControl;
                case NintendoContentMetaConstant.ContentTypeData: return (Byte)NintendoContentFileSystemMetaConstant.ContentTypeData;
                case NintendoContentMetaConstant.ContentTypePublicData: return (Byte)NintendoContentFileSystemMetaConstant.ContentTypePublicData;
                case NintendoContentMetaConstant.ContentTypeMeta: return (Byte)NintendoContentFileSystemMetaConstant.ContentTypeMeta;
                case NintendoContentMetaConstant.ContentTypeHtmlDocument: return (Byte)NintendoContentFileSystemMetaConstant.ContentTypeManual;
                case NintendoContentMetaConstant.ContentTypeLegalInformation: return (Byte)NintendoContentFileSystemMetaConstant.ContentTypeManual;
                case NintendoContentMetaConstant.ContentTypeDeltaFragment: return (Byte)NintendoContentFileSystemMetaConstant.ContentTypeData;
                default: throw new ArgumentException();
            }
        }

        internal byte[] ReadAcid(string npdmPath)
        {
            byte[] acid;
            using (var npdm = new FileStream(npdmPath, FileMode.Open, FileAccess.Read, FileShare.Read, 4096, FileOptions.SequentialScan))
            {
                var head = new byte[0x80];
                npdm.Read(head, 0, head.Length);

                const int offsetAcidOffset = 0x78;
                var acidOffset = BitConverter.ToUInt32(head, offsetAcidOffset);

                const int offsetAcidSize = 0x7C;
                var acidSize = BitConverter.ToUInt32(head, offsetAcidSize);

                acid = new byte[acidSize];
                npdm.Seek(acidOffset, SeekOrigin.Begin);
                npdm.Read(acid, 0, (int)acidSize);
            }
            return acid;
        }

        internal ulong ReadProgramId(string npdmPath)
        {
            byte[] aci;
            using (var npdm = new FileStream(npdmPath, FileMode.Open, FileAccess.Read, FileShare.Read, 4096, FileOptions.SequentialScan))
            {
                var head = new byte[0x80];
                npdm.Read(head, 0, head.Length);

                const int offsetAciOffset = 0x70;
                var aciOffset = BitConverter.ToUInt32(head, offsetAciOffset);

                const int offsetAciSize = 0x74;
                var aciSize = BitConverter.ToUInt32(head, offsetAciSize);

                aci = new byte[aciSize];
                npdm.Seek(aciOffset, SeekOrigin.Begin);
                npdm.Read(aci, 0, (int)aciSize);
            }

            const int offsetAciProgramIdOffset = 0x10;
            return BitConverter.ToUInt64(aci, offsetAciProgramIdOffset);
        }

        private int GetPartitionIndex(string partitionType)
        {
            switch (partitionType)
            {
                case "code": return (int)NintendoContentArchivePartitionType.Code;
                case "data": return (int)NintendoContentArchivePartitionType.Data;
                case "logo": return (int)NintendoContentArchivePartitionType.Logo;
                default: throw new ArgumentException("Invalid partition type: " + partitionType.ToString());
            }
        }

        public static byte[] GetAcidFromDesc(string descFilePath)
        {
            byte[] acidInDesc = null;
            if (descFilePath != null)
            {
                var xml = new XmlDocument();
                xml.Load(descFilePath);
                acidInDesc = Convert.FromBase64String(xml.DocumentElement.SelectSingleNode("/Desc/Acid").InnerText);
            }
            return acidInDesc;
        }

        public static void RetrieveInfoFromDesc(ref NintendoContentFileSystemInfo fileSystemInfo, string descFilePath)
        {
            // Desc ファイルから署名鍵を読み込み
            if (descFilePath != null)
            {
                var xml = new XmlDocument();
                xml.Load(descFilePath);

                // Programコンテンツの場合はDescファイルパスを設定
                if (fileSystemInfo.contentType == (Byte)NintendoContentFileSystemMetaConstant.ContentTypeProgram)
                {
                    fileSystemInfo.descFileName = Path.GetFileName(descFilePath);
                }

                // Desc が署名されていない場合は鍵を null に
                try
                {
                    fileSystemInfo.header2SignKeyModulus = Convert.FromBase64String(xml.DocumentElement.SelectSingleNode("/Desc/RSAKeyValue/Modulus").InnerText);
                    fileSystemInfo.header2SignKeyPrivateExponent = Convert.FromBase64String(xml.DocumentElement.SelectSingleNode("/Desc/RSAKeyValue/D").InnerText);
                    fileSystemInfo.header2SignKeyPublicExponent = Convert.FromBase64String(xml.DocumentElement.SelectSingleNode("/Desc/RSAKeyValue/Exponent").InnerText);
                }
                catch
                {
                    fileSystemInfo.header2SignKeyModulus = null;
                    fileSystemInfo.header2SignKeyPrivateExponent = null;
                    fileSystemInfo.header2SignKeyPublicExponent = null;
                }
            }
        }

        public NintendoContentFileSystemInfo GetFileSystemInfo(string metaType)
        {
            NintendoContentFileSystemInfo fileSystemInfo = new NintendoContentFileSystemInfo();

            using (var adf = new StreamReader(m_adfPath, Encoding.UTF8))
            {
                var yamlStream = new YamlStream();
                yamlStream.Load(adf);

                YamlMappingNode rootNode;
                YamlSequenceNode entries;
                string descPath;
                string contentDetail;
                byte programIndex;

                try
                {
                    rootNode = (YamlMappingNode)yamlStream.Documents[0].RootNode;
                    YamlScalarNode formatType = (YamlScalarNode)rootNode.Children[new YamlScalarNode("formatType")];
                    if (formatType.Value != "NintendoContent")
                    {
                        throw new ArgumentException();
                    }
                    YamlScalarNode contentType = (YamlScalarNode)rootNode.Children[new YamlScalarNode("contentType")];
                    YamlScalarNode programId = (YamlScalarNode)rootNode.Children[new YamlScalarNode("programId")];
                    YamlScalarNode contentIndex = (YamlScalarNode)rootNode.Children[new YamlScalarNode("contentIndex")];
                    entries = (YamlSequenceNode)rootNode.Children[new YamlScalarNode("entries")];

                    fileSystemInfo.contentType    = ConvertToContentTypeByte(contentType.Value);
                    fileSystemInfo.programId      = Convert.ToUInt64(programId.Value, 16);
                    fileSystemInfo.contentIndex   = Convert.ToUInt32(contentIndex.Value);
                    fileSystemInfo.partitionAlignmentType = (int)NintendoContentArchiveUtil.GetAlignmentType(metaType, contentType.Value);

                    contentDetail = contentType.Value;
                }
                catch
                {
                    throw new ArgumentException("invalid format .adf file.");
                }

                // createnca のとき、RightsId の項目がないので別処理
                try
                {
                    YamlScalarNode rightsId = (YamlScalarNode)rootNode.Children[new YamlScalarNode("rightsId")];

                    Func<string, byte[]> ToHexFromString = (string str) =>
                    {
                        int length = str.Length / 2;
                        byte[] bytes = new byte[length];
                        for (int i = 0, j = 0; i < length; i++, j += 2)
                        {
                            bytes[i] = Convert.ToByte(str.Substring(j, 2), 16);
                        }
                        return bytes;
                    };

                    fileSystemInfo.rightsId = ToHexFromString(rightsId.Value);
                }
                catch
                {
                    fileSystemInfo.rightsId = null;
                }

                // keyGeneration はデフォルトで最新の値
                try
                {
                    YamlScalarNode keyGeneration = (YamlScalarNode)rootNode.Children[new YamlScalarNode("keyGeneration")];
                    fileSystemInfo.keyGeneration  = Convert.ToByte(keyGeneration.Value);
                }
                catch
                {
                    fileSystemInfo.keyGeneration  = NintendoContentFileSystemMetaConstant.CurrentKeyGeneration;
                }

                // keyIndex はデフォルトで 0
                try
                {
                    YamlScalarNode keyAreaEncryptionKeyIndex = (YamlScalarNode)rootNode.Children[new YamlScalarNode("keyAreaEncryptionKeyIndex")];
                    fileSystemInfo.keyAreaEncryptionKeyIndex = Convert.ToByte(keyAreaEncryptionKeyIndex.Value);
                }
                catch
                {
                    fileSystemInfo.keyAreaEncryptionKeyIndex = 0;
                }

                try
                {
                    YamlScalarNode isHardwareEncryptionKeyEmbedded = (YamlScalarNode)rootNode.Children[new YamlScalarNode("hardwareEncryptionKey")];
                    fileSystemInfo.isHardwareEncryptionKeyEmbedded = Convert.ToBoolean(isHardwareEncryptionKeyEmbedded.Value);
                }
                catch
                {
                    fileSystemInfo.isHardwareEncryptionKeyEmbedded = false;
                }

                // programIndex はデフォルトで 0
                try
                {
                    YamlScalarNode tmpProgramIndex = (YamlScalarNode)rootNode.Children[new YamlScalarNode("programIndex")];
                    programIndex = Convert.ToByte(tmpProgramIndex.Value);
                }
                catch
                {
                    programIndex = 0;
                }

                try
                {
                    YamlScalarNode headerEncryptionType = (YamlScalarNode)rootNode.Children[new YamlScalarNode("headerEncryptionType")];
                    fileSystemInfo.headerEncryptionType = Convert.ToByte(headerEncryptionType.Value);
                }
                catch
                {
                    fileSystemInfo.headerEncryptionType = (byte)NintendoContentArchiveHeaderEncryptionType.Auto;
                }

                // Program コンテンツには Desc ファイル指定が必須
                if ((NintendoContentArchiveContentType)fileSystemInfo.contentType == NintendoContentArchiveContentType.Program)
                {
                    try
                    {
                        descPath = ((YamlScalarNode)rootNode.Children[new YamlScalarNode("descFilePath")]).Value;
                    }
                    catch
                    {
                        throw new ArgumentException(".desc file is not specified for program content.");
                    }
                }
                else
                {
                    descPath = null;
                }

                // Desc ファイルから ACID と署名鍵を読み込み
                byte[] acidInDesc = null;
                if (descPath != null)
                {
                    RetrieveInfoFromDesc(ref fileSystemInfo, descPath);
                    acidInDesc = GetAcidFromDesc(descPath);
                }

                foreach (YamlMappingNode entry in entries)
                {
                    NintendoContentFileSystemInfo.EntryInfo entryInfo = new NintendoContentFileSystemInfo.EntryInfo();
                    string adfPath = null;
                    foreach (var child in entry)
                    {
                        switch (((YamlScalarNode)child.Key).Value)
                        {
                            case "type":
                                entryInfo.type = ((YamlScalarNode)child.Value).Value;
                                break;
                            case "partitionType":
                                entryInfo.partitionType = ((YamlScalarNode)child.Value).Value;
                                entryInfo.partitionIndex = GetPartitionIndex(entryInfo.partitionType);
                                break;
                            case "formatType":
                                entryInfo.formatType = ((YamlScalarNode)child.Value).Value;
                                break;
                            case "version":
                                entryInfo.version = Convert.ToUInt16(((YamlScalarNode)child.Value).Value);
                                break;
                            case "hashType":
                                entryInfo.hashType = Convert.ToByte(((YamlScalarNode)child.Value).Value);
                                break;
                            case "encryptionType":
                                entryInfo.encryptionType = Convert.ToByte(((YamlScalarNode)child.Value).Value);
                                break;
                            case "generation":
                                entryInfo.generation = Convert.ToUInt32(((YamlScalarNode)child.Value).Value);
                                break;
                            case "path":
                                adfPath = ((YamlScalarNode)child.Value).Value;
                                break;
                            default:
                                throw new ArgumentException("invalid format .adf file. invalid key is specified\n" + entry.ToString());
                        }
                    }
                    if (entryInfo.type == null || entryInfo.formatType == null || adfPath == null)
                    {
                        throw new ArgumentException("invalid format .adf file. \"type\" or \"formatType\" is not specified\n" + entry.ToString());
                    }

                    // TORIAEZU: version は固定値
                    entryInfo.version = 2;
                    entryInfo.hashType = NintendoContentArchiveSource.VerifyHashType(fileSystemInfo, entryInfo);
                    entryInfo.encryptionType = NintendoContentArchiveSource.VerifyEncryptionType(fileSystemInfo, entryInfo);

                    // AES-CTR の IV 先頭 4 バイトに設定する数値を決定
                    if (NintendoContentArchiveUtil.NeedsAesCtrSecureValue(entryInfo))
                    {
                        entryInfo.secureValue = NintendoContentArchiveUtil.GetAesCtrSecureValue(contentDetail, entryInfo.partitionIndex, programIndex);
                    }

                    switch (entryInfo.formatType)
                    {
                        case NintendoContentFileSystemMetaConstant.FormatTypePartitionFs:
                            {
                                var adfReader = new PartitionFsAdfReader(adfPath);
                                var partitionFsInfo = adfReader.GetFileSystemInfo();
                                if (acidInDesc != null && entryInfo.partitionType == "code")
                                {
                                    var npdmEntries = partitionFsInfo.entries.Where(p => p.name == "main.npdm");
                                    if ((NintendoContentArchiveContentType)fileSystemInfo.contentType == NintendoContentArchiveContentType.Program &&
                                        npdmEntries.Count() != 1)
                                    {
                                        throw new ArgumentException("\"main.npdm\" must be included in the code region.");
                                    }
                                    foreach (var npdmEntry in npdmEntries)
                                    {
                                        // desc ファイルと npdm の中の ACID が一致するか検証する
                                        var acidInNpdm = ReadAcid(npdmEntry.path);
                                        if (!acidInNpdm.SequenceEqual(acidInDesc))
                                        {
                                            throw new ArgumentException(".desc file specified differ from the one used to build code.");
                                        }
                                        // programId と npdm の中の ProgramId が一致するか検証する
                                        var programIdInNpdm = ReadProgramId(npdmEntry.path);
                                        if (fileSystemInfo.programId != programIdInNpdm)
                                        {
                                            throw new ArgumentException("program ID specified in .nmeta file differ from the one used to build code.");
                                        }
                                    }
                                }
                                entryInfo.fileSystemInfo = partitionFsInfo;
                                break;
                            }
                        case NintendoContentFileSystemMetaConstant.FormatTypeRomFs:
                            {
                                var adfReader = new RomFsAdfReader(adfPath);
                                entryInfo.fileSystemInfo = adfReader.GetFileSystemInfo();
                                break;
                            }

                        default:
                            throw new NotImplementedException("invalid format .adf file. invalid formatType." + entry.ToString());
                    }

                    fileSystemInfo.fsEntries.Add(entryInfo);
                }
            }

            fileSystemInfo.GenerateExistentFsIndicesFromFsEntries();
            fileSystemInfo.isProdEncryption = false;

            return fileSystemInfo;
        }

        public string GetContentType()
        {
            using (var adf = new StreamReader(m_adfPath, Encoding.UTF8))
            {
                var yamlStream = new YamlStream();
                yamlStream.Load(adf);

                try
                {
                    var rootNode = (YamlMappingNode)yamlStream.Documents[0].RootNode;
                    var formatType = (YamlScalarNode)rootNode.Children[new YamlScalarNode("formatType")];
                    if (formatType.Value != "NintendoContent")
                    {
                        throw new ArgumentException();
                    }
                    var contentType = (YamlScalarNode)rootNode.Children[new YamlScalarNode("contentType")];
                    return contentType.Value;
                }
                catch
                {
                    throw new ArgumentException("invalid format .adf file.");
                }
            }
        }
    }

    public class NintendoContentFileSystemEntryInfoContainer : IEnumerable<NintendoContentFileSystemInfo.EntryInfo>
    {
        public List<NintendoContentFileSystemInfo.EntryInfo> List { get; private set; }
        public int CurrentIndex { get; private set; }
        public int CurrentPartitionIndex { get { return List[CurrentIndex].partitionIndex; } }

        public NintendoContentFileSystemEntryInfoContainer(List<NintendoContentFileSystemInfo.EntryInfo> list)
        {
            List = list;
            CurrentIndex = List.Count - 1;
        }

        public IEnumerator<NintendoContentFileSystemInfo.EntryInfo> GetEnumerator()
        {
            for (int i = List.Count - 1; 0 <= i; --i)
            {
                CurrentIndex = i;
                yield return List[i];
            }
        }

        IEnumerator IEnumerable.GetEnumerator()
        {
            return GetEnumerator();
        }

        public void Update(NintendoContentFileSystemInfo.EntryInfo entryInfo)
        {
            List[CurrentIndex] = entryInfo;
        }
    }

    public class NcaKeyGenerator : IKeyGenerator
    {
        KeyConfiguration m_KeyConfig;

        public NcaKeyGenerator(KeyConfiguration keyConfig)
        {
            m_KeyConfig = keyConfig;
        }

        public virtual byte[] Generate(byte[] encryptedKey, int keyType)
        {
            if (keyType == NintendoContentArchiveKeyType.HeaderEncryptionKey)
            {
                byte[][] headerEncryptedKeys = {
                    new byte[16] { 0x5a, 0x3e, 0xd8, 0x4f, 0xde, 0xc0, 0xd8, 0x26, 0x31, 0xf7, 0xe2, 0x5d, 0x19, 0x7b, 0xf5, 0xd0 },
                    new byte[16] { 0x1c, 0x9b, 0x7b, 0xfa, 0xf6, 0x28, 0x18, 0x3d, 0x71, 0xf6, 0x4d, 0x73, 0xf1, 0x50, 0xb9, 0xd2 },
                };
                if (encryptedKey.SequenceEqual(headerEncryptedKeys[0]))
                {
                    return Aes128XtsCryptoDriver.GetDefaultKey(2);
                }
                else if (encryptedKey.SequenceEqual(headerEncryptedKeys[1]))
                {
                    return Aes128XtsCryptoDriver.GetDefaultKey(3);
                }
                else
                {
                    throw new NotImplementedException();
                }
            }

            Func<byte[], byte[]> decryptKey = (key) =>
            {
                var decryptedKey = new byte[encryptedKey.Length];
                var decryptor = new Aes128CryptoDriver(key);
                decryptor.DecryptBlock(encryptedKey, 0, encryptedKey.Length, decryptedKey, 0);
                return decryptedKey;
            };

            var keyGenMap = new Dictionary<int, Tuple<int, int>>
            {
                { NintendoContentArchiveKeyType.KeyAreaEncryptionKey0, Tuple.Create(0, 0) },
                { NintendoContentArchiveKeyType.KeyAreaEncryptionKey1, Tuple.Create(1, 0) },
                { NintendoContentArchiveKeyType.KeyAreaEncryptionKey2, Tuple.Create(2, 0) },
                { NintendoContentArchiveKeyType.KeyAreaEncryptionKey0Gen2, Tuple.Create(0, 2) },
                { NintendoContentArchiveKeyType.KeyAreaEncryptionKey1Gen2, Tuple.Create(1, 2) },
                { NintendoContentArchiveKeyType.KeyAreaEncryptionKey2Gen2, Tuple.Create(2, 2) },
                { NintendoContentArchiveKeyType.KeyAreaEncryptionKey0Gen3, Tuple.Create(0, 3) },
                { NintendoContentArchiveKeyType.KeyAreaEncryptionKey1Gen3, Tuple.Create(1, 3) },
                { NintendoContentArchiveKeyType.KeyAreaEncryptionKey2Gen3, Tuple.Create(2, 3) },
                { NintendoContentArchiveKeyType.KeyAreaEncryptionKey0Gen4, Tuple.Create(0, 4) },
                { NintendoContentArchiveKeyType.KeyAreaEncryptionKey1Gen4, Tuple.Create(1, 4) },
                { NintendoContentArchiveKeyType.KeyAreaEncryptionKey2Gen4, Tuple.Create(2, 4) },
                { NintendoContentArchiveKeyType.KeyAreaEncryptionKey0Gen5, Tuple.Create(0, 5) },
                { NintendoContentArchiveKeyType.KeyAreaEncryptionKey1Gen5, Tuple.Create(1, 5) },
                { NintendoContentArchiveKeyType.KeyAreaEncryptionKey2Gen5, Tuple.Create(2, 5) },
            };

            try
            {
                var keyParam = keyGenMap[keyType];
                return decryptKey(m_KeyConfig.GetKeyAreaEncryptionKey((uint)keyParam.Item1, (byte)keyParam.Item2).Key);
            }
            catch (KeyNotFoundException)
            {
                // 未知の鍵インデックス・鍵世代の NCA は必ず復号に失敗するが例外にはしないようにする
                return decryptKey(m_KeyConfig.GetRandomAesKey().Key);
            }
        }

        public virtual bool GetUseDevHsm()
        {
            return true;
        }
    };

    // TODO: SourceSinkDriver での処理最適化のため IConnector にする
    public class NintendoContentArchiveSource : ISource
    {
        static public NintendoContentArchiveBuildLog BuildLog = new NintendoContentArchiveBuildLog();

        public long Size { get; private set; }

        private ISource m_source;
        private ISource m_headerSource;
        private ISource m_bodySource;
        private NintendoContentFileSystemInfo m_fileSystemInfo;
        private KeyConfiguration m_keyConfig;
        private long? m_sparseAlignmentSize;

        private const int SectorSize = 512;
        private const int AesXtsBlockSize = 512;
        private const int AesCtrBlockSize = 16;
        private const int FsHeaderCountMax = 4;
        private const long HashBlockSize = 16 * 1024;

        private IEncryptor m_bodyEncryptionKeyEncryptor;
        private IEncryptor m_bodyEncryptionKeyEncryptorHw;
        private IXtsModeEncryptor m_headerEncryptor;
        private ISigner m_header1Signer;
        private ISigner m_header2Signer;
        private ISigner m_acidSigner;
        private ISigner m_nrrSigner;
        private ISigner m_nrrCertificateSigner;
        private ISigner m_acidVerifier;
        private IHashCalculator m_externalKeyHmacCalculator;

        static public byte VerifyHashType(NintendoContentFileSystemInfo fileSystemInfo, NintendoContentFileSystemInfo.EntryInfo entryInfo)
        {
            var type = (NintendoContentArchiveHashType)entryInfo.hashType;

            if (type == NintendoContentArchiveHashType.Auto)
            {
                switch (entryInfo.formatType)
                {
                case NintendoContentFileSystemMetaConstant.FormatTypePartitionFs:
                    type = NintendoContentArchiveHashType.HierarchicalSha256;
                    break;
                case NintendoContentFileSystemMetaConstant.FormatTypeRomFs:
                    type = NintendoContentArchiveHashType.HierarchicalIntegrity;
                    break;
                default:
                    throw new InvalidDataException("invalid format type.");
                }
            }

            if (type != NintendoContentArchiveHashType.HierarchicalSha256 &&
                type != NintendoContentArchiveHashType.HierarchicalIntegrity)
            {
                throw new InvalidDataException("invalid hash type.");
            }

            return (byte)type;
        }

        static public byte VerifyEncryptionType(NintendoContentFileSystemInfo fileSystemInfo, NintendoContentFileSystemInfo.EntryInfo entryInfo)
        {
            var type = (NintendoContentArchiveEncryptionType)entryInfo.encryptionType;

            if (type == NintendoContentArchiveEncryptionType.Auto)
            {
                if (fileSystemInfo.contentType == NintendoContentFileSystemMetaConstant.ContentTypeProgram)
                {
                    switch ((NintendoContentArchivePartitionType)entryInfo.partitionIndex)
                    {
                    case NintendoContentArchivePartitionType.Code:
                    case NintendoContentArchivePartitionType.Data:
                        type = NintendoContentArchiveEncryptionType.AesCtr;
                        break;
                    // ロゴ情報は暗号化しない
                    case NintendoContentArchivePartitionType.Logo:
                        type = NintendoContentArchiveEncryptionType.None;
                        break;
                    default:
                        throw new InvalidDataException("invalid partition index.");
                    }
                }
                else
                {
                    type = NintendoContentArchiveEncryptionType.AesCtr;
                }
            }

            if (fileSystemInfo.contentType == NintendoContentFileSystemMetaConstant.ContentTypeProgram &&
                entryInfo.partitionIndex == (int)NintendoContentArchivePartitionType.Logo)
            {
                if (type != NintendoContentArchiveEncryptionType.None)
                {
                    throw new InvalidDataException("logo must be no encryption type.");
                }
            }

            // TODO: 現在は AesCtrEx も未対応
            if (type != NintendoContentArchiveEncryptionType.None &&
                type != NintendoContentArchiveEncryptionType.AesCtr)
            {
                throw new InvalidDataException("invalid encryption type.");
            }

            return (byte)type;
        }

        private void SetCryptor(KeyConfiguration config)
        {
            // TODO: 以下は本来 nca 毎にセットする必要はないのでリファクタリングしたい
            if (m_fileSystemInfo.isProdEncryption)
            {
                if (m_keyConfig.GetProdKeyAreaEncryptionKey(m_fileSystemInfo.keyGeneration) != null)
                {
                    System.Diagnostics.Debug.Assert(m_fileSystemInfo.keyAreaEncryptionKeyIndex == 0, "key index must be 0 in prod encryption.");
                    m_bodyEncryptionKeyEncryptor = new Aes128CryptoDriver(m_keyConfig.GetProdKeyAreaEncryptionKey(m_fileSystemInfo.keyGeneration).Key);
                }
                else
                {
                    m_bodyEncryptionKeyEncryptor = new HsmAes128CryptoDriver(Aes128KeyIndex.NcaContentKey, m_fileSystemInfo.keyGeneration);
                }

                {
                    byte[][] rawKey = { new byte[16], new byte[16] };
                    byte[][] headerEncryptedKeys = {
                        new byte[16] { 0x5a, 0x3e, 0xd8, 0x4f, 0xde, 0xc0, 0xd8, 0x26, 0x31, 0xf7, 0xe2, 0x5d, 0x19, 0x7b, 0xf5, 0xd0, },
                        new byte[16] { 0x1c, 0x9b, 0x7b, 0xfa, 0xf6, 0x28, 0x18, 0x3d, 0x71, 0xf6, 0x4d, 0x73, 0xf1, 0x50, 0xb9, 0xd2, },
                    };

                    IEncryptor headerEncryptionKeyEncryptor;
                    if (m_keyConfig.GetProdNcaHeaderEncryptionKek() != null)
                    {
                        headerEncryptionKeyEncryptor = new Aes128CryptoDriver(m_keyConfig.GetProdNcaHeaderEncryptionKek().Key);
                    }
                    else
                    {
                        headerEncryptionKeyEncryptor = new HsmAes128CryptoDriver(Aes128KeyIndex.NcaHeader);
                    }

                    headerEncryptionKeyEncryptor.DecryptBlock(headerEncryptedKeys[0], 0, 16, rawKey[0], 0);
                    headerEncryptionKeyEncryptor.DecryptBlock(headerEncryptedKeys[1], 0, 16, rawKey[1], 0);

                    m_headerEncryptor = new Aes128XtsCryptoDriver(rawKey[0], rawKey[1]);
                }

                if (m_keyConfig.GetNcaHeader1SignKey() != null)
                {
                    var rsaKey = m_keyConfig.GetNcaHeader1SignKey();
                    m_header1Signer = new Rsa2048PssSha256SignCryptoDriver(rsaKey.KeyModulus, rsaKey.KeyPublicExponent, rsaKey.KeyPrivateExponent);
                }
                else
                {
                    m_header1Signer = new HsmRsa2048PssSha256SignCryptoDriver(Rsa2048PssSha256KeyIndex.NcaHeader1);
                }

                if ((NintendoContentArchiveContentType)m_fileSystemInfo.contentType != NintendoContentArchiveContentType.Program)
                {
                    m_header2Signer = null;
                }
                else
                {
                    var rsaKey = m_keyConfig.GetRandomRsaKey(); // 乱数鍵
                    m_header2Signer = new Rsa2048PssSha256SignCryptoDriver(rsaKey.KeyModulus, rsaKey.KeyPublicExponent, rsaKey.KeyPrivateExponent);
                }

                if (m_keyConfig.GetAcidSignKey() != null)
                {
                    var rsaKey = m_keyConfig.GetAcidSignKey();
                    m_acidSigner = new Rsa2048PssSha256SignCryptoDriver(rsaKey.KeyModulus, rsaKey.KeyPublicExponent, rsaKey.KeyPrivateExponent);
                }
                else
                {
                    m_acidSigner = new HsmRsa2048PssSha256SignCryptoDriver(Rsa2048PssSha256KeyIndex.Acid);
                }

                {
                    var rsaKey = m_keyConfig.GetRandomRsaKey(); // 乱数鍵
                    m_nrrSigner = new Rsa2048PssSha256SignCryptoDriver(rsaKey.KeyModulus, rsaKey.KeyPublicExponent, rsaKey.KeyPrivateExponent);
                }

                if (m_keyConfig.GetNrrCertificateSignKey() != null)
                {
                    var rsaKey = m_keyConfig.GetNrrCertificateSignKey();
                    m_nrrCertificateSigner = new Rsa2048PssSha256SignCryptoDriver(rsaKey.KeyModulus, rsaKey.KeyPublicExponent, rsaKey.KeyPrivateExponent);
                }
                else
                {
                    m_nrrCertificateSigner = new HsmRsa2048PssSha256SignCryptoDriver(Rsa2048PssSha256KeyIndex.NrrCertificate);
                }

                if (TicketUtility.IsValidRightsId(m_fileSystemInfo.rightsId))
                {
                    if (m_keyConfig.GetProdTitleKeyGenarateKey() != null)
                    {
                        m_externalKeyHmacCalculator = new HmacSha256HashCryptoDriver(m_keyConfig.GetProdTitleKeyGenarateKey().Key);
                    }
                    else
                    {
                        m_externalKeyHmacCalculator = new HsmHmacSha256HashCryptoDriver(HmacSha256KeyIndex.TitleKeyGenarateKey);
                    }
                }
            }
            else
            {
                m_bodyEncryptionKeyEncryptor = new Aes128CryptoDriver(m_keyConfig.GetKeyAreaEncryptionKey(m_fileSystemInfo.keyAreaEncryptionKeyIndex, m_fileSystemInfo.keyGeneration).Key);
                if (m_fileSystemInfo.isHardwareEncryptionKeyEmbedded)
                {
                    m_bodyEncryptionKeyEncryptorHw = new Aes128CryptoDriver(m_keyConfig.GetKeyAreaEncryptionKeyHw(m_fileSystemInfo.keyAreaEncryptionKeyIndex, m_fileSystemInfo.keyGeneration).Key);
                }

                if (m_fileSystemInfo.headerEncryptionType == (byte)NintendoContentArchiveHeaderEncryptionType.None)
                {
                    m_headerEncryptor = new DummyCryptoDriver();
                }
                else
                {
                    m_headerEncryptor = new Aes128XtsCryptoDriver(Aes128XtsCryptoDriver.GetDefaultKey(2), Aes128XtsCryptoDriver.GetDefaultKey(3));
                }
                m_header1Signer = new Rsa2048PssSha256SignCryptoDriver(Rsa2048PssSha256KeyIndex.NcaHeader1);

                if ((NintendoContentArchiveContentType)m_fileSystemInfo.contentType == NintendoContentArchiveContentType.Program &&
                    m_fileSystemInfo.header2SignKeyModulus != null &&
                    m_fileSystemInfo.header2SignKeyPublicExponent != null &&
                    m_fileSystemInfo.header2SignKeyPrivateExponent != null)
                {
                    m_header2Signer = new Rsa2048PssSha256SignCryptoDriver(m_fileSystemInfo.header2SignKeyModulus, m_fileSystemInfo.header2SignKeyPublicExponent, m_fileSystemInfo.header2SignKeyPrivateExponent);
                }
                else
                {
                    m_header2Signer = null;
                }

                m_acidSigner = null;
                m_nrrSigner = new Rsa2048PssSha256SignCryptoDriver(Rsa2048PssSha256KeyIndex.NrrCertificate); // 固定鍵として KeyIndexNrrCertificate を使う
                m_nrrCertificateSigner = new Rsa2048PssSha256SignCryptoDriver(Rsa2048PssSha256KeyIndex.NrrCertificate);

                if (TicketUtility.IsValidRightsId(m_fileSystemInfo.rightsId))
                {
                    m_externalKeyHmacCalculator = new HmacSha256HashCryptoDriver(HmacSha256KeyIndex.TitleKeyGenarateKey);
                }
            }

            m_acidVerifier = new Rsa2048PssSha256SignCryptoDriver(Rsa2048PssSha256KeyIndex.Acid);
        }

        private NintendoContentFileSystemInfo.EntryInfo CreateDefaultLogoPartition()
        {
            var entry = new NintendoContentFileSystemInfo.EntryInfo();
            {
                entry.type = "format";
                entry.partitionType = "logo";
                entry.partitionIndex = (int)NintendoContentArchivePartitionType.Logo;
                entry.formatType = NintendoContentFileSystemMetaConstant.FormatTypePartitionFs;
                entry.version = 2;
                entry.hashType = (int)NintendoContentArchiveHashType.HierarchicalSha256;
                entry.encryptionType = (int)NintendoContentArchiveEncryptionType.None;
                entry.generation = 0;
                entry.secureValue = 0;
            }

            var fileSystemInfo = new PartitionFileSystemInfo();

            var logoFiles = LogoManager.GetDefaultLogoData();

            UInt64 offset = 0;
            foreach (var logoFile in logoFiles)
            {
                var data = logoFile.Data;
                var filename = logoFile.FileName;
                var entryInfo = new PartitionFileSystemInfo.EntryInfo();
                {
                    entryInfo.type = "file";
                    entryInfo.name = filename;
                    entryInfo.offset = offset;
                    entryInfo.size = (ulong)data.Length;
                    entryInfo.sourceInterface = new CliCompatibleSource(new MemorySource(data, 0, data.Length));
                }
                fileSystemInfo.entries.Add(entryInfo);
                offset += entryInfo.size;
            }

            entry.fileSystemInfo = fileSystemInfo;
            return entry;
        }

        private void AddDefaultLogoPartitionIfRequired()
        {
            if ((NintendoContentArchiveContentType)m_fileSystemInfo.contentType != NintendoContentArchiveContentType.Program)
            {
                return;
            }

            foreach (var fsEntry in m_fileSystemInfo.fsEntries)
            {
                if (fsEntry.partitionIndex == (int)NintendoContentArchivePartitionType.Logo)
                {
                    return;
                }
            }

            var entry = CreateDefaultLogoPartition();
            entry.partitionIndex = (int)NintendoContentArchivePartitionType.Logo;
            m_fileSystemInfo.fsEntries.Add(entry);
        }

        private void SetSparseAlignmentSize()
        {
            if (m_fileSystemInfo.fsEntries.Any(value => value.formatType == "SparseFs"))
            {
                // 過去のアライメントを引き継ぐ
                foreach (var entry in m_fileSystemInfo.fsEntries)
                {
                    if (0 < entry.startOffset)
                    {
                        // startOffset が全て HashBlockSize でアライメントされているかチェック
                        if (BitUtility.IsAligned((long)entry.startOffset, (int)HashBlockSize))
                        {
                            m_sparseAlignmentSize = HashBlockSize;
                        }
                        else
                        {
                            m_sparseAlignmentSize = SectorSize;
                            break;
                        }
                    }
                }
            }
        }

        private long GetHashBlockAlignmentSize()
        {
            if (m_sparseAlignmentSize.HasValue)
            {
                return m_sparseAlignmentSize.Value;
            }
            return m_fileSystemInfo.partitionAlignmentType == (int)NintendoContentArchivePartitionAlignmentType.AlignHashBlockSize ? HashBlockSize : SectorSize;
        }

        public NintendoContentArchiveSource(NintendoContentFileSystemInfo fileSystemInfo, KeyConfiguration config, bool needsLogoPartition = false)
        {
            m_fileSystemInfo = fileSystemInfo;
            m_keyConfig = config;
            SetCryptor(m_keyConfig);
            if (needsLogoPartition)
            {
                AddDefaultLogoPartitionIfRequired();
            }
            SetSparseAlignmentSize();

            var bodyEncryptionKey = new BodyEncryptionKey(fileSystemInfo, m_externalKeyHmacCalculator);

            // partitionIndex の順にソート
            fileSystemInfo.fsEntries.Sort((x, y) => (x.partitionIndex.CompareTo(y.partitionIndex)));

            // 製品化の際に NoEncryption 設定の整合性をチェックする
            if (fileSystemInfo.isProdEncryption)
            {
                bool noEncryptionMode = false;
                foreach (var fsEntry in fileSystemInfo.fsEntries)
                {
                    if (fsEntry.encryptionType == (byte)NintendoContentArchiveEncryptionType.None &&
                        fsEntry.partitionIndex != (int)NintendoContentArchivePartitionType.Logo)
                    {
                        noEncryptionMode = true;
                    }
                }
                if (noEncryptionMode ^ (fileSystemInfo.headerEncryptionType == (byte)NintendoContentArchiveHeaderEncryptionType.None))
                {
                    throw new ArgumentException("Encryption types of nca header and nca fs header are not consist.");
                }
            }

            var fsEntries = new NintendoContentFileSystemEntryInfoContainer(fileSystemInfo.fsEntries);
            var fsHeaderHashSourceElements = new List<ConcatenatedSource.Element>();
            var bodySources = new BodySourceElement[fileSystemInfo.fsEntries.Count];

            // DataSource をキャッシュ
            foreach (var entryInfo in fsEntries)
            {
                if (entryInfo.partitionIndex == (int)NintendoContentArchivePartitionType.Data)
                {
                    switch (entryInfo.formatType)
                    {
                        case "PatchFs":
                        case "IscPatchFs":
                        case "SparseFs":
                            // DataSource を持たないのでキャッシュしない
                            break;

                        default:
                            {
                                // DataSource をキャッシュ
                                ISource fsDataSource = GetDataSource(entryInfo);
                                fsEntries.Update(CacheSource(entryInfo, fsDataSource));
                            }
                            break;
                    }
                }
            }

            var buildInfo = new BuildDataInfo((int)GetHashBlockAlignmentSize());

            foreach (var entryInfo in fsEntries)
            {
                if (entryInfo.type == null)
                {
                    continue;
                }

                int index = fsEntries.CurrentIndex;
                bodySources[index].DataOffset = buildInfo.PhysicalOffset - NintendoContentFileSystemMeta.HeaderSize;

                switch (entryInfo.formatType)
                {
                    case "PatchFs":
                        buildInfo.SetData(index, entryInfo.sourceInterface);
                        fsEntries.Update(BuildPatchData(entryInfo, buildInfo));
                        break;

                    case "IscPatchFs":
                        buildInfo.SetData(index, entryInfo.sourceInterface);
                        fsEntries.Update(BuildIscPatchData(entryInfo, buildInfo));
                        break;

                    case "SparseFs":
                        buildInfo.SetData(index, entryInfo.sourceInterface);
                        fsEntries.Update(BuildSparseData(entryInfo, buildInfo, bodyEncryptionKey));
                        break;

                    default:
                        {
                            ISource fsDataSource = GetDataSource(entryInfo);

                            // ACID 再署名
                            if (fileSystemInfo.isProdEncryption &&
                                fileSystemInfo.contentType == NintendoContentFileSystemMetaConstant.ContentTypeProgram &&
                                entryInfo.partitionIndex == (int)NintendoContentArchivePartitionType.Code)
                            {
                                fsDataSource = AcidResignedSource(fsDataSource);
                            }

                            buildInfo.SetData(index, fsDataSource);

                            // 手動 DLL 署名フラグ
                            var needsNrrSign = fileSystemInfo.contentType == NintendoContentFileSystemMetaConstant.ContentTypeProgram &&
                                               entryInfo.partitionIndex == (int)NintendoContentArchivePartitionType.Data;

                            if (entryInfo.patchInfo != null)
                            {
                                fsEntries.Update(BuildDataWithoutHash(entryInfo, buildInfo, needsNrrSign));
                            }
                            else
                            {
                                fsEntries.Update(BuildData(entryInfo, buildInfo, needsNrrSign));
                            }
                        }
                        break;
                }

                // fs n ヘッダのハッシュ計算
                Sha256HierarchicalHashCalculatedSource fsHeaderHashCalculatedSource =
                    new Sha256HierarchicalHashCalculatedSource(buildInfo.HeaderSource, (int)buildInfo.HeaderSource.Size, 1);

                // FS n ヘッダのハッシュを fsHeaderHashSourceElements に追加
                ConcatenatedSource.Element fsHeaderHashElement = new ConcatenatedSource.Element(
                    fsHeaderHashCalculatedSource.GetMasterHashSource(), "fsHeaderHash" + fsEntries.CurrentPartitionIndex, 32 * fsEntries.CurrentPartitionIndex);
                fsHeaderHashSourceElements.Add(fsHeaderHashElement);

                ConcatenatedSource.Element headerElement = new ConcatenatedSource.Element(
                    fsHeaderHashCalculatedSource, "fsHeader" + fsEntries.CurrentPartitionIndex, fsEntries.CurrentPartitionIndex * NintendoContentFileSystemMeta.FsHeaderSize);

                bodySources[index].HeaderElement = headerElement;
                bodySources[index].DataSourceElements = buildInfo.DataSourceElements;
            }

            // 暗号鍵の生成
            MakeBody(fsEntries, bodySources, bodyEncryptionKey, fileSystemInfo.partitionAlignmentType);
            MakeNcaHeader(new ConcatenatedSource(fsHeaderHashSourceElements), bodyEncryptionKey);

            // HeaderSource と BodySource を結合
            {
                List<ConcatenatedSource.Element> elements = new List<ConcatenatedSource.Element>();
                ConcatenatedSource.Element headerElement = new ConcatenatedSource.Element(
                    m_headerSource, "header", 0);
                ConcatenatedSource.Element bodyElement = new ConcatenatedSource.Element(
                    m_bodySource, "body", headerElement.Source.Size);
                elements.Add(headerElement);
                elements.Add(bodyElement);
                m_source = new ConcatenatedSource(elements);
            }
            Size = m_source.Size;
        }

        public ByteData PullData(long offset, int size)
        {
            return m_source.PullData(offset, size);
        }

        public SourceStatus QueryStatus()
        {
            return m_source.QueryStatus();
        }

        internal static ISource GetDataSource(NintendoContentFileSystemInfo.EntryInfo fsEntry)
        {
            if (fsEntry.type == "format")
            {
                if (fsEntry.sourceInterface == null)
                {
                    switch (fsEntry.formatType)
                    {
                        case NintendoContentFileSystemMetaConstant.FormatTypePartitionFs:
                            return new PartitionFsArchiveSource(fsEntry.fileSystemInfo as PartitionFileSystemInfo);
                        case NintendoContentFileSystemMetaConstant.FormatTypeRomFs:
                            return new RomFsArchiveSource(fsEntry.fileSystemInfo as RomFsFileSystemInfo);
                        case NintendoContentFileSystemMetaConstant.FormatTypePatchableRomFs:
                        case NintendoContentFileSystemMetaConstant.FormatTypePatchablePartitionFs:
                            return new NintendoContentPatchFsArchiveSource(fsEntry.fileSystemInfo as NintendoContentPatchFileSystemInfo);
                        default:
                            throw new NotImplementedException();
                    }
                }
                return (ISource)fsEntry.sourceInterface;
            }

            if (fsEntry.type == "source")
            {
                switch (fsEntry.formatType)
                {
                    case "PatchFs":
                    case "IscPatchFs":
                        throw new NotImplementedException();
                    default:
                        return (ISource)fsEntry.sourceInterface;
                }
            }

            throw new NotImplementedException();
        }

        private NintendoContentFileSystemInfo.EntryInfo BuildData(NintendoContentFileSystemInfo.EntryInfo entryInfo, BuildDataInfo buildInfo, bool needsNrrSign)
        {
            var dataSource = buildInfo.DataSource;
            if (needsNrrSign)
            {
                dataSource = NrrSignedSource(dataSource);
            }

            entryInfo.hashType = VerifyHashType(m_fileSystemInfo, entryInfo);
            entryInfo.encryptionType = VerifyEncryptionType(m_fileSystemInfo, entryInfo);
            entryInfo.startOffset = buildInfo.CurrentOffset; // FS n 領域開始オフセットは NCA 全体でのオフセット
            IHierarchicalHashCalculatedSource fsDataHashCalculatedSource;
            switch (entryInfo.hashType)
            {
            case (byte)NintendoContentArchiveHashType.HierarchicalSha256:
                fsDataHashCalculatedSource =
                    new Sha256HierarchicalHashCalculatedSource(dataSource, NintendoContentFileSystemMeta.GetHashBlockSize(dataSource.Size), 2);
                break;

            case (byte)NintendoContentArchiveHashType.HierarchicalIntegrity:
                fsDataHashCalculatedSource =
                    new IntegrityHierarchicalHashCalculatedSource(dataSource);
                break;

            default:
                throw new NotImplementedException();
            }

            long offset = 0;
            ulong hashDataSize = 0; // hashType == NintendoContentArchiveHashType.HierarchicalSha256 のときだけ利用

            // fs n データの階層化ハッシュを取得
            {
                ISource fsHashSource = fsDataHashCalculatedSource.GetLayerHashSource();

                hashDataSize = (ulong)fsHashSource.Size;

                // fs n データの階層化ハッシュを DataSourceElements に追加
                ConcatenatedSource.Element hashElement = new ConcatenatedSource.Element(
                    fsHashSource, "fsHash" + buildInfo.Index, offset);
                buildInfo.DataSourceElements.Add(hashElement);

                offset += hashElement.Source.Size;
            }

            // fs n データが暗号化処理のアラインメントから始まるように調整
            {
                long alignmentSize = GetAlignmentSizeForRealDataOnBuildData(entryInfo);
                if (alignmentSize > 0)
                {
                    int paddingSize = (int)(offset % alignmentSize);
                    if (0 < paddingSize)
                    {
                        ConcatenatedSource.Element hashPaddingElement = new ConcatenatedSource.Element(
                            new PaddingSource(alignmentSize - paddingSize), "fsHashPadding" + buildInfo.Index, offset);
                        buildInfo.DataSourceElements.Add(hashPaddingElement);

                        offset += hashPaddingElement.Source.Size;
                    }
                }
            }

            switch (entryInfo.hashType)
            {
            case (byte)NintendoContentArchiveHashType.HierarchicalSha256:
                entryInfo.SetHashInfo(hashDataSize, (ulong)offset, (ulong)dataSource.Size);
                break;
            case (byte)NintendoContentArchiveHashType.HierarchicalIntegrity:
                entryInfo.SetHashInfo(((IntegrityHierarchicalHashCalculatedSource)fsDataHashCalculatedSource).GetMetaInfo());
                break;
            default:
                throw new NotImplementedException();
            }

            // fs n データを DataSourceElements に追加
            {
                ConcatenatedSource.Element dataElement = new ConcatenatedSource.Element(
                    fsDataHashCalculatedSource, "fsData" + buildInfo.Index, offset);
                buildInfo.DataSourceElements.Add(dataElement);

                offset += dataElement.Source.Size;
            }

            // パディングしておく
            {
                long alignmentSize = GetHashBlockAlignmentSize();
                int paddingSize = (int)(offset % alignmentSize);
                if (0 < paddingSize)
                {
                    ConcatenatedSource.Element dataPaddingElement = new ConcatenatedSource.Element(
                        new PaddingSource(alignmentSize - paddingSize), "fsDataPadding" + buildInfo.Index, offset);
                    buildInfo.DataSourceElements.Add(dataPaddingElement);

                    offset += dataPaddingElement.Source.Size;
                }
            }

            buildInfo.Advance(offset);

            entryInfo.endOffset = buildInfo.CurrentOffset; // FS n 領域終了オフセットは NCA 全体でのオフセット

            // fs n ヘッダの生成（マスターハッシュを埋め込む）
            buildInfo.HeaderSource = fsDataHashCalculatedSource.GetFsHeaderSource(entryInfo);

            return entryInfo;
        }

        // BuildData 用のサブルーチン
        private long GetAlignmentSizeForRealDataOnBuildData(NintendoContentFileSystemInfo.EntryInfo entryInfo)
        {
            long alignmentSize;
            if (m_fileSystemInfo.partitionAlignmentType == (int)NintendoContentArchivePartitionAlignmentType.AlignHashBlockSize)
            {
                alignmentSize = GetHashBlockAlignmentSize();
            }
            else
            {
                switch ((NintendoContentArchiveEncryptionType)entryInfo.encryptionType)
                {
                    case NintendoContentArchiveEncryptionType.Auto:
                        alignmentSize = AesXtsBlockSize;
                        break;

                    case NintendoContentArchiveEncryptionType.AesCtr:
                    case NintendoContentArchiveEncryptionType.AesCtrEx:
                        alignmentSize = AesCtrBlockSize;
                        break;

                    case NintendoContentArchiveEncryptionType.None:
                        alignmentSize = -1;
                        break;

                    default:
                        throw new InvalidDataException();
                }
            }

            return alignmentSize;
        }

        // TODO: リファクタリングで BuildData と統合
        private NintendoContentFileSystemInfo.EntryInfo BuildDataWithoutHash(NintendoContentFileSystemInfo.EntryInfo entryInfo, BuildDataInfo buildInfo, bool needsNrrSign)
        {
            Debug.Assert(entryInfo.patchInfo != null);

            long offset = 0;
            entryInfo.startOffset = buildInfo.CurrentOffset; // FS n 領域開始オフセットは NCA 全体でのオフセット
            var dataSource = buildInfo.DataSource;
            if (needsNrrSign)
            {
                dataSource = NrrSignedSourceForPatch(ref entryInfo.patchInfo.masterHash, entryInfo.fileSystemInfo as NintendoContentPatchFileSystemInfo);
            }

            // FS n データを DataSourceElements に追加
            {
                ConcatenatedSource.Element dataElement = new ConcatenatedSource.Element(
                    dataSource, "fsData" + buildInfo.Index, offset);
                buildInfo.DataSourceElements.Add(dataElement);

                offset += dataElement.Source.Size;
            }

            // セクターサイズでパディングしておく
            {
                int paddingSize = (int)(offset % SectorSize);
                if (paddingSize != 0)
                {
                    ConcatenatedSource.Element dataPaddingElement = new ConcatenatedSource.Element(
                        new PaddingSource(SectorSize - paddingSize), "fsDataPadding" + buildInfo.Index, offset);
                    buildInfo.DataSourceElements.Add(dataPaddingElement);

                    offset += dataPaddingElement.Source.Size;
                }
            }

            buildInfo.Advance(offset);

            entryInfo.endOffset = buildInfo.CurrentOffset; // FS n 領域終了オフセットは NCA 全体でのオフセット

            // FS n ヘッダの生成
            {
                var header = NintendoContentFileSystemMeta.UpdatePatchInfo(entryInfo.patchInfo.fsHeaderRawData, entryInfo.patchInfo, entryInfo.generation, entryInfo.secureValue);
                {
                    var masterHash = entryInfo.patchInfo.masterHash;
                    Array.Copy(masterHash, 0, header, NintendoContentFileSystemMeta.IntegrityMasterHashOffset, masterHash.Length);
                }
                buildInfo.HeaderSource = new MemorySource(header, 0, NintendoContentFileSystemMeta.FsHeaderSize);
            }

            return entryInfo;
        }

        private NintendoContentFileSystemInfo.EntryInfo CacheSource(NintendoContentFileSystemInfo.EntryInfo entryInfo, ISource source)
        {
            entryInfo.sourceInterface = new CliCompatibleSource(source);
            return entryInfo;
        }

        // TODO: リファクタリングで BuildData と統合
        private NintendoContentFileSystemInfo.EntryInfo BuildPatchData(NintendoContentFileSystemInfo.EntryInfo entryInfo, BuildDataInfo buildInfo)
        {
            // パッチ化するデータしか来ない
            Debug.Assert(
                ArchiveReconstructionUtils.IsPatchTargetPartitionForAssersion(
                    (NintendoContentArchiveContentType)m_fileSystemInfo.contentType,
                    (NintendoContentArchivePartitionType)entryInfo.partitionIndex));

            entryInfo.startOffset = buildInfo.CurrentOffset; // FS n 領域開始オフセットは NCA 全体でのオフセット

            var patchSource = (FileSystemArchviePatchSource)buildInfo.Data;

            IndirectStorageSource source = new IndirectStorageSource(IndirectStorageSource.DataCacheType == IndirectStorageSource.CacheType.Memory);

            ISource originalSource = null;
            if (patchSource.ExtraInfo.originalStorageReader == null)
            {
                originalSource = new MemorySource(new byte[0], 0, 0);
            }
            else
            {
                originalSource = new StorageArchiveSource(patchSource.ExtraInfo.originalStorageReader);
                if (IndirectStorageSource.DataCacheType == IndirectStorageSource.CacheType.File)
                {
                    // 比較評価用。使用するなら一時ファイルパスを変更する必要あり
                    originalSource = SourceUtil.MakeFileCacheSource(originalSource, "original.bin");
                }

                // 完全にメモリに載る最短ケース
                // originalSource = SourceUtil.MakeMemoryCacheSource(originalSource);
            }

            // 比較対象から外す領域の指定
            {
                var reader = patchSource.ExtraInfo.originalStorageReader as NintendoContentArchiveStorageReader;
                if (reader == null)
                {
                    source.OldExcludeRange = new List<IndirectStorageStream.ExcludeRange>();
                }
                else
                {
                    source.OldExcludeRange = CollectExcludeRange((RomFsFileSystemArchiveReader)patchSource.ExtraInfo.originalFsReader, reader);
                }
            }
            {
                var reader = patchSource.ExtraInfo.currentStorageReader as NintendoContentArchiveStorageReader;
                if (reader == null)
                {
                    source.NewExcludeRange = new List<IndirectStorageStream.ExcludeRange>();
                }
                else
                {
                    source.NewExcludeRange = CollectExcludeRange((RomFsFileSystemArchiveReader)patchSource.ExtraInfo.currentFsReader, reader);
                }
            }

            // IndirectStorage の作成
            ISource currentSource = new StorageArchiveSource(patchSource.ExtraInfo.currentStorageReader);
            {
                if (IndirectStorageSource.DataCacheType == IndirectStorageSource.CacheType.File)
                {
                    // 比較評価用。使用するなら一時ファイルパスを変更する必要あり
                    currentSource = SourceUtil.MakeFileCacheSource(currentSource, "current.bin");
                }

                // 完全にメモリに載る最短ケース
                // currentSource = SourceUtil.MakeMemoryCacheSource(currentSource);
            }

            source.BinaryMatchHints = patchSource.ExtraInfo.binaryMatchHints;
            source.BinaryMatchHintPaths = patchSource.ExtraInfo.binaryMatchHintPaths;
            source.Build(originalSource, currentSource, patchSource.ExtraInfo.originalHeader);

            // AesCtrEx の世代テーブルを設定
            entryInfo.generationTable = source.GenerationTable;

            long offset = 0;

            // FS n データを DataSourceElements に追加
            {
                ConcatenatedSource.Element dataElement = new ConcatenatedSource.Element(
                    source, "fsData" + buildInfo.Index, offset);
                buildInfo.DataSourceElements.Add(dataElement);

                offset += dataElement.Source.Size;
            }

            // セクターサイズでパディングしておく
            {
                int paddingSize = (int)(offset % SectorSize);
                if (paddingSize != 0)
                {
                    ConcatenatedSource.Element dataPaddingElement = new ConcatenatedSource.Element(
                        new PaddingSource(SectorSize - paddingSize), "fsDataPadding" + buildInfo.Index, offset);
                    buildInfo.DataSourceElements.Add(dataPaddingElement);

                    offset += dataPaddingElement.Source.Size;
                }
            }

            buildInfo.Advance(offset);

            entryInfo.endOffset = buildInfo.CurrentOffset; // FS n 領域終了オフセットは NCA 全体でのオフセット

            // パッチの追加情報生成
            var patchInfo = new NintendoContentArchivePatchInfo()
            {
                indirectSize = source.IndirectTableSize,
                indirectOffset = source.IndirectTableOffset,
                indirectHeader = source.IndirectHeader,
                aesCtrExSize = source.AesCtrExTableSize,
                aesCtrExOffset = source.AesCtrExTableOffset,
                aesCtrExHeader = source.AesCtrExHeader,
            };

            var header = NintendoContentFileSystemMeta.UpdatePatchInfo(patchSource.ExtraInfo.currentHeader, patchInfo, entryInfo.generation, entryInfo.secureValue);

            // FS n ヘッダの生成
            buildInfo.HeaderSource = new MemorySource(header, 0, NintendoContentFileSystemMeta.FsHeaderSize);

            return entryInfo;
        }

        private List<IndirectStorageStream.ExcludeRange> CollectExcludeRange(RomFsFileSystemArchiveReader fsReader, NintendoContentArchiveStorageReader storageReader)
        {
            var ranges = new List<IndirectStorageStream.ExcludeRange>();
            var pattern = new Regex("^(.nrr/).+(.nrr)$");

            // 比較対象から除外する範囲の取得
            foreach (var file in fsReader.ListFileInfo(".nrr/"))
            {
                var name = file.Item1;

                if (pattern.IsMatch(name))
                {
                    var fileOffset = fsReader.GetFileOffset(name);
                    var fileSize = file.Item2;
                    if (fileSize % 16 != 0)
                    {
                        fileSize += 16 - (fileSize % 16);
                    }

                    ranges.Add(new IndirectStorageStream.ExcludeRange { offset = fileOffset, size = fileSize });
                }
            }

            var output = storageReader.CollectExcludeRange(ranges);
            output.Sort((lhs, rhs) => { return lhs.offset == rhs.offset ? 0 : (lhs.offset < rhs.offset ? -1 : 1); });

            // 被っている領域をマージ
            int index = 1;
            while (index < output.Count)
            {
                var previous = output[index - 1];
                var current = output[index];

                if (current.offset <= previous.offset + previous.size)
                {
                    var size = (current.offset + current.size) - previous.offset;
                    if (previous.size < size)
                    {
                        previous.size = size;
                    }

                    output[index - 1] = previous;

                    output.RemoveAt(index);
                }
                else
                {
                    ++index;
                }
            }

            return output;
        }

        // TODO: リファクタリングで BuildData と統合
        private NintendoContentFileSystemInfo.EntryInfo BuildIscPatchData(NintendoContentFileSystemInfo.EntryInfo entryInfo, BuildDataInfo buildInfo)
        {
            // パッチ化するデータしか来ない
            Debug.Assert(
                ArchiveReconstructionUtils.IsPatchTargetPartitionForAssersion(
                    (NintendoContentArchiveContentType)m_fileSystemInfo.contentType,
                    (NintendoContentArchivePartitionType)entryInfo.partitionIndex));

            entryInfo.startOffset = buildInfo.CurrentOffset; // FS n 領域開始オフセットは NCA 全体でのオフセット

            var patchSource = (FileSystemArchiveIscPatchSource)buildInfo.Data;

            RelocatedIndirectStorageSource source = new RelocatedIndirectStorageSource();

            // IndirectStorage の作成
            source.Build(
                patchSource.ExtraInfo.previousRawEntryInfo,
                patchSource.ExtraInfo.previousFsStartOffset,
                patchSource.ExtraInfo.previousSource,
                patchSource.ExtraInfo.previousAesCtrExTable,
                patchSource.ExtraInfo.currentRawEntryInfo,
                patchSource.ExtraInfo.currentFsStartOffset,
                patchSource.ExtraInfo.currentSource,
                (long)buildInfo.CurrentOffset,
                entryInfo.generation);

            // AesCtrEx の世代テーブルを設定
            entryInfo.generationTable = source.GenerationTable;

            long offset = 0;

            // FS n データを DataSourceElements に追加
            {
                ConcatenatedSource.Element dataElement = new ConcatenatedSource.Element(
                    source, "fsData" + buildInfo.Index, offset);
                buildInfo.DataSourceElements.Add(dataElement);

                offset += dataElement.Source.Size;
            }

            // セクターサイズでパディングしておく
            {
                int paddingSize = (int)(offset % SectorSize);
                if (paddingSize != 0)
                {
                    ConcatenatedSource.Element dataPaddingElement = new ConcatenatedSource.Element(
                        new PaddingSource(SectorSize - paddingSize), "fsDataPadding" + buildInfo.Index, offset);
                    buildInfo.DataSourceElements.Add(dataPaddingElement);

                    offset += dataPaddingElement.Source.Size;
                }
            }

            buildInfo.Advance(offset);

            entryInfo.endOffset = buildInfo.CurrentOffset; // FS n 領域終了オフセットは NCA 全体でのオフセット

            // パッチの追加情報生成
            var patchInfo = new NintendoContentArchivePatchInfo()
            {
                indirectSize = source.IndirectTableSize,
                indirectOffset = source.IndirectTableOffset,
                indirectHeader = source.IndirectHeader,
                aesCtrExSize = source.AesCtrExTableSize,
                aesCtrExOffset = source.AesCtrExTableOffset,
                aesCtrExHeader = source.AesCtrExHeader
            };

            var header = NintendoContentFileSystemMeta.UpdatePatchInfo(patchSource.ExtraInfo.currentRawEntryInfo, patchInfo, entryInfo.generation, entryInfo.secureValue);

            // FS n ヘッダの生成
            buildInfo.HeaderSource = new MemorySource(header, 0, NintendoContentFileSystemMeta.FsHeaderSize);

            return entryInfo;
        }

        private NintendoContentFileSystemInfo.EntryInfo BuildSparseData(NintendoContentFileSystemInfo.EntryInfo entryInfo, BuildDataInfo buildInfo, BodyEncryptionKey bodyEncryptionKey)
        {
            // FS n 領域の開始・終了オフセットは変更しない
            if (entryInfo.startOffset != buildInfo.CurrentOffset)
            {
                throw new InvalidOperationException();
            }

            var sparseInfo = ((FileSystemArchiveSparseSource)buildInfo.Data).ExtraInfo;

            SparseStorageSource source = new SparseStorageSource();

            // SparseStorage の作成
            source.Build(sparseInfo.originalReader, sparseInfo.patchReader, buildInfo.Index, buildInfo.PhysicalOffset, new Aes128CtrCryptoDriver(bodyEncryptionKey.SelectAesCtrKey()));

            // 再暗号をしないように書き換える（entryInfo は NcaFsHeader の生成には使われない）
            entryInfo.encryptionType = (byte)NintendoContentArchiveEncryptionType.None;

            // SparseStorage の有無を再設定
            entryInfo.existsSparseLayer = (entryInfo.existsSparseLayer || source.NeedsUpdate);

            long offset = 0;

            // FS n データを DataSourceElements に追加
            {
                ConcatenatedSource.Element dataElement = new ConcatenatedSource.Element(source, "fsData" + buildInfo.Index, offset);
                buildInfo.DataSourceElements.Add(dataElement);

                offset += dataElement.Source.Size;
            }

            buildInfo.Advance(entryInfo.endOffset - entryInfo.startOffset, offset);

            // FS n ヘッダの生成
            buildInfo.HeaderSource = new MemorySource(source.FsHeader, 0, NintendoContentFileSystemMeta.FsHeaderSize);

            return entryInfo;
        }

        private void MakeBody(
            NintendoContentFileSystemEntryInfoContainer fsEntries,
            BodySourceElement[] bodySources,
            BodyEncryptionKey bodyEncryptionKey,
            int alignmentType)
        {
            List<ConcatenatedSource.Element> elements = new List<ConcatenatedSource.Element>();

            // fs ヘッダをまとめて暗号化
            {
                var headerElements = new List<ConcatenatedSource.Element>();
                foreach (var bodyElement in bodySources)
                {
                    headerElements.Add(bodyElement.HeaderElement);
                }

                if (bodySources.Length < FsHeaderCountMax)
                {
                    // 使用しない fs n ヘッダも 0 フィルする必要があるため、n = FsHeaderCountMax - 1 の領域は必ず用意する
                    headerElements.Add(new ConcatenatedSource.Element(new PaddingSource(NintendoContentFileSystemMeta.FsHeaderSize), "fsHeaderPadding", NintendoContentFileSystemMeta.FsHeaderSize * (FsHeaderCountMax - 1)));
                }

                // fs n データを HashBlockSize でアライメントするためにパディングを追加する
                // このパディングは参照されません
                if (GetHashBlockAlignmentSize() == HashBlockSize)
                {
                    long paddingOffset = FsHeaderCountMax * NintendoContentFileSystemMeta.FsHeaderSize + NintendoContentFileSystemMeta.HeaderSize;
                    if (paddingOffset % HashBlockSize != 0)
                    {
                        long paddingSize = HashBlockSize - (paddingOffset % HashBlockSize);
                        var element = new ConcatenatedSource.Element(
                            new PaddingSource(paddingSize), "fsPadding", FsHeaderCountMax * NintendoContentFileSystemMeta.FsHeaderSize);
                        headerElements.Add(element);
                    }
                }

                var headerSource = new Aes128XtsEncryptedSource(
                    new ConcatenatedSource(headerElements), m_headerEncryptor, NintendoContentFileSystemMeta.HeaderSize);
                elements.Add(new ConcatenatedSource.Element(headerSource, "fsHeader", 0));
            }

            foreach (var entryInfo in fsEntries)
            {
                var index = fsEntries.CurrentIndex;
                ISource source = new ConcatenatedSource(bodySources[index].DataSourceElements);
                var key = bodyEncryptionKey.SelectAesCtrKey();
                switch (entryInfo.encryptionType)
                {
                    case (byte)NintendoContentArchiveEncryptionType.None:
                        break;
                    case (byte)NintendoContentArchiveEncryptionType.AesCtr:
                    {
                        // TODO: Counter を ContentType, FS-Index, Generation によって重ならないように求める
                        source = new Aes128CtrEncryptedSource(source, new Aes128CtrCryptoDriver(key), entryInfo.secureValue, entryInfo.generation, (long)entryInfo.startOffset);
                        break;
                    }
                    case (byte)NintendoContentArchiveEncryptionType.AesCtrEx:
                    {
                        source = new Aes128CtrExEncryptedSource(
                            source,
                            new Aes128CtrCryptoDriver(key),
                            entryInfo.secureValue,
                            entryInfo.generationTable,
                            entryInfo.generation, // generationTable の範囲外のデータの鍵世代は m_fileSystemInfo.keyGeneration を設定
                            (long)entryInfo.startOffset);
                        break;
                    }
                    default:
                        throw new InvalidDataException();
                }
                elements.Add(new ConcatenatedSource.Element(source, "fsData" + index, bodySources[index].DataOffset));
            }

            m_bodySource = new ConcatenatedSource(elements);

            m_fileSystemInfo.contentSize = NintendoContentFileSystemMeta.HeaderSize + (ulong)m_bodySource.Size;
        }

        private void MakeNcaHeader(ISource fsHeadersHashSource, BodyEncryptionKey bodyEncryptionKey)
        {
            // 内部鍵の場合（外部鍵の場合は encryptedKeyArea には何も書き込まない）
            if( !TicketUtility.IsValidRightsId(m_fileSystemInfo.rightsId) )
            {
                // BodySource の暗号化鍵を fileSystemInfo に埋め込み
                byte[] encryptedKeyArea = bodyEncryptionKey.GetEncryptedKey(m_fileSystemInfo, m_bodyEncryptionKeyEncryptor, m_bodyEncryptionKeyEncryptorHw);
                Buffer.BlockCopy(encryptedKeyArea, 0, m_fileSystemInfo.encryptedKey, 0, encryptedKeyArea.Length);
            }

            // 署名部分を省いた NCA ヘッダを取得。FS n ヘッダのハッシュを埋め込む。
            ISource unsignedHeaderSource = new AdaptedSource(
                new MemorySource(NintendoContentFileSystemMeta.CreateNcaHeader(m_fileSystemInfo), 512, NintendoContentFileSystemMeta.HeaderSize - 512),
                fsHeadersHashSource, 128, fsHeadersHashSource.Size);

            // 署名部分を省いた NCA ヘッダの署名
            Rsa2048PssSha256SignedSource headerSignedSource = new Rsa2048PssSha256SignedSource(unsignedHeaderSource, m_header1Signer, m_header2Signer);

            // NCA ヘッダ署名 1 を取得
            ISource headerSignatureSource1 = headerSignedSource.GetSignatureValueSource(0);

            // NCA ヘッダ署名 2 を取得
            ISource headerSignatureSource2 = headerSignedSource.GetSignatureValueSource(1);

            // NCA ヘッダの署名と署名部分を省いた FS ヘッダを連結
            List<ConcatenatedSource.Element> headerSourceElements = new List<ConcatenatedSource.Element>();
            headerSourceElements.Add(new ConcatenatedSource.Element(headerSignatureSource1, "headerSign1", 0));
            headerSourceElements.Add(new ConcatenatedSource.Element(headerSignatureSource2, "headerSign2", 256));
            headerSourceElements.Add(new ConcatenatedSource.Element(headerSignedSource, "headerMain", 512));

            ISource headerSource = new ConcatenatedSource(headerSourceElements);

            // HeaderSource の暗号化処理
            m_headerSource = new Aes128XtsEncryptedSource(headerSource, m_headerEncryptor);
        }

        private ISource NrrSignedSource(ISource source)
        {
            var adaptSourceInfos = GetAdaptSourceInfosForNrrSign(source);
            if (adaptSourceInfos.Count == 0)
            {
                return source;
            }
            return new AdaptedSource(source, adaptSourceInfos);
        }

        private ISource NrrSignedSourceForPatch(ref byte[] masterHash, NintendoContentPatchFileSystemInfo patchFsInfo)
        {
            var integrityRomFsSource = patchFsInfo.NewSource;
            var hashSource = new SubSource(integrityRomFsSource, 0, patchFsInfo.HashTargetOffset);
            var romFsSource = new SubSource(integrityRomFsSource, patchFsInfo.HashTargetOffset, integrityRomFsSource.Size - patchFsInfo.HashTargetOffset);

            var adaptSourceInfos = GetAdaptSourceInfosForNrrSign(romFsSource);

            if (adaptSourceInfos.Count == 0)
            {
                return new NintendoContentPatchFsArchiveSource(patchFsInfo);
            }

            var updatedRomFsSource = new AdaptedSource(romFsSource, adaptSourceInfos);

            var hashData = hashSource.PullData(0, (int)hashSource.Size).Buffer.Array;
            Debug.Assert(hashData.Length == hashSource.Size);
            var hashSink = new IntegrityHierarchicalStorageSink(updatedRomFsSource.Size, hashData, masterHash);

            foreach (var adapt in adaptSourceInfos)
            {
                // 更新箇所のハッシュを再計算
                long alignment = HierarchicalIntegrityVerificationStorage.GetHashBlockSize();
                var offset = adapt.Item2 & ~(alignment - 1);
                var size = (adapt.Item3 + (alignment - 1)) & ~(alignment - 1);
                var data = updatedRomFsSource.PullData(offset, (int)size);
                hashSink.PushData(data, offset);
            }

            var updatedHashSource = new IntegrityHierarchicalLayerHashSource(hashSink);
            Debug.Assert(updatedHashSource.Size == patchFsInfo.HashTargetOffset);

            ISource updatedIntegrityRomFsSource;
            {
                var elements = new List<ConcatenatedSource.Element>();
                elements.Add(new ConcatenatedSource.Element(updatedHashSource, "hash", 0));
                elements.Add(new ConcatenatedSource.Element(updatedRomFsSource, "data", updatedHashSource.Size));
                updatedIntegrityRomFsSource = new ConcatenatedSource(elements);
            }

            patchFsInfo.UpdateNewSource(updatedIntegrityRomFsSource);

            {
                var masterHashSource = new IntegrityHierarchicalMasterHashSource(hashSink);
                var updatedMasterHash = masterHashSource.PullData(0, (int)masterHashSource.Size).Buffer.Array;
                Debug.Assert(updatedMasterHash.Length == masterHash.Length);
                masterHash = updatedMasterHash;
            }

            return new NintendoContentPatchFsArchiveSource(patchFsInfo);
        }

        private List<Tuple<ISource, long, long>> GetAdaptSourceInfosForNrrSign(ISource romFsSource)
        {
            var adaptSourceInfos = new List<Tuple<ISource, long, long>>();

            using (Stream stream = new SourceBasedStream(romFsSource))
            {
                var reader = new RomFsFileSystemArchiveReader(stream);

                // .nrr ディレクトリの .nrr ファイルに ApplicationId (ProgramId) を埋め込み署名する

                foreach (var info in reader.ListFileInfo(".nrr/"))
                {
                    var fileName = info.Item1;
                    var fileSize = info.Item2;
                    if (Regex.IsMatch(fileName, "^(.nrr/).+(.nrr)$"))
                    {
                        var nrrData = reader.ReadFile(fileName, 0, fileSize);
                        var nrrSize = fileSize;
                        var nrrBaseOffset = reader.GetFileFragmentList(fileName).Single().Item1;

                        if (nrrData.Length != nrrSize)
                        {
                            throw new InvalidOperationException();
                        }

                        {
                            const int NrrApplicationIdOffset = 0x330;
                            Buffer.BlockCopy(BitConverter.GetBytes(m_fileSystemInfo.programId), 0, nrrData, NrrApplicationIdOffset, 8);
                        }
                        {
                            const int NrrSignTargetOffset = 0x330;
                            const int NrrSignatureOffset = 0x230;
                            const int NrrSignatureSize = 0x100;

                            var sign = m_nrrSigner.SignBlock(nrrData, NrrSignTargetOffset, (int)(nrrSize - NrrSignTargetOffset));
                            if (sign.Length != NrrSignatureSize)
                            {
                                throw new ArgumentException("Invalid signature size. Confirm NrrSignKey.");
                            }
                            Buffer.BlockCopy(sign, 0, nrrData, NrrSignatureOffset, NrrSignatureSize);
                        }
                        {
                            const int NrrCertificateOffset = 0x10;
                            const int NrrCertificateVerifyKeyOffset = NrrCertificateOffset + 0x20;
                            const int NrrCertificateVerifyKeySize = 0x100;

                            var modulus = m_nrrSigner.GetKeyModulus();
                            if (modulus.Length != NrrCertificateVerifyKeySize)
                            {
                                throw new ArgumentException("Invalid key size. Confirm NrrSignKey.");
                            }
                            Buffer.BlockCopy(modulus, 0, nrrData, NrrCertificateVerifyKeyOffset, NrrCertificateVerifyKeySize);

                            const int NrrCertificateSignTargetOffset = NrrCertificateOffset;
                            const int NrrCertificateSignTargetSize = 0x120;
                            const int NrrCertificateSignatureOffset = NrrCertificateSignTargetOffset + 0x120;
                            const int NrrCertificateSignatureSize = 0x100;

                            var sign = m_nrrCertificateSigner.SignBlock(nrrData, NrrCertificateSignTargetOffset, NrrCertificateSignTargetSize);
                            if (sign.Length != NrrCertificateSignatureSize)
                            {
                                throw new ArgumentException("Invalid signature size. Confirm NrrCertificateSignKey.");
                            }
                            Buffer.BlockCopy(sign, 0, nrrData, NrrCertificateSignatureOffset, NrrCertificateSignatureSize);
                        }

                        ISource memorySource = new MemorySource(nrrData, 0, nrrData.Length);
                        adaptSourceInfos.Add(Tuple.Create(memorySource, nrrBaseOffset, nrrSize));
                    }
                }
            }

            return adaptSourceInfos;
        }

        // npdm の中の ACID を検証し NCA ヘッダ署名検証鍵を差し換えて製品鍵で再署名する
        private ISource AcidResignedSource(ISource source)
        {
            byte[] npdmData;
            long npdmBaseOffset;
            long npdmSize;
            using (Stream stream = new SourceBasedStream(source))
            {
                var reader = new PartitionFileSystemArchiveReader(stream);
                var fragmentList = reader.GetFileFragmentList("main.npdm");
                npdmBaseOffset = fragmentList.Single().Item1;
                npdmSize = fragmentList.Single().Item2;
                npdmData = reader.ReadFile("main.npdm", 0, npdmSize);

                if (npdmData.Length != npdmSize)
                {
                    throw new InvalidOperationException();
                }

                const int AcidOffsetDataOffset = 0x78;
                var acidOffset = BitConverter.ToInt32(npdmData, AcidOffsetDataOffset);

                const int AcidSignTargetOffset = 0x100;
                const int AcidSignatureSize = 0x100;

                // ACID の署名対象サイズを取得
                const int AcidSignTargetSizeDataOffset = 0x204;
                var acidSignTargetSize = BitConverter.ToInt32(npdmData, acidOffset + AcidSignTargetSizeDataOffset);

                // ACID の検証
                {
                    var sign = new byte[AcidSignatureSize];
                    Buffer.BlockCopy(npdmData, acidOffset, sign, 0, sign.Length);
                    if (!m_acidVerifier.VerifyBlock(npdmData, acidOffset + AcidSignTargetOffset, acidSignTargetSize, sign))
                    {
                        throw new ArgumentException("Failed to verify acid in main.npdm.");
                    }

                    // 製品化フラグのチェック
                    const int AcidProductionFlagOffset = 0x20C;
                    var productionFlag = BitConverter.ToInt32(npdmData, acidOffset + AcidProductionFlagOffset);
                    if ((productionFlag & 0x1) == 0)
                    {
                        throw new ArgumentException("This application is not allowed to be prodencrypted.");
                    }
                }

                // ACID 内の NCA ヘッダ署名検証鍵を差し換え
                {
                    const int AcidHeader2VerifyKeyOffset = AcidSignTargetOffset;
                    const int AcidHeader2VerifyKeySize = 0x100;

                    byte[] modulus = m_header2Signer.GetKeyModulus();
                    if (modulus.Length != AcidHeader2VerifyKeySize)
                    {
                        throw new ArgumentException("Invalid key size. Confirm NcaHeader2SignKey.");
                    }

                    Buffer.BlockCopy(modulus, 0, npdmData, acidOffset + AcidHeader2VerifyKeyOffset, modulus.Length);
                }

                // ACID を再署名
                {
                    byte[] sign = m_acidSigner.SignBlock(npdmData, acidOffset + AcidSignTargetOffset, acidSignTargetSize);
                    if (sign.Length != AcidSignatureSize)
                    {
                        throw new ArgumentException("Invalid signature size. Confirm AcidSignKey.");
                    }

                    Buffer.BlockCopy(sign, 0, npdmData, acidOffset, sign.Length);
                }
            }

            var npdmSource = new MemorySource(npdmData, 0, npdmData.Length);
            var modifiedSource = new AdaptedSource(source, npdmSource, npdmBaseOffset, npdmSize);
            return modifiedSource;
        }

        private class BuildDataInfo
        {
            public int Index { get; private set; }
            public ISource DataSource { get; private set; }
            public SourceInterface Data { get; private set; }
            public ulong CurrentOffset { get; private set; }
            public long PhysicalOffset { get; private set; }
            public ISource HeaderSource { get; set; }
            public List<ConcatenatedSource.Element> DataSourceElements { get; private set; }

            public BuildDataInfo(int hashBlockAlignment)
            {
                PhysicalOffset = BitUtility.AlignUp(NintendoContentFileSystemMeta.HeaderSize + NintendoContentFileSystemMeta.FsHeaderSize * FsHeaderCountMax, hashBlockAlignment);
                CurrentOffset = (ulong)PhysicalOffset;
            }

            public void SetData(int index, ISource dataSource)
            {
                Index = index;
                DataSource = dataSource;
                Data = null;
                HeaderSource = null;
                DataSourceElements = new List<ConcatenatedSource.Element>();
            }

            public void SetData(int index, SourceInterface data)
            {
                Index = index;
                DataSource = null;
                Data = data;
                HeaderSource = null;
                DataSourceElements = new List<ConcatenatedSource.Element>();
            }

            public void Advance(long size)
            {
                CurrentOffset += (ulong)size;
                PhysicalOffset += size;
            }

            public void Advance(ulong inputSize, long outputSize)
            {
                CurrentOffset += inputSize;
                PhysicalOffset += outputSize;
            }
        }

        private struct BodySourceElement
        {
            public ConcatenatedSource.Element HeaderElement;
            public long DataOffset;
            public List<ConcatenatedSource.Element> DataSourceElements;
        }

        private class BodyEncryptionKey
        {
            public byte[] GetInternalKey(NintendoContentArchiveEncryptionKeyIndex index)
            {
                return m_internalKey[(int)index];
            }

            public byte[] GetExternalKey()
            {
                return m_externalKey;
            }

            public bool UseExternalKey()
            {
                return m_externalKey != null;
            }

            private const int KeyCount = NintendoContentFileSystemMetaConstant.DecryptionKeyIndexMax;
            private const int KeyLength = 16;

            // 内部鍵用。実際に利用しているのは NintendoContentArchiveEncryptionKeyIndex.AesCtr のみだが、
            // 互換性を保つため、KeyCount 分の領域は空けておく
            private byte[][] m_internalKey;
            private byte[] m_externalKey;

            public BodyEncryptionKey(NintendoContentFileSystemInfo fileSystemInfo, IHashCalculator externalKeyHmacCalculator)
            {
                if (externalKeyHmacCalculator != null)
                {
                    // 外部鍵の場合のタイトル鍵取得
                    // (スパース化の場合は fileSystemInfo にオリジナルの情報が設定済みで、同じ鍵が返る)
                    m_externalKey = ExternalContentKeyGenerator.GetNcaExternalContentKey(
                        externalKeyHmacCalculator,
                        TicketUtility.GetContentMetaIdFromRightsId(fileSystemInfo.rightsId),
                        fileSystemInfo.keyGeneration
                    ).Key;
                }
                else
                {
                    // 内部鍵の場合

                    if (fileSystemInfo.isProdEncryption)
                    {
                        if (IsKeyInheritanceRequired(fileSystemInfo))
                        {
                            throw new InvalidOperationException();
                        }

                        // 製品版のときは乱数鍵とする
                        GenerateRamdomInternalKey();
                    }
                    else
                    {
                        if (IsKeyInheritanceRequired(fileSystemInfo))
                        {
                            // 鍵のつけかえをサポートしないソースの場合は、情報元から鍵を継承する
                            InheritInternalKey(fileSystemInfo);
                        }
                        else if (fileSystemInfo.keyAreaEncryptionKeyIndex != 0)
                        {
                            // KeyIndex != 0 のときは乱数鍵とする
                            GenerateRamdomInternalKey();
                        }
                        else
                        {
                            // 開発版かつ KeyIndex == 0 のときは固定鍵とする
                            SetFixedInternalKeyForDevelopment(fileSystemInfo);
                        }
                    }
                }
            }

            private byte[] ToArray()
            {
                if (m_internalKey == null)
                {
                    throw new InvalidOperationException();
                }

                byte[] data = new byte[KeyLength * KeyCount];
                for (int i = 0; i < KeyCount; ++i)
                {
                    byte[] src = m_internalKey[i] != null ? m_internalKey[i] : new byte[KeyLength];
                    Debug.Assert(src.Length == KeyLength);
                    Array.Copy(src, 0, data, i * KeyLength, KeyLength);
                }
                return data;
            }

            public byte[] SelectAesCtrKey()
            {
                return UseExternalKey() ? GetExternalKey() : GetInternalKey(NintendoContentArchiveEncryptionKeyIndex.AesCtr);
            }

            public byte[] GetEncryptedKey(NintendoContentFileSystemInfo fileSystemInfo, IEncryptor encryptorSw, IEncryptor encryptorHw)
            {
                // 必要な鍵領域のみ暗号化、残りは 0 埋め
                byte[] rawKey = ToArray();
                byte[] encryptedKeyArea = new byte[rawKey.Length];
                encryptorSw.EncryptBlock(rawKey, 0, rawKey.Length - KeyLength * 1, encryptedKeyArea, 0); // HW 鍵のみ除外

                if (fileSystemInfo.isHardwareEncryptionKeyEmbedded)
                {
                    // HW 鍵
                    byte[] encryptedKeyAreaHw = new byte[KeyLength];
                    encryptorHw.EncryptBlock(rawKey, (int)NintendoContentArchiveEncryptionKeyIndex.AesCtr * KeyLength, KeyLength, encryptedKeyArea, (int)NintendoContentArchiveEncryptionKeyIndex.AesCtrHw * KeyLength);
                }

                return encryptedKeyArea;
            }

            private void SetFixedInternalKeyForDevelopment(NintendoContentFileSystemInfo fileSystemInfo)
            {
                if (m_internalKey != null)
                {
                    throw new InvalidOperationException();
                }

                // 開発用の固定鍵は "Content+" programId - idOffset (contentMetaId が指定されている場合はそちら) とする
                var keyId = fileSystemInfo.contentMetaId != null ? fileSystemInfo.contentMetaId.Value : fileSystemInfo.programId - (ulong)fileSystemInfo.GetRepresentProgramIdOffset();
                var internalKeyValue = Encoding.ASCII.GetBytes("Content+").Concat(BitConverter.GetBytes(keyId)).ToArray();
                Debug.Assert(internalKeyValue.Length == KeyLength);

                m_internalKey = new byte[KeyCount][];
                m_internalKey[(int)NintendoContentArchiveEncryptionKeyIndex.AesCtr] = internalKeyValue;
            }

            private void GenerateRamdomInternalKey()
            {
                if (m_internalKey != null)
                {
                    throw new InvalidOperationException();
                }

                var internalKeyValue = new byte[KeyLength];
                var rng = new RNGCryptoServiceProvider();
                rng.GetBytes(internalKeyValue);

                m_internalKey = new byte[KeyCount][];
                m_internalKey[(int)NintendoContentArchiveEncryptionKeyIndex.AesCtr] = internalKeyValue;
            }

            private static bool IsKeyInheritanceRequired(NintendoContentFileSystemInfo fileSystemInfo)
            {
                foreach (var fsEntry in fileSystemInfo.fsEntries)
                {
                    var sparseSource = fsEntry.sourceInterface as FileSystemArchiveSparseSource;
                    if (sparseSource != null)
                    {
                        return true;
                    }
                }
                return false;
            }

            private void InheritInternalKey(NintendoContentFileSystemInfo fileSystemInfo)
            {
                if (m_internalKey != null)
                {
                    throw new InvalidOperationException();
                }

                m_internalKey = new byte[KeyCount][];

                foreach (var fsEntry in fileSystemInfo.fsEntries)
                {
                    var sparseSource = fsEntry.sourceInterface as FileSystemArchiveSparseSource;
                    if (sparseSource != null)
                    {
                        var index = (int)NintendoContentArchiveEncryptionKeyIndex.AesCtr;
                        var internalKey = sparseSource.ExtraInfo.originalReader.GetInternalKey(index);
                        m_internalKey[index] = internalKey;
                        return;
                    }
                }

                throw new InvalidDataException();

            }
        }
    }
}
