﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.IO;
using System.Text.RegularExpressions;
using Nintendo.Authoring.FileSystemMetaLibrary;

namespace Nintendo.Authoring.AuthoringLibrary
{
    public class ContentHashSource
    {
        public ISource Source { get; private set; }
        public string Extension { get; private set; }
        public string ContentType { get; private set; }
        public ContentHashSource(ISource source, string extension, string contentType)
        {
            Source = source;
            Extension = extension;
            ContentType = contentType;
        }
        public ContentHashSource(ISource source, string extension) : this(source, extension, string.Empty)
        {
        }
    }

    public class HashNameEntryPartitionFsHeaderSource<T> : ISource
        where T : IPartitionFileSystemMeta, new()
    {
        public long Size { get; private set; }

        private List<ContentHashSource> m_hashSources;
        private PartitionFileSystemInfo m_partFsInfo;
        private byte[] m_buffer;

        public HashNameEntryPartitionFsHeaderSource(List<ContentHashSource> hashSources, PartitionFileSystemInfo partFsInfo, long size)
        {
            Size = size;
            if (hashSources.Count != partFsInfo.entries.Count)
            {
                throw new ArgumentException();
            }
            m_hashSources = hashSources;
            m_partFsInfo = partFsInfo;
        }
        public ByteData PullData(long offset, int size)
        {
            if (m_buffer == null)
            {
                // PartitionFs ヘッダのファイル名にハッシュ値の前半 128 bit を 16 進文字列化し差し込む
                for (int i = 0; i < m_hashSources.Count; i++)
                {
                    // ファイル名を変える必要がない場合
                    if (m_hashSources[i].Extension != ".nca" &&
                        m_hashSources[i].Extension != ".cnmt.nca" &&
                        m_hashSources[i].Extension != ".cnmt.xml" &&
                        m_hashSources[i].Extension != ".jpg" &&
                        m_hashSources[i].Extension != ".cnmt" &&
                        m_hashSources[i].Extension != ".nacp.xml" &&
                        m_hashSources[i].Extension != ".programinfo.xml" &&
                        m_hashSources[i].Extension != ".legalinfo.xml" &&
                        m_hashSources[i].Extension != ".htmldocument.xml" &&
                        m_hashSources[i].Extension != ".compaction.bin")
                    {
                        continue;
                    }

                    if (m_hashSources[i].Source == null || m_hashSources[i].Source.Size != 32)
                    {
                        throw new InvalidOperationException();
                    }

                    ByteData data = m_hashSources[i].Source.PullData(0, 16);
                    if (data.Buffer.Count != 16)
                    {
                        throw new InvalidOperationException();
                    }

                    byte[] buffer = new byte[data.Buffer.Count];
                    Buffer.BlockCopy(data.Buffer.Array, data.Buffer.Offset, buffer, 0, data.Buffer.Count);

                    var entry = m_partFsInfo.entries[i];
                    entry.name = Regex.Replace(entry.name, @"^.{32}", BitConverter.ToString(buffer).Replace("-", string.Empty).ToLower());
                    m_partFsInfo.entries[i] = entry;
                }

                var metaMgr = new T();
                m_buffer = metaMgr.Create(m_partFsInfo);
                if (Size != m_buffer.Length)
                {
                    throw new InvalidOperationException();
                }
            }

            ISource source = new MemorySource(m_buffer, 0, m_buffer.Length);
            return source.PullData(offset, size);
        }

        public SourceStatus QueryStatus()
        {
            SourceStatus sourceStatus = new SourceStatus();
            foreach (var source in m_hashSources)
            {
                if (source.Source == null)
                {
                    continue;
                }
                RangeList rangeList = source.Source.QueryStatus().AvailableRangeList;
                if (rangeList.Count != 1 || (rangeList.Count == 1 &&  rangeList[0].Size != source.Source.Size))
                {
                    return sourceStatus;
                }
            }
            sourceStatus.AvailableRangeList.MergingAdd(new Range(0, Size));
            return sourceStatus;
        }

        public static long GetDummySize(List<string> entryNameList)
        {
            var metaMgr = new T();
            var tmpPartFsInfo = new PartitionFileSystemInfo();

            foreach (var entryName in entryNameList)
            {
                var content = new PartitionFileSystemInfo.EntryInfo();
                content.name = entryName;
                tmpPartFsInfo.entries.Add(content);
            }

            byte[] buffer = metaMgr.Create(tmpPartFsInfo);
            return (long)buffer.Length;
        }
    }

    public class Sha256PartitionFsHashSource
    {
        public ISource Source { get; private set; }
        public int Index { get; private set; }
        public Sha256PartitionFsHashSource(ISource source, int index)
        {
            Source = source;
            Index = index;
        }
    }

    public class HashAdaptedSha256PartitionFsHeaderSource : ISource
    {
        public long Size { get; private set; }

        private ISource m_Source;

        public HashAdaptedSha256PartitionFsHeaderSource(ISource source, List<Sha256PartitionFsHashSource> hashSources)
        {
            var adaptSourceInfos = new List<Tuple<ISource, long, long>>();

            // Sha256PartitionFs ヘッダのハッシュエントリにハッシュ値を差し込む
            foreach (var hashSource in hashSources)
            {
                if (hashSource.Source == null || hashSource.Source.Size != 32)
                {
                    throw new InvalidOperationException();
                }

                adaptSourceInfos.Add(Tuple.Create(hashSource.Source, Sha256PartitionFileSystemMeta.GetEntryHashOffset(hashSource.Index), hashSource.Source.Size));
            }

            m_Source = new AdaptedSource(source, adaptSourceInfos);
            Size = m_Source.Size;
        }

        public ByteData PullData(long offset, int size)
        {
            return m_Source.PullData(offset, size);
        }

        public SourceStatus QueryStatus()
        {
            return m_Source.QueryStatus();
        }
    }
}
