﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.Reflection;
using System.Text;
using System.Xml;
using System.Xml.Serialization;
using Nintendo.Authoring.FileSystemMetaLibrary;

namespace Nintendo.Authoring.AuthoringLibrary
{
    [XmlRoot("CardSpec", IsNullable = false)]
    public class CardSpecModel
    {
        [XmlElement("Size")]
        public string Size { get; set; }

        [XmlElement("ClockRate")]
        public string ClockRate { get; set; }

        [XmlElement("AutoSetSize")]
        public bool AutoSetSize { get; set; }

        [XmlElement("AutoSetClockRate")]
        public bool AutoSetClockRate { get; set; }

        [XmlElement("LaunchFlags")]
        public byte? LaunchFlags { get; set; }
        [XmlIgnore]
        public bool LaunchFlagsSpecified { get { return LaunchFlags != null; } }

        static public CardSpecModel Create(int size, int clockRate, bool autoSetSize, bool autoSetClock, byte? launchFlags)
        {
            if (size != XciInfo.InvalidRomSize && clockRate != XciInfo.InvalidClockRate)
            {
                XciUtils.CheckRomSizeAndClockRate(size, clockRate);
            }

            var model = new CardSpecModel();
            model.Size = size.ToString();
            model.ClockRate = clockRate.ToString();
            model.AutoSetSize = autoSetSize;
            model.AutoSetClockRate = autoSetClock;
            model.LaunchFlags = launchFlags;
            return model;
        }

        public byte[] GetBytes()
        {
            var nameSpace = new XmlSerializerNamespaces();
            nameSpace.Add(String.Empty, String.Empty);

            using (var memoryStream = new MemoryStream())
            {
                var sw = new StreamWriter(memoryStream, Encoding.UTF8);
                var serializer = new XmlSerializer(typeof(CardSpecModel));
                serializer.Serialize(sw, this, nameSpace);
                return memoryStream.ToArray();
            }
        }
    }

    public class CardSpecXmlSource : ISource
    {
        public long Size { get; set; }
        private ISource m_source;

        static public void SetupCardSize(ref bool autoSetSize, ref int cardSize, int requiredCardSize, long appAreaUsedSize)
        {
            if (requiredCardSize == XciInfo.InvalidRomSize)
            {
                // 最大サイズのカードに収まらない場合は警告表示
                Log.Warning(string.Format("Application Area Used: {0} (bytes)", appAreaUsedSize));
                Log.Warning("This nsp can't be published as a game card title because its size exceeds the maximum size of the game card.");
            }

            if (cardSize == 0)
            {
                // カードサイズの自動決定
                autoSetSize = true;
                cardSize = XciUtils.GetRomSizeMinForAutoSet(requiredCardSize);
            }
            else if (autoSetSize && cardSize != requiredCardSize)
            {
                cardSize = XciUtils.GetRomSizeMinForAutoSet(requiredCardSize);
                if (cardSize != XciInfo.InvalidRomSize)
                {
                    Log.Warning(string.Format("On-card patch requires larger card than that of revised application. Card size is automatically adjusted."));
                }
            }
            else
            {
                // カードサイズのチェック
                if (requiredCardSize != XciInfo.InvalidRomSize && cardSize < requiredCardSize)
                {
                    throw new ArgumentException(string.Format("This nsp requires CardSpec/Size = {0}, though it is set as {1}.", requiredCardSize, cardSize));
                }
                else if (requiredCardSize == XciInfo.InvalidRomSize && cardSize != requiredCardSize)
                {
                    // カードサイズが手動設定されているが、ゲームカードに収まらない場合
                    throw new ArgumentException(string.Format("CardSpec/Size in nmeta is invalid value. Please remove CardSpec/Size to set the value automatically."));
                }
            }
        }

        public CardSpecXmlSource(NintendoSubmissionPackageFileSystemInfo nspInfo, KeyConfiguration config, List<ContentMetaLiteModel> additionalContentMetaList, long additionalSize = 0)
        {
            // xci 変換後のサイズの算出（CUP 領域込み、OnCardPatch 込み）
            var xciSizeInfo = ProdEncryptedXciArchive.CalculateXciSize(nspInfo, additionalContentMetaList, false, config);
            var cardSize = XciUtils.GetRomSize(xciSizeInfo.TotalSize + additionalSize);

            var appAreaUsedSize = xciSizeInfo.TotalSize + additionalSize - (XciInfo.UpdatePartitionLimitSize + XciInfo.HeaderReservedSize);

            bool autoSetSize = nspInfo.AutoSetSize;
            bool autoSetClock = nspInfo.AutoSetClockRate;

            if (!nspInfo.SkipSizeCheck)
            {
                int nspInfoCardSize = nspInfo.CardSize;
                SetupCardSize(ref autoSetSize, ref nspInfoCardSize, cardSize, appAreaUsedSize);
                nspInfo.CardSize = nspInfoCardSize;
            }

            // 動作周波数設定が自動になっている場合は、都度カードサイズを元に再設定する
            // パッチによってカードサイズが無効値に自動設定され、動作周波数が手動設定されている場合はこの後 Assert に引っかかる
            if (nspInfo.CardClockRate == 0 || autoSetClock)
            {
                autoSetClock = true;
                if (nspInfo.CardSize == XciInfo.InvalidRomSize)
                {
                    nspInfo.CardClockRate = XciInfo.InvalidClockRate;
                }
                else
                {
                    nspInfo.CardClockRate = XciUtils.GetClockRate(nspInfo.CardSize);
                }
            }

            // カードサイズ、動作周波数のいずれか一方だけが無効値をとることはない
            if (nspInfo.CardSize != XciInfo.InvalidRomSize ^ nspInfo.CardClockRate != XciInfo.InvalidClockRate)
            {
                throw new ArgumentException(string.Format("{0} is set although the other one is invalid.", nspInfo.CardSize != XciInfo.InvalidRomSize ? "CardSize" : "CardClockRate"));
            }

            // CUP, カードヘッダを除いた領域の現在占有量を表示
            if (nspInfo.CardSize != XciInfo.InvalidRomSize)
            {
                var appAreaLimitSize = XciUtils.GetAvailableAreaSize(nspInfo.CardSize) - (XciInfo.UpdatePartitionLimitSize + XciInfo.HeaderReservedSize);
                Log.Info(string.Format("CardSize: {0} (GB), Application Area Used: {1}/{2} (bytes)", nspInfo.CardSize, appAreaUsedSize, appAreaLimitSize));
                var roomSize = appAreaLimitSize - appAreaUsedSize;
                if (roomSize < XciInfo.ApplicationUsableAreaSizeOnUpdatePartition)
                {
                    var msg = AuthoringTool.Resources.ResourceManager.GetString("Info_ExceedsApplicationUsableAreaSizeOnUpdatePartition");
                    Log.Info(string.Format(msg, XciInfo.ApplicationUsableAreaSizeOnUpdatePartition - roomSize));
                }
            }

            var cardSpecXml = CardSpecModel.Create(nspInfo.CardSize, nspInfo.CardClockRate, autoSetSize, autoSetClock, nspInfo.CardLaunchFlags);
            var bytes = cardSpecXml.GetBytes();
            m_source = new MemorySource(bytes, 0, bytes.Length);
            Size = m_source.Size;
        }

        public ByteData PullData(long offset, int size)
        {
            return m_source.PullData(offset, size);
        }

        public SourceStatus QueryStatus()
        {
            return m_source.QueryStatus();
        }
    }
    public class CardSpecXmlUtils
   {
        public static void VerifyCardSpec(string metaFilePath)
        {
            if (metaFilePath == null)
            {
                return;
            }

            var model = MetaFileReader.ReadMetaModel<CardSpecModel>(metaFilePath , "//CardSpec");

            if (model == null)
            {
                return;
            }

            UnpublishableErrorCheckData checkData = new UnpublishableErrorCheckData();
            checkData.CardSpec = model;

            UnpublishableError.VerifyNmetaUnpublishableError(checkData, UnpublishableErrorCheck.CardSpecCheckList);
        }
    }
}
