﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.IO;
using System.IO.Compression;
using System.Linq;
using System.Runtime.CompilerServices;
using System.Diagnostics;
using System.Xml.Linq;
using System.Text.RegularExpressions;
using Nintendo.Authoring.AuthoringLibrary;

[assembly: InternalsVisibleTo("AuthoringToolsOptionTest")]

namespace Nintendo.Authoring.AuthoringTool
{
    using DirecotryConnector = Pair<string, string>;

    internal class OptionDescription
    {
        public OptionDescription(string longName, string shortName, int hasArgument, Action<List<string>> action)
        {
            LongName = longName;
            ShortName = shortName;
            HasArgument = hasArgument;
            Action = action;
        }

        public string LongName { get; set; }
        public string ShortName { get; set; }
        public int HasArgument { get; set; }
        public Action<List<string>> Action { get; set; }
    }

    internal interface IOption
    {
        void ShowSubCommandUsage(string baseName);
        OptionDescription[] GetOptionDescription();
        void ParsePositionalArgument(string[] args);
    }

    internal class OptionUtil
    {
        public static string GetOutputFilePath(string defaultStr, string argStr)
        {
            string output = string.Empty;
            string path = argStr.Replace("\\", "/");
            if (!Path.IsPathRooted(path))
            {
                path = "./" + path;
            }
            if (Directory.Exists(path) && ((File.GetAttributes(path) & FileAttributes.Directory) == FileAttributes.Directory))
            {
                output = path + defaultStr.Substring(1);
            }
            else if (path.EndsWith("/"))
            {
                output = path + defaultStr.Substring(2);
            }
            else
            {
                output = path;
            }
            return output;
        }

        public static void CheckExtension(string path, string ext)
        {
            if (path == null)
            {
                return;
            }

            if (Path.GetExtension(path) != ext)
            {
                throw new InvalidOptionException(string.Format("input file must be {0} file.", ext));
            }
        }

        public static string NormalizePath(string path)
        {
            string newPath = path.Replace("\\", "/");
            if (!Path.IsPathRooted(newPath))
            {
                newPath = "./" + newPath;
            }
            return newPath;
        }

        public static string CheckAndNormalizeFilePath(string path, string optionName)
        {
            string filePath = NormalizePath(path);
            if ((File.GetAttributes(filePath) & FileAttributes.Directory) == FileAttributes.Directory)
            {
                throw new InvalidOptionException(String.Format("file path should be specified for {0}.", optionName));
            }
            return filePath;
        }

        public static string CheckAndNormalizeDirectory(string path, string optionName)
        {
            string dirPath = NormalizePath(path);
            if (!Directory.Exists(dirPath) ||
                (File.GetAttributes(dirPath) & FileAttributes.Directory) != FileAttributes.Directory)
            {
                throw new InvalidOptionException(String.Format("directory path should be specified for {0}.", optionName));
            }

            return dirPath;
        }

        internal delegate void PathSetter(string path);

        public static OptionDescription CreateFilePathOptionDescription(string longName, PathSetter setter)
        {
            return CreateFilePathOptionDescription(longName, null, setter);
        }

        public static OptionDescription CreateFilePathOptionDescription(string longName, string checkExt, PathSetter setter)
        {
            return new OptionDescription(longName, null, 1, (s) =>
                {
                    if (checkExt != null)
                    {
                        CheckExtension(s.First(), checkExt);
                    }
                    setter(CheckAndNormalizeFilePath(s.First(), longName));
                });
        }

        internal static List<Tuple<string, string>> CreateIconFileList(List<string> languageAndPathList)
        {
            var list = new List<Tuple<string, string>>();
            for (int i = 0; i < languageAndPathList.Count; i += 2)
            {
                var language = languageAndPathList[i];
                var iconPath = languageAndPathList[i + 1].Replace("\\", "/");
                if (!Path.IsPathRooted(iconPath))
                {
                    iconPath = "./" + iconPath;
                }
                if ((File.GetAttributes(iconPath) & FileAttributes.Directory) == FileAttributes.Directory)
                {
                    throw new InvalidOptionException("file path should be specified for --icon option.");
                }

                list.Add(Tuple.Create(language, iconPath));
            }
            return list;
        }
    }

    internal class CreateCommandOptionBase
    {
        internal string InputAdfFile { get; set; }
        internal string InputFdfPath { get; set; }
        internal string OutputFile { get; set; }
        internal bool IsSaveAdf { get; set; }
        internal bool IsOnlyAdf { get; set; }

        internal CreateCommandOptionBase()
        {
            InputAdfFile = null;
            InputFdfPath = null;
            OutputFile = "./output";
            IsSaveAdf = false;
            IsOnlyAdf = false;
        }

        public OptionDescription[] GetOptionDescription()
        {
            OptionDescription[] options = new OptionDescription[]
                {
                    new OptionDescription(null, "-o", 1, (s) => { OutputFile = OptionUtil.GetOutputFilePath(OutputFile, s.First()); }),
                    new OptionDescription("--filter", null, 1, (s) => { InputFdfPath = s.First(); }),
                    new OptionDescription("--save-adf", null, 0, (s) => { IsSaveAdf = true; }),
                    new OptionDescription("--only-adf", null, 0, (s) => { IsOnlyAdf = true; }),
                };
            return options;
        }
    }

    internal class NoneOption : IOption
    {
        internal bool IsVerbose { get; private set; }
        internal bool IsShowUsage { get; set; }
        internal bool IsUtf8 { get; set; }

        internal AuthoringConfiguration Config = new AuthoringConfiguration();

        public void ShowSubCommandUsage(string baseName)
        {
            Console.WriteLine("{0}: Authoring Tool for Nintendo SDK", baseName);
            Console.WriteLine("Usage: {0} <subcommand> [options] [args]", baseName);
            Console.WriteLine(string.Empty);
            Console.WriteLine("Available subcommands:");
            Console.WriteLine("  createfs                     Create new file system archive.");
            Console.WriteLine("  createnca                    Create new Nintendo Content Archive.");
            Console.WriteLine("  creatensp                    Create new Nintendo Submission Package.");
            Console.WriteLine("  createnspmeta                Create only meta files for Nintendo Submission Package.");
            Console.WriteLine("  createnspd                   Create new Nintendo Submission Package Directory.");
            Console.WriteLine("  makepatch                    Create patch file from 2 Nintendo Submission Packages.");
            Console.WriteLine("  bundleup                     Create multi-application card indicator nsp file.");
            Console.WriteLine("  convertaoc                   Convert AddOnContent nsp to be available on card.");
            Console.WriteLine("  extract                      Extract files from a Nintendo Submission Package or Nintendo Content Archive.");
            Console.WriteLine("  replace                      Replace a file in a Nintendo Submission Package or Nintendo Content Archive.");
            Console.WriteLine("  replacenspmeta               Replace a file in a Nintendo Submission Package from a nmeta file.");
            Console.WriteLine("  list                         List information about the files included in an archive file.");
            Console.WriteLine("  diffpatch                    Show different files between patches.");
            Console.WriteLine("  getproperty                  Show property of Nintendo Submission Package.");
            Console.WriteLine("  get-unpublishable-error      Show unpublishable error of Nintendo Submission Package.");
            Console.WriteLine("  validatemeta                 Validate meta file.");
            Console.WriteLine("  showversion                  Output tool version to standard output.");
            Console.WriteLine("  help                         Describe the usage of this program or its subcommands.");
            Console.WriteLine("Options:");
            Console.WriteLine("  -v, --verbose                Show detail log.");
            Console.WriteLine("  --utf8                       Show message with UTF-8.");
#if false // 非公開オプション
            Console.WriteLine("  createnacp                   Create new Application Control Property.");
            Console.WriteLine("  mergensp                     Merge the contents included in multiple Nintendo Submission Packages into one nsp file.");
            Console.WriteLine("  get-all-xml-models           Output all xml models.");
            Console.WriteLine("  --keyconfig <path>           Path of key configuration file.");
            Console.WriteLine("  --includes-cnmt              Includes content meta binary to nsp file for debug use.");
#endif
            Console.WriteLine(string.Empty);
            Console.WriteLine("Type '{0} help <subcommand>' for help on a specific subcommand.", baseName);
        }

        public OptionDescription[] GetOptionDescription()
        {
            OptionDescription[] options = new OptionDescription[]
                {
                    new OptionDescription("--help", "-h", 0, (s) => { IsShowUsage = true; } ),
                    new OptionDescription("--verbose", "-v", 0, (s) => { IsVerbose = true; } ),
                    new OptionDescription("--utf8", null, 0, (s) => { IsUtf8 = true; } ),
                    new OptionDescription("--keyconfig", null, 1, (s) =>
                            {
                                Config.KeyConfigFilePath = OptionUtil.GetOutputFilePath(string.Empty, s.First());
                            } ),
                    new OptionDescription("--includes-cnmt", null, 0, (s) => { Config.DebugConfig.EnableContentMetaBinaryExport = true; }),
                };
            return options;
        }

        public void ParsePositionalArgument(string[] args)
        {
            return;
        }
    }

    internal class HelpOption : IOption
    {
        internal List<Option.SubCommandType> SubCommandList { get; private set; }

        internal HelpOption()
        {
            SubCommandList = new List<Option.SubCommandType>();
        }

        public void ShowSubCommandUsage(string baseName)
        {
            Console.WriteLine("help: Describe the usage of this program or its subcommands.");
            Console.WriteLine("Usage: {0} help [subcommand]", baseName);
        }

        public OptionDescription[] GetOptionDescription()
        {
            return null;
        }

        public void ParsePositionalArgument(string[] args)
        {
            for (int i = 0; i < args.Length; i++)
            {
                Option.SubCommandType typeToHelp = Option.GetSubCommandType(args[i]);
                if (typeToHelp != Option.SubCommandType.None)
                {
                    SubCommandList.Add(typeToHelp);
                }
            }
        }
    }


    internal class CreateFsOption : CreateCommandOptionBase, IOption
    {
        internal string InputDir { get; set; }
        internal ArchiveFormatType Format { get; set; }

        internal CreateFsOption()
        {
            InputDir = null;
            InputAdfFile = null;
            Format = ArchiveFormatType.Invalid;
            OutputFile = "./output.fs";
        }

        public void ShowSubCommandUsage(string baseName)
        {
            Console.WriteLine("createfs: Create new file system archive from .adf or specified directory");
            Console.WriteLine("Usage: {0} createfs [-o <outputPath>] <inputAdfPath>", baseName);
            Console.WriteLine("Usage: {0} createfs [-o <outputPath>] [--save-adf] [--only-adf] [--filter <fdfPath>] --format <format> <inputDirPath>", baseName);
            Console.WriteLine(string.Empty);
            Console.WriteLine("Options:");
            Console.WriteLine("  -o <outputPath>              File path to output. If directory path is specified, output to <outputPath>/output.fs.");
            Console.WriteLine("                               If omitted, output to ./output.fs");
            Console.WriteLine("  --save-adf                   Leave .adf which is generated intermediately.");
            Console.WriteLine("  --only-adf                   Create .adf only.");
            Console.WriteLine("  --filter <fdfPath>           Filter files to be included into .adf. <fdfPath> expects a path for a filter description file (.fdf).");
            Console.WriteLine("  --format <format>            Format Type to create archive.");
            Console.WriteLine("                               Available as <format>: partitionfs, romfs");
        }

        public new OptionDescription[] GetOptionDescription()
        {
            OptionDescription[] options = new OptionDescription[]
                {
                    new OptionDescription("--format", null, 1, (s) =>
                        {
                            if (s.First() == "partitionfs")
                            {
                                Format = ArchiveFormatType.PartitionFs;
                            }
                            else if (s.First() == "romfs")
                            {
                                Format = ArchiveFormatType.RomFs;
                            }
                            else
                            {
                                throw new InvalidOptionException(string.Format("invalid option --format."));
                            }
                        }),
                };
            return options.Concat(base.GetOptionDescription()).ToArray();
        }

        public void ParsePositionalArgument(string[] args)
        {
            if (args.Length < 1)
            {
                throw new InvalidOptionException("too few arguments for createfs subcommand.");
            }
            else if (args.Length > 1)
            {
                throw new InvalidOptionException("too many arguments for createfs subcommand.");
            }

            if (Format == ArchiveFormatType.Invalid)
            {
                InputAdfFile = args[0].Replace("\\", "/");
                IsSaveAdf = true;
            }
            else
            {
                InputDir = args[0].Replace("\\", "/").TrimEnd('/');
            }
        }
    }

    internal class CreateNcaOption : CreateCommandOptionBase, IOption
    {
        internal List<DirecotryConnector> InputDirs { get; set; }
        internal string MetaFilePath { get; set; }
        internal string MetaType { get; set; }
        internal string DescFilePath { get; set; }
        internal int KeyAreaEncryptionKeyIndex { get; set; }

        internal byte EncryptionType { get; set; }
        internal string ContentType { get; set; }
        internal byte KeyGeneration { get; set; }
        internal ulong ProgramId { get; set; }
        internal uint ContentIndex { get; set; }
        internal bool NoEncryption { get; set; }

        private List<string> TmpDirs;

        internal CreateNcaOption()
        {
            InputDirs = new List<DirecotryConnector>();
            ContentType = null;
            OutputFile = "./output.nca";
            MetaFilePath = null;
            MetaType = null;
            DescFilePath = null;
            KeyAreaEncryptionKeyIndex = 0;
            TmpDirs = new List<string>();
        }

        ~CreateNcaOption()
        {
            foreach(var dirName in TmpDirs)
            {
                Program.DeleteDirectoryIfExisted(dirName);
            }
        }

        public void ShowSubCommandUsage(string baseName)
        {
            Console.WriteLine("createnca: Create new nintendo content archive from .adf or specified directory and .nmeta and .desc file.");
            Console.WriteLine("Usage: {0} createnca [-o <outputPath>] <inputAdfPath>", baseName);
            Console.WriteLine("Usage: {0} createnca [-o <outputPath>] [--save-adf] [--only-adf] [--filter <fdfPath>] --meta <metaFilePath> --desc <descFilePath> --program <codePath> [<dataPath>] ", baseName);
            Console.WriteLine("Usage: {0} createnca [-o <outputPath>] [--save-adf] [--only-adf] [--filter <fdfPath>] --meta <metaFilePath> --control <controlPath>", baseName);
            Console.WriteLine(string.Empty);
            Console.WriteLine("Options:");
            Console.WriteLine("  -o <outputPath>                      File path to output. If directory path is specified, output to <outputPath>/output.nca.");
            Console.WriteLine("                                       If omitted, output to ./output.nca");
            Console.WriteLine("  --save-adf                           Leave .adf which is generated intermediately.");
            Console.WriteLine("  --only-adf                           Create .adf only.");
            Console.WriteLine("  --filter <fdfPath>                   Filter files to be included into .adf. <fdfPath> expects a path for a filter description file (.fdf).");
            Console.WriteLine("  --meta <metaFilePath>                Location of .nmeta file to input.");
            Console.WriteLine("  --desc <descFilePath>                Location of .desc file to input. This cannot be omitted when create program content.");
            Console.WriteLine("  --program <codePath> [<dataPath>]");
            Console.WriteLine("                                       Location of directory path which includes files to be contained");
            Console.WriteLine("                                       to in a program content as code and data region.");
            Console.WriteLine("                                       dataPath can be omitted.");
            Console.WriteLine("  --control <controlPath>              Location of directory path which includes files to be contained");
            Console.WriteLine("                                       to in a control data content.");
            Console.WriteLine("  --keyindex <index>                   Key index for encryption of content key.");
            Console.WriteLine("  --keygeneration <generation>         Key generation for encryption of content key.");
#if false
            Console.WriteLine("  --no-encryption                      Create .nca without encryption.");
#endif
        }

        internal string ExtractZipFile(string fileName)
        {
            var extension = Path.GetExtension(fileName);
            if (extension.ToLower() != ".zip")
            {
                throw new InvalidOptionException(".zip file should be specified for legal information.");
            }

            string tmpDirPath = Path.GetTempPath();
            string extractDirName = Path.Combine(tmpDirPath, Path.GetRandomFileName()).Replace("\\", "/").TrimEnd('/');

            ZipFile.ExtractToDirectory(fileName, extractDirName);
            TmpDirs.Add(extractDirName);

            return extractDirName;
        }

        internal OptionDescription CreateContentOptionDescription(string longName, string type, int countArgs = 1)
        {
            return new OptionDescription(longName, null, countArgs, (s) =>
                {
                    if (ContentType != null)
                    {
                        throw new InvalidOptionException("cannot set two content types to nca.");
                    }
                    ContentType = type;
                    for (int i = 0; i < s.Count; i++)
                    {
                        string inputPath = s[i].Replace("\\", "/").TrimEnd('/');
                        if (longName == "--legal-information")
                        {
                            inputPath = ExtractZipFile(inputPath);
                        }
                        InputDirs.Add(new DirecotryConnector(inputPath, null));
                    }
                });
        }

        internal OptionDescription CreateContentOptionDescriptionForHtmlDocument(string longName, string outputDir)
        {
            const string ContentTypeName = "HtmlDocument";
            const int countArgs = 1;

            return new OptionDescription(longName, null, countArgs, (s) =>
                {
                    if (ContentType == null)
                    {
                        ContentType = ContentTypeName;
                    }
                    for (int i = 0; i < s.Count; i++)
                    {
                        string inputPath = s[i].Replace("\\", "/").TrimEnd('/');
                        InputDirs.Add(new DirecotryConnector(inputPath, outputDir));
                    }
                });
        }

        public new OptionDescription[] GetOptionDescription()
        {
            OptionDescription[] options = new OptionDescription[]
                {
                    OptionUtil.CreateFilePathOptionDescription("--meta", (path) => { MetaFilePath = path; }),
                    new OptionDescription("--meta-type", null, 1, (type) => { MetaType = type.First(); }),
                    OptionUtil.CreateFilePathOptionDescription("--desc", (path) => { DescFilePath = path; }),
                    CreateContentOptionDescription("--program", "Program", 3),
                    CreateContentOptionDescription("--control", "Control"),
                    CreateContentOptionDescriptionForHtmlDocument("--html-document", "html-document"),
                    CreateContentOptionDescriptionForHtmlDocument("--accessible-urls", "accessible-urls"),
                    CreateContentOptionDescription("--legal-information", "LegalInformation"),
                    CreateContentOptionDescription("--legal-information-dir", "LegalInformation"),
                    CreateContentOptionDescription("--data", "Data"),
                    // VSIの都合で入力しているが、実際は無視する
                    new OptionDescription("--nro", null, 1, (s) => {}),
                    new OptionDescription("--keyindex", null, 1, (s) =>
                        {
                            KeyAreaEncryptionKeyIndex = int.Parse(s.First());
                        }),
                    new OptionDescription("--keygeneration", null, 1, (s) =>
                        {
                            KeyGeneration = (byte)int.Parse(s.First());
                        }),
                    new OptionDescription("--no-encryption", null, 0, (s) =>
                        {
                            NoEncryption = true;
                        }),
                };
            return options.Concat(base.GetOptionDescription()).ToArray();
        }

        public void ParsePositionalArgument(string[] args)
        {
            if (ContentType == null)
            {
                if (args.Length < 1)
                {
                    throw new InvalidOptionException("too few arguments for createnca subcommand.");
                }
                if (args.Length > 1)
                {
                    throw new InvalidOptionException("too many arguments for createnca subcommand.");
                }

                InputAdfFile = args[0].Replace("\\", "/");
                IsSaveAdf = true;

                if (string.IsNullOrEmpty(MetaType))
                {
                    if (File.Exists(InputAdfFile))
                    {
                        var adfReader = new NintendoContentAdfReader(InputAdfFile);
                        MetaType = ContentTypeToContentMetaType(adfReader.GetContentType());
                    }
                }
            }
            else if (string.IsNullOrEmpty(MetaType))
            {
                MetaType = ContentTypeToContentMetaType(ContentType);
            }
        }

        private static string ContentTypeToContentMetaType(string contentType)
        {
            if (contentType == "Program")
            {
                return "Application";
            }
            return contentType;
        }
    }

    internal enum ContentMetaType
    {
        Application,
        AddOnContent,
        Patch,
        SystemUpdate,
        SystemProgram,
        SystemData,
        BootImagePackage,
        BootImagePackageSafe,
        Delta,
    }

    internal class CreateNspOption : CreateCommandOptionBase, IOption
    {
        internal bool GeneratesApplicationControl { get; set; }
        internal bool ErrorUnpublishable { get; private set; }
        internal bool ShowUnpublishableErroFormatXml { get; private set; }
        internal string IgnoreErrorUnpublishable { get; private set; }
        internal string OriginalNspPath { get; private set; }
        internal bool NoCheckDirWarning { get; private set; }

        internal List<NintendoSubmissionPackageContentInfo> NspContentInfos { get; set; }

        private int contentInfoIndex;
        private bool isMetaSpecified;
        private bool isTypeSpecified;

        private bool isUseAdf;
        private bool createPublicSystemData;
        private bool hasTicket;
        private bool forceHasNoTicket;
        private byte? keyGeneration;
        private bool noEncryption;

        private List<string> TmpDirs;

        internal CreateNspOption()
        {
            OutputFile = "./output.nsp";
            GeneratesApplicationControl = false;
            ErrorUnpublishable = false;
            ShowUnpublishableErroFormatXml = false;
            IgnoreErrorUnpublishable = null;
            NoCheckDirWarning = false;

            NspContentInfos = new List<NintendoSubmissionPackageContentInfo>();
            NspContentInfos.Add(new NintendoSubmissionPackageContentInfo());
            contentInfoIndex = 0;
            isMetaSpecified = false;
            isTypeSpecified = false;
            isUseAdf = true;
            createPublicSystemData = false;

            TmpDirs = new List<string>();
        }

        ~CreateNspOption()
        {
            foreach(var dirName in TmpDirs)
            {
                Program.DeleteDirectoryIfExisted(dirName);
            }
        }

        internal string GetApplicationControlGeneratePath()
        {
            Debug.Assert(GeneratesApplicationControl);

            foreach(var resource in NspContentInfos[0].ResourceList)
            {
                if(resource.ContentType.Equals("Control"))
                {
                    return resource.PathList[0].first;
                }
            }

            throw new ArgumentException();
        }

        // TODO: 複数コンテンツセットを NSP に含めるオプション指定
        public void ShowSubCommandUsage(string baseName)
        {
            Console.WriteLine("creatensp: Create new Nintendo Submission Package from .adf or specified directory and .nmeta and .desc file.");
            Console.WriteLine("Usage: {0} creatensp [-o <outputPath>] <inputAdfPath>", baseName);
            Console.WriteLine("Usage: {0} creatensp [-o <outputPath>] [--save-adf] [--only-adf] [--filter <fdfPath>] --meta <metaFilePath> --type <contentMetaType> [--desc <descFilePath> --program <codePath> [<dataPath>]] [--control <controlPath>] [--nro <nroDirectoryPath>] [--error-unpublishable [--xml]] [--no-check-dir-warning]", baseName);
            Console.WriteLine(string.Empty);
            Console.WriteLine("Options:");
            Console.WriteLine("  -o <outputPath>                                File path to output. If directory path is specified, output to <outputPath>/output.nsp.");
            Console.WriteLine("                                                 If omitted, output to ./output.nsp");
            Console.WriteLine("  --save-adf                                     Leave .adf which is generated intermediately.");
            Console.WriteLine("  --only-adf                                     Create .adf only.");
            Console.WriteLine("  --filter <fdfPath>                             Filter files to be included into .adf. <fdfPath> expects a path for a filter description file (.fdf).");
            Console.WriteLine("  --meta <metaFilePath>                          Location of .nmeta file to input.");
            Console.WriteLine("  --desc <descFilePath>                          Location of .desc file to input. This cannot be omitted when containing program content.");
            Console.WriteLine("  --type <contentMetaType>                       Content Meta Type to create package.");
            Console.WriteLine("                                                 Available as <contentMetaType>: Application, SystemProgram");
            Console.WriteLine("  --program <codePath> [<dataPath>]");
            Console.WriteLine("                                                 Location of directory path which includes files to be contained");
            Console.WriteLine("                                                 to in a executable content as code and data region.");
            Console.WriteLine("                                                 dataPath can be omitted.");
            Console.WriteLine("                                                 If codePath indicates a file, it is used as a content just as it is.");
            Console.WriteLine("                                                 In this case, dataPath is ignored.");
            Console.WriteLine("  --control <controlPath>                        Location of directory path which includes files to be contained");
            Console.WriteLine("                                                 to in a control data content.");
            Console.WriteLine("                                                 If manualPath indicates a file, it is used as a content just as it is.");
            Console.WriteLine("  --nro <nroDirectoryPath>                       Location of directory which includes .nro files to be contained.");
            Console.WriteLine("  --error-unpublishable [--xml]                  Throw an exception if the Nintendo Submission Package is not satisfied with conditions to publish it.");
            Console.WriteLine("  --no-check-dir-warning                         Omit to check directory warning.");
#if false // 非公開
            Console.WriteLine("  --ignore-unpublishable-error [ignoreListPath]  Ignore unpublishable error.");
            Console.WriteLine("  --data <dataPath>                              Location of directory path which includes files to be contained");
            Console.WriteLine("                                                 to in a control data content.");
            Console.WriteLine("  --public-system-data                           Create system data content as public for application.");
            Console.WriteLine("                                                 This option is ignored if contentMetaType is specified but "SystemData".");
            Console.WriteLine("  --ticket                                       Create .nsp encrypted by external key.");
            Console.WriteLine("  --no-ticket                                    Create .nsp encrypted by internal key compulsorily. This option is prior to ticket option.");
            Console.WriteLine("  --keyindex <index>                             Key index for encryption of content key.");
            Console.WriteLine("  --keygeneration <generation>                   Key generation for encryption of content key.");
            Console.WriteLine("  --html-document <htmlDocumentPath>             Location of directory which includes gameplay instruction documents.");
            Console.WriteLine("  --accessible-urls <accessibleUrlsPath>         Location of directory which includes files to accept documents to access websites.");
            Console.WriteLine("  --legal-information <legalInformationPath>     Location of zip file which includes software legal information documents.");
            Console.WriteLine("  --no-encryption                                Create .nsp without encryption.");
//            Console.WriteLine("  --icon");
#endif
        }

        private void UpdateElement()
        {
            NspContentInfos.Add(new NintendoSubmissionPackageContentInfo());
            contentInfoIndex++;
            isMetaSpecified = false;
            isTypeSpecified = false;
        }

        private void UpdateContentInfosForAddOnContent(bool aocHasTicket)
        {
            var topContentInfo = NspContentInfos[0];
            var reader = new MetaFileReader(topContentInfo.MetaFilePath, "AddOnContent");

            {
                bool hasDuplicate = reader.GetContentMetaList().GroupBy((meta) => meta.Id).Count() != reader.GetContentMetaList().Count();
                if (hasDuplicate)
                {
                    throw new ArgumentException("Duplicate add-on-contents.");
                }
            }

            var infos = new List<NintendoSubmissionPackageContentInfo>();
            var aocCount = reader.GetContentMetaList().Count();
            for (int i = 0; i < aocCount; ++i)
            {
                var info = new NintendoSubmissionPackageContentInfo();
                infos.Add(info);

                info.MetaFilePath = topContentInfo.MetaFilePath;
                info.MetaType = topContentInfo.MetaType;
                info.HasTicket = true && aocHasTicket;
                var metaReader = new MetaFileReader(info.MetaFilePath, info.MetaType);
                var aocMeta = (AddOnContentMeta)metaReader.GetContentMetaList().ElementAt(i);

                if (aocMeta.FilterDescriptionFilePath != null)
                {
                    var fdfPath = Path.GetFullPath(Path.IsPathRooted(aocMeta.FilterDescriptionFilePath) ? aocMeta.FilterDescriptionFilePath : Path.Combine(Path.GetDirectoryName(info.MetaFilePath), aocMeta.FilterDescriptionFilePath)).Replace("\\", "/");
                    info.FilterDescriptionFilePath = fdfPath;
                }

                var resource = new NintendoSubmissionPackageContentResource();
                resource.ContentType = "PublicData";
                string dataPath = Path.GetFullPath(Path.IsPathRooted(aocMeta.DataPath) ? aocMeta.DataPath : Path.Combine(Path.GetDirectoryName(info.MetaFilePath), aocMeta.DataPath)).Replace("\\", "/");
                resource.PathList.Add(new DirecotryConnector(dataPath, null));

                // ResourceList 内の resource の Index が、nmeta の中の ContentMetaList の Index と一致することを前提とした処理がライブラリ内にあるので注意
                // TODO: リファクタリング
                info.ResourceList.Add(resource);
                SetPublicDataContentType(info.ResourceList);
            }
            NspContentInfos = infos;
        }

        private void UpdateContentInfos(bool aocHasTicket)
        {
            var metaType = NspContentInfos[contentInfoIndex].MetaType;
            if (metaType == "AddOnContent")
            {
                UpdateContentInfosForAddOnContent(aocHasTicket);
            }
            else if (metaType == "Application")
            {
                foreach(var contentInfo in NspContentInfos)
                {
                    if (!string.IsNullOrEmpty(contentInfo.MetaFilePath))
                    {
                        var reader = new MetaFileReader(contentInfo.MetaFilePath, contentInfo.MetaType);
                        var meta = (ApplicationMeta)reader.GetContentMetaList().FirstOrDefault(m => m.ContentType == "Application");
                        var fdfInNmeta = meta?.FilterDescriptionFilePath;
                        if (fdfInNmeta != null)
                        {
                            if(this.InputFdfPath != null)
                            {
                                throw new ArgumentException("It is impossible to specify fdf file from both nmeta and command line option.");
                            }
                            contentInfo.FilterDescriptionFilePath = fdfInNmeta;
                        }
                    }
                }
            }
        }

        internal string ExtractZipFile(string fileName)
        {
            var extension = Path.GetExtension(fileName);
            if (extension.ToLower() != ".zip")
            {
                throw new InvalidOptionException(".zip file should be specified for legal information.");
            }

            string tmpDirPath = Path.GetTempPath();
            string extractDirName = Path.Combine(tmpDirPath, Path.GetRandomFileName()).Replace("\\", "/").TrimEnd('/');

            ZipFile.ExtractToDirectory(fileName, extractDirName);
            TmpDirs.Add(extractDirName);

            return extractDirName;
        }

        internal OptionDescription CreateContentOptionDescription(string longName, string type, int countArgs = 1)
        {
            return new OptionDescription(longName, null, countArgs, (s) =>
                {
                    isUseAdf = false;

                    var resource = new NintendoSubmissionPackageContentResource();
                    resource.ContentType = type;
                    for (int i = 0; i < s.Count; i++)
                    {
                        string inputPath = s[i].Replace("\\", "/").TrimEnd('/');
                        if (longName == "--legal-information")
                        {
                            inputPath = ExtractZipFile(inputPath);
                        }

                        resource.PathList.Add(new DirecotryConnector(inputPath, null));
                    }
                    NspContentInfos[contentInfoIndex].ResourceList.Add(resource);
                });
        }

        internal OptionDescription CreateContentOptionDescriptionForHtmlDocument(string longName, string outputDir)
        {
            const string ContentTypeName = "HtmlDocument";
            const int countArgs = 1;

            return new OptionDescription(longName, null, countArgs, (s) =>
                {
                    isUseAdf = false;

                    var info = NspContentInfos[contentInfoIndex];
                    var resourceList = info.ResourceList.Where(x => x.ContentType.Equals(ContentTypeName));
                    if (resourceList.Count() > 1)
                    {
                        throw new ArgumentException();
                    }

                    NintendoSubmissionPackageContentResource resource;
                    if (resourceList.Count() == 0)
                    {
                        resource = new NintendoSubmissionPackageContentResource();
                        resource.ContentType = ContentTypeName;
                        NspContentInfos[contentInfoIndex].ResourceList.Add(resource);
                    }
                    else
                    {
                        resource = resourceList.ElementAt(0);
                    }

                    for (int i = 0; i < s.Count; i++)
                    {
                        string inputPath = s[i].Replace("\\", "/").TrimEnd('/');

                        resource.PathList.Add(new DirecotryConnector(inputPath, outputDir));
                    }
                });
        }

        public new OptionDescription[] GetOptionDescription()
        {
            OptionDescription[] options = new OptionDescription[]
                {
                    new OptionDescription("--type", null, 1, (s) =>
                        {
                            isUseAdf = false;
                            isTypeSpecified = true;
                            try
                            {
                                Enum.Parse(typeof(ContentMetaType), s.First());
                            }
                            catch (Exception)
                            {
                                throw new InvalidOptionException(string.Format("invalid option --type {0}.", s.First()));
                            }
                            NspContentInfos[contentInfoIndex].MetaType = s.First();

                            // AddOnContent はデフォルトでチケットを持つ
                            if(s.First() == "AddOnContent")
                            {
                                hasTicket = true;
                            }
                        }),
                    OptionUtil.CreateFilePathOptionDescription("--meta", (path) =>
                        {
                            isUseAdf = false;
                            isMetaSpecified = true;
                            NspContentInfos[contentInfoIndex].MetaFilePath = path;
                            {
                                var xml = XDocument.Load(path);
                                var keyGen = xml.Descendants("Meta").Descendants("KeyGeneration");
                                if (!keyGen.Any())
                                {
                                    keyGen = xml.Descendants("NintendoSdkMeta").Descendants("KeyGeneration");
                                }
                                if (keyGen.Any())
                                {
                                    NspContentInfos[contentInfoIndex].KeyGeneration = keyGen.Select((value) => Convert.ToByte(value.Value, 10)).Single();
                                }
                            }
                        }),
                    OptionUtil.CreateFilePathOptionDescription("--desc", (path) =>
                        {
                            isUseAdf = false;
                            NspContentInfos[contentInfoIndex].DescFilePath = path;
                        }),
                    new OptionDescription("--keyindex", null, 1, (s) =>
                        {
                            NspContentInfos[contentInfoIndex].KeyAreaEncryptionKeyIndex = int.Parse(s.First());
                        }),
                    CreateContentOptionDescription("--program", "Program", 3),
                    CreateContentOptionDescription("--control", "Control"),
                    CreateContentOptionDescriptionForHtmlDocument("--html-document", "html-document"),
                    CreateContentOptionDescriptionForHtmlDocument("--accessible-urls", "accessible-urls"),
                    CreateContentOptionDescription("--legal-information", "LegalInformation"),
                    CreateContentOptionDescription("--legal-information-dir", "LegalInformation"),
                    CreateContentOptionDescription("--data", "Data"),
                    new OptionDescription("--nro", null, 1, (s) =>
                        {
                            NspContentInfos[contentInfoIndex].NroDirectoryPath = OptionUtil.CheckAndNormalizeDirectory(s.First(),"--nro");
                        }),
                    new OptionDescription("--icon", null, 32, (s) =>
                        {
                            if(s.Count % 2 != 0)
                            {
                                throw new InvalidOptionException("--icon <language> <iconPath>...");
                            }
                            NspContentInfos[contentInfoIndex].IconList = OptionUtil.CreateIconFileList(s);
                        }),
                    new OptionDescription("--nx-icon", null, 32, (s) =>
                        {
                            if(s.Count % 2 != 0)
                            {
                                throw new InvalidOptionException("--nx-icon <language> <iconPath>...");
                            }
                            NspContentInfos[contentInfoIndex].NxIconList = OptionUtil.CreateIconFileList(s);
                        }),
                    new OptionDescription("--nx-icon-max-size", null, 32, (s) =>
                        {
                            NspContentInfos[contentInfoIndex].NxIconMaxSize = Convert.ToUInt32(s.First());
                        }),
                    new OptionDescription("--public-system-data", null, 0, (s) =>
                        {
                            createPublicSystemData = true;
                        }),
                    new OptionDescription("--error-unpublishable", null, 0, (s) =>
                        {
                            ErrorUnpublishable = true;
                        }),
                    new OptionDescription("--no-check-dir-warning", null, 0, (s) =>
                        {
                            NoCheckDirWarning = true;
                        }),
                    new OptionDescription("--xml", null, 0, (s) =>
                        {
                            ShowUnpublishableErroFormatXml = true;
                        }),
                    new OptionDescription("--ignore-unpublishable-error", null, -1, (s) =>
                        {
                            if (s.Count > 1)
                            {
                                throw new InvalidOptionException("too many arguments for --ignore-unpublishable-error option.");
                            }
                            IgnoreErrorUnpublishable = (s.Count() > 0) ? OptionUtil.CheckAndNormalizeFilePath(s.First(), "--ignore-unpublishable-error") : string.Empty;
                        }),
                    new OptionDescription("--ticket", null, 0, (s) =>
                        {
                            hasTicket = true;
                        }),
                    new OptionDescription("--no-ticket", null, 0, (s) =>
                        {
                            forceHasNoTicket = true;
                        }),
                    new OptionDescription("--keygeneration", null, 1, (s) =>
                        {
                            keyGeneration = Convert.ToByte(s.First());
                        }),
                    new OptionDescription("--no-encryption", null, 0, (s) =>
                        {
                            noEncryption = true;
                        }),
                    OptionUtil.CreateFilePathOptionDescription("--original", (path) =>
                        {
                            OriginalNspPath = path;
                        }),
                };
            return options.Concat(base.GetOptionDescription()).ToArray();
        }

        private void SetPublicDataContentType(List<NintendoSubmissionPackageContentResource> resources)
        {
            for (int i = 0; i < resources.Count; i++)
            {
                var resource = resources[i];
                if (resource.ContentType == "Data")
                {
                    resource.ContentType = "PublicData";
                    resources[i] = resource;
                }
            }
        }

        private void SetHasTicket(List<NintendoSubmissionPackageContentInfo> infos)
        {
            for (int i = 0; i < infos.Count; i++)
            {
                var info = infos[i];
                if (info.MetaType == "Application")
                {
                    info.HasTicket = true;
                    infos[i] = info;
                }
            }
        }

        private void SetKeyGeneration(List<NintendoSubmissionPackageContentInfo> infos)
        {
            for (int i = 0; i < infos.Count; i++)
            {
                var info = infos[i];
                info.KeyGeneration = keyGeneration;
                infos[i] = info;
            }
        }

        private void SetNoEncryption(List<NintendoSubmissionPackageContentInfo> infos)
        {
            for (int i = 0; i < infos.Count; i++)
            {
                var info = infos[i];
                info.NoEncryption = true;
                infos[i] = info;
            }
        }

        public void ParsePositionalArgument(string[] args)
        {
            // AddOnContent の後処理
            UpdateContentInfos(hasTicket && !forceHasNoTicket);
            foreach (var info in NspContentInfos.Where(x => x.MetaType == "AddOnContent"))
            {
                // AddOnContent は常に PublicData
                SetPublicDataContentType(info.ResourceList);
            }

            // --public-system-data の後処理
            if (createPublicSystemData)
            {
                foreach (var info in NspContentInfos.Where(x => x.MetaType == "SystemData"))
                {
                    SetPublicDataContentType(info.ResourceList);
                }
            }

            // NspContentInfos の MetaType が Application であるエントリにチケット有フラグを設定
            if (hasTicket && !forceHasNoTicket)
            {
                SetHasTicket(NspContentInfos);
            }

            if (keyGeneration.HasValue)
            {
                SetKeyGeneration(NspContentInfos);
            }

            if (noEncryption)
            {
                SetNoEncryption(NspContentInfos);
            }

            if (NspContentInfos[0].IconList.Count != 0)
            {
                if (NspContentInfos[0].MetaType != "Application" && NspContentInfos[0].MetaType != "Patch")
                {
                    throw new InvalidOptionException("--icon option should be used with --type Application.");
                }

                if (NspContentInfos[0].HasResource("Control"))
                {
                    throw new InvalidOptionException("--icon option cannot be used with --control option.");
                }
            }

            // Control のデータ自動生成の設定
            // TODO: とりあえず最初の NspContentInfo しか扱わない
            if (isTypeSpecified && (NspContentInfos[0].MetaType == "Application" || NspContentInfos[0].MetaType == "Patch") && ! NspContentInfos[0].HasResource("Control"))
            {
                GeneratesApplicationControl = true;
                var resource = new NintendoSubmissionPackageContentResource();
                resource.ContentType = "Control";
                resource.PathList.Add(new DirecotryConnector(Path.GetDirectoryName(OutputFile).Replace("\\", "/") + "/" + Path.GetFileName(OutputFile) + ".control", null));
                NspContentInfos[0].ResourceList.Add(resource);
            }

            if (!isUseAdf && (!isMetaSpecified || !isTypeSpecified))
            {
                throw new InvalidOptionException("too few options for creatensp subcommand.");
            }
            if (args.Length < 1 && isUseAdf)
            {
                throw new InvalidOptionException("too few arguments for creatensp subcommand.");
            }
            if (args.Length > 1 && isUseAdf)
            {
                throw new InvalidOptionException("too many arguments for creatensp subcommand.");
            }
            if (args.Length > 0 && !isUseAdf)
            {
                throw new InvalidOptionException("too many arguments for creatensp subcommand.");
            }

            if (isUseAdf)
            {
                InputAdfFile = args[0].Replace("\\", "/");
                IsSaveAdf = true;
            }
        }
    }

    internal class CreateNspMetaOption : IOption
    {
        internal string MetaFilePath { get; set; }
        internal string MetaType { get; set; }
        internal string OutputDirectory { get; set; }
        internal string ContentMetaType { get; set; }

        internal List<Tuple<string, string>> IconFileList = new List<Tuple<string, string>>();
        internal List<Tuple<string, string>> NxIconFileList = new List<Tuple<string, string>>();

        const UInt32 DefaultNxIconMaxSize = NintendoSubmissionPackageContentInfo.DefaultNxIconMaxSize;
        internal UInt32 NxIconMaxSize { get; set; }

        internal CreateNspMetaOption()
        {
            OutputDirectory = "./output.nspd";
            NxIconMaxSize = DefaultNxIconMaxSize;
        }

        public void ShowSubCommandUsage(string baseName)
        {
            Console.WriteLine("createnspmeta: Create only meta files for Nintendo Submission Package.");
            Console.WriteLine("Usage: {0} createnspmeta [-o <outputDirectory>] --meta <metaFilePath> --type <contentMetaType>", baseName);
            Console.WriteLine(string.Empty);
            Console.WriteLine("Options:");
            Console.WriteLine("  -o <outputDirectory>                 Directory path to output. If omitted, output to ./output.nspd");
            Console.WriteLine("  --meta <metaFilePath>                Location of .nmeta file to input.");
            Console.WriteLine("  --type <contentMetaType>             Content Meta Type to create package.");
//            Console.WriteLine("  --icon");
//            Console.WriteLine("  --control");
//            Console.WriteLine("  --html-document");
//            Console.WriteLine("  --accessible-urls");
//            Console.WriteLine("  --legal-information");
//            Console.WriteLine("  --data");
        }

        public OptionDescription[] GetOptionDescription()
        {
            OptionDescription[] options = new OptionDescription[]
                {
                    new OptionDescription(null, "-o", 1, (s) => { OutputDirectory = OptionUtil.GetOutputFilePath(".", s.First()); }),
                    new OptionDescription("--type", null, 1, (s) =>
                        {
                            try
                            {
                                Enum.Parse(typeof(ContentMetaType), s.First());
                            }
                            catch (Exception)
                            {
                                throw new InvalidOptionException(string.Format("invalid option --type {0}.", s.First()));
                            }
                            MetaType = s.First();
                        }),
                    OptionUtil.CreateFilePathOptionDescription("--meta", (path) => { MetaFilePath = path; }),
                    new OptionDescription("--icon", null, 32, (s) =>
                        {
                            if(s.Count % 2 != 0)
                            {
                                throw new InvalidOptionException("--icon <language> <iconPath>...");
                            }

                            IconFileList = OptionUtil.CreateIconFileList(s);
                        }),
                    new OptionDescription("--nx-icon", null, 32, (s) =>
                        {
                            if(s.Count % 2 != 0)
                            {
                                throw new InvalidOptionException("--nx-icon <language> <iconPath>...");
                            }

                            NxIconFileList = OptionUtil.CreateIconFileList(s);
                        }),
                    new OptionDescription("--nx-icon-max-size", null, 32, (s) =>
                        {
                            NxIconMaxSize = Convert.ToUInt32(s.First());
                        }),
                };
            return options;
        }

        public void ParsePositionalArgument(string[] args)
        {
            if (IconFileList.Count > 0)
            {
                if (MetaType != "Application" && MetaType != "Patch")
                {
                    throw new InvalidOptionException("--icon option should be used with --type Application.");
                }
            }
        }
    }

    internal class CreateNspdOption : IOption
    {
        internal const string DefaultCodeDirectory = "program0.ncd/code/";
        internal const string DefaultDataDirectory = "program0.ncd/data/";
        internal const string DefaultLogoDirectory = "program0.ncd/logo/";
        internal const string DefaultControlDirectory = "control0.ncd/";
        internal const string DefaultHtmlDocumentDirectory = "htmlDocument0.ncd/htmlDocument";
        internal const string DefaultAccessibleUrlsDirectory = "htmlDocument0.ncd/accessibleUrls";
        internal const string DefaultLegalInformationDirectory = "legalInformation0.ncd/";

        internal string MetaFilePath { get; set; }
        internal string MetaType { get; set; }
        internal string ContentMetaType { get; set; }
        internal string OutputDirectory { get; set; }
        internal string CodeDirectory { get; set; }
        internal string DataDirectory { get; set; }
        internal string LogoDirectory { get; set; }
        internal string ControlDirectory { get; set; }
        internal string HtmlDocumentDirectory { get; set; }
        internal string AccessibleUrlsDirectory { get; set; }
        internal string LegalInformationDirectory { get; set; }
        internal string LegalInformationZipPath { get; set; }
        internal bool DoClean { get; set; }
        internal bool ErrorUnpublishable { get; private set; }
        internal bool ShowUnpublishableErroFormatXml { get; private set; }
        internal string IgnoreErrorUnpublishable { get; set; }

        internal List<Tuple<string, string>> IconFileList = new List<Tuple<string, string>>();
        internal List<Tuple<string, string>> NxIconFileList = new List<Tuple<string, string>>();

        const UInt32 DefaultNxIconMaxSize = NintendoSubmissionPackageContentInfo.DefaultNxIconMaxSize;
        internal UInt32 NxIconMaxSize { get; set; }
        internal string InputFdfPath { get; set; }
        internal bool ForciblyCopyWithFiltering { get; set; }
        internal bool ForciblyCopyCode { get; set; }
        internal bool ForciblyCopyData { get; set; }
        internal bool ForciblyCopyControl { get; set; }
        internal bool ForciblyCopyHtmlDocument { get; set; }
        internal bool ForciblyCopyAccessibleUrls { get; set; }
        internal bool ForciblyCopyLegalInformation { get; set; }

        internal CreateNspdOption()
        {
            NxIconMaxSize = DefaultNxIconMaxSize;
            OutputDirectory = "./output.nspd";
            DoClean = true;
            ErrorUnpublishable = false;
            ShowUnpublishableErroFormatXml = false;
            IgnoreErrorUnpublishable = null;
            InputFdfPath = null;
            ForciblyCopyWithFiltering = false;
            ForciblyCopyCode = false;
            ForciblyCopyData = false;
            ForciblyCopyControl = false;
            ForciblyCopyHtmlDocument = false;
            ForciblyCopyAccessibleUrls = false;
            ForciblyCopyLegalInformation = false;
        }

        public void ShowSubCommandUsage(string baseName)
        {
            Console.WriteLine("createnspd: Create new Nintendo Submission Package Directory.");
            Console.WriteLine("Usage: {0} createnspd [-o <outputDirectory>] [--filter <fdfPath>] [--forcibly-copy-with-filtering] [--forcibly-copy-code] [--forcibly-copy-data] [--forcibly-copy-control] [--forcibly-copy-html-document] [--forcibly-copy-accessible-urls] [--forcibly-copy-legal-information] --meta <metaFilePath> --type <contentMetaType> [--error-unpublishable [--xml]]", baseName);
            Console.WriteLine(string.Empty);
            Console.WriteLine("Options:");
            Console.WriteLine("  -o <outputDirectory>                               Directory path to output. The end of directory name should be \".nspd\"");
            Console.WriteLine("                                                     If omitted, output to ./output.nspd");
            Console.WriteLine("  --filter <fdfPath>                                 Filter files to be included into .nspd. <fdfPath> expects a path for a filter description file (.fdf).");
            Console.WriteLine("  --forcibly-copy-with-filtering                     Copy file instead of creating hard link. This option is validated, only when specified .fdf");
            Console.WriteLine("  --forcibly-copy-code                               Copy the code files instead of creating junction.");
            Console.WriteLine("  --forcibly-copy-data                               Copy the data files instead of creating junction.");
            Console.WriteLine("  --forcibly-copy-control                            Copy the control files instead of creating junction.");
            Console.WriteLine("  --forcibly-copy-html-document                      Copy the html document files instead of creating junction.");
            Console.WriteLine("  --forcibly-copy-accessible-urls                    Copy the accessible website document files instead of creating junction.");
            Console.WriteLine("  --forcibly-copy-legal-information                  Copy the legal information document files instead of creating junction.");
            Console.WriteLine("  --meta <metaFilePath>                              Location of .nmeta file to input.");
            Console.WriteLine("  --type <contentMetaType>                           Content Meta Type to create package.");
            Console.WriteLine("  --program <codePath> [<dataPath>]");
            Console.WriteLine("                                                     Location of directory path which includes files to be contained");
            Console.WriteLine("                                                     to in a executable content as code, and data region.");
            Console.WriteLine("  --code <codePath>                                  Location of directory path which includes files to be contained to in a executable content as code.");
            Console.WriteLine("  --data <dataPath>                                  Location of directory path which includes files to be contained to in a data content.");
            Console.WriteLine("  --control <controlPath>                            Location of directory path which includes files to be contained");
            Console.WriteLine("                                                     to in a control data content.");
            Console.WriteLine("  --error-unpublishable [--xml]                      Throw an exception if the Nintendo Submission Package Directory is not satisfied with conditions to publish it.");
#if false // 非公開
            Console.WriteLine("  --html-document <htmlDocumentPath>                 Location of directory which includes gameplay instruction documents.");
            Console.WriteLine("  --accessible-urls <accessibleUrlsPath>             Location of directory which includes files to accept documents to access websites.");
            Console.WriteLine("  --legal-information <legalInformationPath>         Location of zip file which includes software legal information documents.");
            Console.WriteLine("  --ignore-unpublishable-error [ignoreListPath]      Ignore unpublishable error.");
#endif
        }

        public OptionDescription[] GetOptionDescription()
        {
            OptionDescription[] options = new OptionDescription[]
                {
                    new OptionDescription(null, "-o", 1, (s) => { OutputDirectory = OptionUtil.GetOutputFilePath(".", s.First()).TrimEnd('/'); }),
                    new OptionDescription("--type", null, 1, (s) =>
                        {
                            try
                            {
                                Enum.Parse(typeof(ContentMetaType), s.First());
                            }
                            catch (Exception)
                            {
                                throw new InvalidOptionException(string.Format("invalid option --type {0}.", s.First()));
                            }
                            MetaType = s.First();
                        }),
                    new OptionDescription("--filter", null, 1, (s) =>
                        {
                            InputFdfPath = s.First();
                        }),
                    new OptionDescription("--forcibly-copy-with-filtering", null, 0, (s) =>
                        {
                            ForciblyCopyWithFiltering = true;
                        }),
                    new OptionDescription("--forcibly-copy-code", null, 0, (s) =>
                        {
                            ForciblyCopyCode = true;
                        }),
                    new OptionDescription("--forcibly-copy-data", null, 0, (s) =>
                        {
                            ForciblyCopyData = true;
                        }),
                    new OptionDescription("--forcibly-copy-control", null, 0, (s) =>
                        {
                            ForciblyCopyControl = true;
                        }),
                    new OptionDescription("--forcibly-copy-html-document", null, 0, (s) =>
                        {
                            ForciblyCopyHtmlDocument = true;
                        }),
                    new OptionDescription("--forcibly-copy-accessible-urls", null, 0, (s) =>
                        {
                            ForciblyCopyAccessibleUrls = true;
                        }),
                    new OptionDescription("--forcibly-copy-legal-information", null, 0, (s) =>
                        {
                            ForciblyCopyLegalInformation = true;
                        }),
                    OptionUtil.CreateFilePathOptionDescription("--meta", (path) => { MetaFilePath = path; }),
                    new OptionDescription("--program", null, 3, (s) =>
                    {
                        if (CodeDirectory != null)
                        {
                            throw new InvalidOptionException("--program option cannot be used with --code and --data options.");
                        }
                        CodeDirectory = OptionUtil.GetOutputFilePath(".", s.First());

                        if (s.Count > 1)
                        {
                            if (DataDirectory != null)
                            {
                                throw new InvalidOptionException("--program option cannot be used with --code and --data options.");
                            }
                            DataDirectory = OptionUtil.GetOutputFilePath(".", s[1]);
                        }

                        if (s.Count > 2)
                        {
                            if (LogoDirectory != null)
                            {
                                throw new InvalidOptionException("--program option cannot be used with --code, --data and --logo options.");
                            }
                            LogoDirectory = OptionUtil.GetOutputFilePath(".", s[2]);
                        }
                    }),
                    new OptionDescription("--code", null, 1, (s) =>
                    {
                        if (CodeDirectory != null)
                        {
                            throw new InvalidOptionException("--program option cannot be used with --code and --data options.");
                        }
                        CodeDirectory = OptionUtil.GetOutputFilePath(".", s.First());
                    }),
                    new OptionDescription("--data", null, 1, (s) =>
                    {
                        if (DataDirectory != null)
                        {
                            throw new InvalidOptionException("--program option cannot be used with --code and --data options.");
                        }
                        DataDirectory = OptionUtil.GetOutputFilePath(".", s.First());
                    }),
                    new OptionDescription("--logo", null, 1, (s) =>
                    {
                        if (LogoDirectory != null)
                        {
                            throw new InvalidOptionException("--program option cannot be used with --code, --data and --logo options.");
                        }
                        LogoDirectory = OptionUtil.GetOutputFilePath(".", s.First());
                    }),
                    new OptionDescription("--control", null, 1, (s) => { ControlDirectory = OptionUtil.GetOutputFilePath(".", s.First()); }),
                    new OptionDescription("--html-document", null, 1, (s) => { HtmlDocumentDirectory = OptionUtil.GetOutputFilePath(".", s.First()); }),
                    new OptionDescription("--accessible-urls", null, 1, (s) => { AccessibleUrlsDirectory = OptionUtil.GetOutputFilePath(".", s.First()); }),
                    new OptionDescription("--legal-information", null, 1, (s) => { LegalInformationZipPath = OptionUtil.GetOutputFilePath(".", s.First()); }),
                    new OptionDescription("--legal-information-dir", null, 1, (s) => { LegalInformationDirectory = OptionUtil.GetOutputFilePath(".", s.First()); }),
                    new OptionDescription("--ignore-unpublishable-error", null, -1, (s) =>
                        {
                            if (s.Count > 1)
                            {
                                throw new InvalidOptionException("too many arguments for --ignore-unpublishable-error option.");
                            }
                            IgnoreErrorUnpublishable = (s.Count() > 0) ? OptionUtil.CheckAndNormalizeFilePath(s.First(), "--ignore-unpublishable-error") : string.Empty;
                        }),
                    new OptionDescription("--error-unpublishable", null, 0, (s) =>
                        {
                            ErrorUnpublishable = true;
                        }),
                    new OptionDescription("--xml", null, 0, (s) =>
                        {
                            ShowUnpublishableErroFormatXml = true;
                        }),
                    // VSIの都合で入力しているが、実際は無視する
                    new OptionDescription("--nro", null, 1, (s) => {}),

                    new OptionDescription("--icon", null, 32, (s) =>
                        {
                            if(s.Count % 2 != 0)
                            {
                                throw new InvalidOptionException("--icon <language> <iconPath>...");
                            }

                            IconFileList = OptionUtil.CreateIconFileList(s);
                        }),
                    new OptionDescription("--nx-icon", null, 32, (s) =>
                        {
                            if(s.Count % 2 != 0)
                            {
                                throw new InvalidOptionException("--nx-icon <language> <iconPath>...");
                            }

                            NxIconFileList = OptionUtil.CreateIconFileList(s);
                        }),
                    new OptionDescription("--nx-icon-max-size", null, 32, (s) =>
                        {
                            NxIconMaxSize = Convert.ToUInt32(s.First());
                        }),
                };
            return options;
        }

        public void ParsePositionalArgument(string[] args)
        {
            if (OutputDirectory.Substring(OutputDirectory.Length - 5).ToLower() != ".nspd")
            {
                throw new InvalidOptionException("-o option should be used with file name which ends with \".nspd\"");
            }

            if (MetaType == null || MetaType.Length == 0 ||
                MetaFilePath == null || MetaFilePath.Length == 0)
            {
                throw new InvalidOptionException("createnspd command needs --meta and --type options.");
            }

            if (CodeDirectory == null || CodeDirectory.Length == 0)
            {
                throw new InvalidOptionException("createnspd command needs code directory. Please specify --program or --code option.");
            }

            if (MetaType != "Application" && MetaType != "SystemProgram")
            {
                throw new InvalidOptionException("createnspd command should be used with --type Application or --type SystemProgram");
            }

            if (IconFileList.Count > 0)
            {
                if (MetaType != "Application")
                {
                    throw new InvalidOptionException("--icon option should be used with --type Application.");
                }
            }

            if (NxIconFileList.Count > 0)
            {
                if (MetaType != "Application")
                {
                    throw new InvalidOptionException("--nx-icon option should be used with --type Application.");
                }
            }

            if (LegalInformationDirectory != null && LegalInformationZipPath != null)
            {
                throw new InvalidOperationException("--legal-information should not be used with --legal-information-dir");
            }

            // Metaファイル内の FilterDescriptionFilePath を取得
            var reader = new MetaFileReader(MetaFilePath, MetaType);
            var meta = (ApplicationMeta)reader.GetContentMetaList().FirstOrDefault(m => m.ContentType == "Application");
            var fdfInNmeta = meta?.FilterDescriptionFilePath;
            if (fdfInNmeta != null)
            {
                if (InputFdfPath != null)
                {
                    throw new ArgumentException("It is impossible to specify fdf file from both nmeta and command line option.");
                }
                InputFdfPath = fdfInNmeta;
            }

            string fullPath = Path.GetFullPath(OutputDirectory);
            if ((CodeDirectory != null && Path.GetFullPath(CodeDirectory).IndexOf(fullPath) >= 0) ||
                (DataDirectory != null && Path.GetFullPath(DataDirectory).IndexOf(fullPath) >= 0) ||
                (LogoDirectory != null && Path.GetFullPath(LogoDirectory).IndexOf(fullPath) >= 0) ||
                (ControlDirectory != null && Path.GetFullPath(ControlDirectory).IndexOf(fullPath) >= 0) ||
                (HtmlDocumentDirectory != null && Path.GetFullPath(HtmlDocumentDirectory).IndexOf(fullPath) >= 0) ||
                (AccessibleUrlsDirectory != null && Path.GetFullPath(AccessibleUrlsDirectory).IndexOf(fullPath) >= 0) ||
                (LegalInformationDirectory != null && Path.GetFullPath(LegalInformationDirectory).IndexOf(fullPath) >= 0)
                )
            {
                DoClean = false;
            }
        }
    }

    internal class CreateNacpOption : IOption
    {
        internal string MetaFilePath { get; set; }
        internal string OutputFilePath { get; set; }

        internal CreateNacpOption()
        {
            MetaFilePath = null;
            OutputFilePath = null;
        }

        public void ShowSubCommandUsage(string baseName)
        {
            Console.WriteLine("createnacp: Create nacp from meta file.");
            Console.WriteLine("Usage: {0} createnacp [-o <outputFile>] --meta <metaFilePath>", baseName);
            Console.WriteLine(string.Empty);
            Console.WriteLine("Options:");
            Console.WriteLine("  -o <outputFile>                      File path to output.");
            Console.WriteLine("  --meta <metaFilePath>                Location of .nmeta file to input.");
        }

        public OptionDescription[] GetOptionDescription()
        {
            OptionDescription[] options = new OptionDescription[]
                {
                    new OptionDescription(null, "-o", 1, (s) => { OutputFilePath = OptionUtil.GetOutputFilePath(".", s.First()); }),
                    OptionUtil.CreateFilePathOptionDescription("--meta", (path) => { MetaFilePath = path; }),
                };
            return options;
        }

        public void ParsePositionalArgument(string[] args)
        {
            if (args.Length > 0)
            {
                throw new InvalidOptionException("too many arguments for createnacp subcommand.");
            }
        }
    }

    internal class ExtractNspOption : IOption
    {
        internal string InputNspFile { get; set; }
        internal string OutputDirectory { get; set; }
        internal bool RenameBasedOnContentType { get; set; }

        internal ExtractNspOption()
        {
            InputNspFile = null;
            OutputDirectory = ".";
            RenameBasedOnContentType = false;
        }

        public void ShowSubCommandUsage(string baseName)
        {
            Console.WriteLine("extractnsp: Extract files from a Nintendo Submission Package file.");
            Console.WriteLine("Usage: {0} extractnsp [-o <outputDirectory>] <inputNspPath>", baseName);
            Console.WriteLine(string.Empty);
            Console.WriteLine("Options:");
            Console.WriteLine("  -o <outputDirectory>                 Directory path to output. If omitted, output to current directory");
        }

        public OptionDescription[] GetOptionDescription()
        {
            OptionDescription[] options = new OptionDescription[]
                {
                    new OptionDescription(null, "-o", 1, (s) => { OutputDirectory = OptionUtil.GetOutputFilePath(OutputDirectory, s.First()); }),
                    new OptionDescription(null, "--rename-based-on-contenttype", 0, (s) => { RenameBasedOnContentType = true; })
                };
            return options;
        }

        public void ParsePositionalArgument(string[] args)
        {
            if (args.Length == 0)
            {
                throw new InvalidOptionException("input nsp file must be specified for extractnsp subcommand.");
            }

            InputNspFile = args[0];
        }
    }

    internal class ExtractNsaveOption : IOption
    {
        internal string InputNsaveFile { get; set; }
        internal string OutputDirectory { get; set; }

        internal ExtractNsaveOption()
        {
            InputNsaveFile = null;
            OutputDirectory = ".";
        }

        public void ShowSubCommandUsage(string baseName)
        {
            Console.WriteLine("extractnsave: Extract files from a Nintendo Save data file.");
            Console.WriteLine("Usage: {0} extractnsave [-o <outputDirectory>] <inputNsavePath>", baseName);
            Console.WriteLine(string.Empty);
            Console.WriteLine("Options:");
            Console.WriteLine("  -o <outputDirectory>                 Directory path to output. If omitted, output to current directory");
        }

        public OptionDescription[] GetOptionDescription()
        {
            OptionDescription[] options = new OptionDescription[]
                {
                    new OptionDescription(null, "-o", 1, (s) => { OutputDirectory = OptionUtil.GetOutputFilePath(OutputDirectory, s.First()); }),
                };
            return options;
        }

        public void ParsePositionalArgument(string[] args)
        {
            if (args.Length == 0)
            {
                throw new InvalidOptionException("input nsave file must be specified for extractnsave subcommand.");
            }

            InputNsaveFile = args[0];
        }
    }

    enum ExtractMode
    {
        All,
        TargetStringPath,
        TargetRegexPath,
    };

    internal class ExtractOption : IOption
    {
        internal string InputFile { get; set; }
        internal string OriginalFile { get; set; }
        internal string OutputDirectory { get; set; }
        internal string TargetEntryPath { get; set; }
        internal Regex TargetEntryRegex { get; set; }
        internal ExtractMode Mode { get; set; }
        internal bool ConsoleDump { get; set; }

        internal ExtractOption()
        {
            InputFile = null;
            OriginalFile = null;
            OutputDirectory = ".";
            TargetEntryPath = null;
            TargetEntryRegex = null;
            Mode = ExtractMode.All;
        }

        public void ShowSubCommandUsage(string baseName)
        {
            Console.WriteLine("extract: Extract files from a Nintendo Submission Package file or Nintendo Content Archive file.");
            Console.WriteLine("Usage: {0} extract [-o <outputDirectory>] [--original <originalNspPath>] [--target <targetPath>] <inputNspPath>", baseName);
            Console.WriteLine("Usage: {0} extract [-o <outputDirectory>] [--original <originalNspPath>] [--targetregex <targetRegularExpression>] <inputNspPath>", baseName);
            Console.WriteLine(string.Empty);
            Console.WriteLine("Options:");
            Console.WriteLine("  -o <outputDirectory>                 Directory path to output. If omitted, output to current directory");
            Console.WriteLine("  --target <targetEntryPath>           Path of the target entry to be extracted. If omitted, extract all entries");
            Console.WriteLine("  --targetregex <targetEntryPath>      Regular expression for target entries to be extracted. If omitted, extract all entries");
            Console.WriteLine("  --original <originalNspNcaPath>      Original application nsp path. This is required if <inputNspPath> is a patch.");
        }

        public OptionDescription[] GetOptionDescription()
        {
            OptionDescription[] options = new OptionDescription[]
                {
                    new OptionDescription(null, "-o", 1, (s) => { OutputDirectory = OptionUtil.GetOutputFilePath(OutputDirectory, s.First()); }),
                    new OptionDescription(null, "--target", 1, (s) => { TargetEntryPath = s.First(); }),
                    new OptionDescription(null, "--targetregex", 1, (s) => { TargetEntryRegex = new Regex(s.First()); }),
                    new OptionDescription(null, "--original", 1, (s) => { OriginalFile = OptionUtil.CheckAndNormalizeFilePath(s.First(), "--original"); }),
                    new OptionDescription(null, "--dump", 0, (s) => { ConsoleDump = true; })
                };
            return options;
        }

        public void ParsePositionalArgument(string[] args)
        {
            if (args.Length == 0)
            {
                throw new InvalidOptionException("input nca/nsp file must be specified for extract subcommand.");
            }

            InputFile = args[0];

            VerifyArgument();
        }

        public void VerifyArgument()
        {
            if(TargetEntryPath != null && TargetEntryRegex != null)
            {
                throw new InvalidOptionException("option --target and --targetregex are exclusive.");
            }

            Mode =
                (TargetEntryPath != null) ? ExtractMode.TargetStringPath :
                (TargetEntryRegex != null) ? ExtractMode.TargetRegexPath :
                ExtractMode.All;
        }
    }

    internal class MergeNspOption : IOption
    {
        internal List<string> InputFiles { get; set; }
        internal string OutputFile { get; set; }

        internal MergeNspOption()
        {
            InputFiles = new List<string>();
            OutputFile = "./output.nsp";
        }

        public void ShowSubCommandUsage(string baseName)
        {
            Console.WriteLine("mergensp: Merge the contents included in multiple Nintendo Submission Packages into one nsp file.");
            Console.WriteLine("Usage: {0} mergensp [-o <outputPath>] <inputNspOrNcaPath>...", baseName);
            Console.WriteLine(string.Empty);
            Console.WriteLine("Options:");
            Console.WriteLine("  -o <outputPath>              File path to output. If directory path is specified, output to <outputPath>/output.nsp.");
        }

        public OptionDescription[] GetOptionDescription()
        {
            OptionDescription[] options = new OptionDescription[]
                {
                    new OptionDescription(null, "-o", 1, (s) => { OutputFile = OptionUtil.GetOutputFilePath(OutputFile, s.First()); }),
                };
            return options;
        }

        public void ParsePositionalArgument(string[] args)
        {
            if (args.Length < 2)
            {
                throw new InvalidOptionException("too few arguments for mergensp subcommand.");
            }

            for (int i = 0; i < args.Length; i++)
            {
                InputFiles.Add(OptionUtil.CheckAndNormalizeFilePath(args[i], string.Format("arg[{0}]", i)));
            }
        }
    }

    internal class SplitNspOption : IOption
    {
        internal string InputNspFile { get; set; }
        internal string OutputDirectory { get; set; }
        internal ulong? TargetContentMetaId { get; set; }
        internal uint? TargetVersion { get; set; }

        internal SplitNspOption()
        {
            OutputDirectory = "./";
        }

        public void ShowSubCommandUsage(string baseName)
        {
            Console.WriteLine("splitnsp: Split one nsp file into multiple nsp files with respective content meta id and version.");
            Console.WriteLine("Usage: {0} splitnsp [-o <outputDirectory>] [--id <contentMetaId>] [--version <version>] <inputNspPath>", baseName);
            Console.WriteLine(string.Empty);
            Console.WriteLine("Options:");
            Console.WriteLine("  -o <outputDirectory>         Directory path to output. If omitted, output to current directory.");
            Console.WriteLine("  --id <contentMetaId>         Output only a nsp file has specified contemt meta id.");
            Console.WriteLine("  --version <version>          Output only a nsp file has specified version (not release version.)");
        }

        public OptionDescription[] GetOptionDescription()
        {
            OptionDescription[] options = new OptionDescription[]
                {
                    new OptionDescription(null, "-o", 1, (s) => { OutputDirectory = OptionUtil.GetOutputFilePath(OutputDirectory, s.First()); }),
                    new OptionDescription(null, "--id", 1, (s) => { TargetContentMetaId = Convert.ToUInt64(s.First(), 16); }),
                    new OptionDescription(null, "--version", 1, (s) => { TargetVersion = Convert.ToUInt32(s.First(), 10); }),
                };
            return options;
        }

        public void ParsePositionalArgument(string[] args)
        {
            if (args.Length != 1)
            {
                throw new InvalidOptionException("input nsp file must be specified for splitnsp subcommand.");
            }

            InputNspFile = OptionUtil.CheckAndNormalizeFilePath(args[0], "arg[0]");
        }
    }

    internal class ReplaceOption : IOption
    {
        internal string InputFile { get; set; }
        internal string DescFilePath { get; set; }
        internal string OutputDirectory { get; set; }
        internal string TargetEntryPath { get; set; }
        internal string InputEntryFilePath { get; set; }
        internal string NroDirectoryPath { get; set; }
        internal bool ErrorUnpublishable { get; private set; }
        internal bool ShowUnpublishableErroFormatXml { get; private set; }
        internal string IgnoreErrorUnpublishable { get; private set; }

        // Nca の Replace 用
        internal string MetaType { get; private set; }
        internal string ContentType { get; private set; }

        internal ReplaceOption()
        {
            InputFile = null;
            DescFilePath = null;
            OutputDirectory = ".";
            TargetEntryPath = null;
            InputEntryFilePath = null;
            NroDirectoryPath = null;
            ErrorUnpublishable = false;
            ShowUnpublishableErroFormatXml = false;
            IgnoreErrorUnpublishable = null;
            MetaType = "Application";
            ContentType = "Program";
        }

        public void ShowSubCommandUsage(string baseName)
        {
            Console.WriteLine("replace: Replace files from a Nintendo Submission Package or Nintendo Content Archive file.");
            Console.WriteLine("Usage: {0} replace [-o <outputDirectory>] [--desc <descFilePath>] <inputNspOrNcaPath> <targetEntryPath> <inputEntryFilePath> [--nro <nroDirectoryPath>] [--error-unpublishable [--xml]]", baseName);
            Console.WriteLine("Usage: {0} replace [-o <outputDirectory>] [--desc <descFilePath>] <inputNspOrNcaPath> <replaceRuleListFilePath> [--nro <nroDirectoryPath>] [--error-unpublishable [--xml]]", baseName);
            Console.WriteLine(string.Empty);
            Console.WriteLine("Options:");
            Console.WriteLine("  -o <outputDirectory>                 Directory path to output. If omitted, output to current directory");
            Console.WriteLine("  --desc <descFilePath>                Location of .desc file to input. This cannot be omitted when replacing a program content.");
            Console.WriteLine("  --nro <nroDirectoryPath>             Location of directory which includes .nro files to be contained.");
            Console.WriteLine("  --error-unpublishable [--xml]        Throw an exception if the Nintendo Submission Package is not satisfied with conditions to publish it.");
#if false // 非公開
            Console.WriteLine("  --ignore-unpublishable-error [ignoreListPath]");
            Console.WriteLine("                                       Ignore unpublishable error.");
#endif
        }

        public OptionDescription[] GetOptionDescription()
        {
            OptionDescription[] options = new OptionDescription[]
                {
                    new OptionDescription(null, "-o", 1, (s) => { OutputDirectory = OptionUtil.GetOutputFilePath(OutputDirectory, s.First()); }),
                    new OptionDescription(null, "--desc", 1, (s) => { DescFilePath = s.First(); }),
                    new OptionDescription("--nro", null, 1, (s) =>
                        {
                            NroDirectoryPath = OptionUtil.CheckAndNormalizeDirectory(s.First(),"--nro");
                        }),
                    new OptionDescription(null, "--error-unpublishable", 0, (s) => { ErrorUnpublishable = true; }),
                    new OptionDescription(null, "--xml", 0, (s) => { ShowUnpublishableErroFormatXml = true; }),
                    new OptionDescription("--ignore-unpublishable-error", null, -1, (s) =>
                        {
                            if (s.Count > 1)
                            {
                                throw new InvalidOptionException("too many arguments for --ignore-unpublishable-error option.");
                            }
                            IgnoreErrorUnpublishable = (s.Count() > 0) ? OptionUtil.CheckAndNormalizeFilePath(s.First(), "--ignore-unpublishable-error") : string.Empty;
                        }),
                    new OptionDescription("--meta-type", null, 1, (type) => { MetaType = type.First(); }),
                    new OptionDescription("--content-type", null, 1, (type) => { ContentType = type.First(); }),
                };
            return options;
        }

        public void ParsePositionalArgument(string[] args)
        {
            if (args.Length < 2)
            {
                throw new InvalidOptionException("too few arguments for replace subcommand.");
            }

            InputFile = args[0];
            TargetEntryPath = args[1];
            if (args.Length > 2)
            {
                InputEntryFilePath = args[2];
            }
        }
    }

    internal class ReplaceNspMetaOption : IOption
    {
        internal string InputFile { get; set; }
        internal string OutputDirectory { get; set; }
        internal string DescFilePath { get; set; }
        internal string MetaFilePath { get; set; }
        internal string IgnoreErrorUnpublishable { get; private set; }
        internal string WorkDirectory { get; set; }

        internal ReplaceNspMetaOption()
        {
            InputFile = null;
            OutputDirectory = ".";
            DescFilePath = null;
            MetaFilePath = null;
            IgnoreErrorUnpublishable = null;
        }

        public void ShowSubCommandUsage(string baseName)
        {
            Console.WriteLine("replacenspmeta: Replace a file in a Nintendo Submission Package from a nmeta file.");
            Console.WriteLine("Usage: {0} replacenspmeta <inputNspPath> [-o <outputDirectory>] [--desc <descFilePath>] --meta <metaFilePath>", baseName);
            Console.WriteLine(string.Empty);
            Console.WriteLine("Options:");
            Console.WriteLine("  -o <outputDirectory>                 Directory path to output. If omitted, output to current directory");
            Console.WriteLine("  --desc <descFilePath>                Location of .desc file to input. This cannot be omitted when replacing a program content.");
            Console.WriteLine("  --meta <metaFilePath>                Location of .nmeta file to input.");
#if false // 非公開
            Console.WriteLine("  --ignore-unpublishable-error [ignoreListPath] Ignore unpublishable error.");
#endif
        }

        public OptionDescription[] GetOptionDescription()
        {
            OptionDescription[] options = new OptionDescription[]
                {
                    new OptionDescription(null, "-o", 1, (s) => { OutputDirectory = OptionUtil.GetOutputFilePath(OutputDirectory, s.First()); }),
                    new OptionDescription(null, "--desc", 1, (s) => { DescFilePath = s.First(); }),
                    new OptionDescription(null, "--meta", 1, (s) => { MetaFilePath = s.First(); }),
                    new OptionDescription("--ignore-unpublishable-error", null, -1, (s) =>
                        {
                            if (s.Count > 1)
                            {
                                throw new InvalidOptionException("too many arguments for --ignore-unpublishable-error option.");
                            }
                            IgnoreErrorUnpublishable = (s.Count() > 0) ? OptionUtil.CheckAndNormalizeFilePath(s.First(), "--ignore-unpublishable-error") : string.Empty;
                        }),
                };
            return options;
        }

        public void ParsePositionalArgument(string[] args)
        {
            if (args.Length < 1)
            {
                throw new InvalidOptionException("too few arguments for replacenspmeta subcommand.");
            }

            // 作業用ディレクトリを作成
            while (true)
            {
                WorkDirectory = OutputDirectory + "/" +  Path.GetRandomFileName();
                if (!Directory.Exists(WorkDirectory))
                {
                    break;
                }
            }
            Directory.CreateDirectory(WorkDirectory);

            InputFile = args[0];
        }
    }

    internal class ListOption : IOption
    {
        internal string InputFile { get; set; }
        internal string OriginalFile { get; set; }
        internal string TicketFile { get; set; }
        internal Regex TargetEntryRegex { get; set; }
        internal bool PatchedOnly { get; set; }
        internal bool ShowPatchFragment { get; set; }

        internal ListOption()
        {
            InputFile = null;
            OriginalFile = null;
            TicketFile = null;
            TargetEntryRegex = null;
        }

        public void ShowSubCommandUsage(string baseName)
        {
            Console.WriteLine("list: List information about the files included in an archive file.");
            Console.WriteLine("Usage: {0} list [--targetregex <targetEntryPath>] [--original <originalPath>] <inputNspPath> ", baseName);
            Console.WriteLine(string.Empty);
            Console.WriteLine("Options:");
            Console.WriteLine("  --targetregex <targetEntryPath>      Regular expression to filter the result.");
            Console.WriteLine("  --original <originalPath>            Original application nsp path. This is required if <inputNspPath> is a patch.");
            Console.WriteLine("  --patched-only                       List information for patched file only. This is must be used with --original option.");
            Console.WriteLine("  --patched-only-in-detail             List information for patched file only and list fragment information additionally. This is must be used with --original option.");
        }

        public OptionDescription[] GetOptionDescription()
        {
            OptionDescription[] options = new OptionDescription[]
                {
                    new OptionDescription(null, "--targetregex", 1, (s) => { TargetEntryRegex = new Regex(s.First()); }),
                    new OptionDescription(null, "--original", 1, (s) => { OriginalFile = OptionUtil.CheckAndNormalizeFilePath(s.First(), "--original"); }),
                    new OptionDescription(null, "--ticket", 1, (s) => { TicketFile = OptionUtil.CheckAndNormalizeFilePath(s.First(), "--ticket"); }),
                    new OptionDescription(null, "--patched-only", 0, (s) => { PatchedOnly = true; }),
                    new OptionDescription(null, "--patched-only-in-detail", 0, (s) => { PatchedOnly = true; ShowPatchFragment = true; })
                };
            return options;
        }

        public void ParsePositionalArgument(string[] args)
        {
            if (args.Length != 1)
            {
                throw new InvalidOptionException("input archive file must be specified for list subcommand.");
            }

            InputFile = args[0];

            if (PatchedOnly && OriginalFile == null)
            {
                throw new InvalidOptionException("--patched-only or --patched-only-in-detail option must be used with --original option.");
            }
        }
    }

    internal class DiffPatchOption : IOption
    {
        internal string OriginalFile { get; set; }
        internal string PreviousPatchFile { get; set; }
        internal string PatchFile { get; set; }

        private string NspExtension = ".nsp";

        internal DiffPatchOption()
        {
            OriginalFile = null;
            PreviousPatchFile = null;
            PatchFile = null;
        }

        public void ShowSubCommandUsage(string baseName)
        {
            Console.WriteLine("diffpatch: Show different files between patches.");
            Console.WriteLine("Usage: {0} diffpatch <originalNspPath> <previousPatchNspPath> <patchNspPath> ", baseName);
        }

        public OptionDescription[] GetOptionDescription()
        {
            return null;
        }

        public void ParsePositionalArgument(string[] args)
        {
            if (args.Length != 3)
            {
                throw new InvalidOptionException("input 3 archive files (original application nsp and 2 patch nsp) must be specified for diffpatch subcommand.");
            }

            for (int i = 0; i < args.Length; i++)
            {
                if (Path.GetExtension(args[i]) != NspExtension)
                {
                    throw new InvalidOptionException("input archive file must be .nsp file.");
                }
            }

            OriginalFile = OptionUtil.CheckAndNormalizeFilePath(args[0], "arg[0]");
            PreviousPatchFile = OptionUtil.CheckAndNormalizeFilePath(args[1], "arg[1]");
            PatchFile = OptionUtil.CheckAndNormalizeFilePath(args[2], "arg[2]");
        }
    }

    internal class VerifyOption : IOption
    {
        internal string InputFilePath { get; set; }
        internal string PreviousFilePath { get; set; }

        public VerifyOption()
        {
        }

        public void ShowSubCommandUsage(string baseName)
        {
            Console.WriteLine("verify: Verify input Nintendo Submission Package.");
            Console.WriteLine("Usage: {0} verify <inputNspPath>", baseName);
            Console.WriteLine(string.Empty);
            Console.WriteLine("Options:");
            Console.WriteLine("  --previous <previousNspPath>         Previous(or Original) .nsp file path to input.");
        }

        public OptionDescription[] GetOptionDescription()
        {
            OptionDescription[] options = new OptionDescription[]
                {
                    OptionUtil.CreateFilePathOptionDescription("--previous", (path) => { PreviousFilePath = path; }),
                };
            return options;
        }

        public void ParsePositionalArgument(string[] args)
        {
            if (args.Length != 1)
            {
                throw new InvalidOptionException("input nsp file must be specified for verify subcommand.");
            }

            InputFilePath = OptionUtil.CheckAndNormalizeFilePath(args[0], "arg[0]");

            if (Path.GetExtension(InputFilePath) != ".nsp")
            {
                throw new InvalidOptionException("input archive file must be .nsp file.");
            }

            if (!string.IsNullOrEmpty(PreviousFilePath))
            {
                if (Path.GetExtension(PreviousFilePath) != ".nsp")
                {
                    throw new InvalidOptionException("previous archive file must be .nsp file.");
                }
            }
        }
    }

    internal class ProdEncryptionOption : IOption
    {
        internal string InputNspFile { get { return InputNspFiles[0]; } }
        internal List<string> InputNspFiles { get; set; }
        internal string InputUppNspFile { get; set; }
        internal string InputPatchNspFile { get; set; }
        internal string InputAocNspFile { get; set; }
        internal string InputMultiApplicationCardIndicatorNspFile { get; set; }
        internal string OutputDirectory { get; set; }
        internal uint? RequiredSystemVersion { get; set; }
        internal byte DesiredSafeKeyGen { get; set; }
        internal string InputUppForDesiredSafeKeyGenNspFile { get; set; }

        internal bool CheckIntegrity { get; set; }
        internal bool CreateNspu { get; set; }
        internal bool CreateXci { get; set; }
        internal bool CreateXcie { get; set; }
        internal byte LaunchFlags { get; set; }
        internal bool NoPaddingXci { get; set; }
        internal List<UInt64> ExcludeContents { get; private set; } = new List<UInt64>();
        internal Dictionary<UInt64, byte> KeyGenerations { get; private set; } = new Dictionary<UInt64, byte>();
        internal bool UseKeyForRepairTool { get; set; }
        internal bool KeepGeneration { get; set; }

        private bool NoCreateXcie { get; set; }

        private const string NspExtension = ".nsp";

        internal ProdEncryptionOption()
        {
            InputNspFiles = new List<string>();
            InputUppNspFile = null;
            InputPatchNspFile = null;
            InputAocNspFile = null;
            OutputDirectory = ".";
            CheckIntegrity = true;
            CreateNspu = true;
            CreateXci = false;
            CreateXcie = false;
            NoCreateXcie = false;
            NoPaddingXci = false;
        }

        public void ShowSubCommandUsage(string baseName)
        {
            Console.WriteLine("prodencryption: Create prod-encrypted .nsp, .xci or .xcie file.");
            Console.WriteLine("Usage: {0} prodencryption [-o <outputDirecotry>] [--no-check] [--no-nspu] [--exclude-list <xmlPath>] [--keygen-list <xmlPath>] [--upp <uppNspPath> | --requiredSystemVersion <version>] <inputNspPath>", baseName);
            Console.WriteLine("Usage: {0} prodencryption [-o <outputDirecotry>] [--no-check] --gamecard [--no-padding] [--no-xcie] [--auto-boot] [--history-erase] [--for-repair] [--upp <uppNspPath>] [--patch <patchNspPath>] [--aoc <aocNspPath>] <inputNspPath>", baseName);
            Console.WriteLine("Usage: {0} prodencryption [-o <outputDirecotry>] [--no-check] --multiapplicationgamecard <macNspPath> [--no-padding] [--no-xcie] [--auto-boot] [--history-erase] [--for-repair] [--upp <uppNspPath>] <inputNspPath1> <inputNspPath2> ...", baseName);
            Console.WriteLine(string.Empty);
            Console.WriteLine("Options:");
            Console.WriteLine("  -o <outputDirectory>                 Directory path to output. If omitted, output to current directory");
            Console.WriteLine("  --no-check                           Omit to check integrity of output nsp/xci file.");
            Console.WriteLine("  --no-nspu                            Omit to create nspu file. Must be used without --gamecard option.");
            Console.WriteLine("  --exclude-list <xmlPath>             Specifies xml path which describes contents to be omitted for prodencryption.");
            Console.WriteLine("                                       Must be used without --gamecard and --multiapplicationgamecard option.");
            Console.WriteLine("  --keygen-list <xmlPath>              Specifies xml path which describes keygeneration to be used for prodencryption.");
            Console.WriteLine("                                       Must be used without --gamecard and --multiapplicationgamecard option.");
            Console.WriteLine("  --requiredSystemVersion <version>    Required system version to be used to substitute that of target nsp.");
            Console.WriteLine("  --desired-safe-keygen <generation>   Safe key generation desired to be used for prodencryption.");
            Console.WriteLine("  --upp <uppNspPath>                   Location of update partition nsp file, which is used instead of --requiredSystemVersion option.");
            Console.WriteLine("                                       If --gamecard or --multiapplicationgamecard option is also specified, it is put on gamecard.");
            Console.WriteLine("  --upp-for-desired-safe-keygen <uppNspPath>");
            Console.WriteLine("                                       Location of update partition nsp file, which is used instead of --desired-safe-keygen option.");
            Console.WriteLine("                                       Must be used without --gamecard or --multiapplicationgamecard option.");
            Console.WriteLine("  --gamecard                           Create prod-encrypted xci and xcie file.");
            Console.WriteLine("  --no-padding                         Not insert padding to end of xci and xcie file. Must be used with --gamecard or --multiapplicationgamecard option");
            Console.WriteLine("  --no-xcie                            Omit to create xcie file. Must be used with --gamecard or --multiapplicationgamecard option.");
            Console.WriteLine("  --auto-boot                          Set auto boot flag for game card. Must be used with --gamecard or --multiapplicationgamecard option.");
            Console.WriteLine("  --history-erase                      Set history-erase flag for game card. Must be used with --gamecard or --multiapplicationgamecard option.");
            Console.WriteLine("  --for-repair                         Create xcir file instead of xcie file. Must be used with --gamecard or --multiapplicationgamecard option.");
            Console.WriteLine("  --patch                              Specifies nsp path for on-card-patch. Must be used with --gamecard option.");
            Console.WriteLine("  --aoc                                Specifies nsp path for on-card-aoc. Must be used with --gamecard option.");
            Console.WriteLine("  --keep-generation                    Skip security update process while prodencryption. Must be used with --upp option.");
            Console.WriteLine("  --multiapplicationgamecard <macNspPath>");
            Console.WriteLine("                                       Location of multi-application card indicator nsp file. Must be used without --gamecard option.");
        }

        public OptionDescription[] GetOptionDescription()
        {
            OptionDescription[] options = new OptionDescription[]
                {
                    new OptionDescription(null, "-o", 1, (s) => { OutputDirectory = OptionUtil.GetOutputFilePath(OutputDirectory, s.First()); }),
                    new OptionDescription("--no-check", null, 0, (s) => { CheckIntegrity = false; }),
                    new OptionDescription("--no-nspu", null, 0, (s) => { CreateNspu = false; }),
                    new OptionDescription("--requiredSystemVersion", null, 1, (s) => { RequiredSystemVersion = Convert.ToUInt32(s.First(), 10); }),
                    new OptionDescription("--desired-safe-keygen", null, 1, (s) => { DesiredSafeKeyGen = Convert.ToByte(s.First(), 10); }),
                    new OptionDescription("--gamecard", null, 0, (s) => { CreateXci = true; CreateXcie = true; }),
                    new OptionDescription("--no-xcie", null, 0, (s) => { NoCreateXcie = true; }),
                    new OptionDescription("--auto-boot", null, 0, (s) => { LaunchFlags |= 0x1; }),
                    new OptionDescription("--history-erase", null, 0, (s) => { LaunchFlags |= 0x2; }),
                    new OptionDescription("--repair-time-reviser-tool", null, 0, (s) => { LaunchFlags |= 0x4; }),
                    new OptionDescription("--no-padding", null, 0, (s) => { NoPaddingXci = true; }),
                    new OptionDescription("--for-repair", null, 0, (s) => { UseKeyForRepairTool = true; }),
                    new OptionDescription("--keep-generation", null, 0, (s) => { KeepGeneration = true; }),
                    OptionUtil.CreateFilePathOptionDescription("--multiapplicationgamecard", NspExtension, (path) => { InputMultiApplicationCardIndicatorNspFile = path; CreateXcie = true; }),
                    OptionUtil.CreateFilePathOptionDescription("--upp", NspExtension, (path) => { InputUppNspFile = path; }),
                    OptionUtil.CreateFilePathOptionDescription("--upp-for-desired-safe-keygen", NspExtension, (path) => { InputUppForDesiredSafeKeyGenNspFile = path; }),
                    OptionUtil.CreateFilePathOptionDescription("--patch", NspExtension, (path) => { InputPatchNspFile = path; }),
                    OptionUtil.CreateFilePathOptionDescription("--aoc", NspExtension, (path) => { InputAocNspFile = path; }),
                    new OptionDescription("--exclude-list", null, 1, (s) =>
                        {
                            var xml = XDocument.Load(s.First());
                            ExcludeContents = xml.Descendants("Exclude").Descendants("Content").Descendants("Id").Select((id) => Convert.ToUInt64(id.Value, 16)).ToList();
                        }),
                    new OptionDescription("--keygen-list", null, 1, (s) =>
                        {
                            var xml = XDocument.Load(s.First());
                            foreach (var contentMeta in xml.Descendants("KeyGenerationMin").Descendants("ContentMeta"))
                            {
                                var id = contentMeta.Descendants("Id").Select(x => Convert.ToUInt64(x.Value, 16)).Single();
                                var keyGenerationMin = contentMeta.Descendants("Value").Select(x => Convert.ToByte(x.Value)).Single();
                                KeyGenerations.Add(id, keyGenerationMin);
                            }
                        }),
                };
            return options;
        }

        public void ParsePositionalArgument(string[] args)
        {
            if (args.Length == 0)
            {
                throw new InvalidOptionException("input nsp file must be specified for prodencryption subcommand.");
            }

            if (args.Length >= 2 && InputMultiApplicationCardIndicatorNspFile == null)
            {
                throw new InvalidOptionException("too many arguments for prodencryption subcommand.");
            }

            for (int i = 0 ; i < args.Length; i++)
            {
                var nspFilePath = OptionUtil.CheckAndNormalizeFilePath(args[i], $"arg[i]");
                OptionUtil.CheckExtension(nspFilePath, NspExtension);
                InputNspFiles.Add(nspFilePath);
            }

            if (InputMultiApplicationCardIndicatorNspFile != null)
            {
                if (CreateXci)
                {
                    throw new InvalidOptionException("--multiapplicationgamecard option requires to be used without --gamecard option.");
                }
                if (InputPatchNspFile != null || InputAocNspFile != null)
                {
                    throw new InvalidOptionException("--patch and --aoc option requires to be used with --gamecard option.");
                }
                CreateXci = true;
            }

            if (!CreateXci)
            {
                if (NoCreateXcie || NoPaddingXci || UseKeyForRepairTool || LaunchFlags != 0 || InputPatchNspFile != null || InputAocNspFile != null)
                {
                    throw new InvalidOptionException("--no-padding, --no-xcie, --for-repair, --auto-boot, --history-erase, --repair-time-reviser-tool, --patch and --aoc option requires to be used with --gamecard option.");
                }
            }
            else
            {
                if (!CreateNspu)
                {
                    throw new InvalidOptionException("--no-nspu option requires to be used without --gamecard option.");
                }
            }

            if (KeepGeneration && InputUppNspFile == null)
            {
                throw new InvalidOptionException("--keep-generation option requires to be used with --upp option.");
            }

            if (NoCreateXcie)
            {
                CreateXcie = false;
            }
        }
    }

    internal class ProdEncryptionPatchOption : IOption
    {
        internal string InputNspFile { get; set; }
        internal string InputUppNspFile { get; set; }
        internal string InputOriginalNspFile { get; set; }
        internal string InputOriginalProdNspFile { get; set; }
        internal string InputPreviousProdNspFile { get; set; }
        internal string OutputDirectory { get; set; }
        internal uint? RequiredSystemVersion { get; set; }
        internal byte DesiredSafeKeyGen { get; set; }
        internal string InputUppForDesiredSafeKeyGenNspFile { get; set; }
        internal long SizeThresholdForKeyGenerationUpdate { get; set; }

        internal bool CreateNspu { get; set; }
        internal bool CheckIntegrity { get; set; }
        private const string NspExtension = ".nsp";

        internal ProdEncryptionPatchOption()
        {
            OutputDirectory = ".";
            CreateNspu = true;
            CheckIntegrity = true;
            SizeThresholdForKeyGenerationUpdate = (long)1 * 1024 * 1024 * 1024;
        }

        public void ShowSubCommandUsage(string baseName)
        {
            Console.WriteLine("prodencryption-patch: Create prod-encrypted .nsp for patch.");
            Console.WriteLine("Usage: {0} prodencryption-patch [-o <outputDirecotry>] [--no-check] [--no-nspu] [--upp <uppNspPath> | --requiredSystemVersion <version>] --original <originalNspPath> --original-prod <originalProdNspPath> [--previous-prod <previousProdNspPath>] <inputNspPath>", baseName);
            Console.WriteLine(string.Empty);
            Console.WriteLine("Options:");
            Console.WriteLine("  -o <outputDirecotry>                  Directory path to output. If omitted, output to current directory");
            Console.WriteLine("  --no-check                            Omit to check integrity of output nsp file.");
            Console.WriteLine("  --no-nspu                             Omit to create nspu file.");
            Console.WriteLine("  --requiredSystemVersion <version>     Required system version to be used to substitute that of target nsp.");
            Console.WriteLine("  --upp <uppNspPath>                    Location of update partition nsp file, which is used instead of --requiredSystemVersion option.");
            Console.WriteLine("  --desired-safe-keygen <generation>    Safe key generation desired to be used for prodencryption.");
            Console.WriteLine("                                        This won't be applied if the patch isn't the first or includes content whose size exceeds the threshold (1 GiB as default.)");
            Console.WriteLine("  --upp-for-desired-safe-keygen <uppNspPath>");
            Console.WriteLine("                                        Location of update partition nsp file, which is used instead of --desired-safe-keygen option.");
            Console.WriteLine("  --use-safe-keygen-forcibly            Force to use safe key generation for prodencryption even though the patch includes content whose size exceeds the threshold (1 GiB as default.)");
            Console.WriteLine("  --size-threshold-for-keygen-update    Set size threshold to update key generation or not.");
            Console.WriteLine("  --original <originalNspPath>          Location of original nsp file.");
            Console.WriteLine("  --original-prod <originalProdNspPath> Location of prod-encrypted original nsp file.");
            Console.WriteLine("                                        This can be omitted if --no-check option is specified.");
            Console.WriteLine("  --previous-prod <previousProdNspPath> Location of prod-encrypted previous patch nsp file.");
            Console.WriteLine("                                        This can be omitted if target nsp is the first version of patch.");
        }

        public OptionDescription[] GetOptionDescription()
        {
            OptionDescription[] options = new OptionDescription[]
                {
                    new OptionDescription(null, "-o", 1, (s) => { OutputDirectory = OptionUtil.GetOutputFilePath(OutputDirectory, s.First()); }),
                    new OptionDescription("--no-check", null, 0, (s) => { CheckIntegrity = false; }),
                    new OptionDescription("--no-nspu", null, 0, (s) => { CreateNspu = false; }),
                    new OptionDescription("--requiredSystemVersion", null, 1, (s) => { RequiredSystemVersion = Convert.ToUInt32(s.First(), 10); }),
                    new OptionDescription("--desired-safe-keygen", null, 1, (s) => { DesiredSafeKeyGen = Convert.ToByte(s.First(), 10); }),
                    new OptionDescription("--use-safe-keygen-forcibly", null, 0, (s) => { SizeThresholdForKeyGenerationUpdate = (long)128 * 1024 * 1024 * 1024; }),
                    new OptionDescription("--size-threshold-for-keygen-update", null, 1, (s) => { SizeThresholdForKeyGenerationUpdate = Convert.ToInt64(s.First(), 10); }),
                    OptionUtil.CreateFilePathOptionDescription("--upp", NspExtension, (path) => { InputUppNspFile = path; }),
                    OptionUtil.CreateFilePathOptionDescription("--upp-for-desired-safe-keygen", NspExtension, (path) => { InputUppForDesiredSafeKeyGenNspFile = path; }),
                    OptionUtil.CreateFilePathOptionDescription("--original", NspExtension, (path) => { InputOriginalNspFile = path; }),
                    OptionUtil.CreateFilePathOptionDescription("--original-prod", NspExtension, (path) => { InputOriginalProdNspFile = path; }),
                    OptionUtil.CreateFilePathOptionDescription("--previous-prod", NspExtension, (path) => { InputPreviousProdNspFile = path; }),
                };
            return options;
        }

        public void ParsePositionalArgument(string[] args)
        {
            if (args.Length == 0)
            {
                throw new InvalidOptionException("input nsp file must be specified for prodencryption-patch subcommand.");
            }

            InputNspFile = OptionUtil.CheckAndNormalizeFilePath(args[0], "arg[0]");
            OptionUtil.CheckExtension(InputNspFile, NspExtension);

            if (args.Length >= 2)
            {
                throw new InvalidOptionException("too many arguments for prodencryption-patch subcommand.");
            }

            if (InputOriginalNspFile == null)
            {
                throw new InvalidOptionException("--orignal option must be specified.");
            }

            if (InputOriginalProdNspFile == null && CheckIntegrity)
            {
                throw new InvalidOptionException("--orignal-prod option must be specified.");
            }
        }
    }

    internal class MakePatchOption : IOption
    {
        internal string OutputFile { get; set; }
        internal string MetaFilePath { get; set; }
        internal string DescFilePath { get; set; }
        internal string OriginalFilePath { get; set; }
        internal string CurrentFilePath { get; set; }
        internal string PreviousFilePath { get; set; }
        internal bool IsSaveBuildLog { get; set; }
        internal bool IsNcaFile { get; private set; }
        internal bool WillOutputWithDelta { get; private set; }
        internal bool WillMergeDeltaContent { get; private set; }
        internal string DeltaMetaPath { get; private set; }
        internal bool KeepIntermediates { get; private set; }
        internal int MinimumMatchingSize { get; private set; }
        internal bool NeedsDefragment { get; private set; }
        internal int DefragmentSize { get; private set; }
        internal string CacheDirectory { get; private set; }
        internal int StronglyOptimizeSize { get; private set; }
        internal bool NeedsSparse { get; private set; }
        internal int SparseEraseSize { get; private set; }
        internal int SparseBlockSize { get; private set; }


        internal MakePatchOption()
        {
            OutputFile = "./output.nsp";
            MetaFilePath = null;
            DescFilePath = null;
            OriginalFilePath = null;
            CurrentFilePath = null;
            PreviousFilePath = null;
            IsSaveBuildLog = false;
            IsNcaFile = true;
            WillOutputWithDelta = false;
            WillMergeDeltaContent = false;
            DeltaMetaPath = null;
            KeepIntermediates = false;
            MinimumMatchingSize = 0;
            NeedsDefragment = false;
            DefragmentSize = 0;
            CacheDirectory = null;
            StronglyOptimizeSize = IndirectStorageSource.BlockSize;
            SparseEraseSize = SparseStorageSource.DefaultEraseSize;
            SparseBlockSize = SparseStorageSource.DefaultBlockSize;
        }

        public void ShowSubCommandUsage(string baseName)
        {
            Console.WriteLine("makepatch: Create patch file from Nintendo Submission Packages.");
            Console.WriteLine("Usage: {0} makepatch [-o <outputPath>] [--meta <metaFilePath>] [--minimum-matching-size <Size KiB>] [--cache-directory <cacheDirPath>] --desc <descFilePath> --original <originalNspPath> --current <currentNspPath> [--previous <previousNspPath>]", baseName);
            Console.WriteLine(string.Empty);
            Console.WriteLine("Options:");
            Console.WriteLine("  -o <outputPath>                      File path to output. If directory path is specified, output to <outputPath>/output.nsp.");
            Console.WriteLine("                                       If omitted, output to ./output.nsp");
            Console.WriteLine("  --meta <metaFilePath>                Location of .nmeta file to input.");
            Console.WriteLine("  --minimum-matching-size <Size KiB>   Minimum matching size to refer original. If not specified, use 32KiB.");
            Console.WriteLine("  --cache-directory <cacheDirPath>     Location of directory path to output matching cache.");
            Console.WriteLine("                                       With specifying this option, making patch is accelarated if appropriate cache is generated.");
            Console.WriteLine("  --desc <descFilePath>                Location of .desc file to current input. This cannot be omitted when create program content.");
            Console.WriteLine("  --original <originalNspPath>         Original .nsp file path to input.");
            Console.WriteLine("  --current <currentNspPath>           Current .nsp file path to input.");
            Console.WriteLine("  --previous <previousNspPath>         Previous .nsp file path to input.");
#if false // 非公開
            Console.WriteLine("  --defragment                         Defragment patch.");
            Console.WriteLine("  --defragment-size <Size KiB>         Minimum matching size to defragment patch. If not specified, use 512KiB.");
            Console.WriteLine("  --optimize-size-strongly <Size>      Specify the shift width of the block to increase the possibility of matching.");
            Console.WriteLine("  --save-patch-build-log               Create patch build log files.");
            Console.WriteLine("  --with-delta                         Create patch along with delta.");
            Console.WriteLine("  --delta-meta-path                    Meta path to create delta.");
            Console.WriteLine("  --merge-delta                        Merge delta into patch.");
            Console.WriteLine("  --keep-intermediates                 Not to delete intermediate files.");
            Console.WriteLine("  --do-application-compaction          Create compacted application file.");
            Console.WriteLine("  --compaction-block-size <Size KiB>   Unit size for split original. If not specified, use 64KiB.");
            Console.WriteLine("  --compaction-erase-size <Size KiB>   Minimum erase size to split original. If not specified, use 1MiB.");
#endif
        }

        public OptionDescription[] GetOptionDescription()
        {
            OptionDescription[] options = new OptionDescription[]
                {
                    new OptionDescription(null, "-o", 1, (s) => { OutputFile = OptionUtil.GetOutputFilePath(OutputFile, s.First()); }),
                    OptionUtil.CreateFilePathOptionDescription("--meta", (path) => { MetaFilePath = path; }),
                    OptionUtil.CreateFilePathOptionDescription("--desc", (path) => { DescFilePath = path; }),
                    OptionUtil.CreateFilePathOptionDescription("--original", (path) => { OriginalFilePath = path; }),
                    OptionUtil.CreateFilePathOptionDescription("--current", (path) => { CurrentFilePath = path; }),
                    OptionUtil.CreateFilePathOptionDescription("--previous", (path) => { PreviousFilePath = path; }),
                    OptionUtil.CreateFilePathOptionDescription("--delta-meta-path", (path) => { DeltaMetaPath = path; }),
                    new OptionDescription("--minimum-matching-size", null, 1, (s) => { MinimumMatchingSize = int.Parse(s.First()) * 1024; }),
                    new OptionDescription("--defragment", null, 0, (s) => { NeedsDefragment = true; }),
                    new OptionDescription("--defragment-size", null, 1, (s) => { DefragmentSize = int.Parse(s.First()) * 1024; }),
                    new OptionDescription("--save-patch-build-log", null, 0, (s) => { IsSaveBuildLog = true; }),
                    new OptionDescription("--with-delta", null, 0, (s) => { WillOutputWithDelta = true; }),
                    new OptionDescription("--merge-delta", null, 0, (s) => { WillMergeDeltaContent = true; }),
                    new OptionDescription("--keep-intermediates", null, 0, (s) => { KeepIntermediates = true; }),
                    new OptionDescription("--data-cache-type", null, 1, (s) => { IndirectStorageSource.DataCacheType = (IndirectStorageSource.CacheType)Enum.Parse(typeof(IndirectStorageSource.CacheType), s.First()); }),
                    new OptionDescription("--cache-directory", null, 1, (s) => { CacheDirectory = OptionUtil.NormalizePath(s.First()); }),
                    new OptionDescription("--optimize-size-strongly", null, 1, (s) => { StronglyOptimizeSize = int.Parse(s.First()); }),
                    new OptionDescription("--do-application-compaction", null, 0, (s) => { NeedsSparse = true; }),
                    new OptionDescription("--compaction-block-size", null, 1, (s) => { SparseBlockSize = int.Parse(s.First()) * 1024; }),
                    new OptionDescription("--compaction-erase-size", null, 1, (s) => { SparseEraseSize = int.Parse(s.First()) * 1024; }),
                };
            return options;
        }
        public void VerifyArgument()
        {
            if(OriginalFilePath == OutputFile)
            {
                throw new InvalidOptionException("Original and ouptut are same file");
            }
            if (CurrentFilePath == OutputFile)
            {
                throw new InvalidOptionException("Current and output are same file");
            }
            if (PreviousFilePath == OutputFile)
            {
                throw new InvalidOptionException("Previous and output are same file");
            }
            if (!NeedsDefragment && 0 < DefragmentSize)
            {
                throw new InvalidOptionException("'--defragment' option should be specified.");
            }
            if (CacheDirectory == string.Empty)
            {
                throw new InvalidOptionException("matching cache path should be specified.");
            }
        }

        public void ParsePositionalArgument(string[] args)
        {
            if (string.IsNullOrEmpty(OriginalFilePath))
            {
                throw new InvalidOptionException("original archive file should be specified.");
            }

            if (string.IsNullOrEmpty(CurrentFilePath))
            {
                throw new InvalidOptionException("current archive file should be specified.");
            }

            IsNcaFile = Path.GetExtension(OriginalFilePath) == ".nca";

            if (IsNcaFile)
            {
                if (Path.GetExtension(CurrentFilePath) != ".nca")
                {
                    throw new InvalidOptionException("current archive file must be .nca file.");
                }
            }
            else
            {
                if (Path.GetExtension(OriginalFilePath) != ".nsp")
                {
                    throw new InvalidOptionException("original archive file must be .nca or .nsp file.");
                }

                if (Path.GetExtension(CurrentFilePath) != ".nsp")
                {
                    throw new InvalidOptionException("current archive file must be .nsp file.");
                }

                OutputFile = OutputFile.Replace(".nca", ".nsp");
            }
            VerifyArgument();
        }
    }

    internal class OptimizePatchOption : IOption
    {
        internal string OutputPath { get; set; }
        internal string PreviousFilePath { get; set; }
        internal string CurrentFilePath { get; set; }
        internal string DescFilePath { get; set; }
        internal bool IsSaveBuildLog { get; set; }
        internal bool NeedsDefragment { get; private set; }
        internal int DefragmentSize { get; private set; }
        internal string CacheDirectory { get; private set; }
        internal int StronglyOptimizeSize { get; private set; }

        internal OptimizePatchOption()
        {
            OutputPath = "./output.nsp";
            PreviousFilePath = null;
            CurrentFilePath = null;
            IsSaveBuildLog = false;
            NeedsDefragment = false;
            DefragmentSize = 0;
            CacheDirectory = null;
            StronglyOptimizeSize = IndirectStorageSource.BlockSize;
        }

        public void ShowSubCommandUsage(string baseName)
        {
            Console.WriteLine("optimizepatch: Create an optimized patch from a previous version (optimized) patch file and a next version patch file.");
            Console.WriteLine("Usage: {0} optimizepatch [-o <outputPath>] [--cache-directory <cacheDirPath>] --desc <descFilePath> --previous <previousNspPath> --current <currentNspPath>", baseName);
            Console.WriteLine(string.Empty);
            Console.WriteLine("Options:");
            Console.WriteLine("  -o <outputPath>                      File path to output. If directory path is specified, output to <outputPath>/output.nsp.");
            Console.WriteLine("                                       If omitted, output to ./output.nsp");
            Console.WriteLine("  --cache-directory <cacheDirPath>     Location of directory path to output matching cache.");
            Console.WriteLine("                                       With specifying this option, making patch is accelarated if appropriate cache is generated.");
            Console.WriteLine("  --desc <descFilePath>                Location of .desc file to current input. This cannot be omitted when create program content.");
            Console.WriteLine("  --previous <previousNspPath>         Previous file path to input.");
            Console.WriteLine("  --current <currentNspPath>           Current file path to input.");
#if false // 非公開
            Console.WriteLine("  --defragment                         Defragment patch.");
            Console.WriteLine("  --defragment-size <Size KiB>         Minimum matching size to defragment patch. If not specified, use 512KiB.");
            Console.WriteLine("  --optimize-size-strongly <Size>      Specify the shift width of the block to increase the possibility of matching.");
            Console.WriteLine("  --save-patch-build-log               Create patch build log files.");
#endif
        }

        public OptionDescription[] GetOptionDescription()
        {
            OptionDescription[] options = new OptionDescription[]
            {
                new OptionDescription(null, "-o", 1, (s) => { OutputPath = OptionUtil.GetOutputFilePath(OutputPath, s.First()); }),
                OptionUtil.CreateFilePathOptionDescription("--desc", (path) => { DescFilePath = path; }),
                OptionUtil.CreateFilePathOptionDescription("--previous", (path) => { PreviousFilePath = path; } ),
                OptionUtil.CreateFilePathOptionDescription("--current", (path) => { CurrentFilePath = path; }),
                new OptionDescription("--defragment", null, 0, (s) => { NeedsDefragment = true; }),
                new OptionDescription("--defragment-size", null, 1, (s) => { DefragmentSize = int.Parse(s.First()) * 1024; }),
                new OptionDescription("--save-patch-build-log", null, 0, (s) => { IsSaveBuildLog = true; }),
                new OptionDescription("--cache-directory", null, 1, (s) => { CacheDirectory = OptionUtil.NormalizePath(s.First()); }),
                new OptionDescription("--optimize-size-strongly", null, 1, (s) => { StronglyOptimizeSize = int.Parse(s.First()); }),
            };
            return options;
        }

        public void ParsePositionalArgument(string[] args)
        {
            if (string.IsNullOrEmpty(PreviousFilePath))
            {
                throw new InvalidOptionException("previous archive file should be specified.");
            }

            if (string.IsNullOrEmpty(CurrentFilePath))
            {
                throw new InvalidOptionException("current archive file should be specified.");
            }

            {
                if (Path.GetExtension(PreviousFilePath) != ".nsp")
                {
                    throw new InvalidOptionException("previous archive file must be .nsp file.");
                }

                if (Path.GetExtension(CurrentFilePath) != ".nsp")
                {
                    throw new InvalidOptionException("current archive file must be .nsp file.");
                }

                if (Path.GetExtension(OutputPath) != ".nsp")
                {
                    throw new InvalidOptionException("output archive file must be .nsp file.");
                }
            }

            if (!NeedsDefragment && 0 < DefragmentSize)
            {
                throw new InvalidOptionException("'--defragment' option should be specified.");
            }

            if (CacheDirectory == string.Empty)
            {
                throw new InvalidOptionException("matching cache path should be specified.");
            }
        }
    }

    internal class MakeDeltaOption : IOption
    {
        internal enum OutputType
        {
            Binary,
            DeltaNsp,
        }

        internal string OutputFile { get; set; }
        internal OutputType SubType { get; private set; }
        internal string MetaFilePath { get; set; }
        internal string SourceFilePath { get; set; }
        internal string DestinationFilePath { get; set; }
        internal bool WillSaveBuildLog { get; private set; }
        internal long DeltaCommandSizeMax { get; private set; }
        internal bool WillSaveDelta { get; private set; }

        internal MakeDeltaOption(OutputType subType)
        {
            SubType = subType;
            if (subType == OutputType.Binary)
            {
                OutputFile = "./output.program.bdiff";
            }
            else
            {
                OutputFile = "./output.nsp";
            }
            MetaFilePath = null;
            SourceFilePath = null;
            DestinationFilePath = null;
            DeltaCommandSizeMax = DeltaArchive.DeltaCommandSizeMax;
            WillSaveDelta = false;
            WillSaveBuildLog = false;
        }

        public void ShowSubCommandUsage(string baseName)
        {
            if (SubType == OutputType.Binary)
            {
                Console.WriteLine("bdiff: Create binary patch from .nsp files.");
                Console.WriteLine("Usage: {0} bdiff [-o <outputPath>] --source <sourceNspPath> --destination <destinationNspPath>", baseName);
            }
            else
            {
                Console.WriteLine("MakeDelta: make delta for updating patch");
                Console.WriteLine("Usage: {0} makedelta [-o <outputPath>] [--meta <metaFilePath>] --source <sourceNspPath> --destination <destinationNspPath>", baseName);
            }

            Console.WriteLine(string.Empty);
            Console.WriteLine("Options:");
            Console.WriteLine("  -o <outputPath>                      File path to output. If directory path is specified, output to <outputPath>/output.nsp.");
            Console.WriteLine("                                       If omitted, output to ./output.nsp");
            Console.WriteLine("  --meta <metaFilePath>                Location of .nmeta file to input.");
            Console.WriteLine("  --source <sourceNspPath>             Source nsp path to input.");
            Console.WriteLine("  --destination <destinationNspPath>   Destination nsp path to input.");
        }

        public OptionDescription[] GetOptionDescription()
        {
            OptionDescription[] options = new OptionDescription[]
                {
                    new OptionDescription(null, "-o", 1, (s) => { OutputFile = OptionUtil.GetOutputFilePath(OutputFile, s.First()); }),
                    OptionUtil.CreateFilePathOptionDescription("--meta", (path) => { MetaFilePath = path; }),
                    OptionUtil.CreateFilePathOptionDescription("--source", (path) => { SourceFilePath = path; }),
                    OptionUtil.CreateFilePathOptionDescription("--destination", (path) => { DestinationFilePath = path; }),
                    new OptionDescription("--save-delta", null, 0, (s) => { WillSaveDelta = true; }),
                    new OptionDescription("--command-size-max", null, 1, (s) => { DeltaCommandSizeMax = long.Parse(s.First()); }),
                    new OptionDescription("--save-build-log", null, 0, (s) => { WillSaveBuildLog = true; }),
                };
            return options;
        }

        public void ParsePositionalArgument(string[] args)
        {
            if (args.Length > 0)
            {
                throw new InvalidOptionException("too many arguments for makedelta subcommand.");
            }

            if (string.IsNullOrEmpty(SourceFilePath))
            {
                throw new InvalidOptionException("source nsp file should be specified.");
            }

            if (string.IsNullOrEmpty(DestinationFilePath))
            {
                throw new InvalidOptionException("destinatino nsp file should be specified.");
            }
        }
    }

    internal class MergeDeltaContentOption : IOption
    {
        internal string OutputFile { get; set; }
        internal string PatchFilePath { get; set; }
        internal string MergingFilePath { get; set; }

        internal MergeDeltaContentOption()
        {
            OutputFile = "./output.nsp";
            PatchFilePath = null;
            MergingFilePath = null;
        }

        public void ShowSubCommandUsage(string baseName)
        {
            Console.WriteLine("MergeDeltaContent: Merge delta contents into patch");
            Console.WriteLine("Usage: {0} merge-deltacontent [-o <outputPath>] --patch <patchNspPath> <inputNspPath>", baseName);
            Console.WriteLine(string.Empty);
            Console.WriteLine("Options:");
            Console.WriteLine("  -o <outputPath>                      File path to output. If directory path is specified, output to <outputPath>/output.nsp.");
            Console.WriteLine("                                       If omitted, output to ./output.nsp");
            Console.WriteLine("  --patch <patchNspPath>               Patch nsp path to input.");
        }

        public OptionDescription[] GetOptionDescription()
        {
            OptionDescription[] options = new OptionDescription[]
                {
                    new OptionDescription(null, "-o", 1, (s) => { OutputFile = OptionUtil.GetOutputFilePath(OutputFile, s.First()); }),
                    OptionUtil.CreateFilePathOptionDescription("--patch", (path) => { PatchFilePath = path; }),
                };
            return options;
        }

        public void ParsePositionalArgument(string[] args)
        {
            if (string.IsNullOrEmpty(PatchFilePath))
            {
                throw new InvalidOptionException("source nsp file should be specified.");
            }

            if (args.Length != 1)
            {
                throw new ArgumentException("merging nsp is required");
            }
            MergingFilePath = args[0];
        }
    }

    internal class ValidateMetaFileOption : IOption
    {
        internal string MetaFilePath { get; set; }
        internal string MetaType { get; set; }

        internal ValidateMetaFileOption()
        {
            MetaFilePath = null;
            MetaType = null;
        }

        public void ShowSubCommandUsage(string baseName)
        {
            Console.WriteLine("validatemeta: Validate meta file");
            Console.WriteLine("Usage: {0} validatemeta --meta <metaFilePath> --type <contentMetaType>", baseName);
            Console.WriteLine(string.Empty);
            Console.WriteLine("Options:");
            Console.WriteLine("  --meta <metaFilePath>                          Location of .nmeta file to input.");
            Console.WriteLine("  --type <contentMetaType>                       Content Meta Type to create package with the meta file.");
        }

        public OptionDescription[] GetOptionDescription()
        {
            OptionDescription[] options = new OptionDescription[]
                {
                    OptionUtil.CreateFilePathOptionDescription("--meta", (path) => { MetaFilePath = path; }),
                    new OptionDescription("--type", null, 1, (s) => { MetaType = s.First(); }),
                };
            return options;
        }

        public void ParsePositionalArgument(string[] args)
        {
            if (string.IsNullOrEmpty(MetaFilePath))
            {
                throw new InvalidOptionException("--meta option should be specified.");
            }
            if (string.IsNullOrEmpty(MetaType))
            {
                throw new InvalidOptionException("--type option should be specified.");
            }
        }
    }

    internal class ConvertIconOption : IOption
    {
        internal string InputPath { get; set; }
        internal string outputDirectory { get; set; }
        internal UInt32 NxIconMaxSize { get; set; }

        internal ConvertIconOption()
        {
            outputDirectory = ".";
            NxIconMaxSize = NintendoSubmissionPackageContentInfo.DefaultNxIconMaxSize;
        }

        public void ShowSubCommandUsage(string baseName)
        {
            Console.WriteLine("converticon: convert icon image");
            Console.WriteLine("Usage: {0} converticon <iconFilePath> [-o <outputDirectory>]", baseName);
            Console.WriteLine(string.Empty);
            Console.WriteLine("Options:");
            Console.WriteLine("  -o <outputDirectory>              Directory to output.");
        }

        public OptionDescription[] GetOptionDescription()
        {
            OptionDescription[] options = new OptionDescription[]
                {
                    new OptionDescription(null, "-o", 1, (s) => { outputDirectory = OptionUtil.GetOutputFilePath(outputDirectory, s.First()); }),
                    new OptionDescription("--nx-icon-max-size", null, 32, (s) =>
                    {
                        NxIconMaxSize = Convert.ToUInt32(s.First());
                    }),
                };
            return options;
        }

        public void ParsePositionalArgument(string[] args)
        {
            if (args.Length == 0)
            {
                throw new InvalidOptionException("icon image file path for input required.");
            }

            InputPath = args[0];
        }
    }

    internal class ShowVersionOption : IOption
    {
        internal ShowVersionOption()
        {
        }

        public void ShowSubCommandUsage(string baseName)
        {
            Console.WriteLine("showversion: Output tool version to standard output");
            Console.WriteLine("Usage: {0} showversion", baseName);
        }

        public OptionDescription[] GetOptionDescription()
        {
            return null;
        }

        public void ParsePositionalArgument(string[] args)
        {
        }
    }

    internal class GetAllXmlModelsOption : IOption
    {
        internal GetAllXmlModelsOption()
        {
        }

        public void ShowSubCommandUsage(string baseName)
        {
            Console.WriteLine("get-all-xml-models: Output all xml models");
            Console.WriteLine("Usage: {0} get-all-xml-models", baseName);
        }

        public OptionDescription[] GetOptionDescription()
        {
            return null;
        }

        public void ParsePositionalArgument(string[] args)
        {
        }
    }

    internal class GetNspPropertyOption : IOption
    {
        internal string InputNspFile { get; set; }
        internal string Format { get; set; }

        internal const string YamlFormat = "Yaml";
        internal const string XmlFormat = "Xml";

        internal bool ShowApplyDifference { get; set; }

        public GetNspPropertyOption()
        {
            Format = YamlFormat;
        }

        public void ShowSubCommandUsage(string baseName)
        {
            Console.WriteLine("getproperty: Output property of input Nintendo Submission Package file to standard output.");
            Console.WriteLine("Usage: {0} getproperty <inputNspPath>", baseName);
            Console.WriteLine(string.Empty);
            Console.WriteLine("Options:");
            Console.WriteLine("  --xml    Output property as xml format. If omitted, output as yaml format.");
            Console.WriteLine("  --show-apply-difference    Output property only about whether or not difference of patch is used for the update.");
        }

        public OptionDescription[] GetOptionDescription()
        {
            OptionDescription[] options = new OptionDescription[]
                {
                    new OptionDescription("--xml", null, 0, (s) => { Format = XmlFormat; }),
                    new OptionDescription("--show-apply-difference", null, 0, (s) => { ShowApplyDifference = true; }),
                };
            return options;
        }

        public void ParsePositionalArgument(string[] args)
        {
           if (args.Length != 1)
           {
               throw new InvalidOptionException("input nsp file must be specified for getproperty subcommand.");
           }

           InputNspFile = OptionUtil.CheckAndNormalizeFilePath(args[0], "arg[0]");
        }
    }

    internal class GetUnpublishableErrorOption : IOption
    {
        internal string InputNspFile { get; set; }
        internal string Format { get; set; }

        internal const string TextFormat = "Text";
        internal const string XmlFormat = "Xml";

        public GetUnpublishableErrorOption()
        {
            Format = TextFormat;
        }

        public void ShowSubCommandUsage(string baseName)
        {
            Console.WriteLine("get-unpublishable-error: Output unpublishable error of input Nintendo Submission Package file to standard output.");
            Console.WriteLine("Usage: {0} get-unpublishable-error <inputNspPath>", baseName);
            Console.WriteLine(string.Empty);
            Console.WriteLine("Options:");
            Console.WriteLine("  --xml    Output unpublishable error as xml format. If omitted, output as text format.");
        }

        public OptionDescription[] GetOptionDescription()
        {
            OptionDescription[] options = new OptionDescription[]
                {
                    new OptionDescription("--xml", null, 0, (s) => { Format = XmlFormat; }),
                };
            return options;
        }

        public void ParsePositionalArgument(string[] args)
        {
            if (args.Length != 1)
            {
                throw new InvalidOptionException("input nsp file must be specified for get-unpublishable-error subcommand.");
            }

            InputNspFile = OptionUtil.CheckAndNormalizeFilePath(args[0], "arg[0]");

            bool isNsp = Path.GetExtension(InputNspFile) == ".nsp";
            if (!isNsp)
            {
                throw new InvalidOptionException("input archive file must be .nsp file.");
            }
        }
    }

    internal class EncryptXciOption : IOption
    {
        internal string InputXciFile { get; set; }
        internal string OutputDirectory { get; set; }
        internal bool UseKeyForRepairTool { get; set; }

        public EncryptXciOption()
        {
            OutputDirectory = ".";
        }

        public void ShowSubCommandUsage(string baseName)
        {
            Console.WriteLine("encryptxci: Encrypt xci and output xcie file.");
            Console.WriteLine("Usage: {0} encryptxci [-o <outputDirectory>] [--for-repair] <inputXciPath>", baseName);
            Console.WriteLine(string.Empty);
            Console.WriteLine("Options:");
            Console.WriteLine("  -o <outputDirectory>    Directory to output.");
            Console.WriteLine("  --for-repair            Create xcir file instead of xcie file.");
        }

        public OptionDescription[] GetOptionDescription()
        {
            OptionDescription[] options = new OptionDescription[]
                {
                    new OptionDescription(null, "-o", 1, (s) => { OutputDirectory = OptionUtil.GetOutputFilePath(OutputDirectory, s.First()); }),
                    new OptionDescription("--for-repair", null, 0, (s) => { UseKeyForRepairTool = true; }),
                };
            return options;
        }

        public void ParsePositionalArgument(string[] args)
        {
            if (args.Length != 1)
            {
                throw new InvalidOptionException("input xci file must be specified for encryptxci subcommand.");
            }

            InputXciFile = OptionUtil.CheckAndNormalizeFilePath(args[0], "arg[0]");
            OptionUtil.CheckExtension(InputXciFile, ".xci");
        }
    }

    internal class ConvertAddOnContentForCardOption : IOption
    {
        internal List<string> InputNspFiles { get; set; }
        internal string InputBaseNspFile { get; set; }
        internal string OutputFile { get; set; }
        internal int? CardSize { get; set; }
        internal int? CardClockRate { get; set; }
        internal byte? CardLaunchFlags { get; set; }

        private string NspExtension = ".nsp";

        public ConvertAddOnContentForCardOption()
        {
            InputNspFiles = new List<string>();
            OutputFile = "./output.nsp";
        }

        public void ShowSubCommandUsage(string baseName)
        {
            Console.WriteLine("convertaoc: Convert AddOnContent nsp to be available on card.");
            Console.WriteLine("Usage: {0} convertaoc [-o <outputDirectory>] --base <inputBaseAppNspPath> [--cardsize <size> --cardclockrate <clockrate>] <inputAocNspPath1> [<inputAocNspPath2> ...]", baseName);
            Console.WriteLine(string.Empty);
            Console.WriteLine("Options:");
            Console.WriteLine("  -o <outputPath>         File path to output. If directory path is specified, output to <outputPath>/output.nsp.");
            Console.WriteLine("                          If omitted, output to ./output.nsp");
            Console.WriteLine("  --base                  Location of base Application or Patch nsp, which should be packaged together on OnCardAoc.");
            Console.WriteLine("  --cardsize              Specify card size. If ommited, it is set automatically so all contents can be put on card.");
            Console.WriteLine("  --cardclockrate         Specify card clock rate. If ommited, it is set automatically according to card size.");
        }

        public OptionDescription[] GetOptionDescription()
        {
            OptionDescription[] options = new OptionDescription[]
            {
                new OptionDescription(null, "-o", 1, (s) => { OutputFile = OptionUtil.GetOutputFilePath(OutputFile, s.First()); }),
                OptionUtil.CreateFilePathOptionDescription("--base", NspExtension, (path) => { InputBaseNspFile = path; }),
                new OptionDescription(null, "--cardsize", 1, (s) => { CardSize = Convert.ToInt32(s.First()); }),
                new OptionDescription(null, "--cardclockrate", 1, (s) => { CardClockRate = Convert.ToInt32(s.First()); }),
                new OptionDescription(null, "--cardlaunchflags", 1, (s) => { CardLaunchFlags = Convert.ToByte(s.First()); }),
            };
            return options;
        }

        public void ParsePositionalArgument(string[] args)
        {
            if (InputBaseNspFile == null)
            {
                throw new InvalidOptionException("--base option must be specified.");
            }
            for (int i = 0; i < args.Length; i++)
            {
                if (Path.GetExtension(args[i]) != NspExtension)
                {
                    throw new InvalidOptionException("input archive file must be .nsp file.");
                }
                InputNspFiles.Add(OptionUtil.CheckAndNormalizeFilePath(args[i], string.Format("arg[{0}]", i)));
            }
        }
    }

    internal class CreateMultiProgramApplicationOption : IOption
    {
        internal List<string> InputNspFiles { get; set; }
        internal string OutputFile { get; set; }
        internal int? CardSize { get; set; }
        internal int? CardClockRate { get; set; }
        internal byte? CardLaunchFlags { get; set; }
        internal string IgnoreErrorUnpublishable { get; private set; }

        private string NspExtension = ".nsp";

        public CreateMultiProgramApplicationOption()
        {
            InputNspFiles = new List<string>();
            OutputFile = "./output.nsp";
            IgnoreErrorUnpublishable = null;
        }

        public void ShowSubCommandUsage(string baseName)
        {
            Console.WriteLine("mergeapplication: Merge indexed application nsp files into a multi-program application nsp file.");
            Console.WriteLine("Usage: {0} mergeapplication [-o <outputFilePath>] [--nsp-list-file <textFilePath>] <inputNspPath1> <inputNspPath2> [<inputNspPath3> ...]", baseName);
            Console.WriteLine(string.Empty);
            Console.WriteLine("Options:");
            Console.WriteLine("  -o <outputPath>                File path to output. If directory path is specified, output to <outputPath>/output.nsp.");
            Console.WriteLine("                                 If omitted, output to ./output.nsp");
            Console.WriteLine("  --cardsize                     Specify card size. If ommited, it is set automatically so all contents can be put on card.");
            Console.WriteLine("  --cardclockrate                Specify card clock rate. If ommited, it is set automatically according to card size.");
            Console.WriteLine("  --nsp-list-file                File path of text file whilch lists nsp files. if specified, inputNspPaths are ignored.");
#if false // 非公開
            Console.WriteLine("  --ignore-unpublishable-error [ignoreListPath] Ignore unpublishable error.");
#endif
        }

        public OptionDescription[] GetOptionDescription()
        {
            OptionDescription[] options = new OptionDescription[]
            {
                new OptionDescription(null, "-o", 1, (s) => { OutputFile = OptionUtil.GetOutputFilePath(OutputFile, s.First()); }),
                new OptionDescription(null, "--cardsize", 1, (s) => { CardSize = Convert.ToInt32(s.First()); }),
                new OptionDescription(null, "--cardclockrate", 1, (s) => { CardClockRate = Convert.ToInt32(s.First()); }),
                new OptionDescription(null, "--cardlaunchflags", 1, (s) => { CardLaunchFlags = Convert.ToByte(s.First()); }),
                new OptionDescription(null, "--nsp-list-file", 1, (s) =>
                {
                    InputNspFiles.Clear(); // 念のため
                    var filePath = OptionUtil.CheckAndNormalizeFilePath(s.First(), "--nsp-list-file");
                    using (var fs = File.OpenText(filePath))
                    {
                        while (!fs.EndOfStream)
                        {
                            var line = fs.ReadLine().Trim();
                            if (line.Length > 0)
                            {
                                InputNspFiles.Add(OptionUtil.CheckAndNormalizeFilePath(line, filePath));
                            }
                        }
                    }
                }),
                new OptionDescription("--ignore-unpublishable-error", null, -1, (s) =>
                {
                    if (s.Count > 1)
                    {
                        throw new InvalidOptionException("too many arguments for --ignore-unpublishable-error option.");
                    }
                    IgnoreErrorUnpublishable = (s.Count() > 0) ? OptionUtil.CheckAndNormalizeFilePath(s.First(), "--ignore-unpublishable-error") : string.Empty;
                }),
            };
            return options;
        }

        public void ParsePositionalArgument(string[] args)
        {
            if (args.Length < 2)
            {
                throw new InvalidOptionException("At least two .nsp file should be specified.");
            }

            if (InputNspFiles.Count == 0)
            {
                for (int i = 0; i < args.Length; i++)
                {
                    if (Path.GetExtension(args[i]) != NspExtension)
                    {
                        throw new InvalidOptionException("input archive file must be .nsp file.");
                    }
                    InputNspFiles.Add(OptionUtil.CheckAndNormalizeFilePath(args[i], string.Format("arg[{0}]", i)));
                }
            }
        }
    }

    internal class CreateMultiApplicationCardIndicatorOption : IOption
    {
        internal List<string> InputNspFiles { get; set; }
        internal string OutputFile { get; set; }
        internal ulong Id { get; set; }
        internal int? CardSize { get; set; }
        internal int? CardClockRate { get; set; }
        internal byte? CardLaunchFlags { get; set; }
        internal bool ErrorUnpublishable { get; private set; }
        internal bool ShowUnpublishableErroFormatXml { get; private set; }
        internal string IgnoreErrorUnpublishable { get; private set; }

        private string NspExtension = ".nsp";

        public CreateMultiApplicationCardIndicatorOption()
        {
            InputNspFiles = new List<string>();
            OutputFile = "./output.nsp";
            ErrorUnpublishable = false;
            ShowUnpublishableErroFormatXml = false;
            IgnoreErrorUnpublishable = null;
        }

        public void ShowSubCommandUsage(string baseName)
        {
            Console.WriteLine("bundleup: Create multi-application card indicator nsp file.");
            Console.WriteLine("Usage: {0} bundleup [-o <outputFilePath>] [--cardsize <size> --cardclockrate <clockrate>] --id <MultiApplicationCardId>  [--nsp-list-file <textFilePath>] <inputNspPath1> <inputNspPath2> [<inputNspPath3> ...] [--error-unpublishable [--xml]]", baseName);
            Console.WriteLine(string.Empty);
            Console.WriteLine("Options:");
            Console.WriteLine("  -o <outputPath>                File path to output. If directory path is specified, output to <outputPath>/output.nsp.");
            Console.WriteLine("                                 If omitted, output to ./output.nsp");
            Console.WriteLine("  --id <MultiApplicationCardId>  ID published for this multi-application card.");
            Console.WriteLine("  --cardsize                     Specify card size. If ommited, it is set automatically so all contents can be put on card.");
            Console.WriteLine("  --cardclockrate                Specify card clock rate. If ommited, it is set automatically according to card size.");
            Console.WriteLine("  --nsp-list-file                File path of text file whilch lists nsp files. if specified, inputNspPaths are ignored.");
            Console.WriteLine("  --error-unpublishable [--xml]  Throw an exception if the Nintendo Submission Package is not satisfied with conditions to publish it.");
#if false // 非公開
            Console.WriteLine("  --ignore-unpublishable-error [ignoreListPath]  Ignore unpublishable error.");
#endif
        }

        public OptionDescription[] GetOptionDescription()
        {
            OptionDescription[] options = new OptionDescription[]
            {
                new OptionDescription(null, "-o", 1, (s) => { OutputFile = OptionUtil.GetOutputFilePath(OutputFile, s.First()); }),
                new OptionDescription(null, "--id", 1, (s) => { Id = Convert.ToUInt64(s.First(), 16); }),
                new OptionDescription(null, "--cardsize", 1, (s) => { CardSize = Convert.ToInt32(s.First()); }),
                new OptionDescription(null, "--cardclockrate", 1, (s) => { CardClockRate = Convert.ToInt32(s.First()); }),
                new OptionDescription(null, "--cardlaunchflags", 1, (s) => { CardLaunchFlags = Convert.ToByte(s.First()); }),
                new OptionDescription(null, "--nsp-list-file", 1, (s) =>
                {
                    InputNspFiles.Clear(); // 念のため
                    var filePath = OptionUtil.CheckAndNormalizeFilePath(s.First(), "--nsp-list-file");
                    using (var fs = File.OpenText(filePath))
                    {
                        while (!fs.EndOfStream)
                        {
                            var line = fs.ReadLine().Trim();
                            if (line.Length > 0)
                            {
                                InputNspFiles.Add(OptionUtil.CheckAndNormalizeFilePath(line, filePath));
                            }
                        }
                    }
                }),
                new OptionDescription(null, "--error-unpublishable", 0, (s) => { ErrorUnpublishable = true; }),
                new OptionDescription(null, "--xml", 0, (s) => { ShowUnpublishableErroFormatXml = true; }),
                new OptionDescription("--ignore-unpublishable-error", null, -1, (s) =>
                    {
                        if (s.Count > 1)
                        {
                            throw new InvalidOptionException("too many arguments for --ignore-unpublishable-error option.");
                        }
                        IgnoreErrorUnpublishable = (s.Count() > 0) ? OptionUtil.CheckAndNormalizeFilePath(s.First(), "--ignore-unpublishable-error") : string.Empty;
                    }),
            };
            return options;
        }

        public void ParsePositionalArgument(string[] args)
        {
            if (Id == 0)
            {
                throw new InvalidOptionException("--id option must be specified.");
            }

            if (InputNspFiles.Count == 0)
            {
                for (int i = 0; i < args.Length; i++)
                {
                    if (Path.GetExtension(args[i]) != NspExtension)
                    {
                        throw new InvalidOptionException("input archive file must be .nsp file.");
                    }
                    InputNspFiles.Add(OptionUtil.CheckAndNormalizeFilePath(args[i], string.Format("arg[{0}]", i)));
                }
            }
        }
    }

    internal class SparsifyNspOption : IOption
    {
        internal string OutputOriginalFile { get; set; } = "./output.nsp";
        internal string OutputPatchFile { get; set; } = "./output.patch.nsp";
        internal string OriginalFilePath { get; set; }
        internal string PatchFilePath { get; set; }
        internal string PreviousFilePath { get; set; }
        internal int BlockSize { get; private set; }
        internal int MinimumEraseSize { get; private set; }
        internal bool IsSaveBuildLog { get; set; }

        public void ShowSubCommandUsage(string baseName)
        {
            Console.WriteLine("sparsifynsp: Create compacted application file from Nintendo Submission Packages.");
            Console.WriteLine("Usage: {0} sparsifynsp [-o <outputOriginalPath>] [-op <outputPatchPath>] --original <originalNspPath> --patch <patchNspPath> [--previous <previousNspPath>]", baseName);
            Console.WriteLine(string.Empty);
            Console.WriteLine("Options:");
            Console.WriteLine("  -o <outputOriginalPath>              File path to output original.");
            Console.WriteLine("  -op <outputPatchPath>                File path to output patch.");
            Console.WriteLine("                                       If omitted, output to ./output.nsp");
            Console.WriteLine("  --original <originalNspPath>         Original .nsp file path to input.");
            Console.WriteLine("  --patch <patchNspPath>               Patch .nsp file path to input.");
            Console.WriteLine("  --previous <previousNspPath>         Previous patch .nsp file path to input.");
            Console.WriteLine("  --block-size <Size KiB>              Unit size for split original. If not specified, use 64KiB.");
            Console.WriteLine("  --minimum-erase-size <Size KiB>      Minimum erase size to split original. If not specified, use 1MiB.");
            Console.WriteLine("  --save-patch-build-log               Create patch build log files.");
        }

        public OptionDescription[] GetOptionDescription()
        {
            OptionDescription[] options = new OptionDescription[]
            {
                new OptionDescription(null, "-o", 1, (s) => { OutputOriginalFile = OptionUtil.GetOutputFilePath(OutputOriginalFile, s.First()); }),
                new OptionDescription(null, "-op", 1, (s) => { OutputPatchFile = OptionUtil.GetOutputFilePath(OutputPatchFile, s.First()); }),
                OptionUtil.CreateFilePathOptionDescription("--original", (path) => { OriginalFilePath = path; }),
                OptionUtil.CreateFilePathOptionDescription("--patch", (path) => { PatchFilePath = path; }),
                OptionUtil.CreateFilePathOptionDescription("--previous", (path) => { PreviousFilePath = path; }),
                new OptionDescription("--block-size", null, 1, (s) => { BlockSize = int.Parse(s.First()) * 1024; }),
                new OptionDescription("--minimum-erase-size", null, 1, (s) => { MinimumEraseSize = int.Parse(s.First()) * 1024; }),
                new OptionDescription("--save-patch-build-log", null, 0, (s) => { IsSaveBuildLog = true; }),
            };
            return options;
        }

        public void ParsePositionalArgument(string[] args)
        {
            if (string.IsNullOrEmpty(OriginalFilePath))
            {
                throw new InvalidOptionException("original archive file should be specified.");
            }
            if (Path.GetExtension(OriginalFilePath) != ".nsp")
            {
                throw new InvalidOptionException("original archive file must be .nsp file.");
            }

            if (string.IsNullOrEmpty(PatchFilePath))
            {
                throw new InvalidOptionException("patch archive file should be specified.");
            }
            if (Path.GetExtension(PatchFilePath) != ".nsp")
            {
                throw new InvalidOptionException("patch archive file must be .nsp file.");
            }

            if (!string.IsNullOrEmpty(PreviousFilePath) && Path.GetExtension(PreviousFilePath) != ".nsp")
            {
                throw new InvalidOptionException("previous archive file must be .nsp file.");
            }

            if (OriginalFilePath == PatchFilePath || OriginalFilePath == PreviousFilePath || PatchFilePath == PreviousFilePath)
            {
                throw new InvalidOptionException("input files are same file.");
            }
            if (OutputOriginalFile == OutputPatchFile)
            {
                throw new InvalidOptionException("output files are same file.");
            }
            if (OriginalFilePath == OutputOriginalFile || OriginalFilePath == OutputPatchFile)
            {
                throw new InvalidOptionException("Original and output are same file.");
            }
            if (PatchFilePath == OutputOriginalFile || PatchFilePath == OutputPatchFile)
            {
                throw new InvalidOptionException("Patch and output are same file.");
            }
            if (PreviousFilePath == OutputOriginalFile || PreviousFilePath == OutputPatchFile)
            {
                throw new InvalidOptionException("Previous and output are same file.");
            }

            if (OutputPatchFile == "./output.patch.nsp")
            {
                OutputPatchFile = Path.ChangeExtension(OutputOriginalFile, ".patch.nsp");
            }
        }
    }

    internal class Option
    {
        internal enum SubCommandType
        {
            None = 0,
            CreateFs,
            CreateNca,
            CreateNsp,
            CreateNspMeta,
            CreateNspd,
            CreateNacp,
            MakePatch,
            OptimizePatch,
            BinaryDiff,
            MakeDelta,
            MergeDeltaContent,
            ExtractNsp,
            ExtractNsave,
            MergeNsp,
            Extract,
            Replace,
            ReplaceNspMeta,
            List,
            DiffPatch,
            Verify,
            ProdEncryption,
            ProdEncryptionPatch,
            ValidateMetaFile,
            ConvertIcon,
            ShowVersion,
            GetAllXmlModels,
            GetNspProperty,
            GetUnpublishableError,
            EncryptXci,
            SplitNsp,
            ConvertAddOnContentForCard,
            CreateMultiApplicationCardIndicator,
            CreateMultiProgramApplication,
            SparsifyNsp,
            Help,
        }

        internal static SubCommandType GetSubCommandType(string str)
        {
            switch (str)
            {
                case "createfs": return SubCommandType.CreateFs;
                case "createnca": return SubCommandType.CreateNca;
                case "creatensp": return SubCommandType.CreateNsp;
                case "createnspmeta": return SubCommandType.CreateNspMeta;
                case "createnspd": return SubCommandType.CreateNspd;
                case "createnacp": return SubCommandType.CreateNacp;
                case "makepatch": return SubCommandType.MakePatch;
                case "optimizepatch": return SubCommandType.OptimizePatch;
                case "bdiff": return SubCommandType.BinaryDiff;
                case "makedelta": return SubCommandType.MakeDelta;
                case "merge-deltacontent": return SubCommandType.MergeDeltaContent;
                case "extractnsp": return SubCommandType.ExtractNsp;
                case "extractnsave": return SubCommandType.ExtractNsave;
                case "mergensp": return SubCommandType.MergeNsp;
                case "extract": return SubCommandType.Extract;
                case "replace": return SubCommandType.Replace;
                case "replacenspmeta": return SubCommandType.ReplaceNspMeta;
                case "list": return SubCommandType.List;
                case "diffpatch": return SubCommandType.DiffPatch;
                case "verify": return SubCommandType.Verify;
                case "prodencryption": return SubCommandType.ProdEncryption;
                case "prodencryption-patch": return SubCommandType.ProdEncryptionPatch;
                case "validatemeta": return SubCommandType.ValidateMetaFile;
                case "converticon": return SubCommandType.ConvertIcon;
                case "showversion": return SubCommandType.ShowVersion;
                case "get-all-xml-models": return SubCommandType.GetAllXmlModels;
                case "getproperty":
                case "getnspproperty": return SubCommandType.GetNspProperty;
                case "get-unpublishable-error": return SubCommandType.GetUnpublishableError;
                case "encryptxci": return SubCommandType.EncryptXci;
                case "splitnsp": return SubCommandType.SplitNsp;
                case "convertaoc": return SubCommandType.ConvertAddOnContentForCard;
                case "bundleup": return SubCommandType.CreateMultiApplicationCardIndicator;
                case "mergeapplication": return SubCommandType.CreateMultiProgramApplication;
                case "sparsifynsp": return SubCommandType.SparsifyNsp;
                case "help":      return SubCommandType.Help;
                default:
                    throw new InvalidOptionException(string.Format("subcommand {0} is not implemented.", str));
            }
        }

        internal bool IsVerbose { get { return None.IsVerbose; } }
        internal bool IsShowUsage { get { return None.IsShowUsage; } }
        internal bool IsUtf8 { get { return None.IsUtf8; } }

        internal NoneOption None { get; private set; }
        internal CreateFsOption CreateFs { get; private set; }
        internal CreateNcaOption CreateNca { get; private set; }
        internal CreateNspOption CreateNsp { get; private set; }
        internal CreateNspMetaOption CreateNspMeta { get; private set; }
        internal CreateNspdOption CreateNspd { get; private set; }
        internal CreateNacpOption CreateNacp { get; private set; }
        internal MakePatchOption MakePatch { get; private set; }
        internal OptimizePatchOption OptimizePatch { get; private set; }
        internal MakeDeltaOption MakeDelta { get; private set; }
        internal MergeDeltaContentOption MergeDeltaContent { get; private set; }
        internal ExtractOption Extract { get; private set; }
        internal ExtractNspOption ExtractNsp { get; private set; }
        internal ExtractNsaveOption ExtractNsave { get; private set; }
        internal MergeNspOption MergeNsp { get; private set; }
        internal ReplaceOption Replace { get; private set; }
        internal ReplaceNspMetaOption ReplaceNspMeta { get; private set; }
        internal ListOption List { get; private set; }
        internal DiffPatchOption DiffPatch { get; private set; }
        internal VerifyOption Verify { get; private set; }
        internal ProdEncryptionOption ProdEncryption { get; private set; }
        internal ProdEncryptionPatchOption ProdEncryptionPatch { get; private set; }
        internal HelpOption Help { get; private set; }
        internal ShowVersionOption ShowVersion { get; private set; }
        internal GetAllXmlModelsOption GetAllXmlModels { get; private set; }
        internal GetNspPropertyOption GetNspProperty { get; private set; }
        internal GetUnpublishableErrorOption GetUnpublishableError { get; private set; }
        internal AuthoringConfiguration Config { get { return None.Config; } }
        internal ValidateMetaFileOption ValidateMetaFile { get; private set; }
        internal ConvertIconOption ConvertIcon { get; private set; }
        internal EncryptXciOption EncryptXci { get; private set; }
        internal SplitNspOption SplitNsp { get; private set; }
        internal ConvertAddOnContentForCardOption ConvertAddOnContentForCard { get; private set; }
        internal CreateMultiApplicationCardIndicatorOption CreateMultiApplicationCardIndicator { get; private set; }
        internal CreateMultiProgramApplicationOption CreateMultiProgramApplication { get; private set; }
        internal SparsifyNspOption SparsifyNsp { get; private set; }

        private SubCommandType CurrentSubCommandType { get; set; }

        internal Option()
        {
            None = new NoneOption();
        }

        internal bool HasUtf8Option(string[] args)
        {
            return args.Contains("--utf8") ? true : false;
        }

        internal void Parse(string[] args)
        {
            IOption option;

            CheckWarningOptionalArgument(args);
            string[] restArgs = ParseOptionalArgument(this.None, args);
            SubCommandType type = SubCommandType.None;
            if (restArgs.Length > 0)
            {
                type = GetSubCommandType(restArgs[0]);
            }
            if (type == SubCommandType.None)
            {
                None.IsShowUsage = true;
            }
            if (IsShowUsage)
            {
                option = GetSubCommand(SubCommandType.Help);
                option.ParsePositionalArgument(restArgs);
                return;
            }
            else
            {
                if (type == SubCommandType.Help)
                {
                    None.IsShowUsage = true;
                }
                option = GetSubCommand(type);
                string[] subArgs = restArgs.Skip(1).Take(restArgs.Length - 1).ToArray();
                string[] subRestArgs = ParseOptionalArgument(option, subArgs);
                option.ParsePositionalArgument(subRestArgs);
            }
        }

        internal void ShowUsage()
        {
            string baseName = Path.GetFileName(System.Reflection.Assembly.GetExecutingAssembly().Location);
            if (CurrentSubCommandType != SubCommandType.Help)
            {
                IOption option = GetSubCommand(CurrentSubCommandType);
                option.ShowSubCommandUsage(baseName);
                return;
            }

            if (this.Help.SubCommandList.Count == 0)
            {
                None.ShowSubCommandUsage(baseName);
            }
            else
            {
                foreach (var type in this.Help.SubCommandList)
                {
                    IOption option = GetSubCommand(type);
                    option.ShowSubCommandUsage(baseName);
                }
            }
        }

        private IOption GetSubCommand(SubCommandType type)
        {
            CurrentSubCommandType = type;

            switch (type)
            {
                case SubCommandType.CreateFs:
                    this.CreateFs = new CreateFsOption();
                    return this.CreateFs;
                case SubCommandType.CreateNca:
                    this.CreateNca = new CreateNcaOption();
                    return this.CreateNca;
                case SubCommandType.CreateNsp:
                    this.CreateNsp = new CreateNspOption();
                    return this.CreateNsp;
                case SubCommandType.CreateNspMeta:
                    this.CreateNspMeta = new CreateNspMetaOption();
                    return this.CreateNspMeta;
                case SubCommandType.CreateNspd:
                    this.CreateNspd = new CreateNspdOption();
                    return this.CreateNspd;
                case SubCommandType.CreateNacp:
                    this.CreateNacp = new CreateNacpOption();
                    return this.CreateNacp;
                case SubCommandType.MakePatch:
                    this.MakePatch = new MakePatchOption();
                    return this.MakePatch;
                case SubCommandType.OptimizePatch:
                    this.OptimizePatch = new OptimizePatchOption();
                    return this.OptimizePatch;
                case SubCommandType.BinaryDiff:
                    this.MakeDelta = new MakeDeltaOption(MakeDeltaOption.OutputType.Binary);
                    return this.MakeDelta;
                case SubCommandType.MakeDelta:
                    this.MakeDelta = new MakeDeltaOption(MakeDeltaOption.OutputType.DeltaNsp);
                    return this.MakeDelta;
                case SubCommandType.MergeDeltaContent:
                    this.MergeDeltaContent = new MergeDeltaContentOption();
                    return this.MergeDeltaContent;
                case SubCommandType.Extract:
                    this.Extract = new ExtractOption();
                    return this.Extract;
                case SubCommandType.ExtractNsp:
                    this.ExtractNsp = new ExtractNspOption();
                    return this.ExtractNsp;
                case SubCommandType.ExtractNsave:
                    this.ExtractNsave = new ExtractNsaveOption();
                    return this.ExtractNsave;
                case SubCommandType.MergeNsp:
                    this.MergeNsp = new MergeNspOption();
                    return this.MergeNsp;
                case SubCommandType.Replace:
                    this.Replace = new ReplaceOption();
                    return this.Replace;
                case SubCommandType.ReplaceNspMeta:
                    this.ReplaceNspMeta = new ReplaceNspMetaOption();
                    return this.ReplaceNspMeta;
                case SubCommandType.List:
                    this.List = new ListOption();
                    return this.List;
                case SubCommandType.DiffPatch:
                    this.DiffPatch = new DiffPatchOption();
                    return this.DiffPatch;
                case SubCommandType.Verify:
                    this.Verify = new VerifyOption();
                    return this.Verify;
                case SubCommandType.ProdEncryption:
                    this.ProdEncryption = new ProdEncryptionOption();
                    return this.ProdEncryption;
                case SubCommandType.ProdEncryptionPatch:
                    this.ProdEncryptionPatch = new ProdEncryptionPatchOption();
                    return this.ProdEncryptionPatch;
                case SubCommandType.ValidateMetaFile:
                    this.ValidateMetaFile = new ValidateMetaFileOption();
                    return this.ValidateMetaFile;
                case SubCommandType.ConvertIcon:
                    this.ConvertIcon = new ConvertIconOption();
                    return this.ConvertIcon;
                case SubCommandType.ShowVersion:
                    this.ShowVersion = new ShowVersionOption();
                    return this.ShowVersion;
                case SubCommandType.GetAllXmlModels:
                    this.GetAllXmlModels = new GetAllXmlModelsOption();
                    return this.GetAllXmlModels;
                case SubCommandType.GetNspProperty:
                    this.GetNspProperty = new GetNspPropertyOption();
                    return this.GetNspProperty;
                case SubCommandType.GetUnpublishableError:
                    this.GetUnpublishableError = new GetUnpublishableErrorOption();
                    return this.GetUnpublishableError;
                case SubCommandType.EncryptXci:
                    this.EncryptXci = new EncryptXciOption();
                    return this.EncryptXci;
                case SubCommandType.SplitNsp:
                    this.SplitNsp = new SplitNspOption();
                    return this.SplitNsp;
                case SubCommandType.ConvertAddOnContentForCard:
                    this.ConvertAddOnContentForCard = new ConvertAddOnContentForCardOption();
                    return this.ConvertAddOnContentForCard;
                case SubCommandType.CreateMultiApplicationCardIndicator:
                    this.CreateMultiApplicationCardIndicator = new CreateMultiApplicationCardIndicatorOption();
                    return this.CreateMultiApplicationCardIndicator;
                case SubCommandType.CreateMultiProgramApplication:
                    this.CreateMultiProgramApplication = new CreateMultiProgramApplicationOption();
                    return this.CreateMultiProgramApplication;
                case SubCommandType.SparsifyNsp:
                    this.SparsifyNsp = new SparsifyNspOption();
                    return this.SparsifyNsp;
                case SubCommandType.Help:
                    this.Help = new HelpOption();
                    return this.Help;
                case SubCommandType.None:
                    return this.None;
                default:
                    throw new ArgumentException();
            }
        }

        private void CheckWarningOptionalArgument(string[] args)
        {
            Tuple<string, string>[] OptionList =
            {
                new Tuple<string, string> ( "--legal-information", "LegalInformationFilePath" ),
                new Tuple<string, string> ( "--html-document", "HtmlDocumentPath" ),
                new Tuple<string, string> ( "--accessible-urls", "AccessibleUrlsFilePath" ),
                new Tuple<string, string> ( "--icon", "IconPath" ),
                new Tuple<string, string> ( "--nx-icon", "NxIconPath" ),
            };

            for (int i = 0; i < args.Length; i++)
            {
                if (args[i].StartsWith("-") == false)
                {
                    continue;
                }

                foreach (var option in OptionList)
                {
                    if (args[i] == option.Item1)
                    {
                        Log.Warning(string.Format("{0} option is deprecated. Use the {1} Element in the NMETA file instead.", option.Item1, option.Item2));
                        break;
                    }
                }
            }
        }

        private string[] ParseOptionalArgument(IOption option, string[] args)
        {
            OptionDescription[] optionDescs = option.GetOptionDescription();
            if (optionDescs == null)
            {
                return args;
            }

            if (args.Length == 0)
            {
                None.IsShowUsage = true;
                return args;
            }

            List<string> restArgs = new List<string>();
            for (int i = 0; i < args.Length; i++)
            {
                for (int j = 0; j < optionDescs.Length; j++)
                {
                    var key = args[i];
                    var optionDesc = optionDescs[j];
                    if ((optionDesc.LongName == key) || optionDesc.ShortName ==  key)
                    {
                        if (optionDesc.HasArgument != 0)
                        {
                            List<string> tmpArgs = new List<string>();
                            for (int k = 0; (k < optionDesc.HasArgument) || (-1 == optionDesc.HasArgument); k++)
                            {
                                if ((i < (args.Length - 1)) && !args[i + 1].StartsWith("-"))
                                {
                                    tmpArgs.Add(args[++i]);
                                }
                                else
                                {
                                    break;
                                }
                            }
                            if (!(-1 == optionDesc.HasArgument) && tmpArgs.Count == 0)
                            {
                                throw new InvalidOptionException(string.Format("no option argument for \"{0}\".", key));
                            }
                            optionDesc.Action(tmpArgs);
                        }
                        else
                        {
                            optionDesc.Action(null);
                        }
                        break;
                    }

                    if (j == optionDescs.Length - 1)
                    {
                        restArgs.Add(key);
                    }
                }
            }
            return restArgs.ToArray();
        }
    }
}
