﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <Partition/systemInitializer_ListPartitions.h>
#include <Partition/systemInitializer_Partition.h>
#include <Partition/systemInitializer_PartitionAccessor.h>
#include <nn/utilTool/utilTool_CommandLog.h>
#include <nn/utilTool/utilTool_ResultHandlingUtility.h>
#include "Utility/systemInitializer_FsFile.h"
#include "Utility/systemInitializer_FsBisPartition.h"
#include "Utility/systemInitializer_StorageUtility.h"
#include "Utility/systemInitializer_FileSystemUtility.h"

#include <nn/nn_Abort.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/fs/fs_IStorage.h>

nn::Result OpenBlockStorage(std::unique_ptr<nn::fs::IStorage> *pOut, std::string type, std::string name)
{
    if(type == "storage")
    {
        NN_UTILTOOL_RESULT_DO(
            OpenBlockStorage(pOut, nn::fs::BisPartitionId::UserDataRoot));
    }
    else if(type == "gpt-header")
    {
        NN_UTILTOOL_RESULT_DO(
            OpenGptHeader(pOut));
    }
    else if(type == "gpt-partition-table")
    {
        NN_UTILTOOL_RESULT_DO(
            OpenGptPartitionTable(pOut));
    }
    else if(type == "gpt-partition")
    {
        NN_UTILTOOL_RESULT_DO(
            OpenGptPartition(pOut, name));
    }
    else if(type == "fs-partition")
    {
        NN_UTILTOOL_RESULT_DO(
            OpenBlockStorage(pOut, ConvertToBisPartitionId(name) ));
    }
    else
    {
        NN_ABORT("Unknown type: '%s'", type.c_str());
    }

    NN_RESULT_SUCCESS;
}

nn::Result OpenBlockStorage(std::unique_ptr<nn::fs::IStorage> *pOut, nn::fs::BisPartitionId partitionId)
{
    NN_UTILTOOL_RESULT_DO(
        nn::fs::OpenBisPartition(pOut, partitionId));

    NN_RESULT_SUCCESS;
}

nn::Result OpenGptHeader(std::unique_ptr<nn::fs::IStorage> *pOut)
{
    std::unique_ptr<nn::fs::IStorage> rootStorage;
    NN_UTILTOOL_RESULT_DO(
        nn::fs::OpenBisPartition(&rootStorage, nn::fs::BisPartitionId::UserDataRoot));

    *pOut = MakeSubBlockStorage(std::move(rootStorage), 512, 512);

    NN_RESULT_SUCCESS;
}

nn::Result OpenGptPartitionTable(std::unique_ptr<nn::fs::IStorage> *pOut)
{
    static GptHeader gptHeader;

    std::unique_ptr<nn::fs::IStorage> rootStorage;
    NN_UTILTOOL_RESULT_DO(
        nn::fs::OpenBisPartition(&rootStorage, nn::fs::BisPartitionId::UserDataRoot));

    NN_UTILTOOL_RESULT_DO(
        ReadGptFromStorage(&gptHeader, rootStorage.get()));

    *pOut = MakeSubBlockStorage(std::move(rootStorage), gptHeader.partitionEntryLba * 512, nn::util::align_up(gptHeader.numberOfPartitionEntries * gptHeader.sizeOfPartitionEntry, 512) );

    NN_RESULT_SUCCESS;
}

nn::Result OpenGptPartition(std::unique_ptr<nn::fs::IStorage> *pOut, std::string name)
{
    static GptHeader gptHeader;
    static GptPartitionEntry partitionEntries[GPT_DEFAULT_PARTITION_ENTRIES];

    std::unique_ptr<nn::fs::IStorage> rootStorage;
    NN_UTILTOOL_RESULT_DO(
        nn::fs::OpenBisPartition(&rootStorage, nn::fs::BisPartitionId::UserDataRoot));

    NN_UTILTOOL_RESULT_DO(
        ReadGptFromStorage(&gptHeader, rootStorage.get()));

    NN_ABORT_UNLESS(gptHeader.numberOfPartitionEntries <= GPT_DEFAULT_PARTITION_ENTRIES);

    NN_UTILTOOL_RESULT_DO(
        ReadGptPartitionsFromStorage(partitionEntries, GPT_DEFAULT_PARTITION_ENTRIES, gptHeader, rootStorage.get()));

    for(int i = 0; i < gptHeader.numberOfPartitionEntries; i++)
    {
        const GptPartitionEntry &partition = partitionEntries[i];

        if(GetPartitionName(partition) == name)
        {
            int64_t offset = partition.startingLba * 512;
            int64_t size = (partition.endingLba - partition.startingLba) * 512;

            *pOut = MakeSubBlockStorage(std::move(rootStorage), offset, size);

            NN_RESULT_SUCCESS;
        }
    }

    NN_ABORT("Found no partition: %s", name.c_str());

    NN_RESULT_SUCCESS;
}
