﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <memory>
#include <nn/nn_Result.h>
#include <nn/utilTool/utilTool_CommandFramework.h>
#include <nn/utilTool/utilTool_CommandLog.h>
#include <nn/utilTool/utilTool_ResultHandlingUtility.h>
#include <nn/fs_Base.h>
#include <nn/fs/fs_ApiPrivate.h>
#include <nn/fs/fs_Host.h>
#include "Utility/systemInitializer_FsFile.h"
#include "Utility/systemInitializer_BuildInBlockStorage.h"
#include "Utility/systemInitializer_FileSystemUtility.h"
#include "Utility/systemInitializer_Allocator.h"
#include "cal_Settings.h"
#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Abort.h>
#include <nn/os.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/result/result_HandlingUtility.h>
#include "tool_SystemInitializer.h"
#include <nn/mem/mem_StandardAllocator.h>
#include <nn/htcs.h>
#include <nn/bpc/bpc_BoardPowerControl.h>
#include <nn/settings/factory/settings_ConfigurationId.h> // SDEV の実装分岐の W/A のため
#include <nn/spl/spl_Api.h>
#include <nn/crypto/crypto_Aes128CbcDecryptor.h>

#include <nn/crypto/crypto_RsaOaepEncryptor.h>
#include <nn/crypto/crypto_Sha256Generator.h>

#include <nn/cal/cal.h>

#if defined(NN_TOOL_REPAIR_INITIALIZER)
#include "Utility/systemInitializer_PmicDriver.h"
#include "Utility/systemInitializer_RtcDriver.h"
#include <nn/init.h>
#endif

#if defined NN_TOOL_USE_CMAC_FILE
#include "systemInitializer_CmacFile.h"
#include "systemInitializer_WrapPublicKey.h"
#include <nn/manu/manu_Api.h>
#endif

#if defined(NN_TOOL_SYSTEMUPDATER) || defined(NN_TOOL_SYSTEMUPDATER_ROMFS)

#define TOOL_NAME "SystemUpdater"
#define TOOL_SUMMARY "Update the system by the system initial image."

#elif defined(NN_TOOL_REPAIR_INITIALIZER)

#define TOOL_NAME "SystemUpdaterForRepair"
#define TOOL_SUMMARY "Update the system by the system initial image on memory."

#else

#define TOOL_NAME "SystemInitializer"
#define TOOL_SUMMARY "Initialize the storage by the system initial image."

#endif

#if defined NN_TOOL_USE_CMAC_FILE

class ResultWriter
{
public:
    NN_IMPLICIT ResultWriter(const char* outputPath): m_OutputPath(outputPath) {}

    void WriteResult(nn::Result result)
    {
        FsFile file;
        NN_ABORT_UNLESS_RESULT_SUCCESS(
            file.OpenWrite(m_OutputPath));
        NN_UTIL_SCOPE_EXIT { file.Close(); };

        const size_t BufferSize = 128;
        char buffer[BufferSize];

        int length = 0;
        if (result.IsSuccess())
        {
            length = snprintf(buffer, BufferSize, "Success\n");
        }
        else
        {
            length = snprintf(buffer, BufferSize, "Failed: %08x\n", result.GetInnerValueForDebug());
        }

        if(0 < length)
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(
                file.Write(0, buffer, length, true));
        }
    }
private:
    const char* m_OutputPath;
};

void MakeRequestData(uint8_t *requestData, uint8_t *sessionKey, const uint8_t *wrapPubKeyModulus, uint32_t wrapPubKeyExponent)
{
    uint8_t seed[nn::crypto::Sha256Generator::HashSize] = {0};

    nn::spl::InitializeForCrypto();
    nn::spl::GenerateRandomBytes(seed, sizeof(seed));
    nn::spl::Finalize();

    nn::crypto::RsaOaepEncryptor<Key2048::KEY_SIZE,nn::crypto::Sha256Generator> encryptor;
    encryptor.Initialize(wrapPubKeyModulus, Key2048::KEY_SIZE, &wrapPubKeyExponent, sizeof(wrapPubKeyExponent));
    encryptor.Encrypt(requestData, Key2048::KEY_SIZE, sessionKey, Key128::KEY_SIZE, seed, sizeof(seed));
}

nn::Result WriteRequestData(const char* outputRequestPath, uint8_t *requestData, size_t size)
{
    bool existImage = false;
    NN_UTILTOOL_RESULT_DO(
        FsFile::Exists(&existImage, outputRequestPath));

    if(!existImage)
    {
        NN_UTILTOOL_RESULT_DO(
            FsFile::Create(outputRequestPath));
    }

    FsFile file;
    NN_UTILTOOL_RESULT_DO(
        file.OpenWrite(outputRequestPath));
    NN_UTIL_SCOPE_EXIT { file.Close(); };

    NN_UTILTOOL_RESULT_DO(
        file.Write(0, requestData, size, true));

    NN_RESULT_SUCCESS;
}

nn::Result ReadResponse(const char* inputResponsePath, uint8_t *responseData)
{
    FsFile file;
    nn::Result result;

    for(int i = 0; i < 30; i++)
    {
        nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));

        result = file.OpenRead(inputResponsePath);

        NN_UTILTOOL_LOG_VERBOSE("wait %s", inputResponsePath);

        if(result.IsFailure())
        {
            continue;
        }

        int64_t size;
        result = file.GetSize(&size);
        if(result.IsFailure())
        {
            continue;
        }

        if(size != Key128::KEY_SIZE)
        {
            continue;
        }

        break;
    }
    NN_UTILTOOL_RESULT_DO(result);
    NN_UTIL_SCOPE_EXIT { file.Close(); };

    nn::os::SleepThread(nn::TimeSpan::FromSeconds(3));

    int64_t size;
    NN_UTILTOOL_RESULT_DO(
        file.GetSize(&size));
    NN_ABORT_UNLESS(size == Key128::KEY_SIZE);

    size_t readSize;
    NN_UTILTOOL_RESULT_DO(
        file.Read(&readSize, 0, responseData, Key128::KEY_SIZE));
    NN_ABORT_UNLESS(readSize == Key128::KEY_SIZE);

    NN_RESULT_SUCCESS;
}

void LoadCmacKek(uint8_t *cmacKek, uint8_t *responseData, uint8_t *sessionKey)
{
    uint8_t iv[Key128::KEY_SIZE];
    std::memset(iv, 0, Key128::KEY_SIZE);

    nn::crypto::Aes128CbcDecryptor decryptor;
    decryptor.Initialize(sessionKey, Key128::KEY_SIZE, iv, Key128::KEY_SIZE);
    decryptor.Update(cmacKek, Key128::KEY_SIZE, responseData, Key128::KEY_SIZE);
}

nn::Result GetPublicKey(Key2048 *pWrapPublicKeyModulus, uint32_t *pWrapPublicKeyExponent, Key128 encryptedKey, Key128 iv, Key2048 encryptedPublicKey, const uint32_t exponent)
{
    Key128 kek;
    NN_UTILTOOL_RESULT_DO(
        nn::spl::DecryptAesKey(&kek.data, sizeof(kek), encryptedKey.data, encryptedKey.KEY_SIZE, 0, 0));

    nn::crypto::Aes128CbcDecryptor decryptor;
    decryptor.Initialize(kek.data, kek.KEY_SIZE, iv.data, iv.KEY_SIZE);
    decryptor.Update(pWrapPublicKeyModulus->data, pWrapPublicKeyModulus->KEY_SIZE, encryptedPublicKey.data, encryptedPublicKey.KEY_SIZE);
    *pWrapPublicKeyExponent = exponent;

    NN_RESULT_SUCCESS;
}

nn::Result GetPublicKey(Key2048 *pWrapPublicKeyModulus, uint32_t *pWrapPublicKeyExponent)
{
    nn::spl::InitializeForCrypto();
    NN_UTIL_SCOPE_EXIT {
        nn::spl::Finalize();
    };

    auto socType = nn::spl::GetSocType();
    if (socType == nn::spl::SocType_Mariko)
    {
        if (nn::spl::IsDevelopment())
        {
            NN_UTILTOOL_RESULT_DO(
                GetPublicKey(pWrapPublicKeyModulus, pWrapPublicKeyExponent, EncryptedKeyDevM2, IvDevM2, EncryptedWrapPublicKeyModulusDevM2, WrapPublicKeyExponentDevM2));
        }
        else
        {
            NN_UTILTOOL_RESULT_DO(
                GetPublicKey(pWrapPublicKeyModulus, pWrapPublicKeyExponent, EncryptedKeyProdM4, IvProdM4, EncryptedWrapPublicKeyModulusProdM4, WrapPublicKeyExponentProdM4));
        }
    }
    else
    {
        if (nn::spl::IsDevelopment())
        {
            NN_UTILTOOL_RESULT_DO(
                GetPublicKey(pWrapPublicKeyModulus, pWrapPublicKeyExponent, EncryptedKeyDev, IvDev, EncryptedWrapPublicKeyModulusDev, WrapPublicKeyExponentDev));
        }
        else
        {
            NN_UTILTOOL_RESULT_DO(
                GetPublicKey(pWrapPublicKeyModulus, pWrapPublicKeyExponent, EncryptedKeyProd, IvProd, EncryptedWrapPublicKeyModulusProd, WrapPublicKeyExponentProd));
        }
    }

    NN_RESULT_SUCCESS;
}

nn::Result RequestCmacKek(uint8_t *cmacKek, const char* outputRequestPath, const char* inputResponsePath)
{
    NN_UTILTOOL_LOG_VERBOSE("output request: %s", outputRequestPath);
    NN_UTILTOOL_LOG_VERBOSE("input response: %s", inputResponsePath);

    uint8_t sessionKey[Key128::KEY_SIZE] = {0};
    uint8_t requestData[Key2048::KEY_SIZE] = {0};
    uint8_t responseData[Key128::KEY_SIZE] = {0};

    nn::spl::InitializeForCrypto();
    nn::spl::GenerateRandomBytes(sessionKey, Key128::KEY_SIZE);
    nn::spl::Finalize();

    Key2048 wrapPublicKeyModulus;
    uint32_t wrapPublicKeyExponent;
    GetPublicKey(&wrapPublicKeyModulus, &wrapPublicKeyExponent);

    MakeRequestData(requestData, sessionKey, wrapPublicKeyModulus.data, wrapPublicKeyExponent);

    NN_UTILTOOL_RESULT_DO(
        WriteRequestData(outputRequestPath, requestData, sizeof(requestData)));

    NN_UTILTOOL_RESULT_DO(
        ReadResponse(inputResponsePath, responseData));

    LoadCmacKek(cmacKek, responseData, sessionKey);

    NN_RESULT_SUCCESS;
}
#endif

#if (defined NN_TOOL_WRITE_CAL_IMAGE) || (defined NN_TOOL_USE_CMAC_FILE)
nn::Result ReadRawCalibrationInfo(nn::cal::RawCalibrationInfo *pOut, const char* imagePath)
{
    FsFile file;
    nn::Result result;

    NN_UTILTOOL_RESULT_DO(
        file.OpenRead(imagePath));
    NN_UTIL_SCOPE_EXIT{ file.Close(); };

    int64_t size;
    NN_UTILTOOL_RESULT_DO(
        file.GetSize(&size));

    if (size != sizeof(nn::cal::RawCalibrationInfo) )
    {
        NN_UTILTOOL_RESULT_THROW(nn::fs::ResultInvalidSize());
    }

    size_t readSize;
    NN_UTILTOOL_RESULT_DO(
        file.Read(&readSize, 0, pOut, sizeof(nn::cal::RawCalibrationInfo)));
    NN_ABORT_UNLESS(readSize == sizeof(nn::cal::RawCalibrationInfo));

    NN_RESULT_SUCCESS;
}

nn::Result ValidateCalibrationData(const nn::cal::RawCalibrationInfo& calibrationInfo)
{
    const auto& body = (*reinterpret_cast<const nn::cal::CalibrationInfo*>(calibrationInfo.data)).body;

    NN_UTILTOOL_LOG_MESSAGE("Validate ExtendedEccB233DeviceKey Generation.");
    NN_ABORT_UNLESS(body.extendedEccB233DeviceKeyBlock.encryptionKeyGeneration.IsZero());
    NN_UTILTOOL_LOG_MESSAGE("Validate ExtendedRsa2048DeviceKey Generation.");
    NN_ABORT_UNLESS(body.extendedRsa2048DeviceKeyBlock.encryptionKeyGeneration.IsZero());
    NN_UTILTOOL_LOG_MESSAGE("Validate ExtendedRsa2048ETicketKey Generation.");
    NN_ABORT_UNLESS(body.extendedRsa2048ETicketKeyBlock.encryptionKeyGeneration.IsZero());
    NN_UTILTOOL_LOG_MESSAGE("Validate ExtendedSslKey Generation.");
    NN_ABORT_UNLESS(body.extendedSslKeyBlock.encryptionKeyGeneration.IsZero());
    NN_UTILTOOL_LOG_MESSAGE("Validate ExtendedGameCardKey Generation.");
    NN_ABORT_UNLESS(body.extendedGameCardKeyBlock.encryptionKeyGeneration.IsZero());
    NN_UTILTOOL_LOG_MESSAGE("Validate AmiiboKey Generation.");
    NN_ABORT_UNLESS(body.amiiboKeyBlock.encryptionKeyGeneration.IsZero());
    NN_UTILTOOL_LOG_MESSAGE("Validate AmiiboEcqvBlsKey Generation.");
    NN_ABORT_UNLESS(body.amiiboEcqvBlsKeyBlock.encryptionKeyGeneration.IsZero());

    NN_RESULT_SUCCESS;
}

nn::Result WriteCalImage(const char *calImagePath)
{
    NN_UTILTOOL_LOG_MESSAGE("Write CalImage.");

    static nn::cal::RawCalibrationInfo calibrationInfo = {};

    std::memset(&calibrationInfo, 0, sizeof(calibrationInfo));

    NN_UTILTOOL_RESULT_DO(ReadRawCalibrationInfo(&calibrationInfo, calImagePath));

    NN_UTILTOOL_RESULT_DO(ValidateCalibrationData(calibrationInfo));

    NN_UTILTOOL_RESULT_DO(nn::cal::PutCalibrationData(calibrationInfo));

    NN_UTILTOOL_RESULT_DO(nn::cal::FinishUp());

    NN_RESULT_SUCCESS;
}
#endif

nn::Result InitializeSystem(const char *imagePath, bool isDryRun, bool enableOverwritingProductionInfo, bool enableEraseEmmc, const char* inputCalImagePath, uint8_t *cmacKek)
{

#if (defined NN_TOOL_WRITE_CAL_IMAGE) || (defined NN_TOOL_USE_CMAC_FILE)
    if (std::strcmp(inputCalImagePath, "") != 0)
    {
        NN_UTILTOOL_RESULT_DO(
            WriteCalImage(
                inputCalImagePath));
    }
#endif

#if !(defined NN_TOOL_REPAIR_INITIALIZER)
    if( !IsValidFileSystemEntry(imagePath) )
    {
        NN_UTILTOOL_LOG_ERROR("[ERROR] Invalid path format: '%s'", imagePath);
        return nn::fs::ResultPathNotFound();
    }
#endif
    bool existImage = false;
    NN_UTILTOOL_RESULT_DO(
        FsFile::Exists(&existImage, imagePath));

    NN_UTILTOOL_LOG_VERBOSE("existImage: %d", existImage);

    if(!existImage)
    {
        NN_UTILTOOL_LOG_ERROR("InitialImage is not found: '%s'", imagePath);
        NN_ABORT();
    }

#if defined NN_TOOL_USE_CMAC_FILE

    CmacFile file;
    NN_UTILTOOL_RESULT_DO(
        file.OpenRead(imagePath, Key128::Make(cmacKek)));

    NN_UTIL_SCOPE_EXIT { file.Close(); };
#else
    FsFile file;
    NN_UTILTOOL_RESULT_DO(
        file.OpenRead(imagePath));
    NN_UTIL_SCOPE_EXIT { file.Close(); };
#endif
    NN_UTILTOOL_LOG_DEBUG("image opened");

    std::shared_ptr<IBlockStorage> storage = GetBuildInBlockStorage(isDryRun);

    NN_UTILTOOL_LOG_DEBUG("get interface of block storage");

    NN_UTILTOOL_RESULT_DO(
        WriteInitialImageFromFile(storage.get(), &file, SystemIntializerOption::Make(enableOverwritingProductionInfo, enableEraseEmmc)));

    NN_RESULT_SUCCESS;
}

#if defined NN_TOOL_USE_CMAC_FILE

nn::Result InitializeSystemWithAuthentication(const char *imagePath, bool isDryRun, bool enableOverwritingProductionInfo, bool enableEraseEmmc, const char* outputRequestPath, const char* inputResponsePath, const char* inputCalImagePath)
{
    uint8_t cmacKek[Key128::KEY_SIZE];

    NN_UTILTOOL_RESULT_DO(
        RequestCmacKek(
            cmacKek,
            outputRequestPath,
            inputResponsePath));

    NN_UTILTOOL_RESULT_DO(
        InitializeSystem(
            imagePath,
            isDryRun,
            enableOverwritingProductionInfo,
            enableEraseEmmc,
            inputCalImagePath,
            cmacKek));

    NN_RESULT_SUCCESS;
}

#endif

nn::Result PrintInfos(const char *imagePath)
{
    if( !IsValidFileSystemEntry(imagePath) )
    {
        NN_UTILTOOL_LOG_ERROR("[ERROR] Invalid path format: '%s'", imagePath);
        return nn::fs::ResultPathNotFound();
    }

    bool existImage = false;
    NN_UTILTOOL_RESULT_DO(
        FsFile::Exists(&existImage, imagePath));

    NN_UTILTOOL_LOG_VERBOSE("existImage: %d", existImage);

    if(!existImage)
    {
        NN_UTILTOOL_LOG_ERROR("InitialImage is not found: '%s'", imagePath);
        NN_ABORT();
    }

    FsFile file;
    NN_UTILTOOL_RESULT_DO(
        file.OpenRead(imagePath));
    NN_UTIL_SCOPE_EXIT { file.Close(); };

    NN_UTILTOOL_LOG_DEBUG("image opened");

    InitialImage initialImage;

    NN_UTILTOOL_RESULT_DO(
        initialImage.Initialize(&file));

    InitialImageInfo initialImageInfo;
    NN_UTILTOOL_RESULT_DO(
        initialImage.ReadInfo(&initialImageInfo));
    PrintInitialImageInfo(initialImageInfo);

    InitialImageCommandPartition commandPartition;
    NN_UTILTOOL_RESULT_DO(
        initialImage.ReadCommandPartition(&commandPartition));
    PrintCommandPartition(commandPartition);

    NN_RESULT_SUCCESS;
}

#if defined(NN_TOOL_REPAIR_INITIALIZER)
namespace
{
uintptr_t g_BufferAddressForMalloc = 0;
}
#endif

extern "C" void nninitStartup()
{
#if defined(NN_TOOL_REPAIR_INITIALIZER)
    const size_t MemoryHeapSize = 128 * 1024 * 1024;
    nn::Result result = nn::os::SetMemoryHeapSize(MemoryHeapSize);
    NN_ABORT_UNLESS(result.IsSuccess(), "Cannot set MemoryHeapSize.");

    const size_t BufferSizeForMalloc = 128 * 1024 * 1024;
    result = nn::os::AllocateMemoryBlock( &g_BufferAddressForMalloc, BufferSizeForMalloc );
    NN_ABORT_UNLESS( result.IsSuccess(), "Failed to allocate memory block for InitializeAllocator()." );
    nn::init::InitializeAllocator( reinterpret_cast<void*>(g_BufferAddressForMalloc), BufferSizeForMalloc );
#endif
}

#if defined(NN_TOOL_USE_CMAC_FILE) || defined(NN_TOOL_REPAIR_INITIALIZER)
extern "C" void nndiagStartup()
{
}
#endif

struct ArgumentHeader
{
    int bufferSize;
    int argumentCount;
};

struct ProgramArguments
{
    int bufferSize;
    int argumentCount;
    std::unique_ptr<const char*> arguments;
    std::unique_ptr<char> buffer;
};

struct SingleHtcsConnection
{
    int serverSocket;
    int clientSocket;
};

nn::Result CreateConnectionAsHtcs(SingleHtcsConnection *pOut, const char * serviceName)
{
    NN_UTILTOOL_LOG_DEBUG("Start creating service.");

    pOut->serverSocket = -1;
    pOut->clientSocket = -1;

    while (NN_STATIC_CONDITION(true))
    {
        NN_UTILTOOL_LOG_DEBUG("Wait creating socket.");
        while (NN_STATIC_CONDITION(true))
        {
            int socket = nn::htcs::Socket();
            if (socket >= 0)
            {
                pOut->serverSocket = socket;
                break;
            }

            nn::os::SleepThread( nn::TimeSpan::FromMilliSeconds(10) );
        }

        nn::htcs::SockAddrHtcs serverAddress;
        serverAddress.family = nn::htcs::HTCS_AF_HTCS;
        serverAddress.peerName = nn::htcs::GetPeerNameAny();
        NN_ABORT_UNLESS(std::strlen(serviceName) < nn::htcs::PortNameBufferLength);
        std::strncpy(serverAddress.portName.name, serviceName, nn::htcs::PortNameBufferLength);

        NN_ABORT_UNLESS(nn::htcs::Bind(pOut->serverSocket, &serverAddress) == 0);
        NN_ABORT_UNLESS(nn::htcs::Listen(pOut->serverSocket, 1) == 0);

        NN_UTILTOOL_LOG_DEBUG("Wait connect.");

        while (NN_STATIC_CONDITION(true))
        {
            pOut->clientSocket = nn::htcs::Accept(pOut->serverSocket, nullptr);
            if (pOut->clientSocket <= 0)
            {
                break;
            }

            NN_RESULT_SUCCESS;
        }
    }

    NN_RESULT_SUCCESS;
}

nn::Result ReceiveProgramArguments(ProgramArguments *pOut, const SingleHtcsConnection &connection)
{
    NN_UTILTOOL_LOG_DEBUG("Start receiving arguments header.");

    ArgumentHeader header;
    auto result = nn::htcs::Recv(connection.clientSocket, &header, sizeof(header), nn::htcs::HTCS_MSG_WAITALL);
    if (result <= 0)
    {
        NN_UTILTOOL_RESULT_THROW(nn::htcs::ResultDisconnected());
    }

    NN_ABORT_UNLESS_EQUAL(sizeof(header), static_cast<size_t>(result));

    *pOut = ProgramArguments();

    pOut->bufferSize = header.bufferSize;
    pOut->argumentCount = header.argumentCount;
    pOut->arguments = std::unique_ptr<const char*>(new const char*[header.argumentCount]);
    pOut->buffer = std::unique_ptr<char>(new char[header.bufferSize]);

    NN_UTILTOOL_LOG_DEBUG("Start receiving arguments.");
    result = nn::htcs::Recv(connection.clientSocket, pOut->buffer.get(), header.bufferSize, nn::htcs::HTCS_MSG_WAITALL);
    if (result <= 0)
    {
        NN_UTILTOOL_RESULT_THROW(nn::htcs::ResultDisconnected());
    }
    NN_ABORT_UNLESS_EQUAL(header.bufferSize, result);

    const char * argument = pOut->buffer.get();
    int restlen = pOut->bufferSize;
    for(int i = 0; i < header.argumentCount; i++)
    {
        NN_UTILTOOL_LOG_DEBUG("argument[%d]: %s", i, argument);

        pOut->arguments.get()[i] = argument;

        int len = strnlen(argument, restlen);
        argument += len + 1;
        restlen -= len + 1;
    }

    NN_RESULT_SUCCESS;
}

class InitializeSystemCommand
{
public:
    InitializeSystemCommand() : m_IsPrintInfo(false), m_DryRun(false), m_EnableOverwritingProdcutionInfo(false), m_EnableEraseEmmc(false)
#ifdef NN_TOOL_SYSTEMUPDATER_ROMFS
        , m_MountRomCacheBuffer(nullptr), m_MountRomCacheBufferSize(0)
#endif
    {
    }

    const char* Name()
    {
        return TOOL_NAME;
    }

    const char* Summary()
    {
        return TOOL_SUMMARY;
    }

    const char* Description()
    {
        return "";
    }

    void Arguments(nn::utilTool::SingleCommandInterface &commandInterface)
    {
#if !(defined NN_TOOL_SYSTEMUPDATER_ROMFS) && !(defined NN_TOOL_REPAIR_INITIALIZER)
        commandInterface.AddKeywordArgument(&m_InitialImagePath, 'i', "input", "Input the initial image.", true);
        commandInterface.AddFlagArgument(&m_DryRun, "dry-run", "Run without writing to the storage.");
        commandInterface.AddFlagArgument(&m_IsPrintInfo, "print-info", "Print about the initial image.");
        commandInterface.AddFlagArgument(&m_EnableOverwritingProdcutionInfo, "enable-overwriting-production-info", "For device production.");
        commandInterface.AddFlagArgument(&m_EnableEraseEmmc, "enable-erase-emmc", "For device production.");
#if (defined NN_TOOL_WRITE_CAL_IMAGE) || (defined NN_TOOL_USE_CMAC_FILE)
        commandInterface.AddKeywordArgument(&m_InputCalImage, "cal-image", "set cal image path.", false);
#endif
#if defined NN_TOOL_USE_CMAC_FILE
        // commandInterface.AddKeywordArgument(&m_CmacKek, 'k', "cmac-kek", "set kek as 32-chars for cmac initial image.", true);
        commandInterface.AddKeywordArgument(&m_OutputRequestPath, 'r', "output-request", "set request path as output.", true);
        commandInterface.AddKeywordArgument(&m_InputResponsePath, 's', "input-response", "set response path as input.", true);
        commandInterface.AddKeywordArgument(&m_OutputResultPath,  'o', "output-result",  "set result path as output.",  true);
#endif
#endif
    }

    int Main()
    {
#if defined NN_TOOL_REPAIR_INITIALIZER

#endif
        nn::fs::InitializeWithMultiSessionForTargetTool();
        nn::fs::SetAllocator(Allocate, Deallocate);
#ifdef NN_TOOL_SYSTEMUPDATER_ROMFS
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::QueryMountRomCacheSize(&m_MountRomCacheBufferSize));
        m_MountRomCacheBuffer = Allocate(m_MountRomCacheBufferSize);
        NN_ABORT_UNLESS_NOT_NULL(m_MountRomCacheBuffer);
        NN_UTIL_SCOPE_EXIT
        {
            Deallocate(m_MountRomCacheBuffer, m_MountRomCacheBufferSize);
            m_MountRomCacheBuffer = nullptr;
            m_MountRomCacheBufferSize = 0;
        };
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::MountRom("Contents", m_MountRomCacheBuffer, m_MountRomCacheBufferSize));
        m_InitialImagePath = "Contents:/default.initimg";
#elif defined NN_TOOL_USE_CMAC_FILE
        nn::manu::InitializeUfio();
#elif defined NN_TOOL_REPAIR_INITIALIZER
        // 何もしない
#else
        NN_ABORT_UNLESS_RESULT_SUCCESS(
            nn::fs::MountHostRoot());
#endif

        if ( m_IsPrintInfo )
        {
            auto resultCode = PrintInfos(m_InitialImagePath.c_str());

            if ( resultCode.IsFailure() )
            {
                NN_UTILTOOL_LOG_ERROR("Failed print commands. result = (module: %d, desc: %d)", resultCode.GetModule() , resultCode.GetDescription());
                return 1;
            }

            return 0;
        }
        else
        {
            NN_UTILTOOL_LOG_MESSAGE("Start initializing the system.");
            NN_UTILTOOL_LOG_VERBOSE("imagename: %s", m_InitialImagePath.c_str());

#if defined NN_TOOL_USE_CMAC_FILE
            auto resultCode = InitializeSystemWithAuthentication(
                m_InitialImagePath.c_str(),
                m_DryRun,
                m_EnableOverwritingProdcutionInfo,
                m_EnableEraseEmmc,
                m_OutputRequestPath.c_str(),
                m_InputResponsePath.c_str(),
                m_InputCalImage.c_str()
                );

            auto resultWriter = ResultWriter(m_OutputResultPath.c_str());
            resultWriter.WriteResult(resultCode);
#else
            auto resultCode = InitializeSystem(
                m_InitialImagePath.c_str(),
                m_DryRun,
                m_EnableOverwritingProdcutionInfo,
                m_EnableEraseEmmc,
                m_InputCalImage.c_str(),
                nullptr
                );
#endif

            if ( resultCode.IsSuccess() )
            {
                NN_UTILTOOL_LOG_MESSAGE("Succeeded initializing the system.");

#if !defined NN_TOOL_SYSTEMUPDATER_SAFEMODE_ID
#if !(defined NN_TOOL_USE_CMAC_FILE) && !(defined NN_TOOL_REPAIR_INITIALIZER)

                // TODO : HW Type による実装分岐をやめる
                // SDEV の場合、Target をリブートすると HostBridge がハングする問題があるため、W/A で実装分岐 (SIGLO-23093)
                nn::settings::factory::ConfigurationId1 ConfigurationId1;
                nn::settings::factory::GetConfigurationId1( &ConfigurationId1 );
                if( strncmp( "SDEV", ConfigurationId1.string, 4 ) != 0 )
                {
#endif
                    NN_UTILTOOL_LOG_MESSAGE("Restart now.");

#if (defined NN_TOOL_REPAIR_INITIALIZER)
                    // 修理システム用初期化の場合は直接 PMIC に再起動を仕掛ける
                    nn::os::SleepThread(nn::TimeSpan::FromSeconds(2));
                    nn::systemInitializer::RtcDriver rtcDriver;
                    rtcDriver.MaskRtcAlarm();

                    PmicDriver pmicDriver;
                    pmicDriver.RebootSystem();
#else
                    NN_UTILTOOL_LOG_MESSAGE("Restart now.");
                    nn::os::SleepThread(nn::TimeSpan::FromSeconds(5));
                    nn::bpc::InitializeBoardPowerControl();
                    nn::bpc::RebootSystem();
#endif

#if !(defined NN_TOOL_USE_CMAC_FILE) && !(defined NN_TOOL_REPAIR_INITIALIZER)
                }
#endif
#endif
                return 0;
            }
            else
            {
                NN_UTILTOOL_LOG_ERROR("Failed to initialize the system. result = (module: %d, desc: %d)", resultCode.GetModule() , resultCode.GetDescription());
                return 1;
            }
        }
#ifdef NN_TOOL_SYSTEMUPDATER_ROMFS
        nn::fs::Unmount("Contents");
#endif
    }

private:
    std::string m_InitialImagePath;
    bool m_IsPrintInfo;
    bool m_DryRun;
    bool m_EnableOverwritingProdcutionInfo;
    bool m_EnableEraseEmmc;
    std::string m_CmacKek;
    std::string m_OutputRequestPath;
    std::string m_InputResponsePath;
    std::string m_OutputResultPath;
    std::string m_InputCalImage;
#ifdef NN_TOOL_SYSTEMUPDATER_ROMFS
    void* m_MountRomCacheBuffer;
    size_t m_MountRomCacheBufferSize;
#endif
};

#if defined(NN_TOOL_SYSTEMUPDATER_ROMFS)

extern "C" void nnMain()
{

    const char *emptyArgv[] = { "empty" };
    nn::utilTool::BasicCommandFramework<InitializeSystemCommand>::Run(
        1,
        emptyArgv
    );
}

#elif defined(NN_TOOL_SYSTEMUPDATER)

extern "C" void nnMain()
{
    nn::utilTool::BasicCommandFramework<InitializeSystemCommand>::Run(
        nn::os::GetHostArgc(),
        const_cast<const char**>(nn::os::GetHostArgv())
    );
}

#elif defined(NN_TOOL_REPAIR_INITIALIZER)

extern "C" void nnMain()
{
    const char *emptyArgv[] = { "--verbose" };
    nn::utilTool::BasicCommandFramework<InitializeSystemCommand>::Run(
        1,
        emptyArgv
    );
}

#else

extern "C" void nnMain()
{
    SingleHtcsConnection connection;
    ProgramArguments arguments;

    nn::htcs::Initialize(Allocate, Deallocate);

    NN_UTILTOOL_ABORT_UNLESS_RESULT_SUCCESS(
        CreateConnectionAsHtcs(&connection, "@InitializeSystem"));
    NN_UTIL_SCOPE_EXIT {
        nn::htcs::Close(connection.clientSocket);
        nn::htcs::Close(connection.serverSocket);
    };

    NN_UTILTOOL_ABORT_UNLESS_RESULT_SUCCESS(
        ReceiveProgramArguments(&arguments, connection));

    nn::utilTool::BasicCommandFramework<InitializeSystemCommand>::Run(
        arguments.argumentCount,
        arguments.arguments.get());
}

#endif
