﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "ListFilesInFatPartition.h"
#include "ExportPartition.h"
#include "ListExportablePartitions.h"
#include <nn/fs/fs_Result.h>
#include <nn/fs/fs_Bis.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/utilTool/utilTool_ResultHandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <vector>
#include <functional>

#include "Utility/systemInitializer_FsFile.h"
#include "Utility/systemInitializer_FsBisPartition.h"
#include "Utility/systemInitializer_StorageUtility.h"
#include "Utility/systemInitializer_FileSystemUtility.h"

nn::Result EnumerateDirectoryEntries(std::string rootPath, std::function<nn::Result (std::string, nn::fs::DirectoryEntryType)> &action)
{
    nn::fs::DirectoryHandle directoryHandle;
    NN_UTILTOOL_RESULT_DO(
       nn::fs::OpenDirectory(&directoryHandle, rootPath.c_str(), nn::fs::OpenDirectoryMode_All));
    NN_UTIL_SCOPE_EXIT {nn::fs::CloseDirectory(directoryHandle);};

    int64_t entryCount;
    NN_UTILTOOL_RESULT_DO(
        nn::fs::GetDirectoryEntryCount(&entryCount, directoryHandle));

    std::vector<nn::fs::DirectoryEntry> directoryEntries;
    directoryEntries.resize(entryCount);

    int64_t readEntryCount;
    NN_UTILTOOL_RESULT_DO(
        nn::fs::ReadDirectory(&readEntryCount, &directoryEntries[0], directoryHandle, directoryEntries.size()));

    NN_UTILTOOL_LOG_DEBUG("nn::fs::GetDirectoryEntryCount = %lld", entryCount);
    NN_UTILTOOL_LOG_DEBUG("nn::fs::ReadDirectory = %lld", readEntryCount);

    directoryEntries.resize(readEntryCount);

    if(rootPath[rootPath.length() - 1] != '/')
    {
        rootPath += "/";
    }

    NN_UTILTOOL_RESULT_DO(
        action(rootPath, nn::fs::DirectoryEntryType_Directory));

    for(auto directoryEntry : directoryEntries)
    {
        switch(directoryEntry.directoryEntryType)
        {
        case nn::fs::DirectoryEntryType_File:
            {
                NN_UTILTOOL_LOG_FORCE("file: %s", (rootPath + directoryEntry.name).c_str());
                NN_UTILTOOL_RESULT_DO(
                    action(rootPath + directoryEntry.name, nn::fs::DirectoryEntryType_File));
                break;
            }
        case nn::fs::DirectoryEntryType_Directory:
            {
                NN_UTILTOOL_RESULT_DO(
                    EnumerateDirectoryEntries(rootPath + directoryEntry.name, action));
                break;
            }
        default:
            NN_ABORT();
        }
    }

    NN_RESULT_SUCCESS;
}

nn::Result ListFilesInFatPartition(std::string outputPath, std::string type, std::string name)
{
    if (type != "fs-partition")
    {
        NN_UTILTOOL_LOG_ERROR("No support paritition type(support only fs-partition). type='%s'", type.c_str());
        NN_UTILTOOL_RESULT_THROW(nn::fs::ResultPartitionNotFound());
    }

    bool hasPartition;
    NN_UTILTOOL_RESULT_DO(
        HasExportablePartition(&hasPartition, type, name));

    if (!hasPartition)
    {
        NN_UTILTOOL_LOG_ERROR("Found no partitions(type='%s', name='%s').", type.c_str(), name.c_str());
        NN_UTILTOOL_RESULT_THROW(nn::fs::ResultPartitionNotFound());
    }

    bool exist;
    NN_UTILTOOL_RESULT_DO(
        FsFile::Exists(&exist, outputPath.c_str()));

    if(!exist)
    {
        NN_UTILTOOL_RESULT_DO(
            FsFile::Create(outputPath.c_str()));
    }

    FsFile file;
    NN_UTILTOOL_RESULT_DO(
        file.OpenWrite(outputPath.c_str()));
    NN_UTIL_SCOPE_EXIT { file.Close(); };
    int64_t offset = 0;

    NN_UTILTOOL_LOG_DEBUG("image opened");

    auto bisPartitionId = ConvertToBisPartitionId(name);
    NN_UTILTOOL_RESULT_DO(nn::fs::MountBis(bisPartitionId, nullptr));

    auto mountName = std::string(nn::fs::GetBisMountName(bisPartitionId)) + ":/";

    std::function<nn::Result (std::string, nn::fs::DirectoryEntryType)> writer = [&file, &offset] (std::string fileName, nn::fs::DirectoryEntryType type) -> nn::Result {
            NN_UTILTOOL_LOG_MESSAGE("%s", fileName.c_str());
            fileName += "\n";
            NN_UTILTOOL_RESULT_DO(
                file.Write(offset, fileName.c_str(), fileName.length(), false));
            offset += fileName.length();
            NN_RESULT_SUCCESS;
        };

    NN_UTILTOOL_RESULT_DO(EnumerateDirectoryEntries(mountName, writer));

    NN_UTILTOOL_RESULT_DO(file.Flush());

    NN_RESULT_SUCCESS;
}
