﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Result.h>
#include <nn/sf/sf_HipcServer.h>

#include <nn/spl/spl_Api.h>
#include "spl_GeneralServer.h"

namespace nn { namespace spl {

    class CryptoServer : public GeneralServer
    {
    public:
        explicit CryptoServer(SecureMonitorManager* p)
            : GeneralServer(p)
        {
        }

        virtual ~CryptoServer() NN_NOEXCEPT
        {
            m_Manager.DeallocateAesKeySlots( this );
        }

        nn::Result GenerateAesKek(
            nn::sf::Out<nn::spl::AccessKey> accessKey,
            const nn::spl::detail::KeySource& keySource,
            std::int32_t generation,
            nn::Bit32 option) NN_NOEXCEPT
        {
            return m_Manager.GenerateAesKek(
                &*accessKey,
                &keySource,
                sizeof(keySource),
                generation,
                option );
        }

        nn::Result LoadAesKey(
            std::int32_t slotIndex,
            const nn::spl::AccessKey& accessKey,
            const nn::spl::detail::KeySource& keySource) NN_NOEXCEPT
        {
            return m_Manager.LoadAesKey(slotIndex, this, accessKey, &keySource, sizeof(keySource));
        }

        nn::Result GenerateAesKey(
            nn::sf::Out<nn::spl::detail::AesKey> key,
            const nn::spl::AccessKey& accessKey,
            const nn::spl::detail::KeySource& keySource) NN_NOEXCEPT
        {
            return m_Manager.GenerateAesKey(
                &*key,
                sizeof(*key),
                accessKey,
                &keySource,
                sizeof(keySource));
        }

        nn::Result DecryptAesKey(
            nn::sf::Out<nn::spl::detail::AesKey> key,
            const nn::spl::detail::KeySource& keySource,
            std::int32_t generation,
            nn::Bit32 option) NN_NOEXCEPT
        {
            return m_Manager.DecryptAesKey(
                &*key,
                sizeof(*key),
                &keySource,
                sizeof(keySource),
                generation,
                option );
        }

        nn::Result ComputeCtr(
            const nn::sf::OutBuffer& outBuffer,
            std::int32_t slotIndex,
            const nn::sf::InBuffer& inBuffer,
            const nn::spl::detail::IvIc& initialCounter) NN_NOEXCEPT
        {
            return m_Manager.ComputeCtr(
                outBuffer.GetPointerUnsafe(),
                outBuffer.GetSize(),
                slotIndex,
                this,
                inBuffer.GetPointerUnsafe(),
                inBuffer.GetSize(),
                &initialCounter,
                sizeof(initialCounter) );
        }

        nn::Result ComputeCmac(
            nn::sf::Out<nn::spl::detail::Cmac> pOut,
            std::int32_t slotIndex,
            const nn::sf::InArray<nn::Bit8>& data) NN_NOEXCEPT
        {
            return m_Manager.ComputeCmac(
                &*pOut,
                sizeof(*pOut),
                slotIndex,
                this,
                data.GetData(),
                data.GetLength() );
        }

        nn::Result AllocateAesKeySlot(
            nn::sf::Out<std::int32_t> slotIndex) NN_NOEXCEPT
        {
            return m_Manager.AllocateAesKeySlot( &*slotIndex, this );
        }

        nn::Result DeallocateAesKeySlot(
            std::int32_t slotIndex) NN_NOEXCEPT
        {
            return m_Manager.DeallocateAesKeySlot( slotIndex, this );
        }

        nn::Result GetAesKeySlotAvailableEvent(
            nn::sf::Out<nn::sf::NativeHandle> handle) NN_NOEXCEPT
        {
            auto pEvent = m_Manager.GetAesKeySlotAvailableEvent();
            auto internalHandle = nn::os::GetReadableHandleOfSystemEvent(pEvent->GetBase());
            *handle = nn::sf::NativeHandle(internalHandle, false);
            return ResultSuccess();
        }
    };

}}  // namespace nn::spl
