﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/sm/sm_Result.h>
#include <nn/svc/svc_Tcb.h>
#include <nn/svc/svc_Base.h>
#include <nn/svc/svc_Result.h>
#include <cstring>
#include <nn/diag/detail/diag_DetailStructuredSdkLog.h>
#include <nn/nn_Abort.h>

#include "sm_ServiceManager.h"
#include "sm_WaitList.h"

#define NN_SM_WARN(...)     NN_DETAIL_STRUCTURED_SDK_LOG(sm, Warn,   0, ##__VA_ARGS__)
#define NN_SM_ERROR(...)    NN_DETAIL_STRUCTURED_SDK_LOG(sm, Error,  0, ##__VA_ARGS__)

namespace nn { namespace sm {

    const ServiceName ServiceName::InvalidName = { { 0 } };

    namespace
    {
        const int ProcessCountMax = 64;
        const int ServiceCountMax = 256;
        const size_t AccessControlSizeMax = 512;

        class AccessControlEntry
        {
        public:
            AccessControlEntry(const void* p, size_t size)
            : m_pData(reinterpret_cast<const Bit8*>(p))
            , m_Size(size)
            {
            }

            bool IsValid() const
            {
                if( m_pData == NULL || m_Size == 0 )
                {
                    return false;
                }
                if( 1 + GetNameSize() > m_Size )
                {
                    return false;
                }

                return true;
            }

            size_t GetNameSize() const
            {
                return (*m_pData & 0x7) + 1;
            }

            ServiceName GetName() const
            {
                return ServiceName::Make(
                    reinterpret_cast<const char*>(m_pData + 1), GetNameSize());
            }

            bool IsServer() const
            {
                return (*m_pData >> 7) != 0;
            }

            bool HasWildCard() const
            {
                return *(m_pData + 1 + GetNameSize() - 1) == '*';
            }

            AccessControlEntry GetNext() const
            {
                size_t size = 1 + GetNameSize();
                return AccessControlEntry(m_pData + size, m_Size - size);
            }

        private:
            const Bit8* m_pData;
            size_t      m_Size;
        };

        struct ProcessInfo
        {
            os::ProcessId   processId;
            size_t          accessControlSize;
            Bit8            accessControl[AccessControlSizeMax];
            bool            usedFlag;
        };

        struct ServiceInfo
        {
            ServiceName     name;
            os::ProcessId   ownerId;
            svc::Handle     handle;
            bool            usedFlag;
        };

        ProcessInfo g_Processes[ProcessCountMax];
        ServiceInfo g_Services[ServiceCountMax];


        ProcessInfo* Find(os::ProcessId id)
        {
            for( int i = 0; i < ProcessCountMax; ++i )
            {
                if( g_Processes[i].usedFlag && g_Processes[i].processId == id )
                {
                    return &g_Processes[i];
                }
            }

            return NULL;
        }
        ProcessInfo* FindUnusedProcessInfo()
        {
            for( int i = 0; i < ProcessCountMax; ++i )
            {
                if( g_Processes[i].usedFlag == false )
                {
                    return &g_Processes[i];
                }
            }

            return NULL;
        }
        ServiceInfo* Find(ServiceName name)
        {
            for( int i = 0; i < ServiceCountMax; ++i )
            {
                if( g_Services[i].usedFlag && g_Services[i].name == name )
                {
                    return &g_Services[i];
                }
            }

            return NULL;
        }
        ServiceInfo* FindUnusedServiceInfo()
        {
            for( int i = 0; i < ServiceCountMax; ++i )
            {
                if( g_Services[i].usedFlag == false )
                {
                    return &g_Services[i];
                }
            }

            return NULL;
        }
        void FreeService(ServiceInfo* psi)
        {
            std::memset(psi, 0, sizeof(*psi));
            psi->usedFlag = false; // 冗長だけど念のため明示的に初期化
        }
        bool IsValidName(ServiceName name)
        {
            if( name.value[0] == '\0' )
            {
                return false;
            }
            size_t i = 1;
            for( ; i < sizeof(name); ++i )
            {
                if( name.value[i] == '\0' )
                {
                    break;
                }
            }
            for( ; i < sizeof(name); ++i )
            {
                if( name.value[i] != '\0' )
                {
                    return false;
                }
            }

            return true;
        }
        bool IsPermitted(AccessControlEntry ac, ServiceName name, bool isServer, bool hasWildCard)
        {
            for( ; ac.IsValid(); ac = ac.GetNext() )
            {
                // (server && server) || (client && client)
                if( ac.IsServer() == isServer )
                {
                    // (wildcard && wildcard) || (normal && normal)
                    if( ac.HasWildCard() == hasWildCard )
                    {
                        if( ac.GetName() == name )
                        {
                            return true;
                        }
                    }
                    // (wildcard && normal)
                    else if( ac.HasWildCard() )
                    {
                        auto acName = ac.GetName();
                        if( std::memcmp(&acName, &name, ac.GetNameSize() - 1) == 0 )
                        {
                            return true;
                        }
                    }
                }
            }

            return false;
        }



        Result RegisterServiceNoCheck(
            svc::Handle*    pOut,
            os::ProcessId   id,
            ServiceName     name,
            int             sessionCountMax,
            bool            isLight ) NN_NOEXCEPT
        {
            // 名前のチェック
            if( ! IsValidName(name) )
            {
                return ResultInvalidName();
            }

            // 空き領域確保
            auto psi = FindUnusedServiceInfo();
            if( psi == NULL )
            {
                return ResultMaxService();
            }

            // ポート作成
            Result result;
            svc::Handle serverPort;
            svc::Handle clientPort;
            result = svc::CreatePort(&serverPort, &clientPort, sessionCountMax, isLight, reinterpret_cast<uintptr_t>(psi));
            if( result.IsFailure() )
            {
                return result;
            }

            // エントリ作成
            psi->name = name;
            psi->ownerId = id;
            psi->handle = clientPort;
            psi->usedFlag = true;

            TriggerResume(name);

            *pOut = serverPort;
            return ResultSuccess();
        }

        class InitialProgramIdRange
        {
        public:
            InitialProgramIdRange()
            {
                NN_ABORT_UNLESS_RESULT_SUCCESS(nn::svc::GetSystemInfo(&m_InitialProcessIdMin, nn::svc::SystemInfoType_InitialProcessIdRange, nn::svc::Handle(0), 0));
                NN_ABORT_UNLESS_RESULT_SUCCESS(nn::svc::GetSystemInfo(&m_InitialProcessIdMax, nn::svc::SystemInfoType_InitialProcessIdRange, nn::svc::Handle(0), 1));
                NN_ABORT_UNLESS(m_InitialProcessIdMin <= m_InitialProcessIdMax);
                NN_ABORT_UNLESS(0 < m_InitialProcessIdMin);
            }
            bool IsInitialProgram(os::ProcessId id) const
            {
                NN_ABORT_UNLESS(0 < m_InitialProcessIdMin);
                return (m_InitialProcessIdMin <= id.value && id.value <= m_InitialProcessIdMax);
            }
        private:
            Bit64 m_InitialProcessIdMin;
            Bit64 m_InitialProcessIdMax;
        };
        InitialProgramIdRange s_InitialProgramIdRange;

        bool IsInitialProgram(os::ProcessId id)
        {
            NN_ABORT_UNLESS(id.value != static_cast<Bit64>(-1ll));
            return s_InitialProgramIdRange.IsInitialProgram(id);
        }

    }   // anonymous namespace



    Result RegisterSelfService(
        svc::Handle*    pOut,
        const char*     pName,
        int             sessionCountMax ) NN_NOEXCEPT
    {
        Bit64 pidValue;

        Result result = nn::svc::GetProcessId(&pidValue, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS);
        if( result.IsFailure() )
        {
            return result;
        }

        os::ProcessId pid = { pidValue };
        ServiceName name = ServiceName::Make(pName, std::strlen(pName));

        return RegisterServiceNoCheck(pOut, pid, name, sessionCountMax, false);
    }

    Result RegisterProcess(
        os::ProcessId   id,
        const void*     pAccessRightsLimit,
        size_t          limitSize,
        const void*     pAccessRights,
        size_t          rightsSize ) NN_NOEXCEPT
    {
        // サイズチェック
        if( rightsSize > AccessControlSizeMax )
        {
            return ResultTooLargeAccessControl();
        }

        // 空き領域確保
        auto ppi = FindUnusedProcessInfo();
        if( ppi == NULL )
        {
            return ResultMaxProcess();
        }

        // meta の値が desc の値に収まっているかチェック
        AccessControlEntry limit(pAccessRightsLimit, limitSize);
        AccessControlEntry right(pAccessRights, rightsSize);

        for( AccessControlEntry right(pAccessRights, rightsSize);
             right.IsValid();
             right = right.GetNext() )
        {
            if( ! IsPermitted(limit, right.GetName(), right.IsServer(), right.HasWildCard()) )
            {
                NN_SM_ERROR("desc file does not accept meta value. (pid=%lld)\n", id);
                return ResultNotPermitted();
            }
        }

        // エントリ作成
        ppi->processId = id;
        ppi->accessControlSize = rightsSize;
        ppi->usedFlag = true;
        std::memcpy(ppi->accessControl, pAccessRights, ppi->accessControlSize);

        return ResultSuccess();
    }

    nn::Result UnregisterProcess(
        os::ProcessId   id) NN_NOEXCEPT
    {
        auto ppi = Find(id);
        if( ppi == NULL )
        {
            return ResultInvalidClient();
        }

        // エントリ解放
        ppi->usedFlag = false;
        ppi->processId.value = static_cast<Bit64>(-1ll);

        return ResultSuccess();
    }

    Result GetServiceHandle(
        svc::Handle*    pOut,
        os::ProcessId   id,
        ServiceName     name ) NN_NOEXCEPT
    {
        // 名前のチェック
        if( ! IsValidName(name) )
        {
            return ResultInvalidName();
        }

        if( ! IsInitialProgram(id) )
        {
            // アクセス許可確認
            auto ppi = Find(id);
            if( ppi == NULL )
            {
                NN_SM_WARN("pid=%lld not found\n", id);
                return ResultInvalidClient();
            }

            bool isPermitted = IsPermitted(
                                AccessControlEntry(ppi->accessControl, ppi->accessControlSize),
                                name,
                                false,
                                false );
            if( ! isPermitted )
            {
                NN_SM_WARN("pid=%lld, '%.*s'(client) is not permitted.\n", id, name.Length, name.value);
                return ResultNotPermitted();
            }
        }

        // サービス情報取得
        auto psi = Find(name);
        if( psi == NULL )
        {
            return StartRegisterRetry(name);
        }

        // 接続
        Result result;
        svc::Handle handle;
        result = svc::ConnectToPort(&handle, psi->handle);
        if( result.IsFailure() )
        {
            if( result <= nn::svc::ResultMaxSessions() )
            {
                return ResultMaxSessions();
            }

            return result;
        }

        *pOut = handle;
        return ResultSuccess();
    }

    Result RegisterService(
        svc::Handle*    pOut,
        os::ProcessId   id,
        ServiceName     name,
        int             sessionCountMax,
        bool            isLight ) NN_NOEXCEPT
    {
        // 名前のチェック
        if( ! IsValidName(name) )
        {
            return ResultInvalidName();
        }

        if( ! IsInitialProgram(id) )
        {

            // アクセス許可確認
            auto ppi = Find(id);
            if( ppi == NULL )
            {
                NN_SM_WARN("pid=%lld not found\n", id);
                return ResultInvalidClient();
            }

            bool isPermitted = IsPermitted(
                                AccessControlEntry(ppi->accessControl, ppi->accessControlSize),
                                name,
                                true,
                                false );
            if( ! isPermitted )
            {
                NN_SM_WARN("pid=%lld, '%.*s'(server) is not permitted.\n", id, name.Length, name.value);
                return ResultNotPermitted();
            }
        }

        // 重複登録チェック
        auto psi = Find(name);
        if( psi != NULL )
        {
            return ResultAlreadyRegistered();
        }

        return RegisterServiceNoCheck(pOut, id, name, sessionCountMax, isLight);
    }

    Result UnregisterService(
        os::ProcessId   id,
        ServiceName     name ) NN_NOEXCEPT
    {
        // 名前のチェック
        if( ! IsValidName(name) )
        {
            return ResultInvalidName();
        }

        if( ! IsInitialProgram(id) )
        {
            // アクセス元チェック
            auto ppi = Find(id);
            if( ppi == NULL )
            {
                NN_SM_WARN("pid=%lld not found\n", id);
                return ResultInvalidClient();
            }
        }

        // オーナーチェック
        auto psi = Find(name);
        if( psi == NULL )
        {
            return ResultNotRegistered();
        }

        if( psi->ownerId != id )
        {
            NN_SM_WARN("pid=%lld, '%.*s' cannot be unregistered.\n", id, name.Length, name.value);
            return ResultNotPermitted();
        }

        // エントリ解放
        svc::CloseHandle(psi->handle);
        FreeService(psi);

        return ResultSuccess();
    }

    Result UnregisterAllServices(
        os::ProcessId   id ) NN_NOEXCEPT
    {
        // アクセス元チェック
        auto ppi = Find(id);
        if( ppi == NULL )
        {
            return ResultInvalidClient();
        }

        // オーナーとなっているものをすべて解放
        for( int i = 0; i < ServiceCountMax; ++i )
        {
            if( g_Services[i].usedFlag && g_Services[i].ownerId == id )
            {
                svc::CloseHandle(g_Services[i].handle);
                FreeService(&g_Services[i]);
            }
        }

        return ResultSuccess();
    }


}}  // namespace nn::sm
