﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/sf/sf_HipcServer.h>
#include <nn/sf/sf_ExpHeapAllocator.h>
#include <nn/sf/sf_ObjectFactory.h>

#include <nn/nn_SdkLog.h>
#include <nn/ldr/detail/ldr_ProcessManagerInterface.h>
#include <nn/ldr/detail/ldr_ShellInterface.h>
#include <nn/ldr/detail/ldr_DebugMonitorInterface.h>
#include <nn/ldr/ldr_ProcessManagerApi.h>
#include <nn/ldr/ldr_ShellApi.h>
#include <nn/ldr/ldr_DebugMonitorApi.h>
#include <nn/spl/spl_Api.h>
#include <nn/fs/fs_ResultHandler.h>

#include "ldr_Loader.h"
#include "ldr_Random.h"

namespace
{
    enum PortIndex
    {
        PortIndex_ProcessManager,
        PortIndex_Shell,
        PortIndex_DebugMonitor,
        PortIndex_Count
    };

    static const int ProcessManagerSessionCount = 1;
    static const int ShellSessionCount          = 3;
    static const int DebugMonitorSessionCount   = 2;
    static const int TotalSessionCount = ProcessManagerSessionCount + ShellSessionCount + DebugMonitorSessionCount;

    struct ServerManagerOptions
    {
        static const size_t PointerTransferBufferSize = 1024;
    };

    class ServerManager : public nn::sf::HipcSimpleAllInOneServerManager<TotalSessionCount, PortIndex_Count, ServerManagerOptions>
    {
    private:
        virtual nn::Result OnNeedsToAccept(int portIndex, PortObjectImpl* pPort) NN_NOEXCEPT;
    };


    NN_ALIGNAS(64) nn::Bit8 g_DefaultServerAllocatorBuffer[16 * 1024];

    nn::sf::ExpHeapAllocator  g_DefaultServerAllocator;
    ServerManager g_ServerManager;



    nn::Result ServerManager::OnNeedsToAccept(int portIndex, PortObjectImpl* pPort) NN_NOEXCEPT
    {
        switch(portIndex)
        {
        case PortIndex_ProcessManager:
            {
                nn::sf::SharedPointer<nn::ldr::detail::IProcessManagerInterface> refInterface;

                refInterface = nn::sf::ObjectFactory<
                    nn::sf::ExpHeapAllocator::Policy
                >::CreateSharedEmplaced<
                    nn::ldr::detail::IProcessManagerInterface,
                    nn::ldr::Loader
                >(&g_DefaultServerAllocator);

                return this->AcceptImpl(pPort, refInterface);
            }
            break;

        case PortIndex_Shell:
            {
                nn::sf::SharedPointer<nn::ldr::detail::IShellInterface> refInterface;

                refInterface = nn::sf::ObjectFactory<
                    nn::sf::ExpHeapAllocator::Policy
                >::CreateSharedEmplaced<
                    nn::ldr::detail::IShellInterface,
                    nn::ldr::Loader
                >(&g_DefaultServerAllocator);

                return this->AcceptImpl(pPort, refInterface);
            }
            break;

        case PortIndex_DebugMonitor:
            {
                nn::sf::SharedPointer<nn::ldr::detail::IDebugMonitorInterface> refInterface;

                refInterface = nn::sf::ObjectFactory<
                    nn::sf::ExpHeapAllocator::Policy
                >::CreateSharedEmplaced<
                    nn::ldr::detail::IDebugMonitorInterface,
                    nn::ldr::Loader
                >(&g_DefaultServerAllocator);

                return this->AcceptImpl(pPort, refInterface);
            }
            break;

        default:
            NN_ABORT();
        }
    }



}

namespace nn { namespace ldr {
    bool g_IsDescProductionFlagRequired = true;
}}

extern "C" void nninitStartup()
{
}

extern "C" void nninitInitializeSdkModule()
{
}

extern "C" void nninitFinalizeSdkModule()
{
}

extern "C" void nnMain() NN_NOEXCEPT
{
    nn::fs::SetEnabledAutoAbort(false);

    auto heapHandle = nn::lmem::CreateExpHeap(&g_DefaultServerAllocatorBuffer, sizeof(g_DefaultServerAllocatorBuffer), nn::lmem::CreationOption_NoOption);
    g_DefaultServerAllocator.Attach(heapHandle);

    nn::Result result;

#if defined(NN_DETAIL_LDR_USE_SPL)
    nn::spl::Initialize();
    nn::ldr::g_IsDescProductionFlagRequired = !nn::spl::IsDevelopment();
    nn::spl::Finalize();
#elif defined(NN_BUILD_CONFIG_SPEC_GENERIC)
    nn::ldr::g_IsDescProductionFlagRequired = false;
#endif
    nn::ldr::random::Initialize();

    result = g_ServerManager.InitializePort(PortIndex_ProcessManager,   ProcessManagerSessionCount, nn::ldr::detail::PortNameForProcessManager);
    NN_ABORT_UNLESS(result.IsSuccess(), "result=%08x", result);

    result = g_ServerManager.InitializePort(PortIndex_Shell,            ShellSessionCount,          nn::ldr::detail::PortNameForShell);
    NN_ABORT_UNLESS(result.IsSuccess(), "result=%08x", result);

    result = g_ServerManager.InitializePort(PortIndex_DebugMonitor,     DebugMonitorSessionCount,   nn::ldr::detail::PortNameForDebugMonitor);
    NN_ABORT_UNLESS(result.IsSuccess(), "result=%08x", result);

    g_ServerManager.Start();

    nn::ldr::Loader::Initialize();

    g_ServerManager.LoopAuto();
}
