﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/util/util_UuidApi.h>

#include <nn/nn_Common.h>
#include <nn/crypto/crypto_Sha1Generator.h>
#include <nn/os/os_Random.h>
#include <nn/util/util_BitPack.h>
#include <nn/util/util_UuidTypes.h>

namespace nn {
namespace util {
namespace {

#if NN_BUILD_CONFIG_ENDIAN_SUPPORTS_LITTLE
inline uint64_t ConvertHostToNetwork48(uint64_t v) NN_NOEXCEPT
{
    NN_SDK_REQUIRES((v & 0xFFFF000000000000) == 0);
    return 0
        | ((v & 0x00000000000000FFull) << 40)
        | ((v & 0x000000000000FF00ull) << 24)
        | ((v & 0x0000000000FF0000ull) << 8)
        | ((v & 0x00000000FF000000ull) >> 8)
        | ((v & 0x000000FF00000000ull) >> 24)
        | ((v & 0x0000FF0000000000ull) >> 40);
}
inline uint32_t ConvertHostToNetwork(uint32_t v) NN_NOEXCEPT
{
    return 0
        | ((v & 0x000000FF) << 24)
        | ((v & 0x0000FF00) << 8)
        | ((v & 0x00FF0000) >> 8)
        | ((v & 0xFF000000) >> 24);
}
inline uint16_t ConvertHostToNetwork(uint16_t v) NN_NOEXCEPT
{
    return 0
        | ((v & 0x00FF) << 8)
        | ((v & 0xFF00) >> 8);
}
#elif NN_BUILD_CONFIG_ENDIAN_SUPPORTS_BIG
inline uint64_t ConvertHostToNetwork48(uint64_t v) NN_NOEXCEPT
{
    NN_SDK_REQUIRES((v & 0xFFFF00000000) == 0);
    return v;
}
template <typename T>
inline T ConvertHostToNetwork(T v) NN_NOEXCEPT
{
    return v;
}
#else
#error "Unknown endian type specified"
#endif

struct InternalUuid
{
    util::BitPack32 data[4];

    // data[0]
    typedef nn::util::BitPack32::Field<0,                       32, uint32_t>   TimeLow;
    // data[1]
    typedef nn::util::BitPack32::Field<0,                       16, uint16_t>   TimeMid;
    typedef nn::util::BitPack32::Field<TimeMid::Next,           16, uint16_t>   TimeHiAndVersion;
    // data[2]
    typedef nn::util::BitPack32::Field<0,                       8,  uint8_t>    ClkSeqHiAndRes;
    typedef nn::util::BitPack32::Field<ClkSeqHiAndRes::Next,    8,  uint8_t>    ClkSeqLow;
    typedef nn::util::BitPack32::Field<ClkSeqLow::Next,         16, uint16_t>   NodeLo;
    // data[4]
    typedef nn::util::BitPack32::Field<0,                       32, uint32_t>   NodeHi;

    Uuid Serialize() const NN_NOEXCEPT
    {
        util::BitPack32 serialized[4] = {};
        serialized[0].Set<TimeLow>(ConvertHostToNetwork(data[0].Get<TimeLow>()));
        serialized[1].Set<TimeMid>(ConvertHostToNetwork(data[1].Get<TimeMid>()));
        serialized[1].Set<TimeHiAndVersion>(ConvertHostToNetwork(data[1].Get<TimeHiAndVersion>()));
        serialized[2].Set<ClkSeqHiAndRes>(data[2].Get<ClkSeqHiAndRes>()); // 1byte なので無変換
        serialized[2].Set<ClkSeqLow>(data[2].Get<ClkSeqLow>()); // 1byte なので無変換
        auto nodeLo = static_cast<uint64_t>(data[2].Get<NodeLo>());
        auto nodeHi = static_cast<uint64_t>(data[3].Get<NodeHi>());
        uint64_t node = ConvertHostToNetwork48((nodeHi << 16) | nodeLo);
        serialized[2].Set<NodeLo>(static_cast<uint16_t>(node & 0xFFFFull));
        serialized[3].Set<NodeHi>(static_cast<uint32_t>((node >> 16) & 0xFFFFFFFFull));

        Uuid uuid;
        std::memcpy(uuid.data, serialized, sizeof(uuid.data));
        NN_STATIC_ASSERT(sizeof(uuid.data) == sizeof(serialized));
        return uuid;
    }
};

Uuid GenerateUuidVersion4() NN_NOEXCEPT
{
    const uint16_t Version = 0x4000;
    const uint16_t VersionMask = 0xF000;
    const uint16_t Reserved = 0x80;
    const uint16_t ReservedMask = 0xC0;
    InternalUuid uuid = {};
    os::GenerateRandomBytes(&uuid.data, sizeof(uuid.data));
    uuid.data[1].Set<InternalUuid::TimeHiAndVersion>((uuid.data[1].Get<InternalUuid::TimeHiAndVersion>() & ~VersionMask) | (Version));
    uuid.data[2].Set<InternalUuid::ClkSeqHiAndRes>((uuid.data[2].Get<InternalUuid::ClkSeqHiAndRes>() & ~ReservedMask) | (Reserved));
    return uuid.Serialize();
}
} // ~namespace nn::util::<anonymous>

Uuid GenerateUuidVersion5(const void* pSha1Hash) NN_NOEXCEPT
{
    const uint16_t Version = 0x5000;
    const uint16_t VersionMask = 0xF000;
    const uint16_t Reserved = 0x80;
    const uint16_t ReservedMask = 0xC0;
    InternalUuid uuid = {};
    std::memcpy( &uuid, pSha1Hash, sizeof(uuid.data) );
    uuid.data[1].Set<InternalUuid::TimeHiAndVersion>((uuid.data[1].Get<InternalUuid::TimeHiAndVersion>() & ~VersionMask) | (Version));
    uuid.data[2].Set<InternalUuid::ClkSeqHiAndRes>((uuid.data[2].Get<InternalUuid::ClkSeqHiAndRes>() & ~ReservedMask) | (Reserved));
    return uuid.Serialize();
}

Uuid GenerateUuid() NN_NOEXCEPT
{
    return GenerateUuidVersion4();
}

} // ~namespace nn::util
}
