﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using Nintendo;
using Nintendo.ServiceFramework;
using Nintendo.ServiceFramework.CppCode;
using Nintendo.ServiceFramework.Hipc;
using nn;
using nn.sf;
using std;

namespace nn.spl
{
    [CppRefPath("<nn/spl/spl_Types.h>")]
    [ExternalStruct(16, 1)]
    public struct AccessKey
    {
    }

    [CppRefPath("<nn/spl/spl_Types.h>")]
    [ExternalStruct(4, 4)]
    public struct BootReasonValue
    {
    }
}

namespace nn.spl.detail
{
    [CppRefPath("<nn/spl/detail/spl_InternalTypes.h>")]
    [ExternalStruct(16, 1)]
    public struct KeySource
    {
    }

    [CppRefPath("<nn/spl/detail/spl_InternalTypes.h>")]
    [ExternalStruct(16, 1)]
    public struct AesKey
    {
    }

    [CppRefPath("<nn/spl/detail/spl_InternalTypes.h>")]
    [ExternalStruct(16, 1)]
    public struct IvIc
    {
    }

    [CppRefPath("<nn/spl/detail/spl_InternalTypes.h>")]
    [ExternalStruct(16, 1)]
    public struct Cmac
    {
    }

    [CppRefPath("<nn/spl/detail/spl_InternalTypes.h>")]
    [ExternalStruct(32, 1)]
    public struct GcKey
    {
    }

    [CppRefPath("<nn/spl/detail/spl_InternalTypes.h>")]
    [ExternalStruct(64, 1)]
    public struct GcKeySource
    {
    }

    public interface IGeneralInterface : IServiceObject
    {
        [MethodId(0)]
        Result
            GetConfig(
                Out<Bit64>  pOut,
                int32_t     key
            );

        [MethodId(1)]
        Result
            ModularExponentiate(
                OutArray<Bit8>  result,
                InArray<Bit8>   baseValue,
                InArray<Bit8>   exponent,
                InArray<Bit8>   modulus
            );

        [MethodId(5)]
        Result
            SetConfig(
                int32_t         key,
                Bit64           value );

        [MethodId(7)]
        Result
            GenerateRandomBytes(
                OutArray<Bit8>  result
            );

        [MethodId(11)]
        Result
            IsDevelopment(
                Out<bool>       isDevelopment
            );

        [MethodId(24)]
        Result
            SetBootReason(
                BootReasonValue bootReason
            );

        [MethodId(25)]
        Result
            GetBootReason(
                Out<BootReasonValue> bootReason
            );
    }

    public interface ICryptoInterface : IGeneralInterface
    {
        [MethodId(2)]
        Result
            GenerateAesKek(
                Out<AccessKey>  accessKey,
                KeySource       keySource,
                int32_t         generation,
                Bit32           option
            );

        [MethodId(3)]
        Result
            LoadAesKey(
                int32_t         slotIndex,
                AccessKey       accessKey,
                KeySource       keySource
            );

        [MethodId(4)]
        Result
            GenerateAesKey(
                Out<AesKey>     key,
                AccessKey       accessKey,
                KeySource       keySource
            );

        [MethodId(14)]
        Result
            DecryptAesKey(
                Out<AesKey>     key,
                KeySource       keySource,
                int32_t         generation,
                Bit32           option
            );

        [MethodId(15)]
        Result
            ComputeCtr(
                [NonSecureMapTransfer] OutBuffer  outBuffer,
                int                               slotIndex,
                [NonSecureMapTransfer] InBuffer   inBuffer,
                IvIc                              initialCounter
            );

        [MethodId(16)]
        Result
            ComputeCmac(
                Out<Cmac>       pOut,
                int             slotIndex,
                InArray<Bit8>   data
            );

        [MethodId(21)]
        Result
            AllocateAesKeySlot(
                Out<int32_t>    slotIndex
            );

        [MethodId(22)]
        Result
            DeallocateAesKeySlot(
                int32_t         slotIndex
            );

        [MethodId(23)]
        Result
            GetAesKeySlotAvailableEvent(
                Out<NativeHandle> handle
            );
    }

    public interface IFsInterface : ICryptoInterface
    {
        [MethodId(9)]
        Result
            DecryptAndStoreGcKey(
                InArray<Bit8>   data,
                AccessKey       accessKey,
                KeySource       keySource
            );

        [MethodId(10)]
        Result
            DecryptGcMessage(
                Out<int>        resultSize,
                OutArray<Bit8>  result,
                InArray<Bit8>   cipher,
                InArray<Bit8>   modulus,
                InArray<Bit8>   labelDigest
            );

        [MethodId(12)]
        Result
            GenerateSpecificAesKey(
                Out<AesKey>     key,
                KeySource       keySource,
                int             generation,
                Bit32           purpose
            );

        [MethodId(19)]
        Result
            LoadPreparedAesKey(
                int32_t         slotIndex,
                AccessKey       accessKey
            );

        [MethodId(31)]
        Result
            GetPackage2Hash(
                OutArray<Bit8>  result
            );
    }

    public interface IDeviceUniqueDataInterface : ICryptoInterface
    {
        [MethodId(13)]
        Result
            DecryptDeviceUniqueData(
                OutArray<Bit8>  result,
                InArray<Bit8>   data,
                AccessKey       accessKey,
                KeySource       keySource
            );
    }

    public interface ISslInterface : IDeviceUniqueDataInterface
    {
        [MethodId(26)]
        Result
            DecryptAndStoreSslClientCertKey(
                InArray<Bit8>   data,
                AccessKey       accessKey,
                KeySource       keySource
            );

        [MethodId(27)]
        Result
            ModularExponentiateWithSslClientCertKey(
                OutArray<Bit8>  result,
                InArray<Bit8>   cipher,
                InArray<Bit8>   modulus
            );
    }

    public interface IEsInterface : IDeviceUniqueDataInterface
    {
        [MethodId(17)]
        Result
            LoadEsDeviceKey(
                InArray<Bit8>   data,
                AccessKey       accessKey,
                KeySource       keySource
            );

        [MethodId(18)]
        Result
            PrepareEsTitleKey(
                Out<AccessKey>  accessKey,
                InArray<Bit8>   cipher,
                InArray<Bit8>   modulus,
                InArray<Bit8>   labelDigest,
                int32_t         generation
            );

        [MethodId(20)]
        Result
            PrepareCommonEsTitleKey(
                Out<AccessKey>  accessKey,
                KeySource       keySource,
                int32_t         generation
            );

        [MethodId(28)]
        Result
            DecryptAndStoreDrmDeviceCertKey(
                InArray<Bit8>   data,
                AccessKey       accessKey,
                KeySource       keySource
            );

        [MethodId(29)]
        Result
            ModularExponentiateWithDrmDeviceCertKey(
                OutArray<Bit8>  result,
                InArray<Bit8>   cipher,
                InArray<Bit8>   modulus
            );

        [MethodId(31)]
        Result
            PrepareEsArchiveKey(
                Out<AccessKey>  accessKey,
                InArray<Bit8>   cipher,
                InArray<Bit8>   modulus,
                InArray<Bit8>   labelDigest,
                int32_t         generation
            );

        [MethodId(32)]
        Result
            LoadPreparedAesKey(
                int32_t         slotIndex,
                AccessKey       accessKey
            );
    }

    public interface IManuInterface : IDeviceUniqueDataInterface
    {
        [MethodId(30)]
        Result
            ReencryptDeviceUniqueData(
                OutArray<Bit8>  result,
                InArray<Bit8>   data,
                AccessKey       accessKeyForDecryption,
                KeySource       keySourceForDecryption,
                AccessKey       accessKeyForEncryption,
                KeySource       keySourceForEncryption,
                Bit32           option
            );
    }
}
