﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Abort.h>
#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/os.h>
#include <nn/os/os_Result.public.h>
#include <nn/sf/sf_ObjectFactory.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/nsd/nsd_ApiForMiddleware.h>

#include <nn/socket/socket_Api.h>
#include <nn/socket/resolver/private/resolver_PrivateApi.h>
#include <nn/socket/socket_TypesPrivate.h>
#include <nn/socket/socket_ResolverOptionsPrivate.h>
#include <nn/socket/private/socket_PlatformTypesTranslation.h>

#include "../../detail/socket_Allocator.h"

#include "resolver_CreateClient.h"
#include "resolver_ServiceName.h"
#include "resolver_HipcServer.h"
#include "resolver_Implementation.h"
#include "resolver_Cancel.h"
#include "resolver_ClientServerShared.h"
#include "resolver_OptionsHelper.h"
#include "resolver_DnsServerHelper.h"
#include "resolver_AutoContext.h"

extern "C"
{
#include <siglo/BionicPrivateApi.h>
#include <siglo/ResolverCacheAdditionalFunctionality.h>
};

#include "serializer.h"
#include "serializer_Specializations.h"

//#define LOG_LEVEL LOG_LEVEL_MAX
#define LOG_MODULE_NAME "sfs impl" // NOLINT(preprocessor/const)
#include <nn/socket/resolver/private/resolver_DebugLog.h>

namespace nn { namespace socket {

void DisableCacheByDefault();

namespace resolver {

static const
size_t MaximumResolverOptions = (static_cast<uint32_t>(ResolverOptionKey::RequestMaximumValue) -
                                 static_cast<uint32_t>(ResolverOptionKey::RequestMinimumValue) - 1);

typedef nn::sf::UnmanagedServiceObject<IResolver, Server> ResolverServiceObject;
extern ResolverServiceObject* g_pResolverServerManager;

ResolverImpl::ResolverImpl() :
    m_AccessLock(true)
{
    ZeroDnsServerSettings();
};

ResolverImpl::~ResolverImpl()
{
};

// Move the string into a fqdn structure for nn::nsd::Resolve to use.
void FixupHostName(nn::nsd::Fqdn *nsdResolveName, const char *pHostName)
{
    // Create a local version of the fqdn structure in case Resolve
    // needs 2 entities for safety

    nn::nsd::Fqdn nsdTempName;

    size_t size = strlen(pHostName);
    if (size > nsdTempName.Size - 1)
    {
        size = nsdTempName.Size - 1;
    }

    memcpy(nsdTempName.value, pHostName, size);

    nsdTempName.value[size] = '\0';

    nn::nsd::ResolveEx(nsdResolveName, nsdTempName);
};

/**
 * @brief This macro makes calls to GetPointerUnsafe a little safer
 *
 * @detail
 * Calls to @ref nn::sf::InBuffer::GetPointerUnsafe and @ref
 * nn::sf::OutBuffer::GetPointerUnsafe are unsafe and might return
 * a garbage value. This macro makes those calls a little safer.
 * It does this by always setting pOut to the null pointer value,
 * then setting sizeOut to the size parameter of the buffer, and
 * only if data are present it assigns pOut the value from
 * GetPointerUnsafe.
 *
 * @param pOut This parameter holds the pointer to the buffer on
 * success otherwise it is nullptr.
 *
 * @param sizeOut This parameter contains the size of the buffer
 * which is useful for later checking.
 *
 * @param type type This parameter is the value from
 * @ref ResolverOptionType enum corresponding to the
 * @ref ResolverOptionKey that you want to get from the options.
 * For example: If the key is @ref ResolverOptionKey::RequestCancelHandleInteger
 * then the type must be @ref ResolverOptionType::Integer.
 *
 * @brief bufferObject This is the buffer object to run
 * GetPointerUnsafe on.
 */
#define GET_POINTER_SAFER(pOut, sizeOut, type, bo)                   \
    do                                                               \
    {                                                                \
        pOut = nullptr;                                              \
        if (0 != (sizeOut = bo.GetSize()))                           \
        {                                                            \
            pOut = reinterpret_cast<type>(bo.GetPointerUnsafe());    \
        };                                                           \
    } while (NN_STATIC_CONDITION(false))

/**
 * @brief
 */
#define GET_REQUEST_OPTION(outValue,                                 \
                           rc,                                       \
                           version,                                  \
                           key,                                      \
                           type,                                     \
                           pOptions,                                 \
                           count)                                    \
    do                                                               \
    {                                                                \
        if (-1 == GetRequestOptionValue(outValue,                    \
                                        version,                     \
                                        key,                         \
                                        pOptions,                    \
                                        count))                      \
        {                                                            \
            LogMinor("GetRequestOptionValue failed. "                \
                     "pOptions: %p, version: %u, key: %s, "          \
                     "value: %s, error %u\n",                        \
                     pOptions, version, #key,                        \
                     #outValue, __LINE__ );                          \
            errno = static_cast<int>(nn::socket::Errno::EInval);     \
            ++rc;                                                    \
        };                                                           \
    } while (NN_STATIC_CONDITION(false))


Result ResolverImpl::
SetDnsAddressesPrivateRequest(const nn::sf::InBuffer& socketInBuffer,
                              size_t length) NN_NOEXCEPT
{
#if defined(NN_SDK_BUILD_RELEASE)
    return ResultInternalError();
#else

    DisableCacheByDefault();

    const nn::socket::SockAddrIn* addresses = nullptr;
    if (0 != socketInBuffer.GetSize())
    {
        addresses = reinterpret_cast<const nn::socket::SockAddrIn*>(
            socketInBuffer.GetPointerUnsafe());
        return SetSockaddrDnsAddresses(addresses, length);
    };
    return ResultSuccess();
#endif
};

Result ResolverImpl::
GetDnsAddressPrivateRequest(uint32_t nIndex,
                            const nn::sf::OutBuffer& socketAddress) NN_NOEXCEPT
{
#if defined(NN_SDK_BUILD_RELEASE)
    return ResultInternalError();
#else
    struct nn::socket::SockAddrIn* pSinAddr = nullptr;

    if (0 != socketAddress.GetSize())
    {
        pSinAddr = reinterpret_cast<struct nn::socket::SockAddrIn*>(
            socketAddress.GetPointerUnsafe());
        return GetDnsAddress(nIndex, pSinAddr);
    };
    return ResultSuccess();
#endif
};

Result ResolverImpl::
GetHostByNameRequestWithOptions(Bit64 processId,
                                const nn::sf::InBuffer& nameParam,
                                const nn::sf::OutBuffer& serializedHostEntryOut,
                                const nn::sf::Out<uint32_t> pSerializedHostEntrySizeOut,
                                uint32_t optionVersion,
                                const nn::sf::InBuffer& serializedOptionsInBuffer,
                                uint32_t optionsCount,
                                const nn::sf::Out<int> pHostErrorOut,
                                const nn::sf::Out<int> pErrnoOut) NN_NOEXCEPT
{
    serializer::DNSSerializer serializer;
    Result result = ResultSuccess();

    const uint8_t* pSerializedOptions = nullptr;
    size_t serializedOptionsSize = 0;

    uint8_t* pOutBuffer = nullptr;
    size_t outBufferSize = 0;

    const char* pHostName = nullptr;
    size_t hostNameSize = 0;

    bool useNsdResolve = false;
    nn::nsd::Fqdn nsdResolveName;
    int cancelHandle = 0;
    struct hostent* pHostEntry = nullptr;

    int rc = 0;

    ResolverOption* pOptions = NULL;

    OptionContext mask = OptionContext_Request;

#if ! defined(NN_SDK_BUILD_RELEASE)
    *reinterpret_cast<unsigned*>(&mask) |= OptionContext_PrivateRequest;
#endif

    ResolverAutoContext autoContext;

    *pErrnoOut = errno = 0;
    *pHostErrorOut = static_cast<int>(nn::socket::HErrno::Netdb_Internal);
    *pSerializedHostEntrySizeOut = 0;

    if (optionsCount > MaximumResolverOptions)
    {
        LogMajor("Cannot duplicate resolver options\n");
        goto bail;
    };
    pOptions = reinterpret_cast<ResolverOption*>(alloca(optionsCount * sizeof(*pOptions)));;

    GET_POINTER_SAFER(pSerializedOptions, serializedOptionsSize, const uint8_t*, serializedOptionsInBuffer);
    GET_POINTER_SAFER(pHostName,          hostNameSize,          const char*,    nameParam);
    GET_POINTER_SAFER(pOutBuffer,         outBufferSize,         uint8_t*,       serializedHostEntryOut);

    if (nullptr == pHostName)
    {
        LogMajor("pHostName is NULL\n");
        *pHostErrorOut = HOST_NOT_FOUND;
        *pErrnoOut = static_cast<int>(nn::socket::Errno::EInval);
        goto bail;
    }
    else if (nullptr == pOutBuffer)
    {
        LogMajor("pOutBuffer is NULL\n");
        *pHostErrorOut = static_cast<int>(nn::socket::AiErrno::EAi_Fail);
        *pErrnoOut = static_cast<int>(nn::socket::Errno::EInval);
        goto bail;
    }
    else if (0 != optionsCount)
    {
        if (nullptr == pSerializedOptions)
        {
            LogMajor("Non-zero optionsCount (%u) but null options.\n", optionsCount);
            *pHostErrorOut = static_cast<int>(nn::socket::HErrno::Netdb_Internal);
            *pErrnoOut = static_cast<int>(nn::socket::Errno::EInval);
            goto bail;
        }
        else if ( -1 == ( rc = serializer.FromBuffer(pOptions,
                                                     optionsCount * sizeof(*pOptions),
                                                     pSerializedOptions,
                                                     serializedOptionsSize,
                                                     optionsCount)))
        {
            LogMajor("options frombuffer serializer failed\n");
            goto bail;
        }
        else if (!ValidateOptionsArrayForVersionAndMask(pOptions, optionsCount, optionVersion, mask))
        {
            LogMinor("Options validation failed.\n");
            *pErrnoOut = errno;
            goto bail;
        };
        LogDebug("Options validation succeeded\n");
    };

    GET_REQUEST_OPTION(useNsdResolve, rc, optionVersion,
                       ResolverOptionKey::RequestEnableServiceDiscoveryBoolean,
                       ResolverOptionType::Boolean, pOptions, optionsCount);

    GET_REQUEST_OPTION(cancelHandle, rc, optionVersion,
                       ResolverOptionKey::RequestCancelHandleInteger,
                       ResolverOptionType::Integer, pOptions, optionsCount);

    if (!autoContext.Initialize(processId, cancelHandle, pOptions, optionsCount))
    {
        LogDebug("autoContext initialization failed %s\n", __FUNCTION__);
        *pHostErrorOut = static_cast<int>(nn::socket::HErrno::Try_Again);
        *pErrnoOut = static_cast<int>(nn::socket::Errno::EAgain);
        goto bail;
    };

    UpdateBionicServers();
    if(useNsdResolve)
    {
        FixupHostName(&nsdResolveName, pHostName);
        pHostName = nsdResolveName.value;
    };

    pHostEntry = bionic_gethostbyname(pHostName);
    *pHostErrorOut = static_cast<int>(*nn::socket::GetHError());
    *pErrnoOut = static_cast<uint32_t>(nn::socket::GetLastError());

    if (nullptr == pHostEntry)
    {
        bool isExpired = false;
        bool isCancelled = autoContext.IsSlotCancelledOrExpired(isExpired);
        if (true == isExpired)
        {
            *pHostErrorOut = static_cast<int>(nn::socket::HErrno::Netdb_Internal);
            *pErrnoOut = static_cast<int>(nn::socket::Errno::ETimedOut);
        }
        else if (true == isCancelled)
        {
            *pHostErrorOut = static_cast<int>(nn::socket::HErrno::Netdb_Internal);
            *pErrnoOut = static_cast<int>(nn::socket::Errno::ECanceled);
        }
        else if (static_cast<int>(nn::socket::Errno::ENoMem) == *pErrnoOut)
        {
            *pHostErrorOut = static_cast<int>(nn::socket::HErrno::Try_Again);
        };
    }
    //      and we got something back in a hostent
    // ... but the serializer failed
    else if ( -1 == (rc = serializer.ToBuffer(pOutBuffer,
                                              outBufferSize,
                                              *pHostEntry)))
    {
        // in this case the serializer itself failed; if errno is ENoMem
        // then we tell the client they can try again
        if( nn::socket::Errno::ENoMem == nn::socket::GetLastError() )
        {
            *pHostErrorOut = static_cast<int>(nn::socket::HErrno::Try_Again);
            *pErrnoOut = static_cast<int>(nn::socket::Errno::ENoMem);
        }
        else
        {
            *pHostErrorOut = static_cast<int>(nn::socket::HErrno::No_Recovery);
            *pErrnoOut = static_cast<int>(nn::socket::Errno::EInval);
        };
    }
    else
    {
        // this is the valid number of bytes in the buffer
        *pSerializedHostEntrySizeOut = rc;
    };

bail:
    return result;
}; // NOLINT(impl/function_size)

Result ResolverImpl::GetHostByAddrRequestWithOptions(Bit64 processId,
                                                     const nn::sf::InBuffer& inAddrParam,
                                                     uint32_t inAddrLengthParam,
                                                     int addressFamilyParam,
                                                     const nn::sf::OutBuffer& serializedHostEntryOut,
                                                     const nn::sf::Out<uint32_t> pSerializedHostEntrySizeOut,
                                                     uint32_t optionVersion,
                                                     const nn::sf::InBuffer& serializedOptionsInBuffer,
                                                     uint32_t optionsCount,
                                                     const nn::sf::Out<int> pHostErrorOut,
                                                     const nn::sf::Out<int> pErrnoOut) NN_NOEXCEPT
{
    serializer::DNSSerializer serializer;
    Result result = ResultSuccess();

    const uint8_t* pSerializedOptions = nullptr;
    size_t serializedOptionsSize = 0;

    const uint8_t* pInaddrIn = nullptr;
    size_t inAddrSize = 0;

    uint8_t* pOutBuffer = nullptr;
    size_t outBufferSize = 0;

    int cancelHandle = 0;

    struct hostent* pHostEntry = nullptr;

    int rc = 0;

    ResolverOption* pOptions = NULL;

    OptionContext mask = OptionContext_Request;

#if ! defined(NN_SDK_BUILD_RELEASE)
    *reinterpret_cast<unsigned*>(&mask) |= OptionContext_PrivateRequest;
#endif

    ResolverAutoContext autoContext;

    *pErrnoOut = errno = 0;
    *pHostErrorOut = static_cast<int>(nn::socket::HErrno::Netdb_Internal);
    *pSerializedHostEntrySizeOut = 0;

    if (optionsCount > MaximumResolverOptions)
    {
        LogMajor("Cannot duplicate resolver options\n");
        goto bail;
    };
    pOptions = reinterpret_cast<ResolverOption*>(alloca(optionsCount * sizeof(*pOptions)));;

    GET_POINTER_SAFER(pSerializedOptions, serializedOptionsSize, const uint8_t*, serializedOptionsInBuffer);
    GET_POINTER_SAFER(pInaddrIn,          inAddrSize,            const uint8_t*, inAddrParam);
    GET_POINTER_SAFER(pOutBuffer,         outBufferSize,         uint8_t*,       serializedHostEntryOut);

    if (nullptr == pInaddrIn)
    {
        LogMajor("pInaddrIn is NULL\n");
        *pHostErrorOut = HOST_NOT_FOUND;
        *pErrnoOut = static_cast<int>(nn::socket::Errno::EInval);
        goto bail;
    }
    else if (nullptr == pOutBuffer)
    {
        *pHostErrorOut = HOST_NOT_FOUND;
        *pErrnoOut = static_cast<int>(nn::socket::Errno::EInval);
        goto bail;
    }
    else if (0 != optionsCount)
    {
        if (nullptr == pSerializedOptions)
        {
            LogMajor("Non-zero optionsCount (%u) but null options.\n", optionsCount);
            *pHostErrorOut = static_cast<int>(nn::socket::HErrno::Netdb_Internal);
            *pErrnoOut = static_cast<int>(nn::socket::Errno::EInval);
            goto bail;
        }
        else if ( -1 == ( rc = serializer.FromBuffer(pOptions,
                                                     optionsCount * sizeof(*pOptions),
                                                     pSerializedOptions,
                                                     serializedOptionsSize,
                                                     optionsCount)))
        {
            LogMajor("options frombuffer serializer failed\n");
            goto bail;
        }
        else if (!ValidateOptionsArrayForVersionAndMask(pOptions, optionsCount, optionVersion, mask))
        {
            LogMinor("Options validation failed.\n");
            *pErrnoOut = errno;
            goto bail;
        };
        LogDebug("Options validation succeeded\n");
    };

    GET_REQUEST_OPTION(cancelHandle, rc, optionVersion,
                       ResolverOptionKey::RequestCancelHandleInteger,
                       ResolverOptionType::Integer, pOptions, optionsCount);

    if (!autoContext.Initialize(processId, cancelHandle, pOptions, optionsCount))
    {
        result = ResultSuccess();
        LogDebug("autoContext initialization failed %s\n", __FUNCTION__);
        *pHostErrorOut = static_cast<int>(nn::socket::HErrno::Try_Again);
        *pErrnoOut = static_cast<int>(nn::socket::Errno::EAgain);
        goto bail;
    }

    UpdateBionicServers();
    pHostEntry = bionic_gethostbyaddr(pInaddrIn,
                                      inAddrLengthParam,
                                      addressFamilyParam);

    *pHostErrorOut = static_cast<int>(*nn::socket::GetHError());
    *pErrnoOut = errno;

    if (nullptr == pHostEntry)
    {
        bool isExpired = false;
        bool isCancelled = autoContext.IsSlotCancelledOrExpired(isExpired);
        if (true == isExpired)
        {
            *pHostErrorOut = static_cast<int>(nn::socket::HErrno::Netdb_Internal);
            *pErrnoOut = static_cast<int>(nn::socket::Errno::ETimedOut);
        }
        else if (true == isCancelled)
        {
            *pHostErrorOut = static_cast<int>(nn::socket::HErrno::Netdb_Internal);
            *pErrnoOut = static_cast<int>(nn::socket::Errno::ECanceled);
        }
        else if (static_cast<int>(nn::socket::Errno::ENoMem) == *pErrnoOut)
        {
            *pHostErrorOut = static_cast<int>(nn::socket::HErrno::Try_Again);
        };
    }
    else if (-1 == (rc = serializer.ToBuffer(pOutBuffer,
                                             outBufferSize,
                                             *pHostEntry)))
    {
        // in this case the serializer itself failed; if errno is ENoMem
        // then we tell the client they can try again
        if ( nn::socket::Errno::ENoMem == nn::socket::GetLastError())
        {
            *pHostErrorOut = static_cast<int>(nn::socket::HErrno::Try_Again);
            *pErrnoOut = static_cast<int>(nn::socket::Errno::ENoMem);
        }
        else
        {
            *pHostErrorOut = static_cast<int>(nn::socket::HErrno::No_Recovery);
            *pErrnoOut = static_cast<int>(nn::socket::Errno::EInval);
        };
    }
    else
    {
        *pSerializedHostEntrySizeOut = rc;
    };

    // the service framework call succeeded
    result = ResultSuccess();

bail:
    return result;
}; // NOLINT(impl/function_size)

Result ResolverImpl::GetAddrInfoRequestWithOptions(Bit64 processId,
                                                   const nn::sf::InBuffer& nodeNameParam,
                                                   const nn::sf::InBuffer& serviceNameParam,
                                                   const nn::sf::InBuffer& serializedHintsParam,
                                                   const nn::sf::OutBuffer serializedAddrInfoResultBuffer,
                                                   const nn::sf::Out<uint32_t> pSerializedAddrInfoResultBufferSize,
                                                   const nn::sf::Out<int> pReturnCodeOut,
                                                   uint32_t optionVersion,
                                                   const nn::sf::InBuffer& serializedOptionsInBuffer,
                                                   uint32_t optionsCount,
                                                   const nn::sf::Out<int> pHostErrorOut,
                                                   const nn::sf::Out<int> pErrnoOut) NN_NOEXCEPT
{
    serializer::DNSSerializer serializer;
    Result result = ResultSuccess();

    nn::nsd::Fqdn nsdResolveName;

    const uint8_t* pSerializedOptions = nullptr;
    size_t serializedOptionsSize = 0;

    const char* pNodeName = nullptr;
    size_t nodeNameSize =  0;

    const char* pServiceName = nullptr;
    size_t serviceNameSize = 0;

    const uint8_t* pHintsBuffer = nullptr;
    size_t hintsBufferSize = 0;

    uint8_t* pAddrInfoOutBuffer = nullptr;
    size_t addrInfoOutBufferSize = 0;

    int cancelHandle = 0;
    bool useNsdResolve = false;

    bool didInitializeHints = false;
    struct addrinfo hints = { 0 };
    struct addrinfo* pResults = nullptr;

    int rc = 0;

    ResolverOption* pOptions = NULL;

    OptionContext mask = OptionContext_Request;

#if ! defined(NN_SDK_BUILD_RELEASE)
    *reinterpret_cast<unsigned*>(&mask) |= OptionContext_PrivateRequest;
#endif

    ResolverAutoContext autoContext;

    *pErrnoOut = errno = 0;
    *pHostErrorOut = static_cast<int>(nn::socket::HErrno::Netdb_Internal);
    *pSerializedAddrInfoResultBufferSize = 0;
    *pReturnCodeOut = static_cast<int>(nn::socket::AiErrno::EAi_System);

    if (optionsCount > MaximumResolverOptions)
    {
        LogMajor("Cannot duplicate resolver options\n");
        goto bail;
    };
    pOptions = reinterpret_cast<ResolverOption*>(alloca(optionsCount * sizeof(*pOptions)));;

    GET_POINTER_SAFER(pSerializedOptions, serializedOptionsSize, const uint8_t*, serializedOptionsInBuffer);
    GET_POINTER_SAFER(pNodeName,          nodeNameSize,          const char*,    nodeNameParam);
    GET_POINTER_SAFER(pServiceName,       serviceNameSize,       const char*,    serviceNameParam);
    GET_POINTER_SAFER(pHintsBuffer,       hintsBufferSize,       const uint8_t*, serializedHintsParam);
    GET_POINTER_SAFER(pAddrInfoOutBuffer, addrInfoOutBufferSize, uint8_t*,       serializedAddrInfoResultBuffer);
    LogDebug("pSerializedOptions: %p, optionsCount: %u\n", pSerializedOptions, optionsCount);

    if ( pNodeName == nullptr && pServiceName == nullptr )
    {
        *pReturnCodeOut = static_cast<int>(nn::socket::AiErrno::EAi_NoName);
        *pErrnoOut = static_cast<int>(nn::socket::Errno::EInval);
        goto bail;
    }
    else if (nullptr == pAddrInfoOutBuffer)
    {
        *pReturnCodeOut = static_cast<int>(nn::socket::AiErrno::EAi_Fail);
        *pErrnoOut = static_cast<int>(nn::socket::Errno::EInval);
        goto bail;
    };

    if (nullptr != pHintsBuffer)
    {
        if ( -1 == serializer.FromBuffer(hints, pHintsBuffer, hintsBufferSize))
        {
            *pReturnCodeOut = static_cast<int>(nn::socket::AiErrno::EAi_Fail);
            *pErrnoOut = static_cast<int>(nn::socket::Errno::EInval);
            goto bail;
        };
        didInitializeHints = true;
    };

    if (0 != optionsCount)
    {
        if (nullptr == pSerializedOptions)
        {
            LogMajor("Non-zero optionsCount (%u) but null options.\n", optionsCount);
            *pHostErrorOut = static_cast<int>(nn::socket::HErrno::Netdb_Internal);
            *pErrnoOut = static_cast<int>(nn::socket::Errno::EInval);
            goto bail;
        }
        else if ( -1 == ( rc = serializer.FromBuffer(pOptions,
                                                     optionsCount * sizeof(*pOptions),
                                                     pSerializedOptions,
                                                     serializedOptionsSize,
                                                     optionsCount)))
        {
            LogMajor("options frombuffer serializer failed\n");
            goto bail;
        }
        else if (!ValidateOptionsArrayForVersionAndMask(pOptions, optionsCount, optionVersion, mask))
        {
            LogMinor("Options validation failed.\n");
            *pErrnoOut = errno;
            goto bail;
        };
        LogDebug("Options validation succeeded.\n");
    };

    LogDebug("pOptions: %p, optionsCount: %u\n", pOptions, optionsCount);
    GET_REQUEST_OPTION(useNsdResolve, rc, optionVersion,
                       ResolverOptionKey::RequestEnableServiceDiscoveryBoolean,
                       ResolverOptionType::Boolean, pOptions, optionsCount);

    GET_REQUEST_OPTION(cancelHandle, rc, optionVersion,
                       ResolverOptionKey::RequestCancelHandleInteger,
                       ResolverOptionType::Integer, pOptions, optionsCount);

    if (!autoContext.Initialize(processId, cancelHandle, pOptions, optionsCount))
    {
        LogDebug("autoContext initialization failed %s\n", __FUNCTION__);
        *pReturnCodeOut = static_cast<int>(nn::socket::AiErrno::EAi_Again);
        *pErrnoOut = static_cast<int>(nn::socket::Errno::EAgain);
        goto bail;
    };

    UpdateBionicServers();
    if ( nullptr != pNodeName && useNsdResolve )
    {
        FixupHostName(&nsdResolveName, pNodeName);
        pNodeName = nsdResolveName.value;
    };

    *pReturnCodeOut = bionic_getaddrinfo(pNodeName,
                                         pServiceName,
                                         true == didInitializeHints ? &hints : nullptr,
                                         &pResults);
    *pErrnoOut = static_cast<int>(nn::socket::GetLastError());
    *pHostErrorOut = static_cast<int>(*nn::socket::GetHError());

    if (0 != *pReturnCodeOut)
    {
        bool isExpired = false;
        bool isCancelled = autoContext.IsSlotCancelledOrExpired(isExpired);
        if (true == isExpired)
        {
            *pErrnoOut = static_cast<int>(nn::socket::Errno::ETimedOut);
        }
        else if (true == isCancelled)
        {
            *pErrnoOut = static_cast<int>(nn::socket::Errno::ECanceled);
        }
        else if (static_cast<int>(nn::socket::Errno::ENoMem) == *pErrnoOut)
        {
            *pReturnCodeOut = static_cast<int>(nn::socket::AiErrno::EAi_Again);
            *pErrnoOut = static_cast<int>(nn::socket::Errno::ENoMem);
        };
    }
    else if ( -1 == (rc = serializer.ToBuffer(pAddrInfoOutBuffer,
                                              addrInfoOutBufferSize,
                                              *pResults)))
    {
        if ( nn::socket::Errno::ENoMem == nn::socket::GetLastError())
        {
            *pReturnCodeOut = static_cast<int>(nn::socket::AiErrno::EAi_Memory);
            *pErrnoOut = static_cast<int>(nn::socket::Errno::ENoMem);
        }
        else
        {
            *pReturnCodeOut = static_cast<int>(nn::socket::AiErrno::EAi_Fail);
            *pErrnoOut = static_cast<int>(nn::socket::Errno::EInval);
        };
    }
    else
    {
        *pSerializedAddrInfoResultBufferSize = rc;
    };

bail:
    if ( true == didInitializeHints )
    {
        nn::socket::resolver::serializer::FreeAddrinfoContents(hints);
    };

    if ( nullptr != pResults )
    {
        bionic_freeaddrinfo(pResults);
        pResults = nullptr;
    };

    return result;
}; // NOLINT(impl/function_size)

Result ResolverImpl::GetNameInfoRequestWithOptions(Bit64 processId,
                                                   const nn::sf::InBuffer& socketAddressPointerIn,
                                                   const nn::sf::OutBuffer hostBufferOutParam,
                                                   const nn::sf::OutBuffer serviceBufferOutParam,
                                                   uint32_t flagsParam,
                                                   const nn::sf::Out<int> pReturnCodeOut,
                                                   uint32_t optionVersion,
                                                   const nn::sf::InBuffer& serializedOptionsInBuffer,
                                                   uint32_t optionsCount,
                                                   const nn::sf::Out<int> pHostErrorOut,
                                                   const nn::sf::Out<int> pErrnoOut) NN_NOEXCEPT
{
    serializer::DNSSerializer serializer;
    Result result = ResultSuccess();

    const uint8_t* pSerializedOptions = nullptr;
    size_t serializedOptionsSize = 0;

    const struct sockaddr* pSocketAddress = nullptr;
    size_t socketAddressSize = 0;

    char* pHostBuffer =  nullptr;
    size_t hostBufferSize = 0;

    char* pServiceBuffer = nullptr;
    size_t serviceBufferSize = 0;

    int cancelHandle = 0;

    int rc = 0;

    ResolverOption* pOptions = NULL;

    OptionContext mask = OptionContext_Request;

#if ! defined(NN_SDK_BUILD_RELEASE)
    *reinterpret_cast<unsigned*>(&mask) |= OptionContext_PrivateRequest;
#endif

    ResolverAutoContext autoContext;

    *pErrnoOut = errno = 0;
    *pHostErrorOut = static_cast<int>(nn::socket::HErrno::Netdb_Internal);
    *pReturnCodeOut = static_cast<int>(nn::socket::AiErrno::EAi_System);

    if (optionsCount > MaximumResolverOptions)
    {
        LogMajor("Cannot duplicate resolver options\n");
        goto bail;
    };
    pOptions = reinterpret_cast<ResolverOption*>(alloca(optionsCount * sizeof(*pOptions)));;

    GET_POINTER_SAFER(pSerializedOptions, serializedOptionsSize, const uint8_t*,  serializedOptionsInBuffer);
    GET_POINTER_SAFER(pSocketAddress,     socketAddressSize,     const sockaddr*, socketAddressPointerIn);
    GET_POINTER_SAFER(pHostBuffer,        hostBufferSize,        char*,           hostBufferOutParam);
    GET_POINTER_SAFER(pServiceBuffer,     serviceBufferSize,     char*,           serviceBufferOutParam);

    if (pSocketAddress == nullptr)
    {
        *pReturnCodeOut = static_cast<int>(nn::socket::AiErrno::EAi_Fail);
        goto bail;
    }
    else if (0 == hostBufferSize && 0 == serviceBufferSize)
    {
        *pReturnCodeOut = static_cast<int>(nn::socket::AiErrno::EAi_Fail);
        goto bail;
    }
    else if (sizeof(nn::socket::SockAddrIn) != socketAddressSize)
    {
        // the structure provided might be AF_INET6 or some other, in this case
        *pReturnCodeOut = static_cast<int>(nn::socket::AiErrno::EAi_Family);
        goto bail;
    }
    else if ( AF_INET != pSocketAddress->sa_family )
    {
        *pReturnCodeOut = static_cast<int>(nn::socket::AiErrno::EAi_Family);
        goto bail;
    }
    else if (0 != optionsCount)
    {
        if (nullptr == pSerializedOptions)
        {
            LogMajor("Non-zero optionsCount (%u) but null options.\n", optionsCount);
            *pHostErrorOut = static_cast<int>(nn::socket::HErrno::Netdb_Internal);
            *pErrnoOut = static_cast<int>(nn::socket::Errno::EInval);
            goto bail;
        }
        else if ( -1 == ( rc = serializer.FromBuffer(pOptions,
                                                     optionsCount * sizeof(*pOptions),
                                                     pSerializedOptions,
                                                     serializedOptionsSize,
                                                     optionsCount)))
        {
            LogMajor("options frombuffer serializer failed\n");
            goto bail;
        }
        else if (!ValidateOptionsArrayForVersionAndMask(pOptions, optionsCount, optionVersion, mask))
        {
            LogMinor("Options validation failed.\n");
            *pErrnoOut = errno;
            goto bail;
        };
        LogDebug("Options validation succeeded\n");
    };

    GET_REQUEST_OPTION(cancelHandle,
                       rc,
                       optionVersion,
                       ResolverOptionKey::RequestCancelHandleInteger,
                       ResolverOptionType::Integer,
                       pOptions,
                       optionsCount);

    if (!autoContext.Initialize(processId, cancelHandle, pOptions, optionsCount))
    {
        LogDebug("autoContext initialization failed %s\n", __FUNCTION__);
        *pHostErrorOut = static_cast<int>(nn::socket::HErrno::Try_Again);
        *pErrnoOut = static_cast<int>(nn::socket::Errno::EAgain);
        goto bail;
    };

    UpdateBionicServers();
    *pReturnCodeOut = bionic_getnameinfo(pSocketAddress, socketAddressSize,
                                         pHostBuffer, hostBufferSize,
                                         pServiceBuffer, serviceBufferSize,
                                         flagsParam);
    *pErrnoOut = errno;
    *pHostErrorOut = static_cast<int>(*nn::socket::GetHError());

    if (0 != *pReturnCodeOut)
    {
        bool isExpired = false;
        bool isCancelled = autoContext.IsSlotCancelledOrExpired(isExpired);
        if (true == isExpired)
        {
            *pErrnoOut = static_cast<int>(nn::socket::Errno::ETimedOut);
        }
        else if (true == isCancelled)
        {
            *pErrnoOut = static_cast<int>(nn::socket::Errno::ECanceled);
        }
        else if (static_cast<int>(nn::socket::Errno::ENoMem) == *pErrnoOut)
        {
            *pReturnCodeOut = static_cast<int>(nn::socket::AiErrno::EAi_Again);
            *pErrnoOut = static_cast<int>(nn::socket::Errno::ENoMem);
        };
    };

bail:
    return result;
};  // NOLINT(impl/function_size)

Result ResolverImpl::GetHostStringErrorRequest(uint32_t errorRequestIn,
                                               const nn::sf::OutBuffer& stringErrorResponse) NN_NOEXCEPT
{
    Result result = ResultSuccess();
    const char* errorString = bionic_hstrerror(errorRequestIn);
    size_t len = 0;

    if (nullptr == errorString)
    {
        goto bail;
    };

    len = strlen(errorString) + 1;

    if ( len <= stringErrorResponse.GetSize() )
    {
        memcpy(stringErrorResponse.GetPointerUnsafe(), errorString, len);
    };

bail:
    return result;
}

Result ResolverImpl::ResolverSetOptionRequest(Bit64 processId,
                                              uint32_t version,
                                              const nn::sf::InBuffer& optionBuffer,
                                              nn::sf::Out<int> pReturnCodeOut,
                                              nn::sf::Out<int> pErrnoOut) NN_NOEXCEPT
{
    LogDebug("processId: %lu, version: %d\n", processId, version);

    serializer::DNSSerializer serializer;
    const uint8_t* buffer = nullptr;
    size_t size = 0;
    ResolverOption option;
    OptionContext mask = OptionContext_Set;

#if ! defined(NN_SDK_BUILD_RELEASE)
    *reinterpret_cast<unsigned*>(&mask) |= OptionContext_PrivateSet;
#endif

    *pReturnCodeOut = -1;

    GET_POINTER_SAFER(buffer, size, const uint8_t*, optionBuffer);
    if (nullptr == buffer)
    {
        LogDebug("\n");
        *pErrnoOut = static_cast<int>(nn::socket::Errno::EInval);
        goto bail;
    }
    else if (0 == size)
    {
        LogDebug("No options provided.\n");
    }
    else if ( -1 == serializer.FromBuffer(option, buffer, size))
    {
        LogDebug("\n");
        *pErrnoOut = static_cast<int>(nn::socket::Errno::EInval);
        goto bail;
    }
    else if (!ValidateOptionForVersionAndMask(option, version, mask))
    {
        LogDebug("Validation failed\n");
        *pErrnoOut = errno;
        goto bail;
    };

    switch (static_cast<uint32_t>(option.key))
    {
    case static_cast<uint32_t>(ResolverOptionKey::SetRemoveDomainnameFromCachePointer):
    {
        LogDebug("ResolverOptionKey::SetRemoveDomainnameFromCacheString\n");
        if (option.size > MaximumDomainnameStringBufferSize)
        {
            *pReturnCodeOut = -1;
            *pErrnoOut = static_cast<int>(nn::socket::Errno::EInval);
        }
        else
        {
            const char* domainname = reinterpret_cast<const char*>(option.data.pointerValue);
            *pReturnCodeOut = ::ResolverCacheFindAndPossiblyRemoveHostname(domainname, true);
            *pErrnoOut = errno;
        }
        break;
    };
    case static_cast<uint32_t>(ResolverOptionKey::SetRemoveIpAddressFromCacheUnsigned32):
    {
        LogDebug("ResolverOptionKey::SetRemoveIpAddressFromCacheUnsigned32\n");
        *pReturnCodeOut =::ResolverCacheFindAndPossiblyRemoveIpAddress(option.data.unsigned32Value, true);
        *pErrnoOut = errno;
        break;
    };
    case static_cast<uint32_t>(ResolverOptionKey::SetCancelHandleInteger):
    {
        LogDebug("ResolverOptionKey::SetCancelHandleInteger\n");
        OperationState ignore;
        if ( 0 != (*pReturnCodeOut = nn::socket::resolver::
                   ResolverCancelTable::SetSlotCancelled(option.data.integerValue,
                                                         processId,
                                                         ignore)))
        {
            LogDebug("SetSlotCancelled: "
                     "HandleID: %08x, ProcessID: %08lx, rc: %d, operationState: %s (%d)\n",
                     option.data.integerValue, processId,
                     *pReturnCodeOut,
                     GetOperationStateString(ignore),
                     ignore);
        };

        *pErrnoOut = errno;
        break;
    };
#if ! defined(NN_SDK_BUILD_RELEASE)
    case static_cast<uint32_t>(ResolverOptionPrivateKey::SetFlushCacheBoolean):
    {
        LogDebug("ResolverOptionPrivateKey::PrivateSetFlushCacheBoolean\n");
        ::ResolverCacheFlushUnlocked(::GetResolverCache(ResolverCacheConstantNintendoNetID));
        *pReturnCodeOut = 0;
        *pErrnoOut = 0;
        break;
    };
    case static_cast<uint32_t>(ResolverOptionPrivateKey::SetTimeToLiveForDomainnamePointer):
    {
        LogDebug("ResolverOptionPrivateKey::PrivateSetTimeToLiveForDomainnamePointer\n");
        if (sizeof(struct PrivateSetTimeToLiveForDomainnameArgument) != option.size)
        {
            LogDebug("\n");
            *pReturnCodeOut = -1;
            *pErrnoOut = static_cast<int>(nn::socket::Errno::EInval);
            goto bail;
        };

        const struct PrivateSetTimeToLiveForDomainnameArgument* arg =
            reinterpret_cast<const struct PrivateSetTimeToLiveForDomainnameArgument*>(
                option.data.pointerValue);

        *pReturnCodeOut = ::ResolverCacheUpdateTimeToLiveForHostname(arg->domainname,
                                                                     arg->ttl);
        *pErrnoOut = errno;

        break;
    };
    case static_cast<uint32_t>(ResolverOptionPrivateKey::SetDnsServerAddressesPointer):
    {
        LogDebug("ResolverOptionPrivateKey::PrivateSetDnsServerAddressesPointer\n");
        ::ResolverCacheFlushUnlocked(::GetResolverCache(ResolverCacheConstantNintendoNetID));

        if (sizeof(struct PrivateDnsAddressArrayArgument) != option.size)
        {
            LogDebug("\n");
            *pReturnCodeOut = -1;
            *pErrnoOut = static_cast<int>(nn::socket::Errno::EInval);
            goto bail;
        };

        const struct PrivateDnsAddressArrayArgument* pArg =
            reinterpret_cast<const struct PrivateDnsAddressArrayArgument*>(
                option.data.pointerValue);

        *pReturnCodeOut = -1;
        if (SetSockaddrDnsAddresses(pArg->addresses, pArg->count).IsSuccess())
        {
            *pReturnCodeOut = 0;
        };
        *pErrnoOut = errno;
        break;
    };
#endif
    default:
    {
        LogDebug("default case\n");
        *pReturnCodeOut = -1;
        *pErrnoOut = static_cast<int>(nn::socket::Errno::EInval);
        goto bail;
    };
    };
    *pErrnoOut = 0;
    *pReturnCodeOut = 0;

bail:
    return ResultSuccess();
}; //NOLINT(impl/function_size)

Result ResolverImpl::ResolverGetOptionRequest(Bit64 processId,
                                              uint32_t version,
                                              uint32_t key,
                                              const nn::sf::InBuffer& optionInBuffer,
                                              const nn::sf::OutBuffer& serializedOptionOutBuffer,
                                              nn::sf::Out<uint32_t> outBufferUsed,
                                              nn::sf::Out<int> pReturnCodeOut,
                                              nn::sf::Out<int> pErrnoOut) NN_NOEXCEPT
{
    serializer::DNSSerializer serializer;
    uint8_t* buffer = nullptr;
    size_t size = 0;
    ResolverOption option;
    OptionContext mask = OptionContext_Get;

#if ! defined(NN_SDK_BUILD_RELEASE)
    *reinterpret_cast<unsigned*>(&mask) |= OptionContext_PrivateGet;
#endif

    *pReturnCodeOut = -1;

    GET_POINTER_SAFER(buffer, size, uint8_t*, serializedOptionOutBuffer);
    if (nullptr == buffer)
    {
        LogMajor("ResolverOption out buffer is NULL.\n");
        *pErrnoOut = static_cast<int>(nn::socket::Errno::EInval);
        goto bail;
    };

    memset(buffer, 0, size);

    if (!ValidateKeyForVersionAndMask(static_cast<ResolverOptionKey>(key), version, mask))
    {
        LogDebug("Validation failed\n");
        *pErrnoOut = errno;
        goto bail;
    };
    LogDebug("Validation succeeded: option.key\n");

    switch (static_cast<uint32_t>(key))
    {

#if ! defined(NN_SDK_BUILD_RELEASE)
    case static_cast<uint32_t>(ResolverOptionPrivateKey::GetDnsServerAddressesPointer):
    {
        LogDebug("ResolverOptionPrivateKey::GetDnsServerAddressesPointer\n");
        PrivateDnsAddressArrayArgument arg = { 0 };
        option.key = static_cast<ResolverOptionKey>(key);
        option.type = ResolverOptionType::Pointer;
        option.size = sizeof(struct PrivateDnsAddressArrayArgument);
        option.data.pointerValue = reinterpret_cast<const char*>(&arg);

        for (unsigned idx=0; idx<MaximumDnsServerAddresses; ++idx)
        {
            if (GetDnsAddress(idx, &arg.addresses[idx]).IsSuccess())
            {
                ++arg.count;
            };
        };

        *pReturnCodeOut = 0;
        *pErrnoOut = errno;
        *outBufferUsed = serializer.ToBuffer(buffer, size, option);
        break;
    };

    case static_cast<uint32_t>(ResolverOptionPrivateKey::GetCacheEntryCountForDomainnamePointer):
    {
        LogDebug("ResolverOptionPrivateKey::GetCacheEntryCountIntegerForDomainnamePointer\n");
        const uint8_t* pInOptionBuffer = reinterpret_cast<const uint8_t*>(optionInBuffer.GetPointerUnsafe());
        ResolverOption inOption;
        serializer.FromBuffer(inOption, pInOptionBuffer, optionInBuffer.GetSize());
        const char* domainname = inOption.data.pointerValue;

        *pReturnCodeOut = ::ResolverCacheFindAndPossiblyRemoveHostname(domainname, false);
        *pErrnoOut = errno;

        break;
    };

    case static_cast<uint32_t>(ResolverOptionPrivateKey::GetCacheEntryCountForIpUnsigned32):
    {
        LogDebug("ResolverOptionPrivateKey::GetCacheEntryCountIntegerForIpUnsigned32\n");
        const uint8_t* pInOptionBuffer = reinterpret_cast<const uint8_t*>(optionInBuffer.GetPointerUnsafe());
        ResolverOption inOption;
        serializer.FromBuffer(inOption, pInOptionBuffer, optionInBuffer.GetSize());
        uint32_t ipAddress = inOption.data.unsigned32Value;

        *pReturnCodeOut = ::ResolverCacheFindAndPossiblyRemoveIpAddress(ipAddress, false);
        *pErrnoOut = errno;

        break;
    };

#endif
    default:
    {
        LogDebug("default case\n");
        memset(&option, 0, sizeof(ResolverOption));
        *outBufferUsed = serializer.ToBuffer(buffer, size, option);
        *pReturnCodeOut = -1;
        *pErrnoOut = static_cast<int>(nn::socket::Errno::EInval);
        goto bail;
    };
    };

bail:
    return ResultSuccess();
};

Result ResolverImpl::GetGaiStringErrorRequest(uint32_t errorRequestIn,
                                              const nn::sf::OutBuffer& stringErrorResponse) NN_NOEXCEPT
{
    Result result = ResultSuccess();
    const char* errorString = bionic_gai_strerror(errorRequestIn);
    size_t len = 0;

    if (nullptr == errorString)
    {
        goto bail;
    };

    len = strlen(errorString) + 1;
    if ( len < stringErrorResponse.GetSize() )
    {
        memcpy(stringErrorResponse.GetPointerUnsafe(), errorString, len);
    };

bail:
    return result;
};

//
// SFDL server methods
//
Result ResolverImpl::GetHostByNameRequest(uint32_t cancelHandle,
                                          Bit64 processId,
                                          bool useNsdResolve,
                                          const nn::sf::InBuffer& nameIn,
                                          nn::sf::Out<uint32_t> pHostErrorOut,
                                          nn::sf::Out<uint32_t> pErrnoOut,
                                          const nn::sf::OutBuffer& serializedHostentResponse,
                                          nn::sf::Out<uint32_t> pBufferLengthOut) NN_NOEXCEPT
{
    serializer::DNSSerializer serializer;
    const unsigned int OptionsCount = 2;
    ResolverOption options[OptionsCount];
    memset(options, 0, sizeof(options));

    options[0].key = ResolverOptionKey::RequestEnableServiceDiscoveryBoolean;
    options[0].type = ResolverOptionType::Boolean;
    options[0].size = sizeof(bool);
    options[0].data.booleanValue = useNsdResolve;

    options[1].key = ResolverOptionKey::RequestCancelHandleInteger;
    options[1].type = ResolverOptionType::Integer;
    options[1].size = sizeof(int);
    options[1].data.integerValue = (int)cancelHandle;

    size_t serializedSize = serializer.SizeOf(options, OptionsCount);
    uint8_t* pSerializedBuffer = reinterpret_cast<uint8_t*>(alloca(serializedSize));
    serializer.ToBuffer(pSerializedBuffer,
                        serializedSize,
                        options,
                        OptionsCount);

    nn::sf::InBuffer inBuffer(reinterpret_cast<const char*>(pSerializedBuffer),
                              serializedSize);

    int hostErrorOut = 0;
    int errnoOut = 0;
    Result result = GetHostByNameRequestWithOptions(processId,
                                                    nameIn,
                                                    serializedHostentResponse,
                                                    pBufferLengthOut,
                                                    1,
                                                    inBuffer,
                                                    OptionsCount,
                                                    &hostErrorOut,
                                                    &errnoOut);

    *pHostErrorOut = static_cast<uint32_t>(hostErrorOut);
    *pErrnoOut = static_cast<uint32_t>(errnoOut);

    return result;
};

Result ResolverImpl::GetHostByAddrRequest(uint32_t cancelHandle,
                                          Bit64 processId,
                                          const nn::sf::InBuffer& serializedSocketAddressIn,
                                          uint32_t inAddrLength,
                                          uint32_t addressFamilyParam,
                                          nn::sf::Out<uint32_t> pHostErrorOut,
                                          nn::sf::Out<uint32_t> pErrnoOut,
                                          const nn::sf::OutBuffer& serializedHostEntryOut,
                                          nn::sf::Out<uint32_t> pSerializedHostEntrySizeOut) NN_NOEXCEPT
{
    serializer::DNSSerializer serializer;
    const unsigned int OptionsCount = 1;
    ResolverOption options[OptionsCount];
    memset(options, 0, sizeof(options));

    options[0].key = ResolverOptionKey::RequestCancelHandleInteger;
    options[0].type = ResolverOptionType::Integer;
    options[0].size = sizeof(int);
    options[0].data.integerValue = (int)cancelHandle;

    size_t serializedSize = serializer.SizeOf(options, OptionsCount);
    uint8_t* pSerializedBuffer = reinterpret_cast<uint8_t*>(alloca(serializedSize));
    serializer.ToBuffer(pSerializedBuffer,
                        serializedSize,
                        options,
                        OptionsCount);

    nn::sf::InBuffer inBuffer(reinterpret_cast<const char*>(pSerializedBuffer),
                              serializedSize);

    int hostErrorOut = 0;
    int errnoOut = 0;
    Result result =  GetHostByAddrRequestWithOptions(processId,
                                                     serializedSocketAddressIn,
                                                     inAddrLength,
                                                     addressFamilyParam,
                                                     serializedHostEntryOut,
                                                     pSerializedHostEntrySizeOut,
                                                     1,
                                                     inBuffer,
                                                     OptionsCount,
                                                     &hostErrorOut,
                                                     &errnoOut);

    *pHostErrorOut = static_cast<uint32_t>(hostErrorOut);
    *pErrnoOut = static_cast<uint32_t>(errnoOut);

    return result;
};

Result ResolverImpl::GetAddrInfoRequest(uint32_t cancelHandle,
                                        Bit64 processId,
                                        bool useNsdResolve,
                                        const nn::sf::InBuffer& nodeNameParam,
                                        const nn::sf::InBuffer& serviceNameParam,
                                        const nn::sf::InBuffer& serializedHintsParam,
                                        const nn::sf::OutBuffer& serializedAddrInfoResultBuffer,
                                        nn::sf::Out<uint32_t> pErrnoOut,
                                        nn::sf::Out<int32_t> pReturnCodeOut,
                                        nn::sf::Out<uint32_t> pSerializedAddrInfoResultBufferSize) NN_NOEXCEPT
{
    serializer::DNSSerializer serializer;
    const unsigned int OptionsCount = 2;
    ResolverOption options[OptionsCount];
    memset(options, 0, sizeof(options));

    options[0].key = ResolverOptionKey::RequestEnableServiceDiscoveryBoolean;
    options[0].type = ResolverOptionType::Boolean;
    options[0].size = sizeof(bool);
    options[0].data.booleanValue = useNsdResolve;

    options[1].key = ResolverOptionKey::RequestCancelHandleInteger;
    options[1].type = ResolverOptionType::Integer;
    options[1].size = sizeof(int);
    options[1].data.integerValue = (int)cancelHandle;

    size_t serializedSize = serializer.SizeOf(options, OptionsCount);
    uint8_t* pSerializedBuffer = reinterpret_cast<uint8_t*>(alloca(serializedSize));
    serializer.ToBuffer(pSerializedBuffer,
                        serializedSize,
                        options,
                        OptionsCount);

    nn::sf::InBuffer inBuffer(reinterpret_cast<const char*>(pSerializedBuffer),
                              serializedSize);

    int hostErrorIgnore = 0;
    int errnoOut = 0;
    Result result = GetAddrInfoRequestWithOptions(processId,
                                                  nodeNameParam,
                                                  serviceNameParam,
                                                  serializedHintsParam,
                                                  serializedAddrInfoResultBuffer,
                                                  pSerializedAddrInfoResultBufferSize,
                                                  pReturnCodeOut,
                                                  1,
                                                  inBuffer,
                                                  OptionsCount,
                                                  &hostErrorIgnore,
                                                  &errnoOut);

    *pErrnoOut = static_cast<uint32_t>(errnoOut);
    return result;

}; //NOLINT(impl/function_size)

Result ResolverImpl::GetNameInfoRequest(uint32_t cancelHandle,
                                        Bit64 processId,
                                        const nn::sf::InBuffer& socketAddressPointerIn,
                                        const nn::sf::OutBuffer& hostBufferOutParam,
                                        const nn::sf::OutBuffer& serviceBufferOutParam,
                                        uint32_t flagsParam,
                                        nn::sf::Out<uint32_t> pErrnoOut,
                                        nn::sf::Out<int32_t> pReturnCodeOut) NN_NOEXCEPT
{
    serializer::DNSSerializer serializer;
    const unsigned int OptionsCount = 1;
    ResolverOption options[OptionsCount];
    memset(options, 0, sizeof(options));

    options[0].key = ResolverOptionKey::RequestCancelHandleInteger;
    options[0].type = ResolverOptionType::Integer;
    options[0].size = sizeof(int);
    options[0].data.integerValue = (int)cancelHandle;

    size_t serializedSize = serializer.SizeOf(options, OptionsCount);
    uint8_t* pSerializedBuffer = reinterpret_cast<uint8_t*>(alloca(serializedSize));
    serializer.ToBuffer(pSerializedBuffer,
                        serializedSize,
                        options,
                        OptionsCount);

    nn::sf::InBuffer inBuffer(reinterpret_cast<const char*>(pSerializedBuffer),
                              serializedSize);

    int hostErrorIgnore = 0;
    int errnoOut = 0;
    Result result = GetNameInfoRequestWithOptions(processId,
                                                  socketAddressPointerIn,
                                                  hostBufferOutParam,
                                                  serviceBufferOutParam,
                                                  flagsParam,
                                                  pReturnCodeOut,
                                                  1,
                                                  inBuffer,
                                                  OptionsCount,
                                                  &hostErrorIgnore,
                                                  &errnoOut);

    *pErrnoOut = static_cast<uint32_t>(errnoOut);

    return result;
};

Result ResolverImpl::GetCancelHandleRequest(Bit64 processId,
                                            nn::sf::Out<uint32_t> requestHandleOut) NN_NOEXCEPT
{
    int requestHandle = 0;
    if ( 0 != CreateRandomId(requestHandle) )
    {
        LogDebug("CreateRandomId ID failed for ProcessID: %08lx\n", processId);
    };

    *requestHandleOut = static_cast<uint32_t>(requestHandle);
    return ResultSuccess();
};

Result ResolverImpl::CancelRequest(Bit64 processId,
                                   uint32_t requestHandle) NN_NOEXCEPT
{
    int rc;
    OperationState operationState;

    if ( 0 != (rc = nn::socket::resolver::
               ResolverCancelTable::SetSlotCancelled(requestHandle,
                                                     processId,
                                                     operationState)))
    {
        LogDebug("SetSlotCancelled: HandleID: %08x, ProcessID: %08x, "
                 "rc: %d, operationState: %s (%d)\n",
                 requestHandle, processId, rc,
                 GetOperationStateString(operationState),
                 operationState);
    };

    return ResultSuccess();
};

Result ResolverImpl::CancelAll(Bit64 processId) NN_NOEXCEPT
{
    OperationState operationState;

    nn::socket::resolver::ResolverCancelTable::CancelAllSlotsForProcess(processId, operationState);

    return ResultSuccess();
};

Result ResolverImpl::Finalize() NN_NOEXCEPT
{
    ZeroDnsServerSettings();
    return ResultSuccess();
};


/**  TEMPORARY PRIVATE API **/


NN_OS_EXTERN_C int ResolverGetDnsAddress(unsigned int nIndex, struct sockaddr_in* pSockaddrOut)
{
    int rc = -1;
    if ( nullptr != pSockaddrOut )
    {
        nn::socket::SockAddrIn tempSockAddrIn = {};
        rc = resolver::GetDnsAddress(nIndex, &tempSockAddrIn).IsSuccess() ? 0 : -1;
        LogDebug("DNS Server[%d]: %s:%d\n",
                 nIndex,
                 InetNtoa(pSockaddrOut->sin_addr),
                 InetHtons(pSockaddrOut->sin_port));
        nn::socket::detail::CopyToPlatform(pSockaddrOut, &tempSockAddrIn);
    };
    return rc;
};

NN_OS_EXTERN_C bool ResolverIsRequestCanceled()
{
    ResolverAutoContext & context = ResolverAutoContext::CurrentContext();
    bool isExpired = false;
    return context.IsSlotCancelledOrExpired(isExpired);
};

NN_OS_EXTERN_C bool ResolverIsRequestExpired()
{
    ResolverAutoContext & context = ResolverAutoContext::CurrentContext();

    bool isExpired = false;
    context.IsSlotCancelledOrExpired(isExpired);
    return isExpired;
};

NN_OS_EXTERN_C bool ResolverIsRequestCacheEnabled()
{
    const unsigned optionVersion = 1;
    int rc = 0;

    bool isCacheEnabled = false;

    ResolverAutoContext & context = ResolverAutoContext::CurrentContext();

    GET_REQUEST_OPTION(isCacheEnabled,
                       rc,
                       optionVersion,
                       ResolverOptionKey::RequestEnableDnsCacheBoolean,
                       ResolverOptionType::Boolean,
                       context.GetOptionsArray(),
                       context.GetOptionsCount());

    return isCacheEnabled;
};

}}} // namespace nn::socket::resolver
