﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <new>
#include <type_traits>
#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/sf/sf_HipcServer.h>
#include <nn/sf/sf_ObjectFactory.h>
#include <nn/nn_SystemThreadDefinition.h>
#include <nn/socket/resolver/resolver_Server.h>
#include <nn/socket/resolver/private/resolver_PrivateApi.h>

#include "resolver_ServiceName.h"
#include "resolver_Implementation.h"
#include "resolver_HipcServer.h"
#include "serializer.h"
#include "resolver_Cancel.h"

extern "C"
{
#include <siglo/BionicPrivateApi.h>
// forward declare resolver initialize function
int res_init(void);
}
#include "resolver_ThreadLocalStorage.h"
#include <nn/os/os_Result.public.h>

/**
 * @file
 * @brief resolver server
 *
 * @details resolver server
 */

#define LOG_MODULE_NAME "hipc server" // NOLINT(preprocessor/const)
#include <nn/socket/resolver/private/resolver_DebugLog.h>

namespace nn { namespace socket { namespace resolver {

// resolver service object typedef
typedef nn::sf::UnmanagedServiceObject<IResolver, Server> ResolverServiceObject;

// resolver service object instance
std::aligned_storage<sizeof(ResolverServiceObject),
                     NN_ALIGNOF(ResolverServiceObject)>::type g_ResolverServerManagerStorage;
ResolverServiceObject* g_pResolverServerManager = NULL;

Server::Server() :
    ResolverImpl()
{

}

Server::~Server()
{
}

void Server::SessionThreadEntryStatic(void *pThis)
{
    ((Server *)pThis)->SessionThreadEntry();
};

void Server::SessionThreadEntry()
{
    Result result = tls::Server::Initialize();
    if ( result.IsFailure() )
    {
        NN_SDK_ASSERT(false, "Unable to initialize server TLS");
    }
    else
    {
        g_pResolverServerManager->GetImpl().LoopAuto();
    };
}

Result Server::Initialize()
{
    Result result = ResultInternalError();

    res_init();

    if (RegisterObjectForPort(g_pResolverServerManager->GetShared(),
                              MAX_THREADS,
                              ResolverServiceName).IsFailure() )
    {
        LogDebug("Resolver Server: Unable to initialize for port\n");
        goto bail;
    };

    for (uint32_t index = 0; index < MAX_THREADS; index++)
    {
        CreateNetworkThread(&m_Threads[index], SessionThreadEntryStatic,
                            this, &m_ThreadStacks[index][0],
                            sizeof(m_ThreadStacks[index]),
                            NN_SYSTEM_THREAD_PRIORITY(socket, ResolverIpcServer), -1);
        SetNetworkThreadNamePointer(&m_Threads[index], NN_SYSTEM_THREAD_NAME(socket, ResolverIpcServer));
        StartNetworkThread(&m_Threads[index]);
    };

    Start();
    result = ResultSuccess();

bail:
    return result;
}

Result Server::Wait()
{
    Result result = ResultSuccess(); // should always return success

    std::lock_guard<nn::os::Mutex> lock(m_AccessLock);

    // Synchronize with worker thread exit
    for (uint32_t index = 0; index < MAX_THREADS; index++)
    {
        WaitNetworkThread(&m_Threads[index]);
    }

    return result;
}

Result Server::Finalize() NN_NOEXCEPT
{
    Result result = ResultSuccess(); // should always return success

    std::lock_guard<nn::os::Mutex> lock(m_AccessLock);

    // Stop the server
    RequestStop();

    WaitServer();

    ResolverImpl::Finalize();

    // Synchronize with worker thread exit
    return result;
}

// Initializes the server.
// This must be called when starting the server process.
Result InitializeServer() NN_NOEXCEPT
{
    NN_SDK_ASSERT(!g_pResolverServerManager);
    LogDebug("Resolver server initialization begin\n");

    Result result = ResultInternalError();
    if ((result = nn::socket::resolver::tls::Server::Initialize()).IsFailure() )
    {
        LogDebug("Unable to initialize the resolver server (1/2)\n");
        goto bail;
    }
    else if ( NULL == (g_pResolverServerManager = new (&g_ResolverServerManagerStorage) ResolverServiceObject))
    {
        LogDebug("Unable to allocate the resolver server\n");
        goto bail;
    }
    else if ((result = g_pResolverServerManager->GetImpl().Initialize()).IsFailure() )
    {
        LogDebug("Unable to initialize the resolver server (2/2)\n");
        goto bail;
    }

    result = ResultSuccess();

bail:
    if (result.IsFailure())
    {
        if ( g_pResolverServerManager != NULL )
        {
            g_pResolverServerManager->GetImpl().Finalize();
            delete g_pResolverServerManager;
        };

        nn::socket::resolver::tls::Server::Finalize();
    };
    LogDebug("Resolver server initialization end\n");
    return result;
}

// Initializes the server.
// This must be called when starting the server process.
Result WaitServer() NN_NOEXCEPT
{
    g_pResolverServerManager->GetImpl().Wait();
    return ResultSuccess(); //should always return success
}

// Finalize the server.
Result FinalizeServer() NN_NOEXCEPT
{
    LogDebug("Resolver FinalizeServer begin\n");

    g_pResolverServerManager->GetImpl().Finalize();
    return ResultSuccess();
}

}}} // nn::socket::resolver

