﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "serializer.h"
#include "serializer_Specializations.h"
#include "../../detail/socket_Allocator.h"

/*
  struct hostent {
  char  *h_name;            // official name of host
  char **h_aliases;         // alias list
  int    h_addrtype;        // host address type
  int    h_length;          // length of address
  char **h_addr_list;       // list of addresses (in network byte order)
  }
  #define h_addr h_addr_list[0]     // for backward compatibility
*/

//#define LOG_LEVEL LOG_LEVEL_MAJOR
#define LOG_MODULE_NAME " hostent serializer" // NOLINT(preprocessor/const)
#include <nn/socket/resolver/private/resolver_DebugLog.h>

namespace nn { namespace socket { namespace resolver { namespace serializer {

void FreeHostentContents(nn::socket::HostEnt& hostEntry)
{
    if ( hostEntry.h_name != nullptr)
    {
        socket::detail::Free(hostEntry.h_name);
        hostEntry.h_name = nullptr;
    }

    if ( hostEntry.h_aliases != nullptr )
    {
        unsigned int index = 0;
        for (char* stringPointer = hostEntry.h_aliases[index];
             stringPointer != nullptr;
             stringPointer = hostEntry.h_aliases[index])
        {
            socket::detail::Free(stringPointer);
            hostEntry.h_aliases[index] = nullptr;
            ++index;
        };

        socket::detail::Free(hostEntry.h_aliases);
        hostEntry.h_aliases = nullptr;
    }

    if ( hostEntry.h_addr_list != nullptr )
    {
        // note that we only serialize the s_addr field and not
        // the rest of the struct in_addr packing
        if ( hostEntry.h_length == sizeof(uint32_t) )
        {
            unsigned int index = 0;
            nn::socket::InAddr** ppHostAddrList =
                reinterpret_cast<nn::socket::InAddr**>(hostEntry.h_addr_list);

            for (nn::socket::InAddr* inAddr4Pointer  = ppHostAddrList[index];
                 inAddr4Pointer != nullptr;
                 inAddr4Pointer = ppHostAddrList[index] )
            {
                socket::detail::Free(inAddr4Pointer);
                ppHostAddrList[index] = nullptr;
                ++index;
            }
        }
        else
        {
            NN_SDK_ASSERT(false);
        }
        socket::detail::Free(hostEntry.h_addr_list);
        hostEntry.h_addr_list = nullptr;
    }

    std::memset(reinterpret_cast<void*>(&hostEntry),
                '\0',
                sizeof(hostEntry));
};

void FreeHostentContents(struct hostent& hostEntry)
{
    if ( hostEntry.h_name != NULL)
    {
        socket::detail::Free(hostEntry.h_name);
        hostEntry.h_name = NULL;
    };

    if ( hostEntry.h_aliases != NULL )
    {
        unsigned int index = 0;
        for (char* stringPointer = hostEntry.h_aliases[index];
             stringPointer != NULL;
             stringPointer = hostEntry.h_aliases[index])
        {
            socket::detail::Free(stringPointer);
            hostEntry.h_aliases[index] = NULL;
            ++index;
        };

        socket::detail::Free(hostEntry.h_aliases);
        hostEntry.h_aliases = NULL;
    };

    if ( hostEntry.h_addr_list != NULL )
    {
        // note that we only serialize the s_addr field and not
        // the rest of the struct in_addr packing
        if ( hostEntry.h_length == sizeof(uint32_t) )
        {
            unsigned int index = 0;
            struct in_addr** ppHostAddrList =
                reinterpret_cast<struct in_addr**>(hostEntry.h_addr_list);

            for (struct in_addr* inAddr4Pointer  = ppHostAddrList[index];
                 inAddr4Pointer != NULL;
                 inAddr4Pointer = ppHostAddrList[index] )
            {
                socket::detail::Free(inAddr4Pointer);
                ppHostAddrList[index] = NULL;
                ++index;
            };
        }
        else
        {
            NN_SDK_ASSERT(false);
        }
        socket::detail::Free(hostEntry.h_addr_list);
        hostEntry.h_addr_list = NULL;
    }

    std::memset(reinterpret_cast<void*>(&hostEntry),
                '\0',
                sizeof(struct hostent));
};

template <>
size_t DNSSerializer::SizeOf(const nn::socket::HostEnt& hostentIn)
{
    LogDebug(
        "-------SizeOf------- nn::socket::HostEnt (%p)  ---------------------- \n",
        &hostentIn);

    size_t rc = 0;
    uint32_t junk = 0;

    rc += SizeOf(((const char*)hostentIn.h_name));
    rc += SizeOf(((const char**)hostentIn.h_aliases), junk);
    rc += sizeof(uint32_t); // h_addrtype
    rc += sizeof(uint32_t); // h_length
    rc += SizeOf(((const nn::socket::InAddr**)hostentIn.h_addr_list), junk);

    return rc;
}

template <>
size_t DNSSerializer::SizeOf(const struct hostent& hostentIn)
{
    LogDebug(
        "-------SizeOf------- hostent (%p)  ---------------------- \n",
        &hostentIn);

    size_t rc = 0;
    uint32_t junk = 0;

    rc += SizeOf(((const char*)hostentIn.h_name));
    rc += SizeOf(((const char**)hostentIn.h_aliases), junk);
    rc += sizeof(uint32_t); // h_addrtype
    rc += sizeof(uint32_t); // h_length
    rc += SizeOf(((const in_addr**)hostentIn.h_addr_list), junk);

    return rc;
}

template <>
ssize_t DNSSerializer::FromBuffer(nn::socket::HostEnt& hostentOut,
                                  const uint8_t *      pBufferIn,
                                  size_t               bufferSize)
{
    LogDebug("-------FromBuffer------- nn::socket::HostEnt (%p)  ---------------------- \n",
                   &hostentOut);

    ssize_t rc = -1;
    const uint8_t* pBufferCurrent = pBufferIn;
    uint16_t shortValue = 0;
    nn::socket::InAddr** ppInaddrs = NULL;

    std::memset(reinterpret_cast<void*>(&hostentOut),
                '\0',
                sizeof(hostentOut));

    if (-1 == ( rc = FromBuffer(hostentOut.h_name,
                                pBufferCurrent,
                                bufferSize - (pBufferCurrent - pBufferIn))))
    {
        LogDebug("error: h_name (%d)\n", __LINE__);
        goto bail;
    };
    SERIALIZER_HEXDUMP("h_name: ", pBufferCurrent, rc);
    pBufferCurrent += rc;

    if (-1 == ( rc = FromBuffer(hostentOut.h_aliases,
                                pBufferCurrent,
                                bufferSize - (pBufferCurrent - pBufferIn))))
    {
        LogDebug("error: h_aliases (%d)\n", __LINE__);
        goto bail;
    };
    SERIALIZER_HEXDUMP("h_aliases: ", pBufferCurrent, rc);
    pBufferCurrent += rc;

    if (-1 == ( rc = FromBuffer(shortValue,
                                pBufferCurrent,
                                bufferSize - (pBufferCurrent - pBufferIn))))
    {
        LogDebug("error: h_addrtype (%d)\n", __LINE__);
        goto bail;
    };
    hostentOut.h_addrtype = static_cast<nn::socket::Family>(shortValue);
    SERIALIZER_HEXDUMP("h_addrtype: ", pBufferCurrent, rc);
    pBufferCurrent += rc;

    if (-1 == ( rc = FromBuffer(shortValue,
                                pBufferCurrent,
                                bufferSize - (pBufferCurrent - pBufferIn))))
    {
        LogDebug("error: h_length (%d)\n", __LINE__);
        goto bail;
    };
    hostentOut.h_length = shortValue;
    SERIALIZER_HEXDUMP("h_length: ", pBufferCurrent, rc);
    pBufferCurrent += rc;

    if (-1 == ( rc = FromBuffer(ppInaddrs,
                                pBufferCurrent,
                                bufferSize - (pBufferCurrent - pBufferIn))))
    {
        LogDebug("error: h_addr_list (%d)\n", __LINE__);
        goto bail;
    };
    hostentOut.h_addr_list = (char**)ppInaddrs;
    SERIALIZER_HEXDUMP("h_addr_list: ", pBufferCurrent, rc);
    pBufferCurrent += rc;
    rc = pBufferCurrent - pBufferIn;

bail:
    if ( rc < 0 )
    {
        FreeHostentContents(hostentOut);
    };

    return rc;
}

template <>
ssize_t DNSSerializer::FromBuffer(struct hostent& hostentOut,
                                  const uint8_t * pBufferIn,
                                  size_t          bufferSize)
{
    LogDebug("-------FromBuffer------- hostent (%p)  ---------------------- \n",
                   &hostentOut);

    ssize_t rc = -1;
    const uint8_t* pBufferCurrent = pBufferIn;
    uint16_t shortValue = 0;
    struct in_addr** ppInaddrs = NULL;

    std::memset(reinterpret_cast<void*>(&hostentOut),
                '\0',
                sizeof(struct hostent));

    if (-1 == ( rc = FromBuffer(hostentOut.h_name,
                                pBufferCurrent,
                                bufferSize - (pBufferCurrent - pBufferIn))))
    {
        LogDebug("error: h_name \n");
        goto bail;
    };
    SERIALIZER_HEXDUMP("h_name: ", pBufferCurrent, rc);
    pBufferCurrent += rc;

    if (-1 == ( rc = FromBuffer(hostentOut.h_aliases,
                                pBufferCurrent,
                                bufferSize - (pBufferCurrent - pBufferIn))))
    {
        LogDebug("error: h_aliases \n");
        goto bail;
    };
    SERIALIZER_HEXDUMP("h_aliases: ", pBufferCurrent, rc);
    pBufferCurrent += rc;

    if (-1 == ( rc = FromBuffer(shortValue,
                                pBufferCurrent,
                                bufferSize - (pBufferCurrent - pBufferIn))))
    {
        LogDebug("error: h_addrtype \n");
        goto bail;
    };
    hostentOut.h_addrtype = shortValue;
    SERIALIZER_HEXDUMP("h_addrtype: ", pBufferCurrent, rc);
    pBufferCurrent += rc;

    if (-1 == ( rc = FromBuffer(shortValue,
                                pBufferCurrent,
                                bufferSize - (pBufferCurrent - pBufferIn))))
    {
        LogDebug("error: h_length \n");
        goto bail;
    };
    hostentOut.h_length = shortValue;
    SERIALIZER_HEXDUMP("h_length: ", pBufferCurrent, rc);
    pBufferCurrent += rc;

    if (-1 == ( rc = FromBuffer(ppInaddrs,
                                pBufferCurrent,
                                bufferSize - (pBufferCurrent - pBufferIn))))
    {
        LogDebug("error: h_addr_list \n");
        goto bail;
    };
    hostentOut.h_addr_list = (char**)ppInaddrs;
    SERIALIZER_HEXDUMP("h_addr_list: ", pBufferCurrent, rc);
    pBufferCurrent += rc;
    rc = pBufferCurrent - pBufferIn;

bail:
    if ( rc < 0 )
    {
        FreeHostentContents(hostentOut);
    };

    return rc;
}

template <>
ssize_t DNSSerializer::ToBuffer(uint8_t * const pBufferOut,
                                size_t     sizeOfBuffer,
                                const nn::socket::HostEnt& hostentIn)
{
    LogDebug("-------ToBuffer------- hostent (%p)  ---------------------- \n",
                   &hostentIn);

    size_t requiredSize = SizeOf(hostentIn);
    ssize_t rc = -1;
    uint8_t* pBufferCurrent = pBufferOut;
    uint32_t intValue = 0;
    uint16_t shortValue = 0;

    if ( -1 == (rc = CheckToBufferArguments(pBufferCurrent,
                                            sizeOfBuffer,
                                            requiredSize,
                                            __LINE__)))
    {
        LogDebug("Required size (%ul) unavailable for struct hostent\n", requiredSize);
        goto bail;
    };

    if (-1 == ( rc = ToBuffer(pBufferCurrent,
                              sizeOfBuffer - (pBufferCurrent - pBufferOut),
                              hostentIn.h_name)))
    {
        LogDebug("error: h_name (%d)\n", __LINE__);
        goto bail;
    };
    SERIALIZER_HEXDUMP("h_name: ", pBufferCurrent, rc);
    pBufferCurrent += rc;

    if (-1 == ( rc = ToBuffer(pBufferCurrent,
                              sizeOfBuffer - (pBufferCurrent - pBufferOut),
                              hostentIn.h_aliases)))
    {
        LogDebug("error: h_aliases (%d)\n", __LINE__);
        goto bail;
    };
    SERIALIZER_HEXDUMP("h_aliases: ", pBufferCurrent, rc);
    pBufferCurrent += rc;

    shortValue = static_cast<uint16_t>(hostentIn.h_addrtype);
    if (-1 == ( rc = ToBuffer(pBufferCurrent,
                              sizeOfBuffer - (pBufferCurrent - pBufferOut),
                              shortValue)))
    {
        LogDebug("error: h_addrtype (%d)\n", __LINE__);
        goto bail;
    };
    SERIALIZER_HEXDUMP("h_addrtype: ", pBufferCurrent, rc);
    pBufferCurrent += rc;

    shortValue = static_cast<uint16_t>(hostentIn.h_length);
    if (-1 == ( rc = ToBuffer(pBufferCurrent,
                              sizeOfBuffer - (pBufferCurrent - pBufferOut),
                              shortValue)))
    {
        LogDebug("error: h_length (%d)\n", __LINE__);
        goto bail;
    };
    SERIALIZER_HEXDUMP("h_length: ", pBufferCurrent, rc);
    pBufferCurrent += rc;

    if (hostentIn.h_addrtype == nn::socket::Family::Af_Inet)
    {
        if ( -1 == ( rc = ToBuffer(pBufferCurrent,
                                   sizeOfBuffer - (pBufferCurrent - pBufferOut),
                                   (nn::socket::InAddr**)(hostentIn.h_addr_list) )))
        {
            LogDebug("error: h_addr_list (in4 %d)\n", __LINE__);
            goto bail;
        }
    }
    else if (hostentIn.h_addrtype == nn::socket::Family::Af_Inet6)
    {
        if (-1 == ( rc = ToBuffer(pBufferCurrent,
                                  sizeOfBuffer - (pBufferCurrent - pBufferOut),
                                  (nn::socket::InAddr**)(hostentIn.h_addr_list) )))
        {
            LogDebug("error: h_addr_list (in6 %d)\n", __LINE__);
            goto bail;
        }
    }
    else
    {
        LogDebug("error: h_addrtype (%04x) unsupported, setting to zero\n",
                       hostentIn.h_addrtype);

        intValue = 0;

        if (-1 == ( rc = ToBuffer(pBufferCurrent,
                                  sizeOfBuffer - (pBufferCurrent - pBufferOut),
                                  intValue)))
        {
            LogDebug("error: h_addr_list (NULL / %d)\n", __LINE__);
            goto bail;
        }
    };
    SERIALIZER_HEXDUMP("h_addr_list: ", pBufferCurrent, rc);
    pBufferCurrent += rc;
    rc = pBufferCurrent - pBufferOut;

bail:
    return rc;
}

template <>
ssize_t DNSSerializer::ToBuffer(uint8_t * const pBufferOut,
                                size_t     sizeOfBuffer,
                                const struct hostent& hostentIn)
{
    LogDebug("-------ToBuffer------- hostent (%p)  ---------------------- \n",
                   &hostentIn);

    size_t requiredSize = SizeOf(hostentIn);
    ssize_t rc = -1;
    uint8_t* pBufferCurrent = pBufferOut;
    uint32_t intValue = 0;
    uint16_t shortValue = 0;

    if ( -1 == (rc = CheckToBufferArguments(pBufferCurrent,
                                            sizeOfBuffer,
                                            requiredSize,
                                            __LINE__)))
    {
        LogDebug("Required size (%ul) unavailable for struct hostent\n",
                 requiredSize);
        goto bail;
    };

    if (-1 == ( rc = ToBuffer(pBufferCurrent,
                              sizeOfBuffer - (pBufferCurrent - pBufferOut),
                              hostentIn.h_name)))
    {
        LogDebug("error: h_name \n");
        goto bail;
    };
    SERIALIZER_HEXDUMP("h_name: ", pBufferCurrent, rc);
    pBufferCurrent += rc;

    if (-1 == ( rc = ToBuffer(pBufferCurrent,
                              sizeOfBuffer - (pBufferCurrent - pBufferOut),
                              hostentIn.h_aliases)))
    {
        LogDebug("error: h_aliases \n");
        goto bail;
    };
    SERIALIZER_HEXDUMP("h_aliases: ", pBufferCurrent, rc);
    pBufferCurrent += rc;

    shortValue = hostentIn.h_addrtype;
    if (-1 == ( rc = ToBuffer(pBufferCurrent,
                              sizeOfBuffer - (pBufferCurrent - pBufferOut),
                              shortValue)))
    {
        LogDebug("error: h_addrtype \n");
        goto bail;
    };
    SERIALIZER_HEXDUMP("h_addrtype: ", pBufferCurrent, rc);
    pBufferCurrent += rc;

    shortValue = hostentIn.h_length;
    if (-1 == ( rc = ToBuffer(pBufferCurrent,
                              sizeOfBuffer - (pBufferCurrent - pBufferOut),
                              shortValue)))
    {
        LogDebug("error: h_length \n");
        goto bail;
    };
    SERIALIZER_HEXDUMP("h_length: ", pBufferCurrent, rc);
    pBufferCurrent += rc;

    if (hostentIn.h_addrtype == AF_INET)
    {
        if ( -1 == ( rc = ToBuffer(pBufferCurrent,
                                   sizeOfBuffer - (pBufferCurrent - pBufferOut),
                                   (struct in_addr**)(hostentIn.h_addr_list) )))
        {
            LogDebug("error: h_addr_list (in4)\n");
            goto bail;
        }
    }
    else if (hostentIn.h_addrtype == AF_INET6)
    {
        if (-1 == ( rc = ToBuffer(pBufferCurrent,
                                  sizeOfBuffer - (pBufferCurrent - pBufferOut),
                                  (struct in_addr**)(hostentIn.h_addr_list) )))
        {
            LogDebug("error: h_addr_list (in6)\n");
            goto bail;
        }
    }
    else
    {
        LogDebug("error: h_addrtype (%04x) unsupported, setting to zero\n",
                       hostentIn.h_addrtype);

        intValue = 0;

        if (-1 == ( rc = ToBuffer(pBufferCurrent,
                                  sizeOfBuffer - (pBufferCurrent - pBufferOut),
                                  intValue)))
        {
            LogDebug("error: h_addr_list (NULL)\n");
            goto bail;
        }
    };
    SERIALIZER_HEXDUMP("h_addr_list: ", pBufferCurrent, rc);
    pBufferCurrent += rc;
    rc = pBufferCurrent - pBufferOut;

bail:
    return rc;
}

}}}}// namespace nn::socket::resolver::serializer
