﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "serializer.h"
#include "serializer_Specializations.h"
#include "../../detail/socket_Allocator.h"

/*
  struct addrinfo {
  // ADDRINFO_MAGIC
  int              ai_flags;
  int              ai_family;
  int              ai_socktype;
  int              ai_protocol;
  socklen_t        ai_addrlen;
  struct sockaddr *ai_addr;
  char            *ai_canonname;
  struct addrinfo *ai_next;
  };
*/

//#define LOG_LEVEL LOG_LEVEL_MAJOR
#define LOG_MODULE_NAME "addrinfo serializer" // NOLINT(preprocessor/const)
#include <nn/socket/resolver/private/resolver_DebugLog.h>

namespace nn { namespace socket { namespace resolver { namespace serializer {

static const uint32_t ADDRINFO_MAGIC = 0xBEEFCAFE;

void FreeAddrinfoContents(nn::socket::AddrInfo& addrinfoIn)
{
    nn::socket::AddrInfo* save = nullptr;
    for (nn::socket::AddrInfo* pointer = &addrinfoIn; pointer != nullptr; pointer = save)
    {
        save = pointer->ai_next;

        if (nullptr != pointer->ai_addr)
        {
            switch (pointer->ai_family)
            {
                case nn::socket::Family::Af_Inet:
                {
                    socket::detail::Free((nn::socket::SockAddrIn*)pointer->ai_addr);
                    pointer->ai_addr = nullptr;
                    break;
                }
                case nn::socket::Family::Af_Inet6:
                {
                    socket::detail::Free((struct sockaddr_in6*)pointer->ai_addr);
                    pointer->ai_addr = nullptr;
                    break;
                }
                default:
                {
                    LogDebug("FreeAddrinfoContents: unhandled address family (%02x)\n",
                                   pointer->ai_family);
                    socket::detail::Free(pointer->ai_addr);
                    pointer->ai_addr = nullptr;
                }
            }
        }

        if (NULL != pointer->ai_canonname)
        {
            socket::detail::Free(pointer->ai_canonname);
            pointer->ai_canonname = nullptr;
        }

        if (&addrinfoIn != pointer)
        {
            socket::detail::Free(pointer);
        }
    }
}

void FreeAddrinfoContents(struct addrinfo& addrinfoIn)
{
    struct addrinfo* save = NULL;
    for (struct addrinfo* pointer = &addrinfoIn; pointer != NULL; pointer = save)
    {
        save = pointer->ai_next;

        if (NULL != pointer->ai_addr)
        {
            switch (pointer->ai_family)
            {
                case AF_INET:
                {
                    socket::detail::Free((struct sockaddr_in*)pointer->ai_addr);
                    pointer->ai_addr = NULL;
                    break;
                }
                case AF_INET6:
                {
                    socket::detail::Free((struct sockaddr_in6*)pointer->ai_addr);
                    pointer->ai_addr = NULL;
                    break;
                }
                default:
                {
                    LogMinor("FreeAddrinfoContents: unhandled address family (%02x)\n",
                             pointer->ai_family);
                    socket::detail::Free(pointer->ai_addr);
                    pointer->ai_addr = NULL;
                };
            };
        };

        if (NULL != pointer->ai_canonname)
        {
            socket::detail::Free(pointer->ai_canonname);
            pointer->ai_canonname = NULL;
        };

        if (&addrinfoIn != pointer)
        {
            socket::detail::Free(pointer);
        };
    };
};

size_t AddrInfoSingleSizeOf(const nn::socket::AddrInfo * pAddrinfo)
{
    DNSSerializer serializer;

    LogDebug("-------SizeOf------- addrinfo (%p)  ---------------------- \n",
                   pAddrinfo);

    size_t rc = 0;

    // we want to use the size of types here instead of the members themselves
    rc += sizeof(uint32_t); // ADDRINFO_MAGIC
    rc += sizeof(uint32_t); // ai_flags
    rc += sizeof(uint32_t); // ai_family
    rc += sizeof(uint32_t); // ai_socktype
    rc += sizeof(uint32_t); // ai_protocol
    rc += sizeof(uint32_t); // ai_addrlen

    // ai_addr
    if (nullptr == pAddrinfo->ai_addr)
    {
        rc += sizeof(uint32_t);
    }
    else if (pAddrinfo->ai_family == nn::socket::Family::Af_Inet)
    {
        rc += serializer.SizeOf(*reinterpret_cast<nn::socket::SockAddrIn* >(pAddrinfo->ai_addr));
    }
    else if (pAddrinfo->ai_family == nn::socket::Family::Af_Inet6)
    {
        rc += serializer.SizeOf(*reinterpret_cast<struct sockaddr_in6*>(pAddrinfo->ai_addr));
    }
    else if (0 == pAddrinfo->ai_addrlen)
    {
        LogDebug("address family: %02x, ai_addrlen: %d, setting size=%d\n",
                       pAddrinfo->ai_family,
                       pAddrinfo->ai_addrlen,
                       sizeof(uint32_t));
        rc += sizeof(uint32_t);
    }
    else
    {
        LogDebug("address family: %02x, ai_addrlen: %d, setting size=%d\n",
                       pAddrinfo->ai_family,
                       pAddrinfo->ai_addrlen,
                       pAddrinfo->ai_addrlen);
        rc += pAddrinfo->ai_addrlen;
    };

    // ai_canonname
    if (nullptr == pAddrinfo->ai_canonname)
    {
        rc += sizeof(uint8_t); // nullptr
    }
    else
    {
        rc += serializer.SizeOf(const_cast<const char*>(pAddrinfo->ai_canonname));
    };

    if ( nullptr == pAddrinfo->ai_next)
    {
        rc += sizeof(uint32_t); // NULL byte terminator
    }
    return rc;
}

size_t AddrInfoSingleSizeOf(const struct addrinfo * pAddrinfo)
{
    DNSSerializer serializer;

    LogDebug("-------SizeOf------- addrinfo (%p)  ---------------------- \n",
                   pAddrinfo);

    size_t rc = 0;
    rc += sizeof(uint32_t); // ADDRINFO_MAGIC
    rc += sizeof(uint32_t); // ai_flags
    rc += sizeof(uint32_t); // ai_family
    rc += sizeof(uint32_t); // ai_socktype
    rc += sizeof(uint32_t); // ai_protocol
    rc += sizeof(uint32_t); // ai_addrlen

    // ai_addr
    if (NULL == pAddrinfo->ai_addr)
    {
        rc += sizeof(uint32_t);
    }
    else if (pAddrinfo->ai_family == AF_INET)
    {
        rc += serializer.SizeOf(*reinterpret_cast<struct sockaddr_in*>(pAddrinfo->ai_addr));
    }
    else if (pAddrinfo->ai_family == AF_INET6)
    {
        rc += serializer.SizeOf(*reinterpret_cast<struct sockaddr_in6*>(pAddrinfo->ai_addr));
    }
    else if (0 == pAddrinfo->ai_addrlen)
    {
        LogDebug("address family: %02x, ai_addrlen: %d, setting size=%d\n",
                       pAddrinfo->ai_family,
                       pAddrinfo->ai_addrlen,
                       sizeof(uint32_t));
        rc += sizeof(uint32_t);
    }
    else
    {
        LogDebug("address family: %02x, ai_addrlen: %d, setting size=%d\n",
                       pAddrinfo->ai_family,
                       pAddrinfo->ai_addrlen,
                       pAddrinfo->ai_addrlen);
        rc += pAddrinfo->ai_addrlen;
    };

    // ai_canonname
    if (NULL == pAddrinfo->ai_canonname)
    {
        rc += sizeof(uint8_t); // NULL
    }
    else
    {
        rc += serializer.SizeOf(const_cast<const char*>(pAddrinfo->ai_canonname));
    };

    if ( NULL == pAddrinfo->ai_next)
    {
        rc += sizeof(uint32_t); // NULL byte terminator
    }
    return rc;
};

template <>
size_t DNSSerializer::SizeOf(const nn::socket::AddrInfo & addrinfoHeadIn)
{
    size_t rc = 0;
    const nn::socket::AddrInfo* pAddrinfo = & addrinfoHeadIn;

    while (pAddrinfo != NULL)
    {
        rc += AddrInfoSingleSizeOf(pAddrinfo);
        pAddrinfo = pAddrinfo->ai_next;
    };

    return rc;
};

template <>
size_t DNSSerializer::SizeOf(const struct addrinfo & addrinfoHeadIn)
{
    size_t rc = 0;
    const struct addrinfo* pAddrinfo = & addrinfoHeadIn;

    while (pAddrinfo != NULL)
    {
        rc += AddrInfoSingleSizeOf(pAddrinfo);
        pAddrinfo = pAddrinfo->ai_next;
    };

    return rc;
};

template <>
ssize_t DNSSerializer::ToBufferInternal(uint8_t * const pBufferOut,
                                        size_t sizeOfBuffer,
                                        const nn::socket::AddrInfo & addrinfoIn)
{
    LogDebug("-------ToBufferInternal------- addrinfo (%p)  ---------------------- \n",
                   &addrinfoIn);
    ssize_t rc = -1;
    uint8_t * pBufferCurrent = pBufferOut;
    uint32_t intValue = 0;

    // put magic
    intValue = ADDRINFO_MAGIC;
    if (-1 == ( rc = ToBuffer(pBufferCurrent,
                              sizeOfBuffer - (pBufferCurrent - pBufferOut),
                              intValue)))
    {
        LogDebug("error: magic (%d)\n", __LINE__);
        goto bail;
    };
    SERIALIZER_HEXDUMP("magic: ", pBufferCurrent, rc);
    pBufferCurrent += rc;

    // put ai_flags
    intValue = static_cast<uint32_t>(addrinfoIn.ai_flags);
    if (-1 == ( rc = ToBuffer(pBufferCurrent,
                              sizeOfBuffer - (pBufferCurrent - pBufferOut),
                              intValue)))
    {
        LogDebug("error: ai_flags (%d)\n", __LINE__);
        goto bail;
    };
    SERIALIZER_HEXDUMP("ai_flags: ", pBufferCurrent, rc);
    pBufferCurrent += rc;

    // put ai_family
    intValue = static_cast<uint32_t>(addrinfoIn.ai_family);
    if (-1 == ( rc = ToBuffer(pBufferCurrent,
                              sizeOfBuffer - (pBufferCurrent - pBufferOut),
                              intValue)))
    {
        LogDebug("error: ai_family (%d)\n", __LINE__);
        goto bail;
    };
    SERIALIZER_HEXDUMP("ai_family: ", pBufferCurrent, rc);
    pBufferCurrent += rc;

    // put ai_socktype
    intValue = static_cast<uint32_t>(addrinfoIn.ai_socktype);
    if (-1 == ( rc = ToBuffer(pBufferCurrent,
                              sizeOfBuffer - (pBufferCurrent - pBufferOut),
                              intValue)))
    {
        LogDebug("error: ai_socktype (%d)\n", __LINE__);
        goto bail;
    };
    SERIALIZER_HEXDUMP("ai_socktype: ", pBufferCurrent, rc);
    pBufferCurrent += rc;

    // put ai_protocol
    intValue = static_cast<uint32_t>(addrinfoIn.ai_protocol);
    if (-1 == ( rc = ToBuffer(pBufferCurrent,
                              sizeOfBuffer - (pBufferCurrent - pBufferOut),
                              intValue)))
    {
        LogDebug("error: ai_protocol (%d)\n", __LINE__);
        goto bail;
    };
    SERIALIZER_HEXDUMP("ai_protocol: ", pBufferCurrent, rc);
    pBufferCurrent += rc;

    // put ai_addrlen
    intValue = addrinfoIn.ai_addrlen;
    if (-1 == ( rc = ToBuffer(pBufferCurrent,
                              sizeOfBuffer - (pBufferCurrent - pBufferOut),
                              intValue)))
    {
        LogDebug("error: ai_addrlen (%d)\n", __LINE__);
        goto bail;
    };
    SERIALIZER_HEXDUMP("ai_addrlen: ", pBufferCurrent, rc);
    pBufferCurrent += rc;

    // if ai_addrlen is zero then
    if (NULL == addrinfoIn.ai_addr || 0 == addrinfoIn.ai_addrlen)
    {
        // this is a bad vector to the serializer, so handle it as though it were NULL
        LogDebug("warning: ai_addr %s NULL (%p) but ai_addrlen %s zero (%d)\n",
                       addrinfoIn.ai_addr == NULL ? "is" : "is not",
                       addrinfoIn.ai_addr,
                       addrinfoIn.ai_addrlen == 0 ? "is" : "is not",
                       addrinfoIn.ai_addrlen);
        intValue = 0;
        if ( -1 == ( rc = ToBuffer(pBufferCurrent,
                                   sizeOfBuffer - (pBufferCurrent - pBufferOut),
                                   intValue)))
        {
            LogDebug("error: ai_addr (nn::socket::Family::Af_Inet %d)\n", __LINE__);
            goto bail;
        };
    }
    else if (nn::socket::Family::Af_Inet == addrinfoIn.ai_family)
    {
        if ( -1 == ( rc = ToBuffer(pBufferCurrent,
                                   sizeOfBuffer - (pBufferCurrent - pBufferOut),
                                   *(struct sockaddr_in*)(addrinfoIn.ai_addr))))
        {
            LogDebug("error: ai_addr (nn::socket::Family::Af_Inet %d)\n", __LINE__);
            goto bail;
        };
    }
    else if (nn::socket::Family::Af_Inet6 == addrinfoIn.ai_family)
    {
        if (-1 == ( rc = ToBuffer(pBufferCurrent,
                                  sizeOfBuffer - (pBufferCurrent - pBufferOut),
                                  *(struct sockaddr_in6*)(addrinfoIn.ai_addr))))
        {
            LogDebug("error: ai_addr (nn::socket::Family::Af_Inet6 %d)\n", __LINE__);
            goto bail;
        };
    }
    // we do not specifically know how to handle this address family, so we just
    // copy ai_addrlen bytes into the buffer from the address of ai_addr
    else
    {
        LogDebug("error: unexpected address family (%02x)\n", addrinfoIn.ai_family);
        if ( sizeOfBuffer - (pBufferCurrent - pBufferOut) < addrinfoIn.ai_addrlen )
        {
            rc = -1;
            goto bail;
        };

        memmove(pBufferCurrent, (const uint8_t*)(&addrinfoIn.ai_addr), addrinfoIn.ai_addrlen);
        rc = addrinfoIn.ai_addrlen;
    };

    SERIALIZER_HEXDUMP("ai_addr: ", pBufferCurrent, rc);
    pBufferCurrent += rc;

    // put ai_canonname
    if (-1 == ( rc = ToBuffer(pBufferCurrent,
                              sizeOfBuffer - (pBufferCurrent - pBufferOut),
                              addrinfoIn.ai_canonname)))
    {
        LogDebug("error: ai_canonname (%d)\n", __LINE__);
        goto bail;
    };
    SERIALIZER_HEXDUMP("ai_canonname: ", pBufferCurrent, rc);
    pBufferCurrent += rc;

    // put ai_next
    if ( NULL == addrinfoIn.ai_next )
    {
        intValue = 0;
        if (-1 == ( rc = ToBuffer(pBufferCurrent,
                                  sizeOfBuffer - (pBufferCurrent - pBufferOut),
                                  intValue)))
        {
            LogDebug("error: ai_next (%d)\n", __LINE__);
            goto bail;
        };

        // in this case we do not move the buffer pointer
        SERIALIZER_HEXDUMP("ai_next: ", pBufferCurrent, rc);
        pBufferCurrent += rc;
    };

    rc =  pBufferCurrent - pBufferOut;

bail:
    return rc;
} // NOLINT(impl/function_size)

template <>
ssize_t DNSSerializer::ToBuffer(uint8_t * const pBufferOut,
                                size_t sizeOfBuffer,
                                const nn::socket::AddrInfo & addrinfoIn)
{
    size_t requiredSize = SizeOf(addrinfoIn);
    ssize_t rc = -1;
    uint8_t * pBufferCurrent = pBufferOut;
    memset(pBufferOut, '\0', sizeOfBuffer);

    // argument check
    if ( -1 == (rc = CheckToBufferArguments(pBufferCurrent,
                                            sizeOfBuffer,
                                            requiredSize,
                                            __LINE__)))
    {
        LogDebug("Required size (%ul) unavailable for struct addrinfo\n", requiredSize);
        goto bail;
    };

    for (const nn::socket::AddrInfo* pAddrinfo = &addrinfoIn;
         pAddrinfo != nullptr;
         pAddrinfo = pAddrinfo->ai_next)
    {
        if ( -1 == (rc = ToBufferInternal(pBufferCurrent, sizeOfBuffer, *pAddrinfo)))
        {
            LogDebug("error: ToBufferInternal returned: %d, mark: %d\n",
                           rc, pBufferCurrent - pBufferOut);
            rc = -1;
            goto bail;
        };

        pBufferCurrent += rc;
    };

    rc = pBufferCurrent - pBufferOut;
    SERIALIZER_HEXDUMP("addrinfo ToBuffer: ", pBufferOut, rc);

bail:
    LogDebug("---ToBuffer addrinfo--- expected return: %d, actual: %d ---\n",
                   requiredSize, rc);

    return rc;
};

template <>
ssize_t DNSSerializer::ToBufferInternal(uint8_t * const pBufferOut,
                                        size_t sizeOfBuffer,
                                        const struct addrinfo & addrinfoIn)
{
    LogDebug("-------ToBufferInternal------- addrinfo (%p)  ---------------------- \n",
                   &addrinfoIn);
    ssize_t rc = -1;
    uint8_t * pBufferCurrent = pBufferOut;
    uint32_t intValue = 0;

    // put magic
    intValue = ADDRINFO_MAGIC;
    if (-1 == ( rc = ToBuffer(pBufferCurrent,
                              sizeOfBuffer - (pBufferCurrent - pBufferOut),
                              intValue)))
    {
        LogDebug("magic\n");
        goto bail;
    };
    SERIALIZER_HEXDUMP("magic: ", pBufferCurrent, rc);
    pBufferCurrent += rc;

    // put ai_flags
    intValue = addrinfoIn.ai_flags;
    if (-1 == ( rc = ToBuffer(pBufferCurrent,
                              sizeOfBuffer - (pBufferCurrent - pBufferOut),
                              intValue)))
    {
        LogDebug("ai_flags\n");
        goto bail;
    };
    SERIALIZER_HEXDUMP("ai_flags: ", pBufferCurrent, rc);
    pBufferCurrent += rc;

    // put ai_family
    intValue = addrinfoIn.ai_family;
    if (-1 == ( rc = ToBuffer(pBufferCurrent,
                              sizeOfBuffer - (pBufferCurrent - pBufferOut),
                              intValue)))
    {
        LogDebug("ai_family\n");
        goto bail;
    };
    SERIALIZER_HEXDUMP("ai_family: ", pBufferCurrent, rc);
    pBufferCurrent += rc;

    // put ai_socktype
    intValue = addrinfoIn.ai_socktype;
    if (-1 == ( rc = ToBuffer(pBufferCurrent,
                              sizeOfBuffer - (pBufferCurrent - pBufferOut),
                              intValue)))
    {
        LogDebug("ai_socktype\n");
        goto bail;
    };
    SERIALIZER_HEXDUMP("ai_socktype: ", pBufferCurrent, rc);
    pBufferCurrent += rc;

    // put ai_protocol
    intValue = addrinfoIn.ai_protocol;
    if (-1 == ( rc = ToBuffer(pBufferCurrent,
                              sizeOfBuffer - (pBufferCurrent - pBufferOut),
                              intValue)))
    {
        LogDebug("ai_protocol\n");
        goto bail;
    };
    SERIALIZER_HEXDUMP("ai_protocol: ", pBufferCurrent, rc);
    pBufferCurrent += rc;

    // put ai_addrlen
    intValue = static_cast<int>(addrinfoIn.ai_addrlen);
    if (-1 == ( rc = ToBuffer(pBufferCurrent,
                              sizeOfBuffer - (pBufferCurrent - pBufferOut),
                              intValue)))
    {
        LogDebug("ai_addrlen\n");
        goto bail;
    };
    SERIALIZER_HEXDUMP("ai_addrlen: ", pBufferCurrent, rc);
    pBufferCurrent += rc;

    // if ai_addrlen is zero then
    if (NULL == addrinfoIn.ai_addr || 0 == addrinfoIn.ai_addrlen)
    {
        // this is a bad vector to the serializer, so handle it as though it were NULL
        LogDebug("warning: ai_addr %s NULL (%p) but ai_addrlen %s zero (%d)\n",
                       addrinfoIn.ai_addr == NULL ? "is" : "is not",
                       addrinfoIn.ai_addr,
                       addrinfoIn.ai_addrlen == 0 ? "is" : "is not",
                       addrinfoIn.ai_addrlen);
        intValue = 0;
        if ( -1 == ( rc = ToBuffer(pBufferCurrent,
                                   sizeOfBuffer - (pBufferCurrent - pBufferOut),
                                   intValue)))
        {
            LogDebug("ai_addr (AF_INET)\n");
            goto bail;
        };
    }
    else if (AF_INET == addrinfoIn.ai_family)
    {
        if ( -1 == ( rc = ToBuffer(pBufferCurrent,
                                   sizeOfBuffer - (pBufferCurrent - pBufferOut),
                                   *(struct sockaddr_in*)(addrinfoIn.ai_addr))))
        {
            LogDebug("ai_addr (AF_INET)\n");
            goto bail;
        };
    }
    else if (AF_INET6 == addrinfoIn.ai_family)
    {
        if (-1 == ( rc = ToBuffer(pBufferCurrent,
                                  sizeOfBuffer - (pBufferCurrent - pBufferOut),
                                  *(struct sockaddr_in6*)(addrinfoIn.ai_addr))))
        {
            LogDebug("ai_addr (AF_INET6)\n");
            goto bail;
        };
    }
    // we do not specifically know how to handle this address family, so we just
    // copy ai_addrlen bytes into the buffer from the address of ai_addr
    else
    {
        LogDebug("unexpected address family (%02x)\n", addrinfoIn.ai_family);
        if ( sizeOfBuffer - (pBufferCurrent - pBufferOut) < addrinfoIn.ai_addrlen )
        {
            rc = -1;
            goto bail;
        };

        memmove(pBufferCurrent, (const uint8_t*)(&addrinfoIn.ai_addr), addrinfoIn.ai_addrlen);
        rc = addrinfoIn.ai_addrlen;
    };

    SERIALIZER_HEXDUMP("ai_addr: ", pBufferCurrent, rc);
    pBufferCurrent += rc;

    // put ai_canonname
    if (-1 == ( rc = ToBuffer(pBufferCurrent,
                              sizeOfBuffer - (pBufferCurrent - pBufferOut),
                              addrinfoIn.ai_canonname)))
    {
        LogDebug("ai_canonname\n");
        goto bail;
    };
    SERIALIZER_HEXDUMP("ai_canonname: ", pBufferCurrent, rc);
    pBufferCurrent += rc;

    // put ai_next
    if ( NULL == addrinfoIn.ai_next )
    {
        intValue = 0;
        if (-1 == ( rc = ToBuffer(pBufferCurrent,
                                  sizeOfBuffer - (pBufferCurrent - pBufferOut),
                                  intValue)))
        {
            LogDebug("ai_next\n");
            goto bail;
        };

        // in this case we do not move the buffer pointer
        SERIALIZER_HEXDUMP("ai_next: ", pBufferCurrent, rc);
        pBufferCurrent += rc;
    };

    rc =  pBufferCurrent - pBufferOut;

bail:
    return rc;
} // NOLINT(impl/function_size)

template <>
ssize_t DNSSerializer::ToBuffer(uint8_t * const pBufferOut,
                                size_t sizeOfBuffer,
                                const struct addrinfo & addrinfoIn)
{
    size_t requiredSize = SizeOf(addrinfoIn);
    ssize_t rc = -1;
    uint8_t * pBufferCurrent = pBufferOut;
    memset(pBufferOut, '\0', sizeOfBuffer);

    // argument check
    if ( -1 == (rc = CheckToBufferArguments(pBufferCurrent,
                                            sizeOfBuffer,
                                            requiredSize,
                                            __LINE__)))
    {
        LogDebug("Required size (%ul) unavailable for struct addrinfo\n", requiredSize);
        goto bail;
    };

    for (const struct addrinfo* pAddrinfo = &addrinfoIn;
         pAddrinfo != NULL;
         pAddrinfo = pAddrinfo->ai_next)
    {
        if ( -1 == (rc = ToBufferInternal(pBufferCurrent, sizeOfBuffer, *pAddrinfo)))
        {
            LogDebug("ToBufferInternal returned: %d, mark: %d\n",
                           rc, pBufferCurrent - pBufferOut);
            rc = -1;
            goto bail;
        };

        pBufferCurrent += rc;
    };

    rc = pBufferCurrent - pBufferOut;
    SERIALIZER_HEXDUMP("addrinfo ToBuffer: ", pBufferOut, rc);

bail:
    LogDebug("---ToBuffer addrinfo--- expected return: %d, actual: %d ---\n",
                   requiredSize, rc);

    return rc;
};


template <>
ssize_t DNSSerializer::FromBufferInternal(nn::socket::AddrInfo& addrinfoOut,
                                          const uint8_t  * pBufferIn,
                                          size_t           sizeOfBuffer)
{
    LogDebug("-------FromBufferInternal----- addrinfo (%p)  ---------------------- \n",
                   &addrinfoOut);

    ssize_t rc = -1;
    const uint8_t* pBufferCurrent = pBufferIn;
    uint32_t intValue = 0;

    memset(&addrinfoOut, '\0', sizeof(struct addrinfo));

    // get magic
    if (-1 == ( rc = FromBuffer(intValue,
                                pBufferCurrent,
                                sizeOfBuffer - (pBufferCurrent - pBufferIn))))
    {
        LogDebug("error: magic (%d)\n", __LINE__);
        goto bail;
    }
    else if ( ADDRINFO_MAGIC != intValue )
    {
        LogDebug("error: mismatched magic (%d)\n", __LINE__);
        goto bail;
    }
    SERIALIZER_HEXDUMP("magic: ", pBufferCurrent, rc);
    pBufferCurrent += rc;

    // get ai_flags
    if (-1 == ( rc = FromBuffer(intValue,
                                pBufferCurrent,
                                sizeOfBuffer - (pBufferCurrent - pBufferIn))))
    {
        LogDebug("error: ai_flags (%d)\n", __LINE__);
        goto bail;
    };
    addrinfoOut.ai_flags = static_cast<nn::socket::AddrInfoFlag>(intValue);
    SERIALIZER_HEXDUMP("ai_flags: ", pBufferCurrent, rc);
    pBufferCurrent += rc;

    // get ai_family
    if (-1 == ( rc = FromBuffer(intValue,
                                pBufferCurrent,
                                sizeOfBuffer - (pBufferCurrent - pBufferIn))))
    {
        LogDebug("error: ai_family (%d)\n", __LINE__);
        goto bail;
    };
    addrinfoOut.ai_family = static_cast<nn::socket::Family>(intValue);
    SERIALIZER_HEXDUMP("ai_family: ", pBufferCurrent, rc);
    pBufferCurrent += rc;

    // get ai_socktype
    if (-1 == ( rc = FromBuffer(intValue,
                                pBufferCurrent,
                                sizeOfBuffer - (pBufferCurrent - pBufferIn))))
    {
        LogDebug("error: ai_socktype (%d)\n", __LINE__);
        goto bail;
    };
    addrinfoOut.ai_socktype = static_cast<nn::socket::Type>(intValue);
    SERIALIZER_HEXDUMP("ai_socktype: ", pBufferCurrent, rc);
    pBufferCurrent += rc;

    // get ai_protocol
    if (-1 == ( rc = FromBuffer(intValue,
                                pBufferCurrent,
                                sizeOfBuffer - (pBufferCurrent - pBufferIn))))
    {
        LogDebug("error: ai_protocol (%d)\n", __LINE__);
        goto bail;
    };
    addrinfoOut.ai_protocol = static_cast<nn::socket::Protocol>(intValue);
    SERIALIZER_HEXDUMP("ai_protocol: ", pBufferCurrent, rc);
    pBufferCurrent += rc;

    // get ai_addrlen
    if (-1 == ( rc = FromBuffer(intValue,
                                pBufferCurrent,
                                sizeOfBuffer - (pBufferCurrent - pBufferIn))))
    {
        LogDebug("error: ai_addrlen (%d)\n", __LINE__);
        goto bail;
    };
    addrinfoOut.ai_addrlen = intValue;
    SERIALIZER_HEXDUMP("ai_addrlen: ", pBufferCurrent, rc);
    pBufferCurrent += rc;

    // if ai_addrlen == 0 then we get a uint32_t of zero, we retrieve it here
    // to ensure that we aren't parsing bad data
    if (addrinfoOut.ai_addrlen == 0)
    {
        if (-1 == ( rc = FromBuffer(intValue,
                                    pBufferCurrent,
                                    sizeOfBuffer - (pBufferCurrent - pBufferIn))))
        {
            LogDebug("error: zero ai_addr (%d)\n", __LINE__);
            goto bail;
        };

        if ( 0 != intValue )
        {
            LogDebug("error: zero ai_addr & ai_addr != nullptr (%d)\n", __LINE__);
            rc = -1;
            goto bail;
        };
        addrinfoOut.ai_addr = nullptr;
    }
    // handle ipv4 sockaddr
    else if (nn::socket::Family::Af_Inet == addrinfoOut.ai_family)
    {
        addrinfoOut.ai_addr = static_cast<nn::socket::SockAddr*>(socket::detail::Alloc(sizeof(struct sockaddr_in)));
        if ( nullptr == addrinfoOut.ai_addr )
        {
            LogDebug("error: unable to allocate struct sockaddr_in (%d)\n", __LINE__);
            rc = -1;
            goto bail;
        };
        memset(addrinfoOut.ai_addr, '\0', sizeof(struct sockaddr_in));

        if ( -1 == ( rc = FromBuffer(*reinterpret_cast<nn::socket::SockAddrIn*>(addrinfoOut.ai_addr),
                                     pBufferCurrent,
                                     sizeOfBuffer - (pBufferCurrent - pBufferIn))))
        {
            LogDebug("error: ai_addr (AF_INET %d)\n", __LINE__);
            goto bail;
        };
    }
    // handle ipv6 sockaddr
    else if (nn::socket::Family::Af_Inet6 == addrinfoOut.ai_family)
    {
        addrinfoOut.ai_addr = static_cast<nn::socket::SockAddr*>(socket::detail::Alloc(sizeof(struct sockaddr_in6)));
        if ( nullptr == addrinfoOut.ai_addr )
        {
            rc = -1;
            LogDebug("error: unable to allocate struct sockaddr_in6\n");
            goto bail;
        };
        memset(addrinfoOut.ai_addr, '\0', sizeof(struct sockaddr_in6));

        if (-1 == ( rc = FromBuffer(*(struct sockaddr_in6*)(addrinfoOut.ai_addr),
                                    pBufferCurrent,
                                    sizeOfBuffer - (pBufferCurrent - pBufferIn))))
        {
            LogDebug("error: ai_addr (AF_INET6 %d)\n");
            goto bail;
        };
    }
    // otherwise this is a sockaddr that we do not yet handle and so we just copy
    // ai_addrlen bytes and hope it works
    else
    {
        LogDebug("unhandled address family (%02x)\n", addrinfoOut.ai_family);

        addrinfoOut.ai_addr = static_cast<nn::socket::SockAddr*>(socket::detail::Alloc(addrinfoOut.ai_addrlen));
        if ( nullptr == addrinfoOut.ai_addr )
        {
            LogDebug("error: unable to allocate unknown sockaddr\n", __LINE__);
            rc = -1;
            goto bail;
        };

        memcpy(&addrinfoOut.ai_addr, pBufferCurrent, addrinfoOut.ai_addrlen);
        rc = addrinfoOut.ai_addrlen;
    };
    SERIALIZER_HEXDUMP("ai_addr: ", pBufferCurrent, rc);
    pBufferCurrent += rc;


    // ai_canonname string
    if (-1 == ( rc = FromBuffer(addrinfoOut.ai_canonname,
                                pBufferCurrent,
                                sizeOfBuffer - (pBufferCurrent - pBufferIn))))
    {
        LogDebug("error: ai_canonname (%d)\n", __LINE__);
        goto bail;
    };
    SERIALIZER_HEXDUMP("ai_canonname: ", pBufferCurrent, rc);
    pBufferCurrent += rc;

    // we peek into the buffer here zero as a terminator for the addrinfo ai_next list
    // note: each addrinfo structure is preceeded by ADDRINFO_MAGIC so that
    //       af_flags == 0 (i.e. AF_UNSPEC)
    if (-1 == ( rc = FromBuffer(intValue,
                                pBufferCurrent,
                                sizeOfBuffer - (pBufferCurrent - pBufferIn))))
    {
        LogDebug("error: ai_next (%d)\n", __LINE__);
        goto bail;
    }
    else if (0 == intValue)
    {
        addrinfoOut.ai_next = nullptr;
        SERIALIZER_HEXDUMP("ai_next: ", pBufferCurrent, rc);
        pBufferCurrent += rc;
    }
    else if (ADDRINFO_MAGIC == intValue)
    {
        // allocate new empty addrinfo for next iterative call
        addrinfoOut.ai_next = static_cast<nn::socket::AddrInfo*>(socket::detail::Alloc(sizeof(struct addrinfo)));
        if ( nullptr == addrinfoOut.ai_next)
        {
            LogDebug("error: unable to allocate struct addrinfo\n", __LINE__);
            rc = -1;
            goto bail;
        };

        memset(addrinfoOut.ai_next, '\0', sizeof(struct addrinfo));
        LogDebug("ai_next: Alloc'd %p\n", addrinfoOut.ai_next);
    }
    else
    {
        rc = -1;
        LogDebug("addrinfo stream lost sync\n");
        goto bail;
    };

    rc =  pBufferCurrent - pBufferIn;
    SERIALIZER_HEXDUMP("addrinfo FromBuffer: ", pBufferIn, rc);

bail:
    if ( rc < 0 )
    {
        FreeAddrinfoContents(addrinfoOut);
    };
    return rc;
} // NOLINT(impl/function_size)

template <>
ssize_t DNSSerializer::FromBuffer(nn::socket::AddrInfo& addrinfoOut,
                                  const uint8_t *  pBufferIn,
                                  size_t           sizeOfBuffer)
{
    ssize_t rc = 0;
    const uint8_t* pBufferCurrent = pBufferIn;
    size_t requiredSize = SizeOf(addrinfoOut);

    if ( sizeOfBuffer < requiredSize )
    {
        LogDebug("Buffer too small for FromBuffer");
        nn::socket::SetLastError(Errno::ENoSpc);
        rc = -1;
        goto bail;
    };

    for (nn::socket::AddrInfo* pAddrinfo = &addrinfoOut;
         nullptr != pAddrinfo;
         pAddrinfo = pAddrinfo->ai_next)
    {
        if (-1 == ( rc = FromBufferInternal(*pAddrinfo,
                                            pBufferCurrent,
                                            sizeOfBuffer - (pBufferCurrent - pBufferIn))))
        {
            LogDebug("error: FromBufferInternal returned: %d, mark: %d\n",
                           rc, pBufferCurrent - pBufferIn);
            rc = -1;
            goto bail;
        };

        pBufferCurrent += rc;
    };

    rc = pBufferCurrent - pBufferIn;

bail:
    LogDebug("---FromBuffer addrinfo--- returning: %d, expected: %d ---\n",
                   requiredSize, rc);

    return rc;
} // NOLINT(impl/function_size)


template <>
ssize_t DNSSerializer::FromBufferInternal(struct addrinfo& addrinfoOut,
                                          const uint8_t  * pBufferIn,
                                          size_t           sizeOfBuffer)
{
    LogDebug("-------FromBufferInternal----- addrinfo (%p)  ---------------------- \n",
                   &addrinfoOut);

    ssize_t rc = -1;
    const uint8_t* pBufferCurrent = pBufferIn;
    uint32_t intValue = 0;

    memset(&addrinfoOut, '\0', sizeof(struct addrinfo));

    // get magic
    if (-1 == ( rc = FromBuffer(intValue,
                                pBufferCurrent,
                                sizeOfBuffer - (pBufferCurrent - pBufferIn))))
    {
        LogDebug("magic\n");
        goto bail;
    }
    else if ( ADDRINFO_MAGIC != intValue )
    {
        LogDebug("mismatched magic\n");
        goto bail;
    }
    SERIALIZER_HEXDUMP("magic: ", pBufferCurrent, rc);
    pBufferCurrent += rc;

    // get ai_flags
    if (-1 == ( rc = FromBuffer(intValue,
                                pBufferCurrent,
                                sizeOfBuffer - (pBufferCurrent - pBufferIn))))
    {
        LogDebug("ai_flags\n");
        goto bail;
    };
    addrinfoOut.ai_flags = intValue;
    SERIALIZER_HEXDUMP("ai_flags: ", pBufferCurrent, rc);
    pBufferCurrent += rc;

    // get ai_family
    if (-1 == ( rc = FromBuffer(intValue,
                                pBufferCurrent,
                                sizeOfBuffer - (pBufferCurrent - pBufferIn))))
    {
        LogDebug("ai_family\n");
        goto bail;
    };
    addrinfoOut.ai_family = intValue;
    SERIALIZER_HEXDUMP("ai_family: ", pBufferCurrent, rc);
    pBufferCurrent += rc;

    // get ai_socktype
    if (-1 == ( rc = FromBuffer(intValue,
                                pBufferCurrent,
                                sizeOfBuffer - (pBufferCurrent - pBufferIn))))
    {
        LogDebug("ai_socktype\n");
        goto bail;
    };
    addrinfoOut.ai_socktype = intValue;
    SERIALIZER_HEXDUMP("ai_socktype: ", pBufferCurrent, rc);
    pBufferCurrent += rc;

    // get ai_protocol
    if (-1 == ( rc = FromBuffer(intValue,
                                pBufferCurrent,
                                sizeOfBuffer - (pBufferCurrent - pBufferIn))))
    {
        LogDebug("ai_protocol\n");
        goto bail;
    };
    addrinfoOut.ai_protocol = intValue;
    SERIALIZER_HEXDUMP("ai_protocol: ", pBufferCurrent, rc);
    pBufferCurrent += rc;

    // get ai_addrlen
    if (-1 == ( rc = FromBuffer(intValue,
                                pBufferCurrent,
                                sizeOfBuffer - (pBufferCurrent - pBufferIn))))
    {
        LogDebug("ai_addrlen\n");
        goto bail;
    };
    addrinfoOut.ai_addrlen = intValue;
    SERIALIZER_HEXDUMP("ai_addrlen: ", pBufferCurrent, rc);
    pBufferCurrent += rc;

    // if ai_addrlen == 0 then we get a uint32_t of zero, we retrieve it here
    // to ensure that we aren't parsing bad data
    if (addrinfoOut.ai_addrlen == 0)
    {
        if (-1 == ( rc = FromBuffer(intValue,
                                    pBufferCurrent,
                                    sizeOfBuffer - (pBufferCurrent - pBufferIn))))
        {
            LogDebug("zero ai_addr\n");
            goto bail;
        };

        if ( 0 != intValue )
        {
            LogDebug("zero ai_addr & ai_addr != NULL \n");
            rc = -1;
            goto bail;
        };
        addrinfoOut.ai_addr = NULL;
    }
    // handle ipv4 sockaddr
    else if (AF_INET == addrinfoOut.ai_family)
    {
        addrinfoOut.ai_addr = (struct sockaddr*) socket::detail::Alloc(sizeof(struct sockaddr_in));
        if ( NULL == addrinfoOut.ai_addr )
        {
            LogDebug("unable to allocate struct sockaddr_in\n");
            rc = -1;
            goto bail;
        };
        memset(addrinfoOut.ai_addr, '\0', sizeof(struct sockaddr_in));

        if ( -1 == ( rc = FromBuffer(*(struct sockaddr_in*)(addrinfoOut.ai_addr),
                                     pBufferCurrent,
                                     sizeOfBuffer - (pBufferCurrent - pBufferIn))))
        {
            LogDebug("ai_addr (AF_INET)\n");
            goto bail;
        };
    }
    // handle ipv6 sockaddr
    else if (AF_INET6 == addrinfoOut.ai_family)
    {
        addrinfoOut.ai_addr = (struct sockaddr*) socket::detail::Alloc(sizeof(struct sockaddr_in6));
        if ( NULL == addrinfoOut.ai_addr )
        {
            rc = -1;
            LogDebug("unable to allocate struct sockaddr_in6\n");
            goto bail;
        };
        memset(addrinfoOut.ai_addr, '\0', sizeof(struct sockaddr_in6));

        if (-1 == ( rc = FromBuffer(*(struct sockaddr_in6*)(addrinfoOut.ai_addr),
                                    pBufferCurrent,
                                    sizeOfBuffer - (pBufferCurrent - pBufferIn))))
        {
            LogDebug("ai_addr (AF_INET6)\n");
            goto bail;
        };
    }
    // otherwise this is a sockaddr that we do not yet handle and so we just copy
    // ai_addrlen bytes and hope it works
    else
    {
        LogDebug("unhandled address family (%02x)\n", addrinfoOut.ai_family);

        addrinfoOut.ai_addr = (struct sockaddr*) socket::detail::Alloc(addrinfoOut.ai_addrlen);
        if ( NULL == addrinfoOut.ai_addr )
        {
            LogDebug("unable to allocate unknown sockaddr\n");
            rc = -1;
            goto bail;
        };

        memcpy(&addrinfoOut.ai_addr, pBufferCurrent, addrinfoOut.ai_addrlen);
        rc = addrinfoOut.ai_addrlen;
    };
    SERIALIZER_HEXDUMP("ai_addr: ", pBufferCurrent, rc);
    pBufferCurrent += rc;


    // ai_canonname string
    if (-1 == ( rc = FromBuffer(addrinfoOut.ai_canonname,
                                pBufferCurrent,
                                sizeOfBuffer - (pBufferCurrent - pBufferIn))))
    {
        LogDebug("ai_canonname\n");
        goto bail;
    };
    SERIALIZER_HEXDUMP("ai_canonname: ", pBufferCurrent, rc);
    pBufferCurrent += rc;

    // we peek into the buffer here zero as a terminator for the addrinfo ai_next list
    // note: each addrinfo structure is preceeded by ADDRINFO_MAGIC so that
    //       af_flags == 0 (i.e. AF_UNSPEC)
    if (-1 == ( rc = FromBuffer(intValue,
                                pBufferCurrent,
                                sizeOfBuffer - (pBufferCurrent - pBufferIn))))
    {
        LogDebug("ai_next\n");
        goto bail;
    }
    else if (0 == intValue)
    {
        addrinfoOut.ai_next = NULL;
        SERIALIZER_HEXDUMP("ai_next: ", pBufferCurrent, rc);
        pBufferCurrent += rc;
    }
    else if (ADDRINFO_MAGIC == intValue)
    {
        // allocate new empty addrinfo for next iterative call
        addrinfoOut.ai_next = (struct addrinfo*)socket::detail::Alloc(sizeof(struct addrinfo));
        if ( NULL == addrinfoOut.ai_next)
        {
            LogDebug("unable to allocate struct addrinfo\n");
            rc = -1;
            goto bail;
        };

        memset(addrinfoOut.ai_next, '\0', sizeof(struct addrinfo));
        LogDebug("ai_next: Alloc'd %p\n", addrinfoOut.ai_next);
    }
    else
    {
        rc = -1;
        LogDebug("addrinfo stream lost sync\n");
        goto bail;
    };

    rc =  pBufferCurrent - pBufferIn;
    SERIALIZER_HEXDUMP("addrinfo FromBuffer: ", pBufferIn, rc);

bail:
    if ( rc < 0 )
    {
        FreeAddrinfoContents(addrinfoOut);
    };
    return rc;
} // NOLINT(impl/function_size)

template <>
ssize_t DNSSerializer::FromBuffer(struct addrinfo& addrinfoOut,
                                  const uint8_t *  pBufferIn,
                                  size_t           sizeOfBuffer)
{
    ssize_t rc = 0;
    const uint8_t* pBufferCurrent = pBufferIn;
    size_t requiredSize = SizeOf(addrinfoOut);

    if ( sizeOfBuffer < requiredSize )
    {
        LogDebug("Buffer too small for FromBuffer\n");
        nn::socket::SetLastError(Errno::ENoSpc);
        rc = -1;
        goto bail;
    };

    for (struct addrinfo* pAddrinfo = &addrinfoOut;
         NULL != pAddrinfo;
         pAddrinfo = pAddrinfo->ai_next)
    {
        if (-1 == ( rc = FromBufferInternal(*pAddrinfo,
                                            pBufferCurrent,
                                            sizeOfBuffer - (pBufferCurrent - pBufferIn))))
        {
            LogDebug("FromBufferInternal returned: %d, mark: %d\n",
                           rc, pBufferCurrent - pBufferIn);
            rc = -1;
            goto bail;
        };

        pBufferCurrent += rc;
    };

    rc = pBufferCurrent - pBufferIn;

bail:
    LogDebug("---FromBuffer addrinfo--- returning: %d, expected: %d ---\n", requiredSize, rc);

    return rc;
} // NOLINT(impl/function_size)

}}}}// namespace nn::socket::resolver::serializer
