﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <atomic>
#include <nn/os.h>

#include <ws2tcpip.h>

#include <nn/socket/sys/types.h>
#include <nn/socket/sys/socket.h>
#include <nn/socket/sys/errno.h>
#include <nn/socket/socket_Config.h>
#define USE_WINSOCK_FD_SET
#include <nn/socket/private/socket_PlatformTypesTranslation.h>
#include <nn/socket/socket_Api.h>

#include "socket_Api.h"
#include "socket_Allocator.h"

namespace nn      {
namespace socket  {
namespace detail  {

extern PosixWinSockConverter posixWinSockConverter;

namespace         {

/* Check there is sufficiently large bit space for 'select' emulation call */
static_assert(MAX_SOCKETS_PER_CLIENT < FD_SETSIZE * sizeof(SOCKET) * 8, "FD_SETSIZE not large enough");

static std::atomic<int>  g_InitCounter(0);

// Be sure to initialize to 0 here
int               g_FcntlState[WinsockHandleTableSize] = { };
std::mutex        g_FcntlLock[WinsockHandleTableSize];

// These next two macros are for converting a WinSock fd_set structure that was set using
// POSIX_FD_SET() to appears as if it was set using WinSock's FD_SET().
// POSIX_FD_SET() treats WinSock's fd_set structure like a posix fd_set structure,
// ignoring fd_count and treating fd_array like like fds_bits instead of as an array of SOCKETs.
// TODO: Remove these and the POSIX_FD_SET() macros when the old Select() API is removed.
#define CONVERT_POSIX_TO_WINSOCK_FDSET(socketFd, posixFds, winsockFds)  \
    do                                                                      \
    {                                                                       \
        if ((posixFds) != NULL && POSIX_FD_ISSET((socketFd), (posixFds)))   \
        {                                                                   \
            FD_SET(posixWinSockConverter.PosixToWinsockSocket(socketFd), (winsockFds));           \
        }                                                                   \
    } while (NN_STATIC_CONDITION(false))

#define CONVERT_WINSOCK_TO_POSIX_FDSET(socketFd, posixFds, winsockFds)  \
    do                                                                      \
    {                                                                       \
        if ((posixFds) != NULL &&                                           \
            POSIX_FD_ISSET((socketFd), (posixFds)) &&                       \
           !FD_ISSET(posixWinSockConverter.PosixToWinsockSocket(socketFd), (winsockFds)))         \
        {                                                                   \
            POSIX_FD_CLR((socketFd), (posixFds));                           \
        }                                                                   \
    } while (NN_STATIC_CONDITION(false))

} // namespace

int Select(int numberOfDescriptors, FdSet* pReadPosixDescriptors, FdSet* pWritePosixDescriptors, FdSet* pExceptPosixDescriptors, TimeVal* pTimeout)
NN_NOEXCEPT
{
    fd_set readWinsockDescriptors = {};
    fd_set writeWinsockDescriptors = {};
    fd_set exceptWinsockDescriptors = {};
    int    returnValue = 0;

    // When the 3 file descriptors are empty WinSock will return -1,
    // but we want to match FreeBSD (and, in turn, NX) and return 0.
    if( numberOfDescriptors == 0 )
    {
        return 0;
    }

    CopyToPlatform(&readWinsockDescriptors,   pReadPosixDescriptors);
    CopyToPlatform(&writeWinsockDescriptors,  pWritePosixDescriptors);
    CopyToPlatform(&exceptWinsockDescriptors, pExceptPosixDescriptors);

    // nfds (numberOfDescriptors) is ignored in WinSock and is included only for compatibility
    // with Berkeley socket applications.
    returnValue = ::select(numberOfDescriptors,
                           readWinsockDescriptors.fd_count  ? &readWinsockDescriptors   : nullptr,
                           writeWinsockDescriptors.fd_count ? &writeWinsockDescriptors  : nullptr,
                           &exceptWinsockDescriptors,
                           reinterpret_cast<timeval *>(pTimeout));

    if (returnValue >= 0)
    {
        if( nullptr != pReadPosixDescriptors )
        {
            nn::socket::FdSetZero(pReadPosixDescriptors);
            CopyFromPlatform(pReadPosixDescriptors,   &readWinsockDescriptors);
        }
        if( nullptr != pWritePosixDescriptors )
        {
            nn::socket::FdSetZero(pWritePosixDescriptors);
            CopyFromPlatform(pWritePosixDescriptors,  &writeWinsockDescriptors);
        }
        if( nullptr != pExceptPosixDescriptors )
        {
            nn::socket::FdSetZero(pExceptPosixDescriptors);
            CopyFromPlatform(pExceptPosixDescriptors, &exceptWinsockDescriptors);
        }

    }

    return returnValue;
}

int Select(int numberOfDescriptors, fd_set*  pReadPosixDescriptors, fd_set*  pWritePosixDescriptors, fd_set*  pExceptPosixDescriptors, timeval* pTimeout)
NN_NOEXCEPT
{
    fd_set readWinsockDescriptors = { '\0' };
    fd_set writeWinsockDescriptors = { '\0' };
    fd_set exceptWinsockDescriptors = { '\0' };
    int    posixSocket;
    int    returnValue;

    // When the 3 file descriptors are empty WinSock will return -1,
    // but we want to match FreeBSD (and, in turn, NX) and return 0.
    if( numberOfDescriptors == 0 )
    {
        return 0;
    }

    FD_ZERO(&readWinsockDescriptors);
    FD_ZERO(&writeWinsockDescriptors);
    FD_ZERO(&exceptWinsockDescriptors);

    for (posixSocket = 0; posixSocket < numberOfDescriptors; posixSocket++)
    {
        CONVERT_POSIX_TO_WINSOCK_FDSET(posixSocket, pReadPosixDescriptors,   &readWinsockDescriptors);
        CONVERT_POSIX_TO_WINSOCK_FDSET(posixSocket, pWritePosixDescriptors,  &writeWinsockDescriptors);
        CONVERT_POSIX_TO_WINSOCK_FDSET(posixSocket, pExceptPosixDescriptors, &exceptWinsockDescriptors);
    }

    returnValue = ::select(
                        numberOfDescriptors,
                        readWinsockDescriptors.fd_count  ? &readWinsockDescriptors   : NULL,
                        writeWinsockDescriptors.fd_count ? &writeWinsockDescriptors  : NULL,
                        &exceptWinsockDescriptors,
                        pTimeout
                  );

    if (returnValue >= 0)
    {
        for (posixSocket = 0; posixSocket < numberOfDescriptors; posixSocket++)
        {
            CONVERT_WINSOCK_TO_POSIX_FDSET(posixSocket, pReadPosixDescriptors,   &readWinsockDescriptors);
            CONVERT_WINSOCK_TO_POSIX_FDSET(posixSocket, pWritePosixDescriptors,  &writeWinsockDescriptors);
            CONVERT_WINSOCK_TO_POSIX_FDSET(posixSocket, pExceptPosixDescriptors, &exceptWinsockDescriptors);
        }
    }

    return returnValue;
}

ssize_t Recv(int posixSocket, void* pOutBuffer, size_t outBufferLength, MsgFlag flags)
NN_NOEXCEPT
{
    SOCKET handle = posixWinSockConverter.PosixToWinsockSocket(posixSocket);
    if ( nn::socket::InvalidSocket == handle )
    {
        nn::socket::detail::SetLastError(Errno::EBadf);
        return -1;
    };

    if (outBufferLength == 0)
    {
        return 0;
    }
    // EInval over EFault to be more Cafe-like
    else if (pOutBuffer == nullptr)
    {
        nn::socket::detail::SetLastError(Errno::EInval);
        return -1;
    }
    else if (outBufferLength > UINT_MAX)
    {
        nn::socket::detail::SetLastError(Errno::EFault);
        return -1;
    }

    return ::recv(handle,
                  reinterpret_cast<char*>(pOutBuffer),
                  static_cast<int>(outBufferLength),
                  MapMsgFlagValue(flags));
}

ssize_t Recv(int posixSocket, void* pOutBuffer, size_t outBufferLength, int flags)
NN_NOEXCEPT
{
    SOCKET handle = posixWinSockConverter.PosixToWinsockSocket(posixSocket);
    if ( nn::socket::InvalidSocket == handle )
    {
        nn::socket::detail::SetLastError(Errno::EBadf);
        return -1;
    };

    if (outBufferLength == 0)
    {
        return 0;
    }
    // EInval over EFault to be more Cafe-like
    else if (pOutBuffer == nullptr)
    {
        nn::socket::detail::SetLastError(Errno::EInval);
        return -1;
    }
    else if (outBufferLength > UINT_MAX)
    {
        nn::socket::detail::SetLastError(Errno::EFault);
        return -1;
    }

    return ::recv(handle,
                  reinterpret_cast<char*>(pOutBuffer),
                  static_cast<int>(outBufferLength),
                  flags);
}

ssize_t RecvFromInternal(int posixSocket, void* outBuffer, size_t outBufferLength, MsgFlag flags, SockAddr* pOutAddress, SockLenT* pOutAddressLength)
NN_NOEXCEPT
{
    SOCKET handle = posixWinSockConverter.PosixToWinsockSocket(posixSocket);
    if ( nn::socket::InvalidSocket == handle )
    {
        nn::socket::detail::SetLastError(Errno::EBadf);
        return -1;
    };

    sockaddr sa = {};

    ssize_t bytes = ::recvfrom(handle,
                      reinterpret_cast<char*>(outBuffer),
                      static_cast<int>(outBufferLength),
                      MapMsgFlagValue(flags),
                      (nullptr != pOutAddress) ? &sa : nullptr,
                      reinterpret_cast<socklen_t*>(pOutAddressLength));

    if( nullptr != pOutAddress )
    {
        nn::socket::detail::CopyFromPlatform(reinterpret_cast<SockAddrIn*>(pOutAddress), reinterpret_cast<const sockaddr_in*>(&sa));
    }

    return bytes;
}

ssize_t RecvFromInternal(int posixSocket, void* outBuffer, size_t outBufferLength, int flags, sockaddr* pOutAddress, socklen_t* pOutAddressLength)
NN_NOEXCEPT
{
    SOCKET handle = posixWinSockConverter.PosixToWinsockSocket(posixSocket);
    if ( nn::socket::InvalidSocket == handle )
    {
        nn::socket::detail::SetLastError(Errno::EBadf);
        return -1;
    };

    return ::recvfrom(handle,
                      reinterpret_cast<char*>(outBuffer),
                      static_cast<int>(outBufferLength),
                      flags,
                      pOutAddress,
                      pOutAddressLength);
}

ssize_t RecvFrom(int posixSocket, void* outBuffer, size_t outBufferLength, MsgFlag flags, SockAddr* pOutAddress, SockLenT* pOutAddressLength)
NN_NOEXCEPT
{
    ssize_t rc = -1;
    SOCKET handle = posixWinSockConverter.PosixToWinsockSocket(posixSocket);

    if ( nn::socket::InvalidSocket == handle )
    {
        nn::socket::detail::SetLastError(Errno::EBadf);
        return -1;
    }
    else if (outBufferLength == 0)
    {
        return 0;
    }
    // EInval over EFault to be more Cafe-like
    else if (outBuffer == nullptr)
    {
        nn::socket::detail::SetLastError(Errno::EInval);
        return -1;
    }
    else if (outBufferLength > UINT_MAX)
    {
        nn::socket::detail::SetLastError(Errno::EFault);
        return -1;
    }
    else if (!(flags & MsgFlag::Msg_DontWait))
    {
        rc = RecvFromInternal(posixSocket,
                              outBuffer,
                              outBufferLength,
                              flags,
                              pOutAddress,
                              pOutAddressLength);
    }
    // MSG_DONTWAIT is unsupported on Windows, so we emulate it here
    else
    {
        std::lock_guard<std::mutex> lock(g_FcntlLock[posixSocket]);
        int nonblock = 1;
        bool isDontWait = false;
        nn::socket::Errno realError = nn::socket::Errno::ESuccess;

        // remove flags so ::recvfrom does not fail
        flags &= ~MsgFlag::Msg_DontWait;

        // check fcntlstate to see if client already set non-blocking
        if ( g_FcntlState[posixSocket] == 0 )
        {
            // if not then we set it here
            if ( SOCKET_ERROR == (rc = ::ioctlsocket(handle,
                                                     FIONBIO,
                                                     reinterpret_cast<u_long*>(&nonblock)
                                      )))
            {
                goto bail;
            };

            // follow-up when finished
            isDontWait = true;
        };

        rc = RecvFromInternal(posixSocket,
                              outBuffer,
                              outBufferLength,
                              flags,
                              pOutAddress,
                              pOutAddressLength);

        realError = nn::socket::detail::GetLastError();

        if ( true == isDontWait )
        {
            nonblock = 0;
            ssize_t rcInner = -1;

            do
            {
                rcInner = ::ioctlsocket(handle,
                                        FIONBIO,
                                        reinterpret_cast<u_long*>(&nonblock));
            } while (rcInner == SOCKET_ERROR && nn::socket::detail::GetLastError() == nn::socket::Errno::EInProgress);
        }
        nn::socket::detail::SetLastError(realError);
    }; // end FcntlLock

bail:

    return rc;
}

ssize_t RecvFrom(int posixSocket, void* outBuffer, size_t outBufferLength, int flags, sockaddr* pOutAddress, socklen_t* pOutAddressLength)
NN_NOEXCEPT
{
    ssize_t rc = -1;
    SOCKET handle = posixWinSockConverter.PosixToWinsockSocket(posixSocket);

    if ( nn::socket::InvalidSocket == handle )
    {
        nn::socket::detail::SetLastError(Errno::EBadf);
        return -1;
    }
    else if (!(flags & MSG_DONTWAIT))
    {
        rc = RecvFromInternal(posixSocket,
            outBuffer,
            outBufferLength,
            flags,
            pOutAddress,
            pOutAddressLength);
    }
    // MSG_DONTWAIT is unsupported on Windows, so we emulate it here
    else
    {
        std::lock_guard<std::mutex> lock(g_FcntlLock[posixSocket]);
        int nonblock = 1;
        bool isDontWait = false;
        nn::socket::Errno realError = nn::socket::Errno::ESuccess;

        // remove flags so ::recvfrom does not fail
        flags &= ~MSG_DONTWAIT;

        // check fcntlstate to see if client already set non-blocking
        if ( g_FcntlState[posixSocket] == 0 )
        {
            // if not then we set it here
            if ( SOCKET_ERROR == (rc = ::ioctlsocket(handle,
                FIONBIO,
                reinterpret_cast<u_long*>(&nonblock)
            )))
            {
                goto bail;
            };

            // follow-up when finished
            isDontWait = true;
        };

        rc = RecvFromInternal(posixSocket,
            outBuffer,
            outBufferLength,
            flags,
            pOutAddress,
            pOutAddressLength);

        realError = nn::socket::detail::GetLastError();

        if ( true == isDontWait )
        {
            nonblock = 0;
            ssize_t rcInner = -1;

            do
            {
                rcInner = ::ioctlsocket(handle,
                    FIONBIO,
                    reinterpret_cast<u_long*>(&nonblock));
            } while (rcInner == SOCKET_ERROR && nn::socket::detail::GetLastError() == nn::socket::Errno::EInProgress);
        }
        nn::socket::detail::SetLastError(realError);
    }; // end FcntlLock

bail:

    return rc;
}

ssize_t Send(int posixSocket, const void* buffer, size_t bufferLength, MsgFlag flags)
NN_NOEXCEPT
{
    SOCKET handle = posixWinSockConverter.PosixToWinsockSocket(posixSocket);
    if (nn::socket::InvalidSocket == handle)
    {
        nn::socket::detail::SetLastError(Errno::EBadf);
        return -1;
    };

    return ::send(handle,
                  reinterpret_cast<const char*>(buffer),
                  static_cast<int>(bufferLength),
                  MapMsgFlagValue(flags));
}

ssize_t Send(int posixSocket, const void* buffer, size_t bufferLength, int flags)
NN_NOEXCEPT
{
    SOCKET handle = posixWinSockConverter.PosixToWinsockSocket(posixSocket);
    if (nn::socket::InvalidSocket == handle)
    {
        nn::socket::detail::SetLastError(Errno::EBadf);
        return -1;
    };

    return ::send(handle,
                  reinterpret_cast<const char*>(buffer),
                  static_cast<int>(bufferLength),
                 flags);
}

ssize_t SendTo(int posixSocket, const void* buffer, size_t bufferLength, MsgFlag flags, const SockAddr* pInAddress, SockLenT addressLength)
NN_NOEXCEPT
{
    SOCKET handle = posixWinSockConverter.PosixToWinsockSocket(posixSocket);
    if ( nn::socket::InvalidSocket == handle )
    {
        nn::socket::detail::SetLastError(Errno::EBadf);
        return -1;
    }
    else if ((flags & MsgFlag::Msg_DontWait) != MsgFlag::Msg_None)
    {
        // sendto is inherently non-blocking but Windows ::sendto will fail
        // if you have MSG_DONTWAIT in the flags
        flags &= ~MsgFlag::Msg_DontWait;
    };

    sockaddr sa = {};
    nn::socket::detail::CopyToPlatform(reinterpret_cast<sockaddr_in*>(&sa), reinterpret_cast<const SockAddrIn*>(pInAddress));

    return ::sendto(handle,
                    reinterpret_cast<const char*>(buffer),
                    static_cast<int>(bufferLength),
                    MapMsgFlagValue(flags),
                    (nullptr != pInAddress) ? &sa : nullptr,
                    static_cast<socklen_t>(addressLength));
}

ssize_t SendTo(int posixSocket, const void* buffer, size_t bufferLength, int flags, const sockaddr* pAddress, socklen_t addressLength)
NN_NOEXCEPT
{
    SOCKET handle = posixWinSockConverter.PosixToWinsockSocket(posixSocket);
    if ( nn::socket::InvalidSocket == handle )
    {
        nn::socket::detail::SetLastError(Errno::EBadf);
        return -1;
    }
    else if (flags & MSG_DONTWAIT)
    {
        // sendto is inherently non-blocking but Windows ::sendto will fail
        // if you have MSG_DONTWAIT in the flags
        flags &= ~MSG_DONTWAIT;
    };

    return ::sendto(handle,
                    reinterpret_cast<const char*>(buffer),
                    static_cast<int>(bufferLength),
                    flags,
                    pAddress,
                    addressLength);
}

int Accept(int posixSocket, SockAddr* pOutAddress, SockLenT* pOutAddressLength)
NN_NOEXCEPT
{
    SOCKET handle = posixWinSockConverter.PosixToWinsockSocket(posixSocket);
    if (nn::socket::InvalidSocket == handle)
    {
        nn::socket::detail::SetLastError(Errno::EBadf);
        return -1;
    };

    SOCKET winsockSocket;
    int    newPosixSocket = -1;

    sockaddr sa = {};

    winsockSocket = ::accept(handle,
                            (nullptr != pOutAddress) ? &sa : nullptr,
                             reinterpret_cast<socklen_t*>(pOutAddressLength));

    if( nullptr != pOutAddress )
    {
        nn::socket::detail::CopyFromPlatform(reinterpret_cast<SockAddrIn*>(pOutAddress), reinterpret_cast<const sockaddr_in*>(&sa));
    }

    if (winsockSocket != nn::socket::InvalidSocket)
    {
        newPosixSocket = posixWinSockConverter.AcquirePosixHandle(winsockSocket);
        if (newPosixSocket < 0)
        {
            closesocket(winsockSocket);
            return -1;
        }
    }

    return newPosixSocket;
}

int Accept(int posixSocket, sockaddr* pOutAddress, socklen_t* pOutAddressLength)
NN_NOEXCEPT
{
    SOCKET handle = posixWinSockConverter.PosixToWinsockSocket(posixSocket);
    if (nn::socket::InvalidSocket == handle)
    {
        nn::socket::detail::SetLastError(Errno::EBadf);
        return -1;
    };

    SOCKET winsockSocket;
    int    newPosixSocket = -1;

    winsockSocket = ::accept(handle,
                             pOutAddress,
                             pOutAddressLength);

    if (winsockSocket != nn::socket::InvalidSocket)
    {
        newPosixSocket = posixWinSockConverter.AcquirePosixHandle(winsockSocket);
        if (newPosixSocket < 0)
        {
            closesocket(winsockSocket);
            return -1;
        }
    }

    return newPosixSocket;
}

int Bind(int posixSocket, const SockAddr* pInAddress, SockLenT addressLength)
NN_NOEXCEPT
{
    SOCKET handle = posixWinSockConverter.PosixToWinsockSocket(posixSocket);
    if (nn::socket::InvalidSocket == handle)
    {
        nn::socket::detail::SetLastError(Errno::EBadf);
        return -1;
    };

    sockaddr sa = {};

    if( nullptr != pInAddress )
    {
        nn::socket::detail::CopyToPlatform(reinterpret_cast<sockaddr_in*>(&sa), reinterpret_cast<const SockAddrIn*>(pInAddress));
    }

    return ::bind(handle,
                  (nullptr != pInAddress) ? &sa : nullptr,
                  static_cast<socklen_t>(addressLength));
}

int Bind(int posixSocket, const sockaddr* pAddress, socklen_t addressLength)
NN_NOEXCEPT
{
    SOCKET handle = posixWinSockConverter.PosixToWinsockSocket(posixSocket);
    if (nn::socket::InvalidSocket == handle)
    {
        nn::socket::detail::SetLastError(Errno::EBadf);
        return -1;
    };

    return ::bind(handle,
                  pAddress,
                  addressLength);
}

int Connect(int posixSocket, const SockAddr* pInAddress, SockLenT addressLength)
NN_NOEXCEPT
{
    SOCKET handle = posixWinSockConverter.PosixToWinsockSocket(posixSocket);
    if (nn::socket::InvalidSocket == handle)
    {
        nn::socket::detail::SetLastError(Errno::EBadf);
        return -1;
    };

    sockaddr sa = {};

    if( nullptr != pInAddress )
    {
        nn::socket::detail::CopyToPlatform(reinterpret_cast<sockaddr_in*>(&sa), reinterpret_cast<const SockAddrIn*>(pInAddress));
    }

    return ::connect(handle,
                     (nullptr != pInAddress) ? &sa : nullptr,
                     static_cast<SockLenT>(addressLength));
}

int Connect(int posixSocket, const sockaddr* pAddress, socklen_t addressLength)
NN_NOEXCEPT
{
    SOCKET handle = posixWinSockConverter.PosixToWinsockSocket(posixSocket);
    if (nn::socket::InvalidSocket == handle)
    {
        nn::socket::detail::SetLastError(Errno::EBadf);
        return -1;
    };

    return ::connect(handle,
                     pAddress,
                     addressLength);
}

int GetPeerName(int posixSocket, SockAddr* pOutAddress, SockLenT* pOutAddressLength)
NN_NOEXCEPT
{
    SOCKET handle = posixWinSockConverter.PosixToWinsockSocket(posixSocket);
    if (nn::socket::InvalidSocket == handle)
    {
        nn::socket::detail::SetLastError(Errno::EBadf);
        return -1;
    };

    sockaddr sa = {};

    int result = ::getpeername(handle,
                               (nullptr != pOutAddress) ? &sa : nullptr,
                               reinterpret_cast<socklen_t*>(pOutAddressLength));

    if( nullptr != pOutAddress )
    {
        nn::socket::detail::CopyFromPlatform(reinterpret_cast<SockAddrIn*>(pOutAddress), reinterpret_cast<const sockaddr_in*>(&sa));
    }

    return result;
}

int GetPeerName(int posixSocket, sockaddr* pOutAddress, socklen_t* pOutAddressLength)
NN_NOEXCEPT
{
    SOCKET handle = posixWinSockConverter.PosixToWinsockSocket(posixSocket);
    if (nn::socket::InvalidSocket == handle)
    {
        nn::socket::detail::SetLastError(Errno::EBadf);
        return -1;
    };

    return ::getpeername(handle,
                         pOutAddress,
                         pOutAddressLength);
}

int GetSockName(int posixSocket, SockAddr* pOutAddress, SockLenT* pOutAddressLength)
NN_NOEXCEPT
{
    SOCKET handle = posixWinSockConverter.PosixToWinsockSocket(posixSocket);
    if ( nn::socket::InvalidSocket == handle )
    {
        nn::socket::detail::SetLastError(Errno::EBadf);
        return -1;
    };

    sockaddr sa = {};

    int result = ::getsockname(handle,
                               (nullptr != pOutAddress) ? &sa : nullptr,
                               reinterpret_cast<socklen_t*>(pOutAddressLength));

    if( nullptr != pOutAddress )
    {
        nn::socket::detail::CopyFromPlatform(reinterpret_cast<SockAddrIn*>(pOutAddress), reinterpret_cast<const sockaddr_in*>(&sa));
    }

    return result;
}

int GetSockName(int posixSocket, sockaddr* pOutAddress, socklen_t* pOutAddressLength)
NN_NOEXCEPT
{
    SOCKET handle = posixWinSockConverter.PosixToWinsockSocket(posixSocket);
    if ( nn::socket::InvalidSocket == handle )
    {
        nn::socket::detail::SetLastError(Errno::EBadf);
        return -1;
    };

    return ::getsockname(handle,
                         pOutAddress,
                         pOutAddressLength);
}

int GetSockOpt(int posixSocket, Level level, Option optionName, void* pOutOptionValue, SockLenT* pOutOptionLength)
NN_NOEXCEPT
{
    union SocketOptionValue
    {
        linger      optionLinger;
        timeval     optionTimeval;
    };

    SOCKET handle = posixWinSockConverter.PosixToWinsockSocket(posixSocket);
    if ( nn::socket::InvalidSocket == handle )
    {
        nn::socket::detail::SetLastError(Errno::EBadf);
        return -1;
    };

    int                 result              = -1;
    SocketOptionValue   socketOptionValue   = {};
    socklen_t           optionValueLength   = 0;
    char*               ptrOptionValue      = nullptr;

    optionValueLength = *pOutOptionLength;

    // handle special options that have their own struct
    switch( optionName )
    {
    case Option::So_Linger:
    case Option::So_Nn_Linger:
        if( *pOutOptionLength < sizeof(socketOptionValue.optionLinger) )
        {
            nn::socket::detail::SetLastError(Errno::EInval);
            result = -1;
            goto exit;
        }
        optionValueLength = sizeof(socketOptionValue.optionLinger);
        ptrOptionValue = reinterpret_cast<char*>(&socketOptionValue.optionLinger);
        break;

    case Option::So_SndTimeo:
    case Option::So_RcvTimeo:
        if( *pOutOptionLength < sizeof(socketOptionValue.optionTimeval) )
        {
            nn::socket::detail::SetLastError(Errno::EInval);
            result = -1;
            goto exit;
        }
        optionValueLength = sizeof(socketOptionValue.optionTimeval);
        ptrOptionValue = reinterpret_cast<char*>(&socketOptionValue.optionTimeval);
        break;

    case Option::Tcp_Info:
        // unsupported in Windows - let ::getsockopt() handle it
        // fall through

    default:
        ptrOptionValue = reinterpret_cast<char*>(pOutOptionValue);
        break;
    }

    result = ::getsockopt(handle,
                          static_cast<int>(level),
                          static_cast<int>(optionName),
                          ptrOptionValue,
                          &optionValueLength);
    *pOutOptionLength = static_cast<SockLenT>(optionValueLength);

    // handle special options that have their own struct
    switch( optionName )
    {
    case Option::So_Linger:
    case Option::So_Nn_Linger:
        *pOutOptionLength = sizeof(socketOptionValue.optionLinger);
        nn::socket::detail::CopyFromPlatform(reinterpret_cast<Linger*>(pOutOptionValue), &socketOptionValue.optionLinger);
        break;

    case Option::So_SndTimeo:
    case Option::So_RcvTimeo:
        *pOutOptionLength = sizeof(socketOptionValue.optionTimeval);
        nn::socket::detail::CopyFromPlatform(reinterpret_cast<TimeVal*>(pOutOptionValue), &socketOptionValue.optionTimeval);
        break;

    default:
        // nothing special to do for other options
        break;
    }

exit:
    return result;
}

int GetSockOpt(int posixSocket, int level, int optionName, void* pOutOptionValue, socklen_t* pOutOptionLength)
NN_NOEXCEPT
{
    SOCKET handle = posixWinSockConverter.PosixToWinsockSocket(posixSocket);
    if ( nn::socket::InvalidSocket == handle )
    {
        nn::socket::detail::SetLastError(Errno::EBadf);
        return -1;
    };

    return ::getsockopt(handle,
                        level,
                        optionName,
                        reinterpret_cast<char*>(pOutOptionValue),
                        pOutOptionLength);
}

int Listen(int posixSocket, int backlog)
NN_NOEXCEPT
{
    SOCKET handle = posixWinSockConverter.PosixToWinsockSocket(posixSocket);
    if ( nn::socket::InvalidSocket == handle )
    {
        nn::socket::detail::SetLastError(Errno::EBadf);
        return -1;
    };

    return ::listen(handle,
                    backlog);
}

int SetSockOpt(int posixSocket, Level level, Option optionName, const void* pInOptionValue, SockLenT inOptionLength)
NN_NOEXCEPT
{
    union SocketOptionValue
    {
        linger  optionLinger;
        DWORD   optionTimeoutMilliseconds;
    };

    SOCKET handle = posixWinSockConverter.PosixToWinsockSocket(posixSocket);
    if ( nn::socket::InvalidSocket == handle )
    {
        nn::socket::detail::SetLastError(Errno::EBadf);
        return -1;
    };

    int                 result              = 0;
    SocketOptionValue   socketOptionValue   = {};
    socklen_t           optionValueLength   = inOptionLength;
    const char*         ptrOptionValue      = nullptr;

    // handle special options that have their own struct
    switch( optionName )
    {
    case Option::So_Linger:
    case Option::So_Nn_Linger:
        if( inOptionLength < sizeof(socketOptionValue.optionLinger) )
        {
            nn::socket::detail::SetLastError(Errno::EInval);
            result = -1;
            goto exit;
        }
        optionValueLength = sizeof(socketOptionValue.optionLinger);
        nn::socket::detail::CopyToPlatform(&socketOptionValue.optionLinger, reinterpret_cast<const Linger*>(pInOptionValue));
        ptrOptionValue = reinterpret_cast<const char*>(&socketOptionValue.optionLinger);
        break;

    case Option::So_SndTimeo:
    case Option::So_RcvTimeo:
        if( inOptionLength < sizeof(TimeVal) )
        {
            nn::socket::detail::SetLastError(Errno::EInval);
            result = -1;
            goto exit;
        }
        optionValueLength = sizeof(socketOptionValue.optionTimeoutMilliseconds);
        socketOptionValue.optionTimeoutMilliseconds = (reinterpret_cast<const TimeVal*>(pInOptionValue)->tv_sec * 1000) + (reinterpret_cast<const TimeVal*>(pInOptionValue)->tv_usec / 1000);
        ptrOptionValue = reinterpret_cast<const char*>(&socketOptionValue.optionTimeoutMilliseconds);
        break;

    default:
        ptrOptionValue = reinterpret_cast<const char*>(pInOptionValue);
        break;
    }

    result = ::setsockopt(handle,
                          MapLevelValue(level),
                          MapOptionValue(level, optionName),
                          ptrOptionValue,
                          optionValueLength);

exit:
    return result;
}

int SetSockOpt(int posixSocket, int level, int optionName, const void* pOptionValue, socklen_t optionLength)
NN_NOEXCEPT
{
    SOCKET handle = posixWinSockConverter.PosixToWinsockSocket(posixSocket);
    if ( nn::socket::InvalidSocket == handle )
    {
        nn::socket::detail::SetLastError(Errno::EBadf);
        return -1;
    };

    DWORD   optionTimeoutMilliseconds;
    bool    bTimeOption = (optionName == SO_RCVTIMEO || optionName == SO_SNDTIMEO);

    if( bTimeOption )
    {
        bTimeOption = true;
        optionTimeoutMilliseconds = (reinterpret_cast<const timeval*>(pOptionValue)->tv_sec * 1000) + (reinterpret_cast<const timeval*>(pOptionValue)->tv_usec / 1000);
        optionLength = sizeof(optionTimeoutMilliseconds);
    }

    return ::setsockopt(handle,
                        level,
                        optionName,
                        reinterpret_cast<const char*>((!bTimeOption)? pOptionValue : &optionTimeoutMilliseconds),
                        optionLength);
}

int SockAtMark(int posixSocket)
NN_NOEXCEPT
{
    SOCKET handle = posixWinSockConverter.PosixToWinsockSocket(posixSocket);
    if ( nn::socket::InvalidSocket == handle )
    {
        nn::socket::detail::SetLastError(Errno::EBadf);
        return -1;
    };

    int atMark;
    return ::ioctlsocket(handle,
                         SIOCATMARK,
                         reinterpret_cast<unsigned long*>(&atMark)) < 0 ? (-1) : atMark;
}

int Shutdown(int posixSocket, ShutdownMethod how)
NN_NOEXCEPT
{
    SOCKET handle = posixWinSockConverter.PosixToWinsockSocket(posixSocket);
    if ( nn::socket::InvalidSocket == handle )
    {
        nn::socket::detail::SetLastError(Errno::EBadf);
        return -1;
    };

    return ::shutdown(handle, MapShutdownMethodValue(how));
}

int Shutdown(int posixSocket, int how)
NN_NOEXCEPT
{
    SOCKET handle = posixWinSockConverter.PosixToWinsockSocket(posixSocket);
    if ( nn::socket::InvalidSocket == handle )
    {
        nn::socket::detail::SetLastError(Errno::EBadf);
        return -1;
    };

    return ::shutdown(handle, how);
}

int Socket(Family domain, Type type, Protocol protocol)
NN_NOEXCEPT
{
    SOCKET winsockSocket;
    int    posixSocket = -1;

    winsockSocket = ::socket(MapFamilyValue(domain),
                             MapTypeValue(type),
                             MapProtocolValue(protocol));

    if (winsockSocket != nn::socket::InvalidSocket)
    {
        posixSocket = posixWinSockConverter.AcquirePosixHandle(winsockSocket);
        if (posixSocket < 0)
        {
            nn::socket::Errno error = nn::socket::GetLastError();
            ::closesocket(winsockSocket);
            nn::socket::SetLastError(error);
        }
    }

    return posixSocket;
}

int Socket(int domain, int type, int protocol)
NN_NOEXCEPT
{
    SOCKET winsockSocket;
    int    posixSocket = -1;

    winsockSocket = ::socket(domain,
                             type,
                             protocol);

    if (winsockSocket != nn::socket::InvalidSocket)
    {
        posixSocket = posixWinSockConverter.AcquirePosixHandle(winsockSocket);
        if (posixSocket < 0)
        {
            nn::socket::Errno error = nn::socket::GetLastError();
            ::closesocket(winsockSocket);
            nn::socket::SetLastError(error);
        }
    }

    return posixSocket;
}

int SocketExempt(int domain, int type, int protocol)
NN_NOEXCEPT
{
    NN_UNUSED(domain);
    NN_UNUSED(type);
    NN_UNUSED(protocol);
    nn::socket::detail::SetLastError(Errno::EOpNotSupp);
    return -1;
}

int SocketExempt(Family domain, Type type, Protocol protocol)
NN_NOEXCEPT
{
    NN_UNUSED(domain);
    NN_UNUSED(type);
    NN_UNUSED(protocol);
    nn::socket::detail::SetLastError(Errno::EOpNotSupp);
    return -1;
}

int Write(int posixSocket, const void* buffer, size_t bufferLength)
NN_NOEXCEPT
{
    SOCKET handle = posixWinSockConverter.PosixToWinsockSocket(posixSocket);
    if (nn::socket::InvalidSocket == handle)
    {
        nn::socket::detail::SetLastError(Errno::EBadf);
        return -1;
    };

    return ::send(handle,
                  reinterpret_cast<const char*>(buffer),
                  static_cast<int>(bufferLength),
                  0);
}

int Read(int posixSocket, void* outBuffer, size_t outBufferLength)
NN_NOEXCEPT
{
    SOCKET handle = posixWinSockConverter.PosixToWinsockSocket(posixSocket);
    if (nn::socket::InvalidSocket == handle)
    {
        nn::socket::detail::SetLastError(Errno::EBadf);
        return -1;
    };

    if (outBufferLength == 0)
    {
        return 0;
    }
    // EInval over EFault to be more Cafe-like
    else if (outBuffer == nullptr)
    {
        nn::socket::detail::SetLastError(Errno::EInval);
        return -1;
    }
    else if (outBufferLength > UINT_MAX)
    {
        nn::socket::detail::SetLastError(Errno::EFault);
        return -1;
    }

    return ::recv(handle,
                  reinterpret_cast<char*>(outBuffer),
                  static_cast<int>(outBufferLength),
                  0);
}

int Close(int posixSocket)
NN_NOEXCEPT
{
    static std::mutex s_CloseLock;
    int rc = -1;
    SOCKET handle = static_cast<SOCKET>(nn::socket::InvalidSocket);
    {
        std::lock_guard<std::mutex> lock(s_CloseLock);
        handle = posixWinSockConverter.PosixToWinsockSocket(posixSocket);
        if (nn::socket::InvalidSocket == handle)
        {
            goto bail;
        };
        posixWinSockConverter.ReleasePosixHandle(posixSocket);
    };

    rc =  ::closesocket(handle);

bail:
    return rc;
}

int Poll(PollFd* pSocketDescriptors, NfdsT numberOfDescriptors, int timeoutMilliseconds)
NN_NOEXCEPT
{
    unsigned long   i = 0;
    int             returnValue = -1;

    // if a silly big number is passed in for numberOfDescriptors, then new will throw an exception, so let's keep it realistic
    if( numberOfDescriptors > 1024 )
    {
        numberOfDescriptors = 1024;
    }

    pollfd*         pPosixSocketDescriptors = new pollfd[numberOfDescriptors];

    if( nullptr == pPosixSocketDescriptors )
    {
        nn::socket::detail::SetLastError(Errno::ENoMem);
        return -1;
    }

    for (i = 0; i < numberOfDescriptors; i++)
    {
        pSocketDescriptors[i].fd =
            static_cast<int>(posixWinSockConverter.PosixToWinsockSocket(pSocketDescriptors[i].fd));

        if ( -1 == pSocketDescriptors[i].fd )
        {
            nn::socket::detail::SetLastError(Errno::EInval);
            goto bail;
        }

        // POLLPRI does not work on Win32 in the same way that it does on BSD / the target
        pSocketDescriptors[i].events &= ~PollEvent::PollPri;

        CopyToPlatform(&pPosixSocketDescriptors[i], &pSocketDescriptors[i]);
    }

    returnValue = WSAPoll((nullptr != pSocketDescriptors)? pPosixSocketDescriptors : nullptr, numberOfDescriptors, timeoutMilliseconds);

    if (returnValue >= 0)
    {
        for (i = 0; i < numberOfDescriptors; i++)
        {
            CopyFromPlatform(&pSocketDescriptors[i], &pPosixSocketDescriptors[i]);

            // On Win32, POLLIN is defined to be POLLRDNORM | POLLRDBAND and POLLOUT is defined to be POLLWRNORM,
            // so POLLRDNORM, POLLRDBAND, or POLLWRNORM may be set even if the caller wasn't explicitly checking for them.
            // Therefore, we unset them if they weren't being checked for.
            if( ((pSocketDescriptors[i].revents | PollEvent::PollRdNorm) != PollEvent::PollNone)
             && ((pSocketDescriptors[i].events | PollEvent::PollRdNorm) == PollEvent::PollNone) )
            {
                pSocketDescriptors[i].revents &= ~PollEvent::PollRdNorm;
            }
            if( ((pSocketDescriptors[i].revents | PollEvent::PollRdBand) != PollEvent::PollNone)
             && ((pSocketDescriptors[i].events | PollEvent::PollRdBand) == PollEvent::PollNone) )
            {
                pSocketDescriptors[i].revents &= ~PollEvent::PollRdBand;
            }
            if( ((pSocketDescriptors[i].revents | PollEvent::PollWrNorm) != PollEvent::PollNone)
             && ((pSocketDescriptors[i].events | PollEvent::PollWrNorm) == PollEvent::PollNone) )
            {
                pSocketDescriptors[i].revents &= ~PollEvent::PollWrNorm;
            }

            pSocketDescriptors[i].fd =
                posixWinSockConverter.WinsockToPosixSocket(pSocketDescriptors[i].fd);

            if ( -1 == pSocketDescriptors[i].fd )
            {
                nn::socket::detail::SetLastError(Errno::EBadf);
                goto bail;
            }
        }
    }

bail:

    delete[] pPosixSocketDescriptors;

    return returnValue;
}

int Poll(pollfd* pSocketDescriptors, nfds_t numberOfDescriptors, int timeoutMilliseconds)
NN_NOEXCEPT
{
    int i;
    int returnValue = -1;

    for (i = 0; i < numberOfDescriptors; i++)
    {
        pSocketDescriptors[i].fd =
            posixWinSockConverter.PosixToWinsockSocket(static_cast<int>(pSocketDescriptors[i].fd));

        if ( -1 == pSocketDescriptors[i].fd )
        {
            nn::socket::detail::SetLastError(Errno::EInval);
            goto bail;
        }

        // POLLPRI does not work on Win32 in the same way that it does on BSD / the target
        pSocketDescriptors[i].events &= ~POLLPRI;
    }

    returnValue = WSAPoll(pSocketDescriptors, numberOfDescriptors, timeoutMilliseconds);

    if (returnValue >= 0)
    {
        for (i = 0; i < numberOfDescriptors; i++)
        {
            pSocketDescriptors[i].fd =
                posixWinSockConverter.WinsockToPosixSocket(pSocketDescriptors[i].fd);

            if ( -1 == pSocketDescriptors[i].fd )
            {
                nn::socket::detail::SetLastError(Errno::EBadf);
                goto bail;
            };
        }
    }

bail:
    return returnValue;
}

int Fcntl(int posixSocket, FcntlCommand command, ...)
NN_NOEXCEPT
{
    int     returnValue;
    int     iocmd;
    va_list args;

    SOCKET handle = posixWinSockConverter.PosixToWinsockSocket(posixSocket);
    if ( nn::socket::InvalidSocket == handle )
    {
        nn::socket::detail::SetLastError(Errno::EBadf);
        return -1;
    };

    std::lock_guard<std::mutex> lock(g_FcntlLock[posixSocket]);

    switch (command)
    {
    case FcntlCommand::F_SetFl:
        va_start(args, command);
        iocmd = ((static_cast<FcntlFlag>(va_arg(args, int)) & FcntlFlag::O_NonBlock) != FcntlFlag::None) ? 1 : 0;
        va_end(args);
        returnValue = ::ioctlsocket(handle,
                                    FIONBIO,
                                    reinterpret_cast<u_long*>(&iocmd)
        );
        if (returnValue == 0)
        {
            g_FcntlState[posixSocket] = iocmd;
        }
        break;
    case FcntlCommand::F_GetFl:
        returnValue = g_FcntlState[posixSocket] ? static_cast<int>(FcntlFlag::O_NonBlock) : 0;
        break;
    default:
        nn::socket::detail::SetLastError(Errno::EOpNotSupp);
        returnValue  = -1;
        break;
    }

    return returnValue;
}

int Fcntl(int posixSocket, int command, ...)
NN_NOEXCEPT
{
    int     returnValue;
    int     iocmd;
    va_list args;

    SOCKET handle = posixWinSockConverter.PosixToWinsockSocket(posixSocket);
    if ( nn::socket::InvalidSocket == handle )
    {
        nn::socket::detail::SetLastError(Errno::EBadf);
        return -1;
    };

    std::lock_guard<std::mutex> lock(g_FcntlLock[posixSocket]);

    switch (command)
    {
    case F_SETFL:
        va_start(args, command);
        iocmd = (va_arg(args, int) & O_NONBLOCK) ? 1 : 0;
        va_end(args);
        returnValue = ::ioctlsocket(handle,
                                    FIONBIO,
                                    reinterpret_cast<u_long*>(&iocmd)
                );
        if (returnValue == 0)
        {
            g_FcntlState[posixSocket] = iocmd;
        }
        break;
    case F_GETFL:
        returnValue = g_FcntlState[posixSocket] ? O_NONBLOCK : 0;
        break;
    default:
        nn::socket::detail::SetLastError(Errno::EOpNotSupp);
        returnValue  = -1;
        break;
    }

    return returnValue;
}

int InetPton(Family family, const char* pAddressString, void* pOutAddressBuffer)
NN_NOEXCEPT
{
    TCHAR szAddress[512];

    if (MultiByteToWideChar(
            CP_ACP,
            0,
            pAddressString,
            -1,
            szAddress,
            sizeof(szAddress) / sizeof(szAddress[0])) == 0)
    {
        return 0;
    }
    // Do not accept IPV6 addresses
    if (family == Family::Af_Inet6)
    {
        nn::socket::detail::SetLastError(Errno::EAfNoSupport);
        return 0;
    }
    return ::InetPtonW(MapFamilyValue(family), szAddress, pOutAddressBuffer);
}

int InetPton(int family, const char* pAddressString, void* pOutAddressBuffer)
NN_NOEXCEPT
{
    TCHAR szAddress[512];

    if (MultiByteToWideChar(
            CP_ACP,
            0,
            pAddressString,
            -1,
            szAddress,
            sizeof(szAddress) / sizeof(szAddress[0])) == 0)
    {
        return 0;
    }
    // Do not accept IPV6 addresses
    if (family == AF_INET6)
    {
        nn::socket::detail::SetLastError(Errno::EAfNoSupport);
        return 0;
    }
    return ::InetPtonW(family, szAddress, pOutAddressBuffer);
}

const char* InetNtop(Family family, const void* pNetworkAddress, char* addressStringBuffer, SockLenT addressStringBufferLength)
NN_NOEXCEPT
{
    char* rc = nullptr;
    int returnValue = 0;
    PCWSTR stringValue = nullptr;

    TCHAR szAddress[512];
    if (Family::Af_Inet != family)
    {
        nn::socket::detail::SetLastError(Errno::EAfNoSupport);
        goto bail;
    }
    else if (nullptr == pNetworkAddress)
    {
        nn::socket::detail::SetLastError(Errno::EInval);
        goto bail;
    }
    else if (nullptr == addressStringBuffer)
    {
        nn::socket::detail::SetLastError(Errno::ENoSpc);
        goto bail;
    }
    else if (nullptr == (stringValue = ::InetNtopW(MapFamilyValue(family),
                                                   (PVOID)pNetworkAddress,
                                                   szAddress,
                                                   sizeof(szAddress) / sizeof(szAddress[0]))))
    {
        goto bail;
    }
    else if (static_cast<int>(wcslen(stringValue)) >=
                  static_cast<int>(addressStringBufferLength))
    {
        nn::socket::detail::SetLastError(Errno::ENoSpc);
        goto bail;
    }
    else if (0 == (returnValue = WideCharToMultiByte(CP_ACP,
                                                     0,
                                                     szAddress,
                                                     -1,
                                                     addressStringBuffer,
                                                     static_cast<socklen_t>(addressStringBufferLength),
                                                     nullptr,
        nullptr)))
    {
        goto bail;
    };

    rc = addressStringBuffer;

bail:
    return rc;
}

const char* InetNtop(int family, const void* pNetworkAddress, char* addressStringBuffer, socklen_t addressStringBufferLength)
NN_NOEXCEPT
{
    char* rc = NULL;
    int returnValue = 0;
    PCWSTR stringValue = NULL;

    TCHAR szAddress[512];
    if (AF_INET != family)
    {
        nn::socket::detail::SetLastError(Errno::EAfNoSupport);
        goto bail;
    }
    else if (NULL == pNetworkAddress)
    {
        nn::socket::detail::SetLastError(Errno::EInval);
        goto bail;
    }
    else if (NULL == addressStringBuffer)
    {
        nn::socket::detail::SetLastError(Errno::ENoSpc);
        goto bail;
    }
    else if (NULL == (stringValue = ::InetNtopW(family,
                                                (PVOID)pNetworkAddress,
                                                szAddress,
                                                sizeof(szAddress) / sizeof(szAddress[0]))))
    {
        goto bail;
    }
    else if (static_cast<int>(wcslen(stringValue)) >=
                  static_cast<int>(addressStringBufferLength))
    {
        nn::socket::detail::SetLastError(Errno::ENoSpc);
        goto bail;
    }
    else if (0 == (returnValue = WideCharToMultiByte(CP_ACP,
                                                     0,
                                                     szAddress,
                                                     -1,
                                                     addressStringBuffer,
                                                     addressStringBufferLength,
                                                     NULL,
                                                     NULL)))
    {
        goto bail;
    };

    rc = addressStringBuffer;

bail:
    return rc;
}

int InetAton(const char* addressStringBuffer, InAddr* pOutNetworkAddress)
NN_NOEXCEPT
{
    in_addr posixNetworkAddress = {};

    int result = ::inet_pton(MapFamilyValue(Family::Af_Inet), addressStringBuffer, (nullptr != pOutNetworkAddress)? &posixNetworkAddress : nullptr);
    nn::socket::detail::CopyFromPlatform(pOutNetworkAddress, &posixNetworkAddress);

    if (result == 1)
    {
        return result;
    }

    nn::socket::detail::SetLastError(Errno::EAfNoSupport);
    return 0;
}

int InetAton(const char* addressStringBuffer, in_addr* pOutNetworkAddress)
NN_NOEXCEPT
{
    int ret = ::inet_pton(AF_INET,addressStringBuffer, pOutNetworkAddress);
    if (ret == 1)
    {
        return ret;
    }
    nn::socket::detail::SetLastError(Errno::EAfNoSupport);
    return 0;
}

// VS2013 update 3 deprecated inet_addr and inet_ntoa
#pragma warning(push)
#pragma warning(disable : 4996)
char* InetNtoa(InAddr networkAddress)
NN_NOEXCEPT
{
    in_addr posixNetworkAddress = {};
    nn::socket::detail::CopyToPlatform(&posixNetworkAddress, &networkAddress);

    return ::inet_ntoa(posixNetworkAddress);
}
char* InetNtoa(in_addr networkAddress)
NN_NOEXCEPT
{
    return ::inet_ntoa(networkAddress);
}
#pragma warning(pop)

uint16_t InetHtons(uint16_t hostValue)
NN_NOEXCEPT
{
    return ::htons(hostValue);
}

uint32_t InetHtonl(uint32_t hostValue)
NN_NOEXCEPT
{
    return ::htonl(hostValue);
}

uint16_t InetNtohs(uint16_t networkValue)
NN_NOEXCEPT
{
    return ::ntohs(networkValue);
}

uint32_t InetNtohl(uint32_t networkValue)
NN_NOEXCEPT
{
    return ::ntohl(networkValue);
}

void SetLastError(Errno error)
NN_NOEXCEPT
{
    WSASetLastError(MapErrnoValue(error));
}

void SetLastErrno(int posixError)
NN_NOEXCEPT
{
    int winsockError;

    switch (posixError)
    {
    case EINTR:              winsockError = WSAEINTR;           break;
    case EBADF:              winsockError = WSAEBADF;           break;
    case EACCES:             winsockError = WSAEACCES;          break;
    case EFAULT:             winsockError = WSAEFAULT;          break;
    case EINVAL:             winsockError = WSAEINVAL;          break;
    case EMFILE:             winsockError = WSAEMFILE;          break;
    case EWOULDBLOCK:        winsockError = WSAEWOULDBLOCK;     break;
    case EINPROGRESS:        winsockError = WSAEINPROGRESS;     break;
    case EALREADY:           winsockError = WSAEALREADY;        break;
    case ENOTSOCK:           winsockError = WSAENOTSOCK;        break;
    case EDESTADDRREQ:       winsockError = WSAEDESTADDRREQ;    break;
    case EMSGSIZE:           winsockError = WSAEMSGSIZE;        break;
    case EPROTOTYPE:         winsockError = WSAEPROTOTYPE;      break;
    case ENOPROTOOPT:        winsockError = WSAENOPROTOOPT;     break;
    case EPROTONOSUPPORT:    winsockError = WSAEPROTONOSUPPORT; break;
    case ESOCKTNOSUPPORT:    winsockError = WSAESOCKTNOSUPPORT; break;
    case EOPNOTSUPP:         winsockError = WSAEOPNOTSUPP;      break;
    case EPFNOSUPPORT:       winsockError = WSAEPFNOSUPPORT;    break;
    case EAFNOSUPPORT:       winsockError = WSAEAFNOSUPPORT;    break;
    case EADDRINUSE:         winsockError = WSAEADDRINUSE;      break;
    case EADDRNOTAVAIL:      winsockError = WSAEADDRNOTAVAIL;   break;
    case ENETDOWN:           winsockError = WSAENETDOWN;        break;
    case ENETUNREACH:        winsockError = WSAENETUNREACH;     break;
    case ENETRESET:          winsockError = WSAENETRESET;       break;
    case ECONNABORTED:       winsockError = WSAECONNABORTED;    break;
    case ECONNRESET:         winsockError = WSAECONNRESET;      break;
    case ENOBUFS:            winsockError = WSAENOBUFS;         break;
    case EISCONN:            winsockError = WSAEISCONN;         break;
    case ENOTCONN:           winsockError = WSAENOTCONN;        break;
    case ESHUTDOWN:          winsockError = WSAESHUTDOWN;       break;
    case ETOOMANYREFS:       winsockError = WSAETOOMANYREFS;    break;
    case ETIMEDOUT:          winsockError = WSAETIMEDOUT;       break;
    case ECONNREFUSED:       winsockError = WSAECONNREFUSED;    break;
    case ELOOP:              winsockError = WSAELOOP;           break;
    case ENAMETOOLONG:       winsockError = WSAENAMETOOLONG;    break;
    case EHOSTDOWN:          winsockError = WSAEHOSTDOWN;       break;
    case EHOSTUNREACH:       winsockError = WSAEHOSTUNREACH;    break;
    case ENOTEMPTY:          winsockError = WSAENOTEMPTY;       break;
    case EPROCLIM:           winsockError = WSAEPROCLIM;        break;
    case EUSERS:             winsockError = WSAEUSERS;          break;
    case EDQUOT:             winsockError = WSAEDQUOT;          break;
    case ESTALE:             winsockError = WSAESTALE;          break;
    case EREMOTE:            winsockError = WSAEREMOTE;         break;
    case ECANCELED:          winsockError = WSAECANCELLED;      break;
    default:                 winsockError = posixError;         break;
    }

    WSASetLastError(winsockError);
}

Errno GetLastError()
NN_NOEXCEPT
{
    return MapErrnoValue(WSAGetLastError());
}

int GetLastErrno()
NN_NOEXCEPT
{
    int posixError;
    int winsockError = WSAGetLastError();
    switch (winsockError)
    {
    case WSAEINTR:           posixError = EINTR;                break;
    case WSAEBADF:           posixError = EBADF;                break;
    case WSAEACCES:          posixError = EACCES;               break;
    case WSAEFAULT:          posixError = EINVAL;               break;  // Remapped
    case WSAEINVAL:          posixError = EINVAL;               break;
    case WSAEMFILE:          posixError = EMFILE;               break;
    case WSAEWOULDBLOCK:     posixError = EWOULDBLOCK;          break;
    case WSAEINPROGRESS:     posixError = EINPROGRESS;          break;
    case WSAEALREADY:        posixError = EALREADY;             break;
    case WSAENOTSOCK:        posixError = EBADF;                break;  // Remapped
    case WSAEDESTADDRREQ:    posixError = EDESTADDRREQ;         break;
    case WSAEMSGSIZE:        posixError = EMSGSIZE;             break;
    case WSAEPROTOTYPE:      posixError = EPROTOTYPE;           break;
    case WSAENOPROTOOPT:     posixError = ENOPROTOOPT;          break;
    case WSAEPROTONOSUPPORT: posixError = EPROTONOSUPPORT;      break;
    case WSAESOCKTNOSUPPORT: posixError = ESOCKTNOSUPPORT;      break;
    case WSAEOPNOTSUPP:      posixError = EOPNOTSUPP;           break;
    case WSAEPFNOSUPPORT:    posixError = EPFNOSUPPORT;         break;
    case WSAEAFNOSUPPORT:    posixError = EAFNOSUPPORT;         break;
    case WSAEADDRINUSE:      posixError = EADDRINUSE;           break;
    case WSAEADDRNOTAVAIL:   posixError = EADDRNOTAVAIL;        break;
    case WSAENETDOWN:        posixError = ENETDOWN;             break;
    case WSAENETUNREACH:     posixError = ENETUNREACH;          break;
    case WSAENETRESET:       posixError = ENETRESET;            break;
    case WSAECONNABORTED:    posixError = ECONNABORTED;         break;
    case WSAECONNRESET:      posixError = ECONNRESET;           break;
    case WSAENOBUFS:         posixError = ENOBUFS;              break;
    case WSAEISCONN:         posixError = EISCONN;              break;
    case WSAENOTCONN:        posixError = ENOTCONN;             break;
    case WSAESHUTDOWN:       posixError = ESHUTDOWN;            break;
    case WSAETOOMANYREFS:    posixError = ETOOMANYREFS;         break;
    case WSAETIMEDOUT:       posixError = EAGAIN;               break;   // Remapped
    case WSAECONNREFUSED:    posixError = ECONNREFUSED;         break;
    case WSAELOOP:           posixError = ELOOP;                break;
    case WSAENAMETOOLONG:    posixError = ENAMETOOLONG;         break;
    case WSAEHOSTDOWN:       posixError = EHOSTDOWN;            break;
    case WSAEHOSTUNREACH:    posixError = EHOSTUNREACH;         break;
    case WSAENOTEMPTY:       posixError = ENOTEMPTY;            break;
    case WSAEPROCLIM:        posixError = EPROCLIM;             break;
    case WSAEUSERS:          posixError = EUSERS;               break;
    case WSAEDQUOT:          posixError = EDQUOT;               break;
    case WSAESTALE:          posixError = ESTALE;               break;
    case WSAEREMOTE:         posixError = EREMOTE;              break;
    case WSAECANCELLED:      posixError = ECANCELED;            break;
    default:                 posixError = winsockError;         break;
    }
    return posixError;
}

ssize_t RecvMsg(int posixSocket, MsgHdr* pOutMessage, MsgFlag flags)
NN_NOEXCEPT
{
    posixSocket;
    pOutMessage;
    flags;
    nn::socket::detail::SetLastError(Errno::EOpNotSupp);
    return -1;
}

ssize_t RecvMsg(int posixSocket, msghdr* pOutMessage, int flags)
NN_NOEXCEPT
{
    posixSocket;
    pOutMessage;
    flags;
    nn::socket::detail::SetLastError(Errno::EOpNotSupp);
    return -1;
}

ssize_t RecvMMsg(int posixSocket, MMsgHdr* pMsgvec, size_t vlen, MsgFlag flags, nn::TimeSpan *pTimeout)
NN_NOEXCEPT
{
    posixSocket;
    pMsgvec;
    vlen;
    flags;
    pTimeout;
    nn::socket::detail::SetLastError(Errno::EOpNotSupp);
    return -1;
}

ssize_t RecvMMsg(int posixSocket, mmsghdr* pMsgvec, size_t vlen, int flags, nn::TimeSpan *pTimeout)
NN_NOEXCEPT
{
    posixSocket;
    pMsgvec;
    vlen;
    flags;
    pTimeout;
    nn::socket::detail::SetLastError(Errno::EOpNotSupp);
    return -1;
}

ssize_t SendMsg(int posixSocket, const MsgHdr* pMessage, MsgFlag flags)
NN_NOEXCEPT
{
    posixSocket;
    pMessage;
    flags;
    nn::socket::detail::SetLastError(Errno::EOpNotSupp);
    return -1;
}

ssize_t SendMsg(int posixSocket, const msghdr* pMessage, int flags)
NN_NOEXCEPT
{
    posixSocket;
    pMessage;
    flags;
    nn::socket::detail::SetLastError(Errno::EOpNotSupp);
    return -1;
}

ssize_t SendMMsg(int posixSocket, const MMsgHdr* msgvec, size_t vlen, MsgFlag flags)
NN_NOEXCEPT
{
    posixSocket;
    msgvec;
    vlen;
    flags;
    nn::socket::detail::SetLastError(Errno::EOpNotSupp);
    return -1;
}

ssize_t SendMMsg(int posixSocket, const mmsghdr* msgvec, size_t vlen, int flags)
NN_NOEXCEPT
{
    posixSocket;
    msgvec;
    vlen;
    flags;
    nn::socket::detail::SetLastError(Errno::EOpNotSupp);
    return -1;
}

int Ioctl(int posixSocket, IoctlCommand command, void* pData, size_t dataLength)
NN_NOEXCEPT
{
    SOCKET handle = posixWinSockConverter.PosixToWinsockSocket(posixSocket);
    if ( nn::socket::InvalidSocket == handle )
    {
        nn::socket::detail::SetLastError(Errno::EBadf);
        return -1;
    };

    uint32_t ioctlCommand = MapIoctlCommandValue(command);
    if( -1 == ioctlCommand )
    {
        nn::socket::detail::SetLastError(Errno::EInval);
        return -1;
    }

    dataLength;
    return ::ioctlsocket(handle,
                         ioctlCommand,
                         reinterpret_cast<u_long*>(pData));
}

int Ioctl(int posixSocket, uint32_t command, void* pData, size_t dataLength)
NN_NOEXCEPT
{
    SOCKET handle = posixWinSockConverter.PosixToWinsockSocket(posixSocket);
    if ( nn::socket::InvalidSocket == handle )
    {
        nn::socket::detail::SetLastError(Errno::EBadf);
        return -1;
    };

    // WinSock does not support FIONWRITE or FIONSPACE; FIONREAD is all that's left from our API
    if( FIONREAD != command )
    {
        nn::socket::detail::SetLastError(Errno::EInval);
        return -1;
    }

    dataLength;
    return ::ioctlsocket(handle,
                         command,
                         reinterpret_cast<u_long*>(pData));
}

int Open(const char* path, OpenFlag flags)
NN_NOEXCEPT
{
    path;
    flags;
    nn::socket::detail::SetLastError(Errno::EOpNotSupp);
    return -1;
}

int Open(const char* path, int flags)
NN_NOEXCEPT
{
    path;
    flags;
    nn::socket::detail::SetLastError(Errno::EOpNotSupp);
    return -1;
}

int Sysctl(int* pMibEntries, size_t mibEntryCount, void* pOldValue, size_t* pOldValueLength, void* pNewValue, size_t newValueLength)
NN_NOEXCEPT
{
    pMibEntries;
    mibEntryCount;
    pOldValue;
    pOldValueLength;
    pNewValue;
    newValueLength;
    nn::socket::detail::SetLastError(Errno::EOpNotSupp);
    return -1;
}

int DuplicateSocket(int socket, uint64_t ownerProcessId)
NN_NOEXCEPT
{
    socket;
    ownerProcessId;
    nn::socket::detail::SetLastError(Errno::EOpNotSupp);
    return -1;
}

int GetResourceStatistics(StatisticsType type, void* outBuffer, size_t outBufferLength, uint32_t options)
NN_NOEXCEPT
{
    NN_UNUSED(type);
    NN_UNUSED(outBuffer);
    NN_UNUSED(outBufferLength);
    NN_UNUSED(options);
    nn::socket::detail::SetLastError(Errno::EOpNotSupp);
    return -1;
}

int ShutdownAllSockets(bool forced)
NN_NOEXCEPT
{
    NN_UNUSED(forced);
    nn::socket::detail::SetLastError(Errno::EOpNotSupp);
    return -1;
}

void* Alloc(size_t size)
NN_NOEXCEPT
{
    return reinterpret_cast<void*>(new char[size]);
}

void* AllocAligned(size_t size, size_t alignment)
NN_NOEXCEPT
{
    size;
    alignment;
    return nullptr;
}

void* Calloc(size_t numberOfElements, size_t size)
NN_NOEXCEPT
{
    const size_t totalSize = numberOfElements * size;
    void *pBuf = Alloc(totalSize);
    if( nullptr != pBuf )
    {
        std::memset(pBuf, 0, totalSize);
    }
    return pBuf;
}

void* Realloc(void* address, size_t size)
NN_NOEXCEPT
{
    address;
    size;
    return nullptr;
}

void Free(void* address)
NN_NOEXCEPT
{
    delete[] address;
}

Result Initialize(const nn::socket::Config& config)
NN_NOEXCEPT
{
    int     result;
    WSADATA wsaData;
    WORD    wVersionRequested = MAKEWORD(2, 2);

    NN_UNUSED(config);

    g_InitCounter++;

    // winsock uses internal init counter
    result = WSAStartup(wVersionRequested, &wsaData);
    NN_ABORT_UNLESS(result == 0, "Failed to initialize winsock, error %d\n", result);

    return ResultSuccess();
}

Result Initialize(
    void* memoryPool, size_t memoryPoolSize,
    size_t allocatorPoolSize, int concurrentLimit)NN_NOEXCEPT
{
    Result result;
    nn::socket::ConfigDefault config(memoryPool, memoryPoolSize, allocatorPoolSize);
    config.SetConcurrencyCountMax(concurrentLimit);
    result = nn::socket::detail::Initialize(config);
    return result;
}

Result Finalize()
NN_NOEXCEPT
{
    g_InitCounter--;
    NN_ABORT_UNLESS(g_InitCounter >= 0, "Mismatched Socket Initialize/Finalize calls");
    WSACleanup();
    return ResultSuccess();
}

}}} // namespace nn::socket::detail
