﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "sf_HipcEmulatedNamedPipePort.h"

#include <mutex>
#include <cstdlib>
#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>

#include <nn/sf/sf_ISharedObject.h>

#include <nn/os/os_Event.h>
#include <nn/os/os_Semaphore.h>
#include <nn/os/os_MultipleWait.h>
#include <nn/util/util_Uuid.h>
#include <nn/util/util_IntrusiveList.h>
#include <nn/mem/mem_StandardAllocator.h>

#include <tchar.h>

#include "detail/sf_HipcWindowsThreadpoolWork.h"


namespace nn { namespace sf { namespace hipc {

namespace {

template <typename T>
SharedPointer<T> MakeSharedAttached(T* p) NN_NOEXCEPT
{
    return SharedPointer<T>(p, false);
}

// sf_HipcDirectApi-os.win32.cpp からコピー
mem::StandardAllocator& GetStandardAllocator() NN_NOEXCEPT
{
    static std::aligned_storage<1024 * 1024>::type g_Buffer;
    NN_FUNCTION_LOCAL_STATIC(mem::StandardAllocator, g_Allocator, (&g_Buffer, sizeof(g_Buffer)));
    return g_Allocator;
}

void* Allocate(size_t size) NN_NOEXCEPT
{
    return GetStandardAllocator().Allocate(size);
}

void Deallocate(void* p, size_t) NN_NOEXCEPT
{
    GetStandardAllocator().Free(p);
}

}  // namespace unnnamed


HipcEmulatedNamedPipeClientPort::HipcEmulatedNamedPipeClientPort(const TCHAR* pipeName) NN_NOEXCEPT
    : m_StateLock(false)
    , m_pCurrentNamedPipe(nullptr)
{
    ::_tcsncpy_s(m_PipeName, pipeName, _TRUNCATE);
}

HipcEmulatedNamedPipeClientPort::~HipcEmulatedNamedPipeClientPort() NN_NOEXCEPT
{
    if (m_pCurrentNamedPipe)
    {
        m_pCurrentNamedPipe->Release();
        m_pCurrentNamedPipe = nullptr;
    }
}

HipcEmulatedNamedPipeClientPort* HipcEmulatedNamedPipeClientPort::Create(const TCHAR* pipeName)  NN_NOEXCEPT
{
    return Factory<HipcEmulatedNamedPipeClientPort>::Create(pipeName);
}

void HipcEmulatedNamedPipeClientPort::AttachConnectEvent(nn::os::MultiWaitHolderType* pHolder) NN_NOEXCEPT
{
    {
        std::lock_guard<os::Mutex> lk(m_StateLock);
        if (m_pCurrentNamedPipe == nullptr)
        {
            m_pCurrentNamedPipe = Factory<detail::HipcWindowsNamedPipe>::Create();
            m_pCurrentNamedPipe->InitializeAsClient(m_PipeName);
        }
    }
    m_pCurrentNamedPipe->AttachConnectEvent(pHolder);
}

HipcEmulatedClientSession* HipcEmulatedNamedPipeClientPort::Connect(bool blocking) NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> lk(m_StateLock);

    if (m_pCurrentNamedPipe == nullptr)
    {
        m_pCurrentNamedPipe = Factory<detail::HipcWindowsNamedPipe>::Create();
        m_pCurrentNamedPipe->InitializeAsClient(m_PipeName);
    }
    if (!m_pCurrentNamedPipe->Connect(blocking))
    {
        return nullptr;
    }

    auto ret = Factory<HipcEmulatedNamedPipeClientSession>::Create(m_pCurrentNamedPipe);
    m_pCurrentNamedPipe = nullptr;
    return ret;
}


class HipcEmulatedServerPortAcceptablePipeNotificationWorker
    : public detail::HipcWindowsThreadpoolWork
{
    // ServerPort の AttachAcceptEvent 対応のため、複数の名前付きパイプのどれか 1 つについて ConnectNamedPipe が返ったら
    // シグナルするようなイベントが欲しい。名前付きパイプの同時待ちを行うスレッドを立てることで対応する
private:
    class NamedPipeEntry
        : public util::IntrusiveListBaseNode<NamedPipeEntry>
    {
    private:
        SharedPointer<detail::HipcWindowsNamedPipe> m_pNamedPipe;
    public:
        explicit NamedPipeEntry(SharedPointer<detail::HipcWindowsNamedPipe> pNamedPipe) NN_NOEXCEPT
            : m_pNamedPipe(std::move(pNamedPipe))
        {
        }
        detail::HipcWindowsNamedPipe* Get() { return m_pNamedPipe.Get(); }
    };

private:
    NamedPipeEntry* m_NamedPipes;
    int m_PipeCount;

    os::Event m_RefreshEvent;  // 「同時待ちするパイプが変わった」ことを通知する手段と「スレッドの終了」を通知する手段を兼ねる。どちらの通知かは m_ShouldExit で判断
    bool m_ShouldExit;

    os::Semaphore m_AcceptablePipeSemaphore;
    util::IntrusiveList<NamedPipeEntry, util::IntrusiveListBaseNodeTraits<NamedPipeEntry>> m_ListeningPipes;
    util::IntrusiveList<NamedPipeEntry, util::IntrusiveListBaseNodeTraits<NamedPipeEntry>> m_AcceptablePipes;
    util::IntrusiveList<NamedPipeEntry, util::IntrusiveListBaseNodeTraits<NamedPipeEntry>> m_AcceptedPipes;
    os::Mutex m_ListeningPipesLock;
    os::Mutex m_AcceptablePipesLock;
    os::Mutex m_AcceptedPipesLock;

protected:
    HipcEmulatedServerPortAcceptablePipeNotificationWorker(SharedPointer<detail::HipcWindowsNamedPipe>* pipes, int pipeCount) NN_NOEXCEPT
        : m_PipeCount(pipeCount)
        , m_RefreshEvent(os::EventClearMode_ManualClear)
        , m_ShouldExit(false)
        , m_AcceptablePipeSemaphore(0, pipeCount)
        , m_ListeningPipes()
        , m_AcceptablePipes()
        , m_AcceptedPipes()
        , m_ListeningPipesLock(false)
        , m_AcceptablePipesLock(false)
        , m_AcceptedPipesLock(false)
    {
        m_NamedPipes = reinterpret_cast<NamedPipeEntry*>(Allocate(sizeof(NamedPipeEntry) * pipeCount));
        for (int i = 0; i < pipeCount; i++)
        {
            new (&m_NamedPipes[i]) NamedPipeEntry(pipes[i]);
        }

        this->SubmitThreadpoolWork();
    }

    ~HipcEmulatedServerPortAcceptablePipeNotificationWorker() NN_NOEXCEPT
    {
        m_ShouldExit = true;
        m_RefreshEvent.Signal();
        this->WaitForThreadpoolWork(true);

        for (int i = 0; i < m_PipeCount; i++)
        {
            m_NamedPipes[i].~NamedPipeEntry();
        }
        Deallocate(m_NamedPipes, sizeof(NamedPipeEntry) * m_PipeCount);
    }

public:
    void AttachAcceptEvent(os::MultiWaitHolderType* pHolder) NN_NOEXCEPT
    {
        os::InitializeMultiWaitHolder(pHolder, m_AcceptablePipeSemaphore.GetBase());
    }

    void WaitAccept() NN_NOEXCEPT
    {
        os::WaitAny(m_AcceptablePipeSemaphore.GetBase());
    }

    SharedPointer<detail::HipcWindowsNamedPipe> Accept() NN_NOEXCEPT
    {
        m_AcceptablePipeSemaphore.Acquire();
        NamedPipeEntry* pEntry;
        {
            std::lock_guard<os::Mutex> lk(m_AcceptablePipesLock);
            NamedPipeEntry& entry = m_AcceptablePipes.front();
            m_AcceptablePipes.pop_front();
            pEntry = &entry;
        }
        NN_ABORT_UNLESS(pEntry->Get()->Accept(false));
        {
            std::lock_guard<os::Mutex> lk(m_AcceptedPipesLock);
            m_AcceptedPipes.push_back(*pEntry);
        }
        return SharedPointer<detail::HipcWindowsNamedPipe>(pEntry->Get(), true);
    }

    void NotifyPipeStateChanged() NN_NOEXCEPT
    {
        m_RefreshEvent.Signal();
    }

private:
    virtual void RunThreadpoolWork(PTP_CALLBACK_INSTANCE pInstance) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_UNUSED(pInstance);

        os::MultiWaitType multiWait;
        os::MultiWaitHolderType refreshEventHolder;
        os::MultiWaitHolderType* acceptableEventHolders = reinterpret_cast<os::MultiWaitHolderType*>(Allocate(sizeof(os::MultiWaitHolderType) * m_PipeCount));

        os::InitializeMultiWait(&multiWait);
        os::InitializeMultiWaitHolder(&refreshEventHolder, m_RefreshEvent.GetBase());

        for (int i = 0; i < m_PipeCount; i++)
        {
            m_ListeningPipes.push_back(m_NamedPipes[i]);
        }

        while (!m_ShouldExit)
        {
            {
                std::lock_guard<os::Mutex> lk(m_AcceptedPipesLock);
                for (auto it = m_AcceptedPipes.begin(); it != m_AcceptedPipes.end(); it++)
                {
                    if (!(*it).Get()->IsConnected())
                    {
                        NamedPipeEntry& entry = *it;
                        it = m_AcceptablePipes.erase(it);
                        m_ListeningPipes.push_back(entry);
                    }
                }
            }
            int acceptableEventHolderUsedCount = 0;
            {
                std::lock_guard<os::Mutex> lk(m_ListeningPipesLock);
                for (auto it = m_ListeningPipes.begin(); it != m_ListeningPipes.end(); it++)
                {
                    (*it).Get()->AttachAcceptEvent(&acceptableEventHolders[acceptableEventHolderUsedCount]);
                    os::SetMultiWaitHolderUserData(&acceptableEventHolders[acceptableEventHolderUsedCount], reinterpret_cast<uintptr_t>(&(*it)));
                    os::LinkMultiWaitHolder(&multiWait, &acceptableEventHolders[acceptableEventHolderUsedCount]);
                    acceptableEventHolderUsedCount++;
                }
            }

            os::LinkMultiWaitHolder(&multiWait, &refreshEventHolder);

            os::MultiWaitHolderType* pSignaled = os::WaitAny(&multiWait);
            os::UnlinkAllMultiWaitHolder(&multiWait);
            for (int i = 0; i < acceptableEventHolderUsedCount; i++)
            {
                os::FinalizeMultiWaitHolder(&acceptableEventHolders[i]);
            }

            if (pSignaled == &refreshEventHolder)
            {
                m_RefreshEvent.Clear();
            }
            else
            {
                NamedPipeEntry* pEntry = reinterpret_cast<NamedPipeEntry*>(os::GetMultiWaitHolderUserData(pSignaled));
                {
                    std::lock_guard<os::Mutex> lk(m_ListeningPipesLock);
                    m_ListeningPipes.erase(m_ListeningPipes.iterator_to(*pEntry));
                }
                {
                    std::lock_guard<os::Mutex> lk(m_AcceptablePipesLock);
                    m_AcceptablePipes.push_back(*pEntry);
                }
                m_AcceptablePipeSemaphore.Release();
            }
        }

        os::FinalizeMultiWaitHolder(&refreshEventHolder);
        os::FinalizeMultiWait(&multiWait);

        Deallocate(acceptableEventHolders, sizeof(os::MultiWaitHolderType) * m_PipeCount);
    }
};


HipcEmulatedNamedPipeServerPort::HipcEmulatedNamedPipeServerPort(const TCHAR* pPipeName, int maxSessions) NN_NOEXCEPT
    : m_StateLock(true)
    , m_MaxSessions(maxSessions)
    , m_pNotificationWorker(nullptr)
{
    m_NamedPipes = reinterpret_cast<SharedPointer<detail::HipcWindowsNamedPipe>*>(
        Allocate(sizeof(SharedPointer<detail::HipcWindowsNamedPipe>) * maxSessions));

    for (int i = 0; i < maxSessions; i++)
    {
        new (&m_NamedPipes[i]) SharedPointer<detail::HipcWindowsNamedPipe>(
            Factory<detail::HipcWindowsNamedPipe>::Create(), false);
        m_NamedPipes[i]->InitializeAsServer(pPipeName, m_MaxSessions);
    }

    m_pChildSessions = reinterpret_cast<HipcEmulatedNamedPipeServerSession**>(
        Allocate(sizeof(HipcEmulatedServerSession*) * maxSessions));

    std::memset(m_pChildSessions, 0, sizeof(HipcEmulatedServerSession*) * maxSessions);
}

HipcEmulatedNamedPipeServerPort::~HipcEmulatedNamedPipeServerPort() NN_NOEXCEPT
{
    if (m_pNotificationWorker)
    {
        m_pNotificationWorker->Release();
        m_pNotificationWorker = nullptr;
    }

    UnregisterAllChildSession();
    Deallocate(m_pChildSessions, sizeof(HipcEmulatedServerSession*) * m_MaxSessions);

    for (int i = 0; i < m_MaxSessions; i++)
    {
        m_NamedPipes[i].~SharedPointer<detail::HipcWindowsNamedPipe>();
    }
    Deallocate(m_NamedPipes, sizeof(SharedPointer<detail::HipcWindowsNamedPipe>) * m_MaxSessions);
}

HipcEmulatedNamedPipeServerPort* HipcEmulatedNamedPipeServerPort::Create(const TCHAR* pPipeName, int maxSessions) NN_NOEXCEPT
{
    return Factory<HipcEmulatedNamedPipeServerPort>::Create(pPipeName, maxSessions);
}

void HipcEmulatedNamedPipeServerPort::StartAccepting() NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> lk(m_StateLock);

    for (int i = 0; i < m_MaxSessions; i++)
    {
        if (!m_NamedPipes[i]->IsConnected())
        {
            m_NamedPipes[i]->Listen();
        }
    }
    NN_SDK_ASSERT(m_pNotificationWorker == nullptr);
    m_pNotificationWorker
        = Factory<HipcEmulatedServerPortAcceptablePipeNotificationWorker>::Create(m_NamedPipes, m_MaxSessions);
}

void HipcEmulatedNamedPipeServerPort::StopAccepting() NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> lk(m_StateLock);

    NN_SDK_ASSERT_NOT_NULL(m_pNotificationWorker);
    m_pNotificationWorker->Release();
    m_pNotificationWorker = nullptr;

    for (int i = 0; i < m_MaxSessions; i++)
    {
        // Listen 中のパイプについて Listen を停止させる
        if (!m_NamedPipes[i]->IsConnected())
        {
            m_NamedPipes[i]->Disconnect();
        }
    }
}

void HipcEmulatedNamedPipeServerPort::AttachAcceptEvent(nn::os::MultiWaitHolderType* pHolder) NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> lk(m_StateLock);

    NN_SDK_ASSERT_NOT_NULL(m_pNotificationWorker);
    m_pNotificationWorker->AttachAcceptEvent(pHolder);
}

void HipcEmulatedNamedPipeServerPort::WaitAccept() NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> lk(m_StateLock);

    NN_SDK_ASSERT_NOT_NULL(m_pNotificationWorker);
    m_pNotificationWorker->WaitAccept();
}

HipcEmulatedServerSession* HipcEmulatedNamedPipeServerPort::Accept() NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> lk(m_StateLock);

    NN_SDK_ASSERT_NOT_NULL(m_pNotificationWorker);

    auto connectedPipe = m_pNotificationWorker->Accept();

    auto pServerSession = Factory<HipcEmulatedNamedPipeServerSession>::Create(std::move(connectedPipe));
    RegisterChildSession(pServerSession);
    return pServerSession;
}

void HipcEmulatedNamedPipeServerPort::OnSessionClosing(
    HipcEmulatedNamedPipeServerSession* pSession, detail::HipcWindowsNamedPipe* pNamedPipe) NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> lk(m_StateLock);

    UnregisterChildSession(pSession);
    pNamedPipe->Disconnect();
    pNamedPipe->Listen();
    if (m_pNotificationWorker)
    {
        m_pNotificationWorker->NotifyPipeStateChanged();
    }
}

void HipcEmulatedNamedPipeServerPort::RegisterChildSession(HipcEmulatedNamedPipeServerSession* pSession) NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> lk(m_StateLock);

    for (int i = 0; i < m_MaxSessions; i++)
    {
        if (m_pChildSessions[i])
        {
            if (m_pChildSessions[i] == pSession)
            {
                return;
            }
        }
        else
        {
            m_pChildSessions[i] = pSession;
            pSession->SetNotificationReceiver(this);
            return;
        }
    }
    NN_SDK_ASSERT(false);
}

void HipcEmulatedNamedPipeServerPort::UnregisterChildSession(HipcEmulatedNamedPipeServerSession* pSession) NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> lk(m_StateLock);

    for (int i = 0; i < m_MaxSessions; i++)
    {
        if (m_pChildSessions[i] == pSession)
        {
            m_pChildSessions[i] = nullptr;
            pSession->SetNotificationReceiver(nullptr);
            return;
        }
    }
    NN_SDK_ASSERT(false);
}

void HipcEmulatedNamedPipeServerPort::UnregisterAllChildSession() NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> lk(m_StateLock);

    for (int i = 0; i < m_MaxSessions; i++)
    {
        if (m_pChildSessions[i])
        {
            m_pChildSessions[i]->SetNotificationReceiver(nullptr);
            m_pChildSessions[i] = nullptr;
        }
    }
}


std::pair<
    HipcEmulatedNamedPipeServerPort*,
    HipcEmulatedNamedPipeClientPort*
> CreateHipcEmulatedNamedPipePortPair(int maxSessions) NN_NOEXCEPT
{
    const int pipeNameLengthMax = detail::HipcWindowsNamedPipe::PipeNameLengthMax;

    TCHAR pipeName[pipeNameLengthMax];
    {
        nn::util::Uuid uuid = nn::util::GenerateUuid();

#ifdef UNICODE
        char pipeNameChars[pipeNameLengthMax];
        int p = ::_snprintf_s(pipeNameChars, pipeNameLengthMax, _TRUNCATE, "\\\\.\\pipe\\NintendoSDK$nn_sf_<anonymous>_");
        NN_SDK_ASSERT(p >= 0 && p <= pipeNameLengthMax);
        uuid.ToString(&pipeNameChars[p], pipeNameLengthMax - p);
        std::mbstowcs(pipeName, pipeNameChars, pipeNameLengthMax);
#else
        int p = ::_snprintf_s(pipeName, pipeNameLengthMax, _TRUNCATE, "\\\\.\\pipe\\NintendoSDK$nn_sf_<anonymous>_");
        NN_SDK_ASSERT(p >= 0 && p <= pipeNameLengthMax);
        uuid.ToString(&pipeName[p], pipeNameLengthMax - p);
#endif
    }

    HipcEmulatedNamedPipeServerPort* pServerPort = HipcEmulatedNamedPipeServerPort::Create(pipeName, maxSessions);
    HipcEmulatedNamedPipeClientPort* pClientPort = HipcEmulatedNamedPipeClientPort::Create(pipeName);

    pServerPort->StartAccepting();

    return std::make_pair(pServerPort, pClientPort);
}

}}}  // namespace nn::sf::hipc
