﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>
#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/sf/hipc/sf_HipcHandleTypes.h>

#include "sf_HipcEmulatedInProcessPort.h"
#include "sf_HipcEmulatedInProcessServiceResolver.h"

namespace nn { namespace sf { namespace hipc {

Result HipcEmulatedInProcessServiceResolver::RegisterService(HipcServerPortHandle* pOut, const char* name, int32_t maxSessions) NN_NOEXCEPT
{
    auto pair = CreateHipcEmulatedInProcessPortPair(maxSessions);
    auto serverPort = pair.first;
    auto clientPort = pair.second;
    {
        std::lock_guard<decltype(m_Mutex)> lk(m_Mutex);
        RegisterPortImpl(clientPort, name);
        m_Condition.Broadcast();
    }
    *pOut = serverPort;
    NN_RESULT_SUCCESS;
}

void HipcEmulatedInProcessServiceResolver::UnregisterService(const char* name) NN_NOEXCEPT
{
    HipcClientPortHandle port;
    {
        std::lock_guard<decltype(m_Mutex)> lk(m_Mutex);
        port = reinterpret_cast<HipcClientPortHandle>(UnregisterPortImpl(name));
    }
    port->Release();
}

bool HipcEmulatedInProcessServiceResolver::IsServiceRegistered(const char* name) NN_NOEXCEPT
{
    HipcClientPortHandle port;
    std::lock_guard<decltype(m_Mutex)> lk(m_Mutex);
    return GetPortImpl(&port, name);
}

HipcClientPortHandle HipcEmulatedInProcessServiceResolver::GetServicePortImpl(const char* name) NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lk(m_Mutex);
    HipcClientPortHandle portHandle;
    while (!GetPortImpl(&portHandle, name))
    {
        m_Condition.Wait(m_Mutex);
    }
    portHandle->AddReference();
    return portHandle;
}

}}}
