﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/sf/hipc/sf_HipcDirectApi.h>

#include <nn/nn_Common.h>
#include <nn/nn_Abort.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/result/result_HandlingUtility.h>

#include <nn/svc/svc_Result.h>
#include <nn/svc/svc_Base.h>
#include <nn/svc/svc_Server.h>
#include <nn/svc/svc_Handle.h>
#include <nn/svc/svc_Tcb.h>
#include <nn/svc/svc_ThreadLocalRegion.h>
#include <nn/os/os_SdkMultipleWaitApi.h>
#include <nn/sf/hipc/sf_HipcResult.h>
#include <cstring>

namespace nn { namespace sf { namespace hipc {

namespace {

    inline void CloseHandleImpl(svc::Handle handle) NN_NOEXCEPT
    {
        auto result = svc::CloseHandle(handle);
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    }

    inline void* GetMessageBufferOnTlsImpl() NN_NOEXCEPT
    {
        return svc::GetThreadLocalRegion()->messageBuffer;
    }

}

Result CreatePort(HipcServerPortHandle* pOutServer, HipcClientPortHandle* pOutClient, int maxSessions) NN_NOEXCEPT
{
    return svc::CreatePort(pOutServer, pOutClient, maxSessions, false, 0);
}

void CloseServerPortHandle(HipcServerPortHandle handle) NN_NOEXCEPT
{
    CloseHandleImpl(handle);
}

void AttachWaitHolderForAccept(nn::os::MultiWaitHolderType* pHolder, HipcServerPortHandle port) NN_NOEXCEPT
{
    os::InitializeMultiWaitHolder(pHolder, reinterpret_cast<nn::os::NativeHandle&>(port));
}

Result AcceptSession(HipcServerSessionHandle* pOut, HipcServerPortHandle port) NN_NOEXCEPT
{
    return svc::AcceptSession(pOut, port);
}

Result CreateSession(HipcServerSessionHandle* pOutServer, HipcClientSessionHandle* pOutClient) NN_NOEXCEPT
{
    NN_RESULT_TRY(svc::CreateSession(pOutServer, pOutClient, false, 0))
        NN_RESULT_CATCH(svc::ResultOutOfResource)
        {
            NN_RESULT_THROW(hipc::ResultOutOfSession());
        }
    NN_RESULT_END_TRY
    NN_RESULT_SUCCESS;
}

void CloseServerSessionHandle(HipcServerSessionHandle handle) NN_NOEXCEPT
{
    CloseHandleImpl(handle);
}

void AttachWaitHolderForReceive(nn::os::MultiWaitHolderType* pHolder, HipcServerSessionHandle session) NN_NOEXCEPT
{
    os::InitializeMultiWaitHolder(pHolder, reinterpret_cast<nn::os::NativeHandle&>(session));
}

namespace {

    Result ReceiveImpl(HipcServerSessionHandle session, void* messageBuffer, size_t messageBufferSize) NN_NOEXCEPT
    {
        if (GetMessageBufferOnTlsImpl() == messageBuffer)
        {
            NN_SDK_ASSERT(messageBufferSize == MessageBufferSizeOnTls, "[SF-Internal]");
            int32_t dummy;
            return svc::ReplyAndReceive(&dummy, &session, 1, svc::INVALID_HANDLE_VALUE, -1);
        }
        else
        {
            int32_t dummy;
            return svc::ReplyAndReceiveWithUserBuffer(&dummy, reinterpret_cast<uintptr_t>(messageBuffer), messageBufferSize, &session, 1, svc::INVALID_HANDLE_VALUE, -1);
        }
    }

}


// ビルド互換性維持用
Result Receive(bool* pClosed, HipcServerSessionHandle session, void* messageBuffer, size_t messageBufferSize) NN_NOEXCEPT
{
    NN_RESULT_TRY(ReceiveImpl(session, messageBuffer, messageBufferSize))
        NN_RESULT_CATCH(svc::ResultSessionClosed)
        {
            *pClosed = true;
            NN_RESULT_SUCCESS;
        }
    NN_RESULT_END_TRY
    *pClosed = false;
    NN_RESULT_SUCCESS;
}

Result Receive(ReceiveResult* pReceiveResult, HipcServerSessionHandle session, void* messageBuffer, size_t messageBufferSize) NN_NOEXCEPT
{
    NN_RESULT_TRY(ReceiveImpl(session, messageBuffer, messageBufferSize))
        NN_RESULT_CATCH(svc::ResultSessionClosed)
        {
            *pReceiveResult = ReceiveResult::Close;
            NN_RESULT_SUCCESS;
        }
        NN_RESULT_CATCH(svc::ResultReceptionSpecificationListBroken)
        {
            *pReceiveResult = ReceiveResult::Retry;
            NN_RESULT_SUCCESS;
        }
    NN_RESULT_END_TRY
    *pReceiveResult = ReceiveResult::Process;
    NN_RESULT_SUCCESS;
}

namespace {

    inline Result HandleReplyResult(Result result) NN_NOEXCEPT
    {
        NN_RESULT_TRY(result)
            NN_RESULT_CATCH(svc::ResultTimeout)
            {
                // タイムアウトが通常時の動作
                NN_RESULT_SUCCESS;
            }
            NN_RESULT_CATCH(svc::ResultSessionClosed)
            {
                // クライアント側が破棄されてもかまわない
                // なお、この際、ハンドルは一通り巻き戻される
                NN_RESULT_SUCCESS;
            }
        NN_RESULT_END_TRY
        // success はありえない
        NN_ABORT("[SF-Internal]");
    }

}

Result Reply(HipcServerSessionHandle session, void* messageBuffer, size_t messageBufferSize) NN_NOEXCEPT
{
    if (GetMessageBufferOnTlsImpl() == messageBuffer)
    {
        NN_SDK_ASSERT(messageBufferSize == MessageBufferSizeOnTls, "[SF-Internal]");
        int32_t dummy;
        return HandleReplyResult(svc::ReplyAndReceive(&dummy, &session, 0, session, 0));
    }
    else
    {
        int32_t dummy;
        return HandleReplyResult(svc::ReplyAndReceiveWithUserBuffer(&dummy, reinterpret_cast<uintptr_t>(messageBuffer), messageBufferSize, &session, 0, session, 0));
    }
}

void AttachWaitHolderForReply(nn::os::MultiWaitHolderType* pHolder, HipcClientRequestHandle request) NN_NOEXCEPT
{
    os::InitializeMultiWaitHolder(pHolder, reinterpret_cast<nn::os::NativeHandle&>(request));
}

}}}
