﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "sdmmc_DeviceDetector.h"
#include "sdmmc_Log.h"
#include <nn/nn_SystemThreadDefinition.h>
#include <nn/nn_Abort.h>

namespace nn { namespace sdmmc {
namespace detail {

bool DeviceDetector::IsCurrentInserted() NN_NOEXCEPT
{
    nn::gpio::GpioValue currentGpioValue = nn::gpio::GetValue(&m_GpioPadSession);
    if (currentGpioValue == m_GpioValueInserted)
    {
        return true;
    }
    else
    {
        return false;
    }
}

void DeviceDetector::HandleDeviceStatus(bool isLastInserted, bool isCurrentInserted) NN_NOEXCEPT
{
    if ((!isLastInserted) && (!isCurrentInserted))
    {
        // 抜 -> ... -> 抜 の場合は何もしない
        NN_DETAIL_SDMMC_DETECTOR_LOG("Insert (LOST)\n");
        NN_DETAIL_SDMMC_DETECTOR_LOG("Remove (ignore)\n");
    }
    else if ((!isLastInserted) && isCurrentInserted)
    {
        // 抜 -> ... -> 挿 の場合
        NN_DETAIL_SDMMC_DETECTOR_LOG("Insert\n");
        if (m_InsertedCallback != nullptr)
        {
            m_InsertedCallback(m_pInsertedCallbackParameter);
        }
    }
    else if (isLastInserted && (!isCurrentInserted))
    {
        // 挿 -> ... -> 抜 の場合
        NN_DETAIL_SDMMC_DETECTOR_LOG("Remove\n");
        if (m_RemovedCallback != nullptr)
        {
            m_RemovedCallback(m_pRemovedCallbackParameter);
        }
    }
    else    // (isLastInserted && isCurrentInserted)
    {
        // 挿 -> ... -> 挿 の場合
        NN_DETAIL_SDMMC_DETECTOR_LOG("Remove (LOST)\n");
        if (m_RemovedCallback != nullptr)
        {
            m_RemovedCallback(m_pRemovedCallbackParameter);
        }
        NN_DETAIL_SDMMC_DETECTOR_LOG("Insert\n");
        if (m_InsertedCallback != nullptr)
        {
            m_InsertedCallback(m_pInsertedCallbackParameter);
        }
    }
}

void DeviceDetector::DetectorThread() NN_NOEXCEPT
{
    // gpioEvent が挿抜でシグナルされるよう設定
    nn::gpio::Initialize();
    nn::gpio::OpenSession(&m_GpioPadSession, m_GpioPadName);
    nn::gpio::SetDirection(&m_GpioPadSession, nn::gpio::Direction_Input);
    nn::gpio::SetDebounceTime(&m_GpioPadSession, m_GpioDebounceMilliSeconds);
    nn::gpio::SetDebounceEnabled(&m_GpioPadSession, true);
    nn::gpio::SetInterruptMode(&m_GpioPadSession, nn::gpio::InterruptMode_AnyEdge);
    nn::os::SystemEventType gpioEvent;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::gpio::BindInterrupt(&gpioEvent, &m_GpioPadSession));

    // waiter にスレッド終了イベントと gpioEvent を登録
    nn::os::MultiWaitType waiter;
    nn::os::MultiWaitHolderType detectorThreadEndEventHolder;
    nn::os::MultiWaitHolderType requestSleepAwakeEventHolder;
    nn::os::MultiWaitHolderType gpioEventHolder;
    nn::os::InitializeMultiWait(&waiter);
    nn::os::InitializeMultiWaitHolder(&detectorThreadEndEventHolder, &m_DetectorThreadEndEvent);
    nn::os::LinkMultiWaitHolder(&waiter, &detectorThreadEndEventHolder);
    nn::os::InitializeMultiWaitHolder(&requestSleepAwakeEventHolder, &m_RequestSleepAwakeEvent);
    nn::os::LinkMultiWaitHolder(&waiter, &requestSleepAwakeEventHolder);
    nn::os::InitializeMultiWaitHolder(&gpioEventHolder, &gpioEvent);
    nn::os::LinkMultiWaitHolder(&waiter, &gpioEventHolder);

    // 最初の状態を確定
    nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(m_GpioDebounceMilliSeconds));    // 最初の状態が定まる時間待つ
    bool isCurrentInserted = IsCurrentInserted();
    NN_DETAIL_SDMMC_DETECTOR_LOG("%s\n", (isCurrentInserted ? "Inserted" : "Removed"));
    m_IsLastInserted = isCurrentInserted;

    // 以降は状態取得可能
    m_State = State_Awake;
    nn::os::SignalEvent(&m_ReadyDeviceStatusEvent);

    // 割り込み許可
    nn::gpio::SetInterruptEnable(&m_GpioPadSession, true);

    while (true)
    {
        nn::os::MultiWaitHolderType* signaledHolder = nn::os::WaitAny(&waiter);

        bool isChanged = false;
        if (signaledHolder == &detectorThreadEndEventHolder)
        {
            nn::os::ClearEvent(&m_DetectorThreadEndEvent);
            m_State = State_Finalized;
            break;
        }
        else if (signaledHolder == &requestSleepAwakeEventHolder)
        {
            nn::os::ClearEvent(&m_RequestSleepAwakeEvent);
            m_State = State_Sleep;
            nn::os::SignalEvent(&m_AcceptedSleepAwakeEvent);
            NN_DETAIL_SDMMC_DETECTOR_LOG("Put DeviceDetector to sleep\n");

            nn::os::UnlinkMultiWaitHolder(&gpioEventHolder);

            signaledHolder = nn::os::WaitAny(&waiter);

            nn::os::LinkMultiWaitHolder(&waiter, &gpioEventHolder);

            nn::os::ClearEvent(&m_RequestSleepAwakeEvent);
            m_State = State_Awake;
            nn::os::SignalEvent(&m_AcceptedSleepAwakeEvent);
            NN_DETAIL_SDMMC_DETECTOR_LOG("Awaken DeviceDetector with m_IsForcedDetectionEvent = %d\n", m_IsForcedDetectionEvent);

            if (signaledHolder == &detectorThreadEndEventHolder)
            {
                nn::os::ClearEvent(&m_DetectorThreadEndEvent);
                m_State = State_Finalized;
                break;
            }
            else    // (signaledHolder == &requestSleepAwakeEventHolder)
            {
                if (m_IsForcedDetectionEvent
                    || nn::gpio::IsWakeEventActive(m_GpioPadName)
                    || nn::os::TryWaitSystemEvent(&gpioEvent)
                    || (m_IsLastInserted != IsCurrentInserted()))
                {
                    isChanged = true;
                }
            }
        }
        else    // (signaledHolder == &gpioEventHolder)
        {
            isChanged = true;
        }

        if (isChanged)
        {
            if (m_DeviceDetectionEventCallback != nullptr)
            {
                // 上位レイヤに挿抜イベントがあったことを通知
                m_DeviceDetectionEventCallback(m_pDeviceDetectionEventCallbackParameter);
            }

            // イベントのクリア
            nn::os::ClearSystemEvent(&gpioEvent);

            // 割り込みステータスをクリア
            nn::gpio::ClearInterruptStatus(&m_GpioPadSession);

            // 割り込み許可状態に戻す
            nn::gpio::SetInterruptEnable(&m_GpioPadSession, true);

            isCurrentInserted = IsCurrentInserted();
            HandleDeviceStatus(m_IsLastInserted, isCurrentInserted);
            m_IsLastInserted = isCurrentInserted;
        }
    }

    // 割り込み禁止
    nn::gpio::SetInterruptEnable(&m_GpioPadSession, false);

    // waiter の後片付け
    nn::os::UnlinkMultiWaitHolder(&gpioEventHolder);
    nn::os::FinalizeMultiWaitHolder(&gpioEventHolder);
    nn::os::UnlinkMultiWaitHolder(&requestSleepAwakeEventHolder);
    nn::os::FinalizeMultiWaitHolder(&requestSleepAwakeEventHolder);
    nn::os::UnlinkMultiWaitHolder(&detectorThreadEndEventHolder);
    nn::os::FinalizeMultiWaitHolder(&detectorThreadEndEventHolder);
    nn::os::FinalizeMultiWait(&waiter);

    // gpio の後片付け
    nn::gpio::UnbindInterrupt(&m_GpioPadSession);
    nn::gpio::CloseSession(&m_GpioPadSession);
    nn::gpio::Finalize();
}

void DeviceDetector::Initialize(CallbackInfos* pCallbackInfos) NN_NOEXCEPT
{
    // Initialize() と Finalize() は同じスレッドから重複せずにコールされる想定
    NN_ABORT_UNLESS(m_State == State_Finalized);
    m_State = State_Initializing;

    m_InsertedCallback = pCallbackInfos->insertedCallback;
    m_pInsertedCallbackParameter = pCallbackInfos->pInsertedCallbackParameter;
    m_RemovedCallback = pCallbackInfos->removedCallback;
    m_pRemovedCallbackParameter = pCallbackInfos->pRemovedCallbackParameter;

    // 最初の挿抜状態が確定するまでロック
    nn::os::InitializeEvent(&m_ReadyDeviceStatusEvent, false, nn::os::EventClearMode_ManualClear);

    // スリープ制御イベント
    nn::os::InitializeEvent(&m_RequestSleepAwakeEvent, false, nn::os::EventClearMode_ManualClear);
    nn::os::InitializeEvent(&m_AcceptedSleepAwakeEvent, false, nn::os::EventClearMode_ManualClear);

    // 挿抜監視スレッドを起動
    nn::os::InitializeEvent(&m_DetectorThreadEndEvent, false, nn::os::EventClearMode_ManualClear);
    nn::os::CreateThread(&m_DetectorThread, DetectorThreadEntry, this,
        m_DetectorThreadStack, sizeof(m_DetectorThreadStack), NN_SYSTEM_THREAD_PRIORITY(sdmmc, DeviceDetector));
    nn::os::SetThreadNamePointer(&m_DetectorThread, NN_SYSTEM_THREAD_NAME(sdmmc, DeviceDetector));
    nn::os::StartThread(&m_DetectorThread);
}

void DeviceDetector::Finalize() NN_NOEXCEPT
{
    // Initialize() と Finalize() は同じスレッドから重複せずにコールされる想定
    NN_ABORT_UNLESS(m_State != State_Finalized);

    // 挿抜監視スレッドを停止
    nn::os::SignalEvent(&m_DetectorThreadEndEvent);
    nn::os::WaitThread(&m_DetectorThread);
    nn::os::DestroyThread(&m_DetectorThread);
    nn::os::FinalizeEvent(&m_DetectorThreadEndEvent);

    nn::os::FinalizeEvent(&m_AcceptedSleepAwakeEvent);
    nn::os::FinalizeEvent(&m_RequestSleepAwakeEvent);

    nn::os::FinalizeEvent(&m_ReadyDeviceStatusEvent);
}

void DeviceDetector::PutToSleep() NN_NOEXCEPT
{
    // PutToSleep() と Awaken() は同じスレッドから重複せずにコールされる想定
    nn::os::SignalEvent(&m_RequestSleepAwakeEvent);
    nn::os::WaitEvent(&m_AcceptedSleepAwakeEvent);
    nn::os::ClearEvent(&m_AcceptedSleepAwakeEvent);
}

void DeviceDetector::Awaken(bool isForcedDetectionEvent) NN_NOEXCEPT
{
    // PutToSleep() と Awaken() は同じスレッドから重複せずにコールされる想定
    m_IsForcedDetectionEvent = isForcedDetectionEvent;
    nn::os::SignalEvent(&m_RequestSleepAwakeEvent);
    nn::os::WaitEvent(&m_AcceptedSleepAwakeEvent);
    nn::os::ClearEvent(&m_AcceptedSleepAwakeEvent);
}

bool DeviceDetector::IsInserted() NN_NOEXCEPT
{
    bool isInserted = false;

    switch (m_State)
    {
    case State_Initializing:
        // 最初の gpio の設定が終わるまで待つ
        NN_DETAIL_SDMMC_DETECTOR_LOG("wait to become ready...\n");
        nn::os::WaitEvent(&m_ReadyDeviceStatusEvent);
        NN_FALL_THROUGH;                        // 待った後は Awake と同じ
    case State_Awake:
        isInserted = IsCurrentInserted();       // 最新の状態取得
        break;
    case State_Sleep:
        NN_FALL_THROUGH;                        // Sleep 中は Finalize() 後と同じ
    case State_Finalized:
        isInserted = m_IsLastInserted;          // 最後に認識した状態取得
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
    NN_DETAIL_SDMMC_DETECTOR_LOG("return %s\n", (isInserted ? "Inserted" : "Removed"));
    return isInserted;
}

void DeviceDetector::RegisterDetectionEventCallback(DeviceDetectionEventCallback callback, void* pParameter) NN_NOEXCEPT
{
    m_pDeviceDetectionEventCallbackParameter = pParameter;
    m_DeviceDetectionEventCallback = callback;
}

void DeviceDetector::UnregisterDetectionEventCallback() NN_NOEXCEPT
{
    m_DeviceDetectionEventCallback = nullptr;
}

} // namespace detail {
}} // namespace nn { namespace sdmmc {
