﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/dd.h>

namespace{

const int NumberOfGpioPerOneControllerBitWidth = 5;
const int NumberOfControllerBitWidth = 3;

const int NumberOfPortPerOneControllerBitWidth = 2;
const int NumberOfPortPerOneController = (1u << NumberOfPortPerOneControllerBitWidth);

const int NumberOfbitPerOnePortBitWidth = 3;
const int NumberOfbitPerOnePort = (1u << NumberOfbitPerOnePortBitWidth);
}

namespace nn{
namespace pinmux{
namespace driver{
namespace detail{

/**
* @brief Tegra 向けの GPIO パッド番号計算用マクロ
*
* @details Tegra 向けの GPIO パッド番号計算用マクロです。
*/
#define NN_GPIO_GET_PAD_NUM(port, bit) ((port) * 8 + (bit))

/**
* @brief Tegra 向けの GPIO ポート一覧
*
* @details Tegra 向けの GPIO ポート一覧です。GPIO 番号計算用マクロを使用し、GPIO パッド番号を算出するために使用します。
*/
enum GpioPadPort
{
    GpioPadPort_PA = 0,
    GpioPadPort_PB,
    GpioPadPort_PC,
    GpioPadPort_PD,

    GpioPadPort_PE = 4,
    GpioPadPort_PF,
    GpioPadPort_PG,
    GpioPadPort_PH,

    GpioPadPort_PI = 8,
    GpioPadPort_PJ,
    GpioPadPort_PK,
    GpioPadPort_PL,

    GpioPadPort_PM = 12,
    GpioPadPort_PN,
    GpioPadPort_PO,
    GpioPadPort_PP,

    GpioPadPort_PQ = 16,
    GpioPadPort_PR,
    GpioPadPort_PS,
    GpioPadPort_PT,

    GpioPadPort_PU = 20,
    GpioPadPort_PV,
    GpioPadPort_PW,
    GpioPadPort_PX,

    GpioPadPort_PY = 24,
    GpioPadPort_PZ,
    GpioPadPort_PAA,
    GpioPadPort_PBB,

    GpioPadPort_PCC = 28,
    GpioPadPort_PDD,
    GpioPadPort_PEE,
    GpioPadPort_PFF,
};


// IO マッピング関連
// GPIO
const nn::dd::PhysicalAddress   GpioPhysicalAddress   = 0x06000d000ull;
const size_t                    GpioAddressSize       = 0x1000;

// GPIO コントローラー番号の列挙体
enum GpioController
{
    GpioController_GPIO1 = 0,
    GpioController_GPIO2,
    GpioController_GPIO3,
    GpioController_GPIO4,
    GpioController_GPIO5,
    GpioController_GPIO6,
    GpioController_GPIO7,
    GpioController_GPIO8,
    GpioController_NumOfGpioController,
};

// GPIO コントローラー(レジスタ)の列挙体
enum GpioRegisterType
{
    GpioRegisterType_GPIO_CNF = 0,
    GpioRegisterType_GPIO_OE,
    GpioRegisterType_GPIO_OUT,
    GpioRegisterType_GPIO_IN,
    GpioRegisterType_GPIO_INT_STA,
    GpioRegisterType_GPIO_INT_ENB,
    GpioRegisterType_GPIO_INT_LVL,
    GpioRegisterType_GPIO_INT_CLR,
    GpioRegisterType_GPIO_DB_CTRL,
    GpioRegisterType_GPIO_DB_CNT,
};

// GPIO_CNF レジスタにセットする値
enum GpioCnf
{
    GpioCnf_Sfio = 0,
    GpioCnf_Gpio = 1,
};

// GPIO_OUT レジスタにセットする値
enum GpioOut
{
    GpioOut_Low = 0,
    GpioOut_High = 1,
};

// GPIO_OE レジスタにセットする値
enum GpioOe
{
    GpioOe_Input = 0,
    GpioOe_Output = 1,
};

// Jetson の InterruptMode 設定用レジスタ(GPIO_INT_LVL)に設定する Bit 列(BIT_0 を起点としているので、シフトして使う)。
// (Tegra_K1_TRM_DP06905001v02p.pdf p.283)
enum InternalInterruptMode
{
    InternalInterruptMode_LowLevel    = (0 << 16) | (0 << 8) | (0 << 0),
    InternalInterruptMode_HighLevel   = (0 << 16) | (0 << 8) | (1 << 0),
    InternalInterruptMode_RisingEdge  = (0 << 16) | (1 << 8) | (1 << 0),
    InternalInterruptMode_FallingEdge = (0 << 16) | (1 << 8) | (0 << 0),
    InternalInterruptMode_AnyEdge     = (1 << 16) | (1 << 8) | (0 << 0),
    InternalInterruptMode_BitMask     = (1 << 16) | (1 << 8) | (1 << 0),
};

inline int ConvertInternalGpioPadNumberToControllerNumber(int padNumber)
{
    return (padNumber >> NumberOfGpioPerOneControllerBitWidth);
}

inline int ConvertInternalGpioPadNumberToPortNumber(int padNumber)
{
    return (padNumber >> NumberOfControllerBitWidth);
}

inline int ConvertInternalGpioPadNumberToBitPosition(int padNumber)
{
    return padNumber & (NumberOfbitPerOnePort - 1u);
}

inline int ConvertPortNumberToAddressOffset(int number)
{
    return number & (NumberOfPortPerOneController - 1u);
}

// Per-Pin Mask Write をするためのアドレスとビットのオフセット値
const uintptr_t MaskWriteAddressOffset = 0x0080;
const int       MaskWriteBitOffset = 8;

// 各GPIO レジスタのアクセスアドレスを計算する関数 (Tegra_K1_TRM_DP06905001v02p.pdf p.278)
inline nn::Bit32* GetGpioRegAccessAddress(GpioRegisterType regType, int padNumber, uintptr_t gpioVirtualAddress)
{
    int controllerNum = ConvertInternalGpioPadNumberToControllerNumber(padNumber);
    int portNum  = ConvertInternalGpioPadNumberToPortNumber(padNumber);
    int offset   = ConvertPortNumberToAddressOffset(portNum);

    // Debaunce 系のレジスタだけ特定のレジスタの Upper Offset を使っている
    switch(regType)
    {
    // このレジスタは GpioRegisterType_GPIO_IN の Upper Offset を使っているので、MaskedWrite を行うので、GPIO_IN のアドレスをそのまま返す
    case GpioRegisterType_GPIO_DB_CTRL:
        return reinterpret_cast<nn::Bit32*>(gpioVirtualAddress + (controllerNum) * 0x0100 + (GpioRegisterType_GPIO_IN) * 0x0010 + (offset) * 0x004);

    // このレジスタは GpioRegisterType_GPIO_INT_CLR の Upper Offset を使っているので、GPIO_INT_CLR のアドレス + Offset を返す
    case GpioRegisterType_GPIO_DB_CNT:
        return reinterpret_cast<nn::Bit32*>(gpioVirtualAddress + (controllerNum) * 0x0100 + (GpioRegisterType_GPIO_INT_CLR) * 0x0010 + (offset) * 0x004
                                                                                                                                + MaskWriteAddressOffset);
    default:
        return reinterpret_cast<nn::Bit32*>(gpioVirtualAddress + (controllerNum) * 0x0100 + (regType) * 0x0010 + (offset) * 0x004);
    }
}


// Tegra の GPIO レジスタに備わっている MaskedWrite 機能を使って特定ピン(ビット)だけ設定する関数
// Read-Modify-Write していないので、必ず MaskedWrite 機能を持ったレジスタに使用すること。
inline void SetBitForTegraMaskedWrite(int value, int pos, nn::Bit32* gpioVirtualAddress)
{
    uintptr_t address = reinterpret_cast<uintptr_t>(gpioVirtualAddress) + MaskWriteAddressOffset;

    *reinterpret_cast<volatile nn::Bit32*>(address) = (1 << ( MaskWriteBitOffset + pos)) | (value << pos);
}

// Tegra の GPIO レジスタに備わっている MaskedWrite 機能を使って書き込む関数
inline void WriteTegraMaskedWrite32(Bit32 value, Bit32 mask, nn::Bit32* gpioAccessAddress)
{
    uintptr_t address = reinterpret_cast<uintptr_t>(gpioAccessAddress) + MaskWriteAddressOffset;

    *reinterpret_cast<volatile nn::Bit32*>(address) = (mask << MaskWriteBitOffset) | (value);
}

} // detail
} // driver
} // pinmux
} // nn
