﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <type_traits>
#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>

namespace nn { namespace osdbg {
namespace detail {

//-----------------------------------------------------------------------------
//  ILP32/LP64 ABI 環境でのサイズに合致する型
//
typedef uint32_t    ptr32_t;
typedef uint64_t    ptr64_t;

typedef uint32_t    size32_t;
typedef uint64_t    size64_t;

typedef uint32_t    uintptr32_t;
typedef uint64_t    uintptr64_t;

//-----------------------------------------------------------------------------
//  構造体のサイズを ILP32/LP64 それぞれのビルド条件に合わせて算出するマクロ
//  baseSize は ILP32/LP64 で変化しない部分のベースサイズ、
//  pointerNum はポインタ変数の数を指定する。
//
#define NN_OSDBG_CALC_OBJECT_SIZE_ILP32(baseSize, pointerNum)    \
            ((baseSize) + (pointerNum) * sizeof(::nn::osdbg::detail::ptr32_t))
#define NN_OSDBG_CALC_OBJECT_SIZE_LP64(baseSize, pointerNum)    \
            ((baseSize) + (pointerNum) * sizeof(::nn::osdbg::detail::ptr64_t))


//-----------------------------------------------------------------------------
//  ILP32/LP64 環境で指定サイズとアライメントを持つ領域を確保する構造体
//
//  - baseSize  は ILP32/LP64 で変化しない部分のベースサイズ
//  - PtrNum    はポインタ変数の数
//  - Alignment は内部オブジェクトのアライメントのサイズ
//
template<size_t BaseSize, int PtrNum, size_t Alignment>
struct AlignedStorageIlp32
{
    typename std::aligned_storage<NN_OSDBG_CALC_OBJECT_SIZE_ILP32(BaseSize, PtrNum), Alignment>::type   storage;
};

template<size_t BaseSize, int PtrNum, size_t Alignment>
struct AlignedStorageLp64
{
    typename std::aligned_storage<NN_OSDBG_CALC_OBJECT_SIZE_LP64(BaseSize, PtrNum), Alignment>::type   storage;
};


//-----------------------------------------------------------------------------
//  type1 と type2 の型のサイズが合致することを静的に判定
#define NN_OSDBG_STATIC_ASSERT_SIZEOF(type1, type2)     \
        static_assert(sizeof(type1) == sizeof(type2),   \
            "サイズ違いを検出しました")

//-----------------------------------------------------------------------------
//  type1 と type2 の型のアライメントが合致することを静的に判定
#define NN_OSDBG_STATIC_ASSERT_ALIGNOF(type1, type2)            \
        static_assert(NN_ALIGNOF(type1) == NN_ALIGNOF(type2),   \
            "アライメントサイズ違いを検出しました")

//-----------------------------------------------------------------------------
//  type1 と type2 の型のメンバ member の offsetof が合致することを静的に判定
#define NN_OSDBG_STATIC_ASSERT_OFFSETOF(type1, type2, member)               \
        static_assert(offsetof(type1, member) == offsetof(type2, member),   \
            "オフセット違いを検出しました")

//-----------------------------------------------------------------------------

}   // namespace detail
}}  // namespace nn::osdbg

